/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       SOCKLIB.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/6/26
**
** DESCRIPTION: Contains the implementation of the Socket
**              Library.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.2  $
** WORKFILE:    $Workfile:   SOCKLIB.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/SOCK/VCS/SOCKLIB.C_V  $
**              
**                 Rev 1.2   01 Feb 1991 11:24:44   etstjan
**              No explicit note
**              
**                 Rev 1.1   21 Nov 1990 14:32:28   etstjan
**              No explicit note
**              
**                 Rev 1.0   21 Nov 1990 12:27:42   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/SOCK/VCS/SOCKLIB.C_V   1.2   01 Feb 1991 11:24:44   etstjan  $";
#endif

#include      <stdlib.h>
#include      <string.h>
#include      <stdio.h>
#include      <power3.h>
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "iplib.h"                /* include ip library */
#include      "iplayer.h"              /* iplayer header file */
#include      "socklib.h"              /* this file's header file */
 
/* global variables */
PROTSTRUCT    *RawProt;                /* 'protocol' for raw socket  */
PROTSTRUCT    *UDPProt;                /* UDP protocol switch */
PROTSTRUCT    *Protocs;                /* linked list of protocols */

/* prototypes */
int    RawService(USERREQ ThisReq, SOCKET *ThisSocket,
                  DATALINK *ThisLink, SO_ADDRESS *FromAddress,
                  SO_ADDRESS *ToAddress);
int    UDPService(USERREQ ThisReq, SOCKET *ThisSocket,
                  DATALINK *ThisLink, SO_ADDRESS *FromAddress,
                  SO_ADDRESS *ToAddress);
void   AppendSocket(SOCKET *ThisSock, PROTSTRUCT *ThisProt);
void   RemoveSocket(SOCKET *ThisSocket, PROTSTRUCT *ThisProt);
void   InsertSocket(SOCKET *DestSocket, SOCKET *SourceSocket,
                  PROTSTRUCT *ThisProt);
void   MoveSocket(SOCKET *DestSocket, SOCKET *SourceSocket,
                  PROTSTRUCT *ThisProt);
int    SoBindLoc(SOCKET *ThisSocket, SO_ADDRESS *ThisAddress, 
                 PROTSTRUCT *ThisProt);
int    SoRemoteConnect(SOCKET *ThisSocket, SO_ADDRESS *ThisAddress,
                       PROTSTRUCT *ThisProt);
void   SocketSort(SOCKET *ThisSocket, PROTSTRUCT *ThisProt);
int    ProtService(USERREQ ThisReq, SOCKET *ThisSocket,
                   DATALINK *ThisLink, SO_ADDRESS *FromAddress,
                   SO_ADDRESS *ToAddress, PROTSTRUCT *ThisProt);

/**************************************************************
** NAME:        CreateSocket
** SYNOPSIS:    SOCKET *CreateSocket(int Type,
**                                   int Domain,
**                                   int Protocol);
** DESCRIPTION: Creates a new socket. At this moment, the
**              following constraints apply:
**              Type should be SOCK_DGRAM or SOCK_RAW;
**              Domain should be PF_INET;
**              Protocol should be 0 (autodetect) or UDP.
** RETURNS:     NULL  --> Failed to create socket:
**                        - parameter error;
**                        - no buffer space.     
**              else  --> Pointer to the newly created
**                        socket.
**************************************************************/
SOCKET *CreateSocket(int Type, int Domain, int Protocol)
{
  SOCKET    *TmpSock;

  Domain = Domain;
  Protocol = Protocol;

  if ((TmpSock = IPBufGet(sizeof(SOCKET))) == NULL) {
    return NULL;
    }
  memset(TmpSock, 0, sizeof(SOCKET));  /* clear socket */

  switch (Type) {
    case SOCK_DGRAM :
      TmpSock->So_Prot = UDPProt;
      TmpSock->So_Type = SOCK_DGRAM;
      break;
    case SOCK_RAW :
      TmpSock->So_Prot = RawProt;
      TmpSock->So_Type = SOCK_RAW;
      break;
    }
  if ((*(TmpSock->So_Prot->UserReq))
       (PRU_ATTACH, TmpSock, NULL, NULL, NULL) != NOERR) {
    IPBufFree(TmpSock);
    TmpSock = NULL;
    }
  return TmpSock;
}


/**************************************************************
** NAME:        FreeSoPack
** SYNOPSIS:    void FreeSoPack(SO_PACKLIST *ThisSoPack);
**
** DESCRIPTION: Frees everything associated with the
**              SO_PACKLIST ThisSoPack.
** RETURNS:  
**************************************************************/
void FreeSoPack(SO_PACKLIST *ThisSoPack)
{
  IPBufFree(ThisSoPack->Source);
  DiscardAll(ThisSoPack->Packet);
  IPBufFree(ThisSoPack);
}

/**************************************************************
** NAME:        _RelSocket
** SYNOPSIS:    int _RelSocket(SOCKET *ThisSocket);
**           
** DESCRIPTION: Releases ThisSocket. The protocol is requested
**              to detach the socket. If no error occures in
**              detaching the socket, the memory space for
**              the socket is freed as well.
** RETURNS:     NOERR -->   no error
**              else         error code
**************************************************************/
int _RelSocket(SOCKET *ThisSocket)
{
  int          RetCode;

  (*(ThisSocket->So_Prot->UserReq))(PRU_SHUTDOWN, ThisSocket, NULL, NULL, NULL);

  if (ThisSocket->So_Options != NULL) {
    ThisSocket->So_Options->SaveData = 0;
    DiscardAll(ThisSocket->So_Options);
    }

  RetCode = (*(ThisSocket->So_Prot->UserReq))
             (PRU_DETACH, ThisSocket, NULL, NULL, NULL);
  if (RetCode == NOERR) {
    RetCode = IPBufFree(ThisSocket);
    }
  return RetCode;
}

/**************************************************************
** NAME:        InitSockets
** SYNOPSIS:    int InitSockets(void);
**  
** DESCRIPTION: Initializes the Socket Layer.
**              The following protocols are initialized:
**              - RAW protocol (direct IP service);
**              - UDP protocol.
** RETURNS:     NOERR -->  no error
**              else        error code
**************************************************************/
int InitSockets(void)
{
  if (((RawProt = IPBufGet(sizeof(PROTSTRUCT))) == NULL) ||
      ((UDPProt = IPBufGet(sizeof(PROTSTRUCT))) == NULL)) {
    if (RawProt != NULL) {
      IPBufFree(RawProt);
      }
    return NOSPACE;
    }
  /* clear protocol switches */
  memset(RawProt, 0, sizeof(PROTSTRUCT));
  memset(UDPProt, 0, sizeof(PROTSTRUCT));

  /* initialize 'RAW' protocol */
  strncpy(RawProt->PrName, "Raw", 15);
  RawProt->PrType = IP;                /* dummy for 'raw' protocol */
  RawProt->PrFlags |= PR_ATOMIC;
  RawProt->UserReq = RawService;       /* set raw service routine */
  RawProt->IPRec = IPRawRec;           /* Raw IP receive function */

  /* initialize 'UDP' protocol */
  strncpy(UDPProt->PrName, "UDP", 15);
  UDPProt->PrType = UDP;               /* type = UDP protocol */
  UDPProt->PrFlags |= PR_ATOMIC;
  UDPProt->UserReq = UDPService;       /* set raw service routine */
  UDPProt->IPRec = UDPRec;             /* UDP IP receive function */

  Protocs = RawProt;                   /* set protocol linked list */
  RawProt->PrNext = UDPProt;
  UDPProt->PrNext = NULL;

  return NOERR;
}


/**************************************************************
** NAME:        AppendSocket
** SYNOPSIS:    void AppendSocket(SOCKET *ThisSock,
**                                PROTSTRUCT *ThisProt)
**   
** DESCRIPTION: Appends ThisSock to the tail of the
**              socket queue of ThisProt.     
** RETURNS: 
**************************************************************/
void AppendSocket(SOCKET *ThisSock, PROTSTRUCT *ThisProt)
{
  ThisSock->So_RMaxLen = DEFRECMAX;         /* set receive queue lenght */
  if (ThisProt->PrSocket == NULL) {         /* and append socket at tail */
    ThisProt->PrSocket = ThisSock;
    }
  else {
    ThisProt->LastSock->So_Next = ThisSock; /* maintain double linked list */
    ThisSock->So_Prev = ThisProt->LastSock;
    }
  ThisProt->LastSock = ThisSock;
  ThisSock->So_Next = NULL;
}

/**************************************************************
** NAME:        RemoveSocket
** SYNOPSIS:    void RemoveSocket(SOCKET *ThisSocket,
**                                PROTSTRUCT *ThisProt);
**  
** DESCRIPTION: Removes ThisSocket from the socket list
**              of ThisProt.
** RETURNS:  
**************************************************************/
void RemoveSocket(SOCKET *ThisSocket, PROTSTRUCT *ThisProt)
{
  if (ThisSocket->So_Prev == NULL) {
    ThisProt->PrSocket = ThisSocket->So_Next;
    }
  else {
    ThisSocket->So_Prev->So_Next = ThisSocket->So_Next;
    }
  if (ThisSocket->So_Next == NULL) {
    ThisProt->LastSock = ThisSocket->So_Prev;
    }
  else {
    ThisSocket->So_Next->So_Prev = ThisSocket->So_Prev;
    }
}

/**************************************************************
** NAME:        InsertSocket
** SYNOPSIS:    void InsertSocket(SOCKET *DestSocket,
**                                SOCKET *SourceSocket,
**                                PROTSTRUCT *ThisProt)
**              
** DESCRIPTION: Inserts SourceSocket in front of DestSocket
**              in the linked Socket list. If necessary, the
**              protocol structure ThisProt is adjusted.
** RETURNS:     
**************************************************************/
void InsertSocket(SOCKET *DestSocket, SOCKET *SourceSocket,
                  PROTSTRUCT *ThisProt)
{
  if (DestSocket->So_Prev == NULL) {
    ThisProt->PrSocket = SourceSocket;
    }
  else {
    DestSocket->So_Prev->So_Next = SourceSocket;
    }
  SourceSocket->So_Prev = DestSocket->So_Prev;
  SourceSocket->So_Next = DestSocket;
  DestSocket->So_Prev = SourceSocket;

}

/**************************************************************
** NAME:        MoveSocket
** SYNOPSIS:    void MoveSocket(SOCKET *DestSocket,
**                              SOCKET *SourceSocket,
**                              PROTSTRUCT *ThisProt);
**
** DESCRIPTION: Places SourceSocket ahead of DestSocket.
**              The pointers in the protocol structure
**              ThisProt are adjusted if necessary.
** RETURNS:
**************************************************************/
void MoveSocket(SOCKET *DestSocket, SOCKET *SourceSocket,
                PROTSTRUCT *ThisProt)
{
  if (DestSocket != SourceSocket) {    /* if equal then ready */
    /* first, remove SourceSocket */
    RemoveSocket(SourceSocket, ThisProt);

    /* then insert socket in front of DestSocket */
    InsertSocket(DestSocket, SourceSocket, ThisProt);
  }
}


/**************************************************************
** NAME:        SocketSort
** SYNOPSIS:    void SocketSort(SOCKET *ThisSocket,
**                              PROTSTRUCT *ThisProt);
**     
** DESCRIPTION: Places ThisSocket in the socket list of
**              ThisProt. The socket list is sorted on
**              local port and local address respectively.
**              Before this procedure is called, the socket
**              should be placed in the part of the socket
**              list where the sockets with unbounds local
**              reside (at the end).
** RETURNS:     
**************************************************************/
void SocketSort(SOCKET *ThisSocket, PROTSTRUCT *ThisProt)
{
  SOCKET *TmpSock;

  for (TmpSock = ThisProt->PrSocket;
       TmpSock != ThisSocket;
       TmpSock = TmpSock->So_Next) {
    if (!LOCPORTBOUND(TmpSock) ||
        (TmpSock->So_LocAdd.Port > ThisSocket->So_LocAdd.Port)) {
      MoveSocket(TmpSock, ThisSocket, ThisProt);
      break;
      }
    else {                             /* socket found with same port nr */
      if ((TmpSock->So_LocAdd.Port == ThisSocket->So_LocAdd.Port) &&
          (!LOCADDBOUND(TmpSock) ||
           (LOCADDBOUND(ThisSocket) &&
            (ntohl(TmpSock->So_LocAdd.IPAddress) >
                   ntohl(ThisSocket->So_LocAdd.IPAddress))))) {
         MoveSocket(TmpSock, ThisSocket, ThisProt);
         break;
         }
      }
    }
}

/**************************************************************
** NAME:        SoBindLoc
** SYNOPSIS:    int SoBindLoc(SOCKET *ThisSocket,
**                            SO_ADDRESS *ThisAddress,
**                            PROTSTRUCT ThisProt)
**
** DESCRIPTION: Binds ThisAddress to ThisSocket according to
**              the rules:
**              - bound only if Port != 0 or IPAddress != 0;
**              - the binding flags are set as well.
**              - the selected port is placed in
**                ThisAddress->Port
** RETURNS:     NOERR  -->  everything just fine
**************************************************************/
int SoBindLoc(SOCKET *ThisSocket, SO_ADDRESS *ThisAddress, 
              PROTSTRUCT *ThisProt)
{
  int    ColPoints;
  SOCKET *TmpSock  = ThisProt->PrSocket;
  USHORT FreeSock  = IPPORT_USERRES;  /* first non reserved port */

  if (ThisAddress->Port != 0) {        /* if socket specified */
    ThisSocket->So_LocAdd.Port = ThisAddress->Port;
    }
  else {                               /* search non reserved socket */
    while((TmpSock != NULL) && LOCPORTBOUND(TmpSock)) {
      if (TmpSock->So_LocAdd.Port < IPPORT_USERRES) {
        TmpSock = TmpSock->So_Next;
        }
      else {
        if (TmpSock->So_LocAdd.Port == FreeSock) {
          if (++FreeSock == 0xffff) {
            return NOSOCKLEFT;
            }
          TmpSock = TmpSock->So_Next;
          }
        else {
          break;
          }
        }
      }
    ThisSocket->So_LocAdd.Port = ThisAddress->Port = FreeSock;
    }
  ThisSocket->So_LocAdd.IPAddress = ThisAddress->IPAddress;

  if ((TmpSock = So_Locate(ThisSocket->So_LocAdd.Port,
                           ThisSocket->So_LocAdd.IPAddress,
                           ThisSocket->So_RemAdd.Port,
                           ThisSocket->So_RemAdd.IPAddress,
                           ThisProt, &ColPoints)) != NULL) {
    return SOCKINUSE;
    }

  ThisSocket->So_State |= SS_LOCPORT;
  if (ThisAddress->IPAddress != INADDR_ANY) 
    ThisSocket->So_State |= SS_LOCALBOUND;
  else                                        /* debind local address */
    ThisSocket->So_State &= ~SS_LOCALBOUND;
  SocketSort(ThisSocket, ThisProt);
  return NOERR;  
}

/**************************************************************
** NAME:        SoRemoteConnect
** SYNOPSIS:    int SoRemoteConnect(SOCKET *ThisSocket,
**                                  SO_ADDRESS *ThisAddress,
**                                  PROTSTRUCT *ThisProt);
**          
** DESCRIPTION: Sets the remote address part of ThisSocket.
**              The binding flags are set as well.
**              The following rules apply:
**              - Port is set only if ThisAddress.Port != 0,
**                else the remote port is debounded;
**              - Address is set only if ThisAddress.
**                IPAddress != 0, else the remote address
**                is debounded.
**              - If the local port is not bounded yet,
**                the same port as the connected port will
**                be chosen.
**              - If the local address is not bounded yet,
**                a host address will be selected.
**
** RETURNS:     NOERR     -->  Everything OK
**              SOCKINUSE -->  This socket spec has already
**                             been used.
**************************************************************/
int SoRemoteConnect(SOCKET *ThisSocket, SO_ADDRESS *ThisAddress,
                    PROTSTRUCT *ThisProt)
{
  int         RetCode;
  SO_ADDRESS  LocAddr;


  if ((ThisSocket->So_RemAdd.Port = ThisAddress->Port) != 0) {
    ThisSocket->So_State |= SS_REMPORT;           /* if port specified */
    }
  else {
    ThisSocket->So_State &= ~SS_REMPORT;
    }

  if ((ThisSocket->So_RemAdd.IPAddress = ThisAddress->IPAddress) != INADDR_ANY) {
    ThisSocket->So_State |= SS_REMOTEBOUND;
    }
  else {
    ThisSocket->So_State &= ~SS_REMOTEBOUND;
    }

  if (!LOCALBOUND(ThisSocket)) {
    /* socket binding required */
    LocAddr.Port = ThisSocket->So_LocAdd.Port;
    if ((RetCode = IPGetHostAdd(&(LocAddr.IPAddress),
                                ThisAddress->IPAddress, 0)) != NO_ERR) {
      return RetCode;
      }
    return (*(ThisProt->UserReq))(PRU_BIND, ThisSocket, NULL, &LocAddr, NULL);
    }
  return NO_ERR;
}

/**************************************************************
** NAME:        UDPService
** SYNOPSIS:    int UDPService(USERREQ ThisReq,
**                             SOCKET *ThisSocket,
**                             DATALINK *ThisLink,
**                             SO_ADDRESS *FromAddress,
**                             SO_ADDRESS *ToAddress);
**  
** DESCRIPTION: Test function
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int UDPService(USERREQ ThisReq, SOCKET *ThisSocket,
               DATALINK *ThisLink, SO_ADDRESS *FromAddress,
               SO_ADDRESS *ToAddress)
{
  return ProtService(ThisReq, ThisSocket, ThisLink,
                     FromAddress, ToAddress, UDPProt);
}

/**************************************************************
** NAME:        RawService
** SYNOPSIS:    int RawService(USERREQ ThisReq,
**                             SOCKET *ThisSocket,
**                             DATALINK *ThisLink,
**                             SO_ADDRESS *FromAddress,
**                             SO_ADDRESS *ToAddress);
** 
** DESCRIPTION: Test function
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int RawService(USERREQ ThisReq, SOCKET *ThisSocket,
               DATALINK *ThisLink, SO_ADDRESS *FromAddress,
               SO_ADDRESS *ToAddress)
{
  return ProtService(ThisReq, ThisSocket, ThisLink,
                     FromAddress, ToAddress, RawProt);
}

/**************************************************************
** NAME:        ProtService
** SYNOPSIS:    int ProtService(USERREQ ThisReq,
**                              SOCKET *ThisSocket,
**                              DATALINK *ThisLink,
**                              SO_ADDRESS *FromAddress,
**                              SO_ADDRESS *ToAddress,
**                              PROTSTRUCT *ThisProt);
**    
** DESCRIPTION: Implements services for the UDP en Raw socket
**              protocol structure. This function is usable
**              only for these protocols.
** RETURNS:     NOERR -->   no error
**              else        error code
**************************************************************/
int ProtService(USERREQ ThisReq, SOCKET *ThisSocket,
                DATALINK *ThisLink, SO_ADDRESS *FromAddress,
                SO_ADDRESS *ToAddress, PROTSTRUCT *ThisProt)
{
  static USHORT UDPIdent = 0;
  static USHORT RawIdent = 0;
  CIPHEAD       IPHead;
  CUDPHEAD      UDPHead;
  DATALINK      *HeadPoint, *OptLink;
  USHORT        TempCheck;
  SO_ADDRESS    SaveAdd = { 0, 0 };
  SO_PACKLIST   *SockList;
  SO_PACKLIST   *OldSock;
  int           RetCode;

  switch (ThisReq) {
    case PRU_ATTACH :                  /* attach new socket to protocol */
      AppendSocket(ThisSocket, ThisProt);
      return NOERR;

    case PRU_BIND :                     /* bind local address to socket */
      if (LOCPORTBOUND(ThisSocket)) {              /* debind if bounded */
        SaveAdd = ThisSocket->So_LocAdd;          /* save if bind fails */
        ProtService(PRU_DEBIND, ThisSocket, NULL, NULL, NULL, ThisProt);
        }
      if (((RetCode = SoBindLoc(ThisSocket, FromAddress, ThisProt))
            != NOERR) && (SaveAdd.Port != 0)) {
        SoBindLoc(ThisSocket, &SaveAdd, ThisProt);  /* if error, restore */
        }
      return RetCode;

    case PRU_DEBIND :                           /* unbind local address  */
      SockList = ThisSocket->RecQueue;          /* first, remove waiting */
      while (SockList != NULL) {                /* packets               */
        OldSock = SockList;
        SockList = SockList->NextPack;
        FreeSoPack(OldSock);
        }
      ThisSocket->RecQueue = NULL;
      ThisSocket->So_State &= ~(SS_LOCALBOUND | SS_LOCPORT);
      ThisSocket->So_LocAdd.Port = 0;
      ThisSocket->So_LocAdd.IPAddress = INADDR_ANY;
      RemoveSocket(ThisSocket, ThisProt);
      AppendSocket(ThisSocket, ThisProt);
      return NOERR;

    case PRU_CONNECT :               /* connect remote address to socket */
      return SoRemoteConnect(ThisSocket, FromAddress, ThisProt);

    case PRU_DISCONNECT :       /* disconnect remote address from socket */
      ThisSocket->So_RemAdd.Port = 0;
      ThisSocket->So_RemAdd.IPAddress = INADDR_ANY;
      ThisSocket->So_State &= ~(SS_REMOTEBOUND | SS_REMPORT);
      return NOERR;

    case PRU_SHUTDOWN :                   /* remove socket from protocol */
      ProtService(PRU_DISCONNECT, ThisSocket, NULL, NULL, NULL, ThisProt);
      ProtService(PRU_DEBIND, ThisSocket, NULL, NULL, NULL, ThisProt);
      return NOERR;

    case PRU_DETACH :                     /* remove socket from protocol */
      RemoveSocket(ThisSocket, ThisProt);
      return NOERR;

    case PRU_SENDTO :
      if (FromAddress->IPAddress != INADDR_ANY) {
        IPHead.Source = FromAddress->IPAddress;
        }
      else {
        if ((RetCode = IPGetHostAdd(&(IPHead.Source),
                                    ToAddress->IPAddress, 0)) != NO_ERR) {
          return RetCode;
          }
        }
      IPHead.Destin = ToAddress->IPAddress;

      if (ThisProt == UDPProt) {       /* UDP protocol service */
        if ((HeadPoint = IPDataLinkGet(8)) == NULL) {
            IPReport(NOSPACE, NULL, ThisLink);
            return NOBUFS;
            }
        HeadPoint->ThisType = UDPHD;
        HeadPoint->NextOne = ThisLink;

        /* fill the UDP header */
        UDPHead.SPort = FromAddress->Port;
        UDPHead.DPort = ToAddress->Port;
        UDPHead.DLength = 8 + ThisLink->Length;
        CudpToByte(HeadPoint->DataStart, &UDPHead);

        /* fill the IP header */
        IPHead.Tos = UDPTOS;
        IPHead.Ident = UDPIdent++;
        IPHead.Protocol = UDP;

        /* determine the checksum */
        TempCheck = UDPCheckSum(&IPHead, HeadPoint);
        *(USHORT *)(HeadPoint->DataStart + 6) =
                (TempCheck == 0) ? 0xffff : TempCheck;
        }
      else {                           /* Raw protocol service */
        /* fill the IP header */
        IPHead.Tos = RAWTOS;
        IPHead.Ident = RawIdent++;
        /* set protocol specifier */
        IPHead.Protocol = (BYTE)(ThisSocket->So_RemAdd.Port & 0xff);
        HeadPoint = ThisLink;
        }
      IPHead.Flag = 0;
      IPHead.Time = TTL;
      IPHead.OLength = IPHead.OpState = 0;
      if (ThisSocket->So_Options != NULL) {
        if ((OptLink = IPBufGet(sizeof(DATALINK))) == NULL) {
          IPReport(NOSPACE, NULL, HeadPoint);
          return NOBUFS;
          }
        /* make copy of datalink, data is saved 'cuz SaveData = 1 */
        *OptLink = *ThisSocket->So_Options;
        OptLink->NextOne = HeadPoint;
        HeadPoint = OptLink;
        }
      return IPSend(&IPHead, HeadPoint, HOPS_ALLOWED);

    default :
      return ILL_REQUEST;
    }
}


/**************************************************************
** NAME:        So_Locate
** SYNOPSIS:    SOCKET *So_Locate(USHORT DPort, ADDRESS_T DAddr,
**                                USHORT SPort, ADDRESS_T SAddr,
**                                PROTSTRUCT *ThisProt,
**                                int *ColPoints)
**              
** DESCRIPTION: Locates a socket from the protocol structure
**              ThisProt. The socket that best matches the
**              given parameters is selected.
**              The socket found gets points for matching
**              items, returned in ColPoints.
**              DPort --> 8 pnts   DAddr --> 4 pnts
**              SPort --> 2 pnts   SAddr --> 1pnt.
**              Exact match gives 15 points.
** RETURNS:     NULL --> no fitting socket found
**              else     a pointer to the selected socket
**************************************************************/
SOCKET *So_Locate(USHORT DPort, ADDRESS_T DAddr,
                  USHORT SPort, ADDRESS_T SAddr,
                  PROTSTRUCT *ThisProt, int *ColPoints)
{
  SOCKET  *TmpPnt;
  SOCKET  *MaxSock  = NULL;
  int     MaxPoints = 0;
  int     Collected = 0;

  for (TmpPnt = ThisProt->PrSocket;
       (TmpPnt != NULL) &&             /* locate socket with same port */
       LOCPORTBOUND(TmpPnt) &&
       (TmpPnt->So_LocAdd.Port <= DPort);
       TmpPnt = TmpPnt->So_Next) {
    Collected = 0;
    if (TmpPnt->So_LocAdd.Port == DPort) {
      Collected = 8;
      if (LOCADDBOUND(TmpPnt)) {
        if (TmpPnt->So_LocAdd.IPAddress == DAddr) Collected += 4;
        else continue;
        }
      if (REMPORTBOUND(TmpPnt)) {
        if (TmpPnt->So_RemAdd.Port == SPort) Collected += 2;
        else continue;
        }
      if (REMADDBOUND(TmpPnt)) {
        if (TmpPnt->So_RemAdd.IPAddress == SAddr) Collected += 1;
        else continue;
        }
      if (Collected > MaxPoints) {
        MaxPoints = Collected;
        MaxSock = TmpPnt;
        }
      }
    }
  return ((*ColPoints = MaxPoints) > 0) ? MaxSock : NULL;
}

/**************************************************************
** NAME:        MessToSock
** SYNOPSIS:    void MessToSock(SOCKET *DestSocket,
**                              CIPHEAD *ThisHeader,
**                              DATALINK *ThisPacket)
**             
** DESCRIPTION: Append ThisHeader and ThisPacket to the
**              receive queue of DestSocket.
**              If the receive queue has reached maximum
**              length or if there is no buffer space left,
**              ThisHeader and ThisPacket will be freed.
** RETURNS:     
**************************************************************/
void MessToSock(SOCKET *DestSocket, CIPHEAD *ThisHeader,
                DATALINK *ThisPacket)
{
  SO_PACKLIST  *TmpPack;

  if (DestSocket->So_RCurLen >= DestSocket->So_RMaxLen) {
    /* socket queue is full */
    IPStat.QueueFull++;
    IPReport(QUEUEFULL, ThisHeader, ThisPacket);
    return;
    }                                  
  else {                         /* append this message to socket list */
    if ((TmpPack = IPBufGet(sizeof(SO_PACKLIST))) == NULL) {
      IPReport(NOSPACE, ThisHeader, ThisPacket);
      return;
      }
    else {
      TmpPack->NextPack = NULL;
      TmpPack->Source = ThisHeader;
      TmpPack->Packet = ThisPacket;
      if (DestSocket->RecQueue == NULL) {
        DestSocket->RecQueue = TmpPack;
        }
      else {
        DestSocket->RecQEnd->NextPack = TmpPack;
        }
      DestSocket->RecQEnd = TmpPack;
      DestSocket->So_RCurLen++;
      }
    }
}


/**************************************************************
** NAME:        SetIPOptions
** SYNOPSIS:    int SetIPOptions(SOCKET *ThisSocket,
**                               char *OptVal, int OptLen);
**             
** DESCRIPTION: Connects the options in 'OptVal' to
**              'ThisSocket'. The length of the options is
**              'OptLen'. All options are placed in an IP
**              packet as received in 'OptVal', with the
**              exception of Source Routed packets, in which
**              case the first hop is removed and used as
**              first hop destination.
** RETURNS:     NO_ERR   -->  everything ok
**              else     -->  error code (see ipcodes.h)
**************************************************************/
int SetIPOptions(SOCKET *ThisSocket, char *OptVal, int OptLen)
{
  if (ThisSocket->So_Options != NULL) {
    ThisSocket->So_Options->SaveData = 0;
    DiscardAll(ThisSocket->So_Options);
    ThisSocket->So_Options = NULL;
    }
  if (OptLen > 0) {
    if ((ThisSocket->So_Options = IPDataLinkGet(OptLen)) == NULL) {
      return NOSPACE;
      }
    ThisSocket->So_Options->ThisType = OPTIONSHD;
    ThisSocket->So_Options->SaveData = 1;
    memcpy(ThisSocket->So_Options->DataStart, OptVal, OptLen);
    }
  return NOERR;
}


