/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       dpdev.c
**     SYSTEM   NAME:       PACKET DISPATCHER
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1989/11/30
**
** DESCRIPTION: Dispatcher's device driver functions
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   DPDEV.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPDEV.C_V  $
**              
**                 Rev 1.1   01 Feb 1991 14:21:06   etstjan
**              
**                 Rev 1.0   26 Oct 1990 12:27:54   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/DPDEV.C_V   1.1   01 Feb 1991 14:21:06   etstjan  $";
#endif
/*
    The device functions
*/

#include <stdio.h>                          /* for NULL */
#include <string.h>                         /* for str*() */
#include <error.h>                          /* for ERR_DEB */
#include <pkt.h>                            /* for packet */
#include "dpinc.h"                          /* main include file */

extern  PKTUPCALL   DpDevReceive[];         /* defined in dprvc.asm */


DPDEV DpDevActive[DPMAXDEV];
static int DpDevCount;

int DpDevStartTypeFilt (int Dev, int Filt, int Mode);
int DpDevStopTypeFilt (int Dev, int Filt);

static int DpDevAdd (int PktInt)
{
    DPDEV *pDev;
    PKTINFO *pInfo;
    int Dev;

    ERR_DEB (ERR_NPD, 4, "DpDevAdd Enter <PktInt=%d>",PktInt);
    for (Dev=0; Dev<DPMAXDEV && DpDevActive[Dev].PktInt!=0; Dev++);                              
    if (Dev >= DPMAXDEV)                       /* search first free */
        return (-1);
    if (Dev >= DpDevCount)
        DpDevCount = Dev+1;
    pDev=&(DpDevActive[Dev]);
    pInfo=PktDriverInfo(PktInt);
    strncpy(pDev->Name, pInfo->Name, DPNAMELEN);
    pDev->Name[DPNAMELEN] = '\0';              /* just to be sure */
    pDev->Class = pInfo->Class;
    pDev->Type = pInfo->Type;
    pDev->PktInt = PktInt;
    pDev->TypeFiltCount = 0;
    ERR_DEB( ERR_NPD, 4, "DpDevAdd Leave okay <Dev=%d>",Dev);
    return 0;
}

/**************************************************************
** NAME:        DpDevInit    
** SYNOPSIS:    int DpDevInit (void)
** DESCRIPTION: Searches for devices intializes them.
**              The receive mode is switched off and a filter
**              for all packets is activated.
** RETURNS:     0   no error
**              -1  search for packet drivers failed
**************************************************************/
int DpDevInit (void)
{
    int Dev;
    DPSTAT *pStat;

    ERR_DEB(ERR_NPD, 4, "DpDevInit");
    DpDevCount = 0;
    if (PktInit(DpDevAdd))
        return (-1);
    for (Dev=0; Dev<DpDevCount; Dev++)
        DpDevAddTypeFilt (Dev, 0);
    pStat = &(DpDevActive[Dev].Stat);
    pStat->TotalPackets=0;             /* reset statistics */
    pStat->TotalBytes=0;        
    pStat->MissedPackets=0;        
    pStat->MissedBytes=0;        
    return (0);
}

/**************************************************************
** NAME:        DpDevEnd            
** SYNOPSIS:    void DpDevEnd (void)
** DESCRIPTION: Switches off devices, releases active filters.
** RETURNS:     void
**************************************************************/
void DpDevEnd (void)
{
    int Dev, Filt;
    DPDEV   *pDev;

    ERR_DEB(ERR_NPD, 4, "DpDevEnd");
    for (Dev=0; Dev<DpDevCount; Dev++)
    {
        pDev= &(DpDevActive[Dev]);
        for (Filt=0; Filt<pDev->TypeFiltCount; Filt++)
            DpDevRemoveTypeFilt (Dev, Filt);
    }
    DpDevCount=0;
}


/**************************************************************
** NAME:        DpDevStart      
** SYNOPSIS:    int DpDevStart (void)
** DESCRIPTION: Starts all devices in promiscuous mode.
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**              -3  illegal mode
**************************************************************/
int DpDevStart (void)
{
    int Ret = 0;
    int Dev;

    ERR_DEB(ERR_NPD, 4, "DpDevStart");
    for (Dev=0; Dev<DpDevCount; Dev++)
        Ret |= DpDevStartDevice (Dev, PKT_MODE_ALL);
    return (Ret);
}

/**************************************************************
** NAME:        DpDevStop      
** SYNOPSIS:    int DpDevStop (void)
** DESCRIPTION: Stops all devices.
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevStop (void)
{
    int Ret = 0;
    int Dev;

    ERR_DEB(ERR_NPD, 4, "DpDevStop");
    for (Dev=0; Dev<DpDevCount; Dev++)
        Ret |= DpDevStopDevice (Dev);
    return (Ret);
}

/**************************************************************
** NAME:        DpDevStartDevice    
** SYNOPSIS:    int DpDevStartDevice (int Dev, int Mode)
** DESCRIPTION: Starts a device in a specified mode.
**              PKT_MODE_NORMAL   :packets to this interface
**              PKT_MODE_BROAD    :+ broadcast
**              PKT_MODE_LIMMULTI :+ limited multicast
**              PKT_MODE_MULTI    :+ all multicast
**              PKT_MODE_ALL      :all packets
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**              -3  illegal mode
**************************************************************/
int DpDevStartDevice (int Dev, int Mode)
{
    DPDEV *pDev;
    int Filt;

    ERR_DEB(ERR_NPD, 4, "DpDevStartDevice");
    if (Dev<0 || Dev>=DpDevCount)            /* check valid arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);    
    if (pDev->PktInt==0)    
        return (-2);
    if (Mode<PKT_MODE_NORMAL || Mode>PKT_MODE_ALL)
        return (-3);
    for (Filt=0; Filt<pDev->TypeFiltCount; Filt++)
        DpDevStartTypeFilt (Dev, Filt, Mode);
    return (0);
}

/**************************************************************
** NAME:        DpDevStopDevice   
** SYNOPSIS:    int DpDevStopDevice (int Dev)
** DESCRIPTION: Switches off the specified device.
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevStopDevice (int Dev)
{
    DPDEV *pDev;
    int Filt;

    ERR_DEB(ERR_NPD, 4, "DpDevStopDevice");
    if (Dev<0 || Dev>=DpDevCount)                /* check valid arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    for (Filt=0; Filt<pDev->TypeFiltCount; Filt++)
        DpDevStopTypeFilt (Dev, Filt);
    return (0);
}

/**************************************************************
** NAME:        DpDevAddTypeFilt     
** SYNOPSIS:    int DpDevAddTypeFilt (int Dev, int Filt)
** DESCRIPTION: Activates a type filter for a device.
** RETURNS:     Filter handle.
**              -1  device number too big
**              -2  no packet interrupt for that device
**              -3  no filter available
**************************************************************/
int DpDevAddTypeFilt (int Dev, int Filt)
{
    DPDEV *pDev;
    DPFILTHANDLE *pHandle;
    
    ERR_DEB(ERR_NPD, 4, "DpDevAddTypeFilt");
    if (Dev<0 || Dev>=DpDevCount)           /* check valid arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);    
    if (pDev->PktInt==0)    
        return (-2);
    if (pDev->TypeFiltCount>=DPMAXTYPEFILT)
        return (-3);
    pHandle = &(pDev->TypeFiltActive[(pDev->TypeFiltCount)++]);
    pHandle->Filt = Filt;
    pHandle->Handle = 0;
    return (0);
}

/**************************************************************
** NAME:        DpDevRemoveTypeFilt    
** SYNOPSIS:    int DpDevRemoveTypeFilt (int Dev, int Filt)
** DESCRIPTION: Removes a filter from a device.
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**              -3  no filter handle
**************************************************************/
int DpDevRemoveTypeFilt (int Dev, int Filt)
{
    DPDEV *pDev;

    ERR_DEB(ERR_NPD, 4, "DpDevRemoveTypeFilt");
    if (Dev<0 || Dev>=DpDevCount)        /* check valid arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    if (Filt>=pDev->TypeFiltCount)
        return (-3);
    (pDev->TypeFiltCount)--;
    while (Filt<pDev->TypeFiltCount)
    {
        pDev->TypeFiltActive[Filt]=pDev->TypeFiltActive[Filt+1];
        Filt++;
    }
    return (0);
}


/**************************************************************
** NAME:        DpDevStartTypeFilt     
** SYNOPSIS:    int DpDevStartTypeFilt (int Dev, int Filt, int Mode)
** DESCRIPTION: Starts a type filter in a specified mode.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int DpDevStartTypeFilt (int Dev, int Filt, int Mode)
{
    int Ret = 0;
    DPDEV *pDev;
    DPFILTHANDLE *pHandle;
    DPTYPEFILT *pFilt;
    
    ERR_DEB(ERR_NPD, 4, "DpDevStartTypeFilt");
    pDev = &(DpDevActive[Dev]);
    pHandle = &(pDev->TypeFiltActive[Filt]);
    pFilt = DpTypeFiltAvailable[pHandle->Filt];
    if (pHandle->Handle==0)
    {
        Ret |= PktAccessType (pDev->PktInt, pFilt->TypeData, pFilt->TypeLen,    
            DpDevReceive[Dev], &(pHandle->Handle));
        Ret |= PktSetRcvMode (pDev->PktInt, pHandle->Handle, Mode);
    }
    return (Ret);
}

/**************************************************************
** NAME:        DpDevStopTypeFilt   
** SYNOPSIS:    int DpDevStopTypeFilt (int Dev, int Filt)
** DESCRIPTION: Stops a type filter.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int DpDevStopTypeFilt (int Dev, int Filt)
{
    int Ret = 0;
    DPDEV *pDev;
    DPFILTHANDLE *pHandle;
    
    ERR_DEB(ERR_NPD, 4, "DpDevStopTypeFilt");
    pDev = &(DpDevActive[Dev]);
    pHandle = &(pDev->TypeFiltActive[Filt]);
    if (pHandle->Handle!=0)
    {
        Ret |= PktSetRcvMode (pDev->PktInt, pHandle->Handle, PKT_MODE_BROAD);
        Ret |= PktReleaseType (pDev->PktInt, pHandle->Handle);
        pHandle->Handle=0;
    }
    return (Ret);
}


/**************************************************************
** NAME:        DpDevGetTypeFiltCount    
** SYNOPSIS:    int DpDevGetTypeFiltCount (int Dev)
** DESCRIPTION: Returns number of type filters for device.
** RETURNS:     Number of type filters
**              -1  No device
**              -2  Device not active
**************************************************************/
int DpDevGetTypeFiltCount (int Dev)
{
    DPDEV *pDev;
    
    if (Dev<0 || Dev>=DpDevCount)         /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    return (pDev->TypeFiltCount);
}

/**************************************************************
** NAME:        DpDevGetTypeFilt                 
** SYNOPSIS:    int DpDevGetTypeFilt (int Dev, int Filt)
** DESCRIPTION: Returns number of type filter.
**              This is an index in DpFilterAvailable.
** RETURNS:     Number of type filter.
**              -1  No device
**              -2  Device not active
**              -3  No filter
**************************************************************/
int DpDevGetTypeFilt (int Dev, int Filt)
{
    DPDEV *pDev;
    
    if (Dev<0 || Dev>=DpDevCount)         /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    if (Filt<0 || Filt>=pDev->TypeFiltCount)
        return (-3);
    return (pDev->TypeFiltActive[Filt].Filt);
}


/**************************************************************
** NAME:        DpDevGetStatistics
** SYNOPSIS:    DPSTAT *DpDevGetStatistics (int Dev)
** DESCRIPTION: Returns a ptr to the device statistics struct.
**              DPSTAT {
**                  unsigned long TotalPackets;
**                  unsigned long TotalBytes
**                  unsigned long MissedPackets;
**                  unsigned long MissedBytes;
**              }
** RETURNS:     Pointer to statistics structure
**              NULL   invalid device
**************************************************************/
DPSTAT *DpDevGetStatistics (int Dev)
{
    DPDEV *pDev;
    
    if (Dev<0 || Dev>=DpDevCount)         /* check arguments */
        return (NULL);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (NULL);
    return (&(pDev->Stat));
}


/**************************************************************
** NAME:        DpDevResetStatistics               
** SYNOPSIS:    int DpDevResetStatistics (int Dev)
** DESCRIPTION: Resets statistics for a device.
**              DPSTAT {
**                  unsigned long TotalPackets;
**                  unsigned long TotalBytes
**                  unsigned long MissedPackets;
**                  unsigned long MissedBytes;
**              }
** RETURNS:     0   no error
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevResetStatistics (int Dev)
{
    DPDEV *pDev;

    if (Dev<0 || Dev>=DpDevCount)            /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    pDev->Stat.TotalPackets=0;             /* reset statistics */
    pDev->Stat.TotalBytes=0;        
    pDev->Stat.MissedPackets=0;        
    pDev->Stat.MissedBytes=0;
    return (0);
}

/**************************************************************
** NAME:        DpDevGetName
** SYNOPSIS:    char *DpDevGetName (int Dev)
** DESCRIPTION: Returns device name.
** RETURNS:     Pointer to device name
**              NULL  invalid device
**************************************************************/
char *DpDevGetName (int Dev)
{
    DPDEV *pDev;
    
    if (Dev<0 || Dev>=DpDevCount)             /* check arguments */
        return (NULL);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (NULL);
    return (pDev->Name);        
}

/**************************************************************
** NAME:        DpDevGetClass
** SYNOPSIS:    int DpDevGetClass (int Dev)
** DESCRIPTION: Returns device class.
** RETURNS:     Device class
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevGetClass (int Dev)
{
    DPDEV *pDev;

    if (Dev<0 || Dev>=DpDevCount)             /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    return (pDev->Class);        
}

/**************************************************************
** NAME:        DpDevGetType
** SYNOPSIS:    int DpDevGetType (int Dev)
** DESCRIPTION: Returns device type.
** RETURNS:     Device type
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevGetType (int Dev)
{
    DPDEV *pDev;

    if (Dev<0 || Dev>=DpDevCount)             /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    return (pDev->Type);        
}

/**************************************************************
** NAME:        DpDevGetIntr
** SYNOPSIS:    int DpDevGetIntr (int Dev)
** DESCRIPTION: Returns device packet interrupt.
** RETURNS:     Device packet interrupt
**              -1  device number too big
**              -2  no packet interrupt for that device
**************************************************************/
int DpDevGetIntr (int Dev)
{
    DPDEV *pDev;

    if (Dev<0 || Dev>=DpDevCount)             /* check arguments */
        return (-1);
    pDev = &(DpDevActive[Dev]);
    if (pDev->PktInt==0)    
        return (-2);
    return (pDev->PktInt);        
}


/**************************************************************
** NAME:        DpDevGetCount    
** SYNOPSIS:    int DpDevGetCount (void)
** DESCRIPTION: Returns number of active devices.
** RETURNS:     number of active devices
**************************************************************/
int DpDevGetCount (void)
{
    return (DpDevCount);
}
