/************************************************************************
 * exehex.c -	This program converts Microsoft .EXE files to Intel	*
 *		HEX files suitable for burning into EPROM.  Like the	*
 *		loader in MSDOS, exehex performs any necessary		*
 *		relocation of segments in the .EXE file.		*
 ************************************************************************/

/************************************************************************
 * Copyright 1991, Charles F. Harris, C.F. Harris - Consulting.		*
 *									*
 * You may use this program for any purpose Commercial or Private with	*
 * following exceptions:						*
 *									*
 * 1) You may not sell this program without my written permission.	*
 * 2) You must leave this header comment unmodified, and intact.	*
 * 3) You must clearly identify any modifications that you make.	*
 *									*
 * This program is provided without any warantees or support, and is	*
 * to be used at your own risk.						*
 ************************************************************************/

/************************************************************************
 * This program was designed to run on 80x86 type machines, and to be	*
 * compiled with TURBO C++ V1.00. (that is not to say that it is	*
 * written in C++)							*
 * I can be reached at: chuck@eng.umd.edu				*
 * or at:	Chuck Harris						*
 *		C.F. Harris - Consulting				*
 *		9308 Canterbury Riding					*
 *		Laurel, Maryland 20723					*
 ************************************************************************/

#include <dir.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <io.h>

#define ERR(s, c)	if(opterr){\
	char errbuf[2];\
	errbuf[0] = c; errbuf[1] = '\n';\
	(void) write(2, argv[0], (unsigned)strlen(argv[0]));\
	(void) write(2, s, (unsigned)strlen(s));\
	(void) write(2, errbuf, 2);}

#define DATAREC 0
#define EXADDR	2
#define ENDREC	1


struct EXEHDR {
	unsigned signature;		/* linker signature */
	unsigned image_rsize;		/* length of image mod 512 */
	unsigned file_nblks;		/* number of 512 byte blocks in file */
	unsigned num_rels;		/* number of relocation items */
	unsigned header_size;		/* size in paragraphs */
	unsigned minheap_size;		/* min num paragraphs in heap */
	unsigned maxheap_size;		/* max num paragraphs in heap */
	unsigned stackseg_offset;	/* offset of stack seg in load module */
	unsigned initial_SP;		/* initial value of stack register */
	unsigned checksum;		/* neg sum of all words in file */
	unsigned initial_IP;		/* initial value of instruction ptr */
	unsigned codeseg_offset;	/* offset of code seg in load module */
	unsigned reltbl_offset;		/* offset of first relocation item */
	unsigned overlay_num;		/* overlay number (0 == resident part) */
} exehdr;

typedef struct RELTBL {
	unsigned rel_offset;
	unsigned rel_segment;
} RELTBL;

int	opterr = 1;
int	optind = 1;
int	optopt;
char	*optarg;

char *tempfile = "exehexXXXXXX";	/* template for mktemp() */
char *tmpfname = 0;			/* name of temp file */

FILE *fpi,*fpo,*fpt;
/*
 * prototypes for functions defined in this file
 */
void discover_options(int argc, char **argv);
int getopt(int argc, char **argv, char *opts);
void open_files(int argc, char **argv);
void read_exe_header(void);
void print_exe_header(void);
void read_relocation_table(void);
void copy_image_to_temp(void);
void relocate_image(void);
void write_hex_file(void);
void write_extended_data_record(unsigned address, unsigned char datalen, unsigned char data[]);
void write_data_record(unsigned address, unsigned char datalen, unsigned char data[]);
void write_extended_address_record(unsigned usba);
void write_end_record(unsigned start);
void build_hex_record(unsigned char type, unsigned loadaddress, unsigned char datalen, unsigned char data[]);
unsigned hex_hi_hi(unsigned n);
unsigned hex_hi_lo(unsigned n);
unsigned hex_lo_hi(unsigned n);
unsigned hex_lo_lo(unsigned n);
unsigned hi_byte(unsigned n);
unsigned lo_byte(unsigned n);
void ABORT(char *reason, ...);
void remove_temp_file(void);
void how_to_use_message(void);
/*
 * a few global variables...
 */
unsigned char intelstring[600];
unsigned segment = 0, offset = 0, estart = 0;
unsigned bytesline = 0x20;		/* # bytes per line */
long imagelen = 0;
unsigned debug = 0;
unsigned no_end_record = 0;

RELTBL *reloc_tbl;

main(argc,argv)
int argc;
char **argv;
{
	discover_options(argc,argv);
	open_files(argc,argv);
	read_exe_header();
	read_relocation_table();
	copy_image_to_temp();
	relocate_image();
	write_hex_file();
	remove_temp_file();
}

void
discover_options(argc,argv)
int argc;
char **argv;
{
	char *ourargs = "o:O:s:S:e:E:l:L:dDN?Hh";
	int opt;

	while((opt=getopt(argc,argv,ourargs)) != EOF){
		switch(opt){
		case 'd':
		case 'D':
			fprintf(stdout,"DEBUG level is: %d\n",++debug);
			break;
		case 's':
		case 'S':
			sscanf(optarg,"%x",&segment);
			if(debug>=1){
				fprintf(stdout,"Segment= 0x%X\n",segment);
			}
			break;
		case 'o':
		case 'O':
			sscanf(optarg,"%x",&offset);
			if(debug>=1){
				fprintf(stdout,"Offset= 0x%X\n",offset);
			}
			break;
		case 'e':
		case 'E':
			sscanf(optarg,"%x",&estart);
			if(debug>=1){
				fprintf(stdout,"EPROM Start Seg= 0x%X\n",estart);
			}
			break;
		case 'N':
			no_end_record++;
			if(debug>=1){
				fprintf(stdout,"No End Record\n");
			}
			break;
		case 'l':
		case 'L':
			sscanf(optarg,"%x",&bytesline);
			if(debug>=1){
				fprintf(stdout,"Bytes/Line= 0x%X\n",bytesline);
			}
			if(bytesline>0xff){
				ABORT("Too many bytes/line (0xff max)");
			}
			break;
		case '?':
		case 'h':
		case 'H':
			how_to_use_message();
			break;
		default:
			ABORT("Unknown argument");
		}
	}
}

/*
 * getopt a wonderful little function that handles the command line.
 * available courtesy of AT&T.
 */

int
getopt(argc, argv, opts)
int	argc;
char	**argv, *opts;
{
	static int sp = 1;
	register int c;
	register char *cp;

	if(sp == 1)
		if(optind >= argc ||
		   argv[optind][0] != '-' || argv[optind][1] == '\0')
			return(EOF);
		else if(strcmp(argv[optind], "--") == NULL) {
			optind++;
			return(EOF);
		}
	optopt = c = argv[optind][sp];
	if(c == ':' || (cp=strchr(opts, c)) == NULL) {
		ERR(": illegal option -- ", c);
		if(argv[optind][++sp] == '\0') {
			optind++;
			sp = 1;
		}
		return('?');
	}
	if(*++cp == ':') {
		if(argv[optind][sp+1] != '\0')
			optarg = &argv[optind++][sp+1];
		else if(++optind >= argc) {
			ERR(": option requires an argument -- ", c);
			sp = 1;
			return('?');
		} else
			optarg = argv[optind++];
		sp = 1;
	} else {
		if(argv[optind][++sp] == '\0') {
			sp = 1;
			optind++;
		}
		optarg = NULL;
	}
	return(c);
}

void
open_files(argc,argv)
int argc;
char **argv;
{
	int argind, files_found;

	argc--,argv++;			/* skip the program name */
	for(argind=0,files_found=0; argind<argc; argind++){
		if(argv[argind][0] == '-') continue;
		if(debug>=1){
			fprintf(stdout,"Argument is: %s\n",argv[argind]);
		}
		files_found++;
		
		if(files_found==1 && !(fpi=fopen(argv[argind],"rb"))){
			ABORT("opening %s",argv[argind]);
		}
		else if(files_found==2 && !(fpo=fopen(argv[argind],"wb"))){
			ABORT("opening %s",argv[argind]);
		}
		else if(files_found > 2){
			ABORT("too many file arguments\n");
		}
	}
	if(files_found == 0){
		ABORT("need a file to work on!");
	}
	else if(files_found == 1){
		fpo = stdout;
	}
	tmpfname = mktemp(tempfile);
	if(!(fpt=fopen(tmpfname,"wb+"))){
		ABORT("can't open temporary file: %s",tmpfname);
	}
}

void
read_exe_header()
{
	if(fread(&exehdr,sizeof(struct EXEHDR),1,fpi) != 1){
		ABORT("couldn't read exe header");
	}
	if(debug>=1){
		print_exe_header();
	}
	if(exehdr.signature != 0x5a4d){
		ABORT("bad exe signature:[0x%X]",exehdr.signature);
	}
	imagelen  = (long)exehdr.file_nblks*512 - (long)exehdr.header_size*16;
	imagelen -= (exehdr.image_rsize == 4) ? 0 : exehdr.image_rsize;
	if(debug>=1){
		fprintf(stdout,"binary image length is: 0x%lX,(%ldd)\n",imagelen,imagelen);
	}
}

void
print_exe_header()
{
	fprintf(stdout,"EXE HEADER:\n");
	fprintf(stdout,"->signature       = 0x%X\n",exehdr.signature);
	fprintf(stdout,"->image_rsize     = 0x%X\n",exehdr.image_rsize);
	fprintf(stdout,"->file_nblks      = 0x%X\n",exehdr.file_nblks);
	fprintf(stdout,"->num_rels        = 0x%X\n",exehdr.num_rels);
	fprintf(stdout,"->header_size     = 0x%X\n",exehdr.header_size);
	fprintf(stdout,"->minheap_size    = 0x%X\n",exehdr.minheap_size);
	fprintf(stdout,"->maxheap_size    = 0x%X\n",exehdr.maxheap_size);
	fprintf(stdout,"->stackseg_offset = 0x%X\n",exehdr.stackseg_offset);
	fprintf(stdout,"->initial_SP      = 0x%X\n",exehdr.initial_SP);
	fprintf(stdout,"->checksum        = 0x%X\n",exehdr.checksum);
	fprintf(stdout,"->initial_IP      = 0x%X\n",exehdr.initial_IP);
	fprintf(stdout,"->codeseg_offset  = 0x%X\n",exehdr.codeseg_offset);
	fprintf(stdout,"->reltbl_offset   = 0x%X\n",exehdr.reltbl_offset);
	fprintf(stdout,"->overlay_num     = 0x%X\n",exehdr.overlay_num);
}

void
read_relocation_table()
{
	int i;

	if(exehdr.num_rels == 0) return;	/* can't do much here */

	reloc_tbl = (RELTBL *)malloc(exehdr.num_rels * sizeof(RELTBL));
	if(!reloc_tbl){
		ABORT("can't malloc reloc_tbl");
	}
	if(fseek(fpi,(long)exehdr.reltbl_offset,SEEK_SET) != 0){
		ABORT("couldn't fseek to relocation tbl");
	}
	if(fread(reloc_tbl, sizeof(RELTBL) * exehdr.num_rels, 1, fpi) != 1){
		ABORT("couldn't read relocation table");
	}
	if(debug>=1){
		fprintf(stdout,"RELOCATION TABLE\n");
		for(i=0;i<exehdr.num_rels;i++){
			fprintf(stdout,"seg= 0x%X, off= 0x%X\n",reloc_tbl[i].rel_segment, reloc_tbl[i].rel_offset);
		}
	}
}

void
copy_image_to_temp()
{
	unsigned char bytes;
	unsigned char data[32];

	if(fseek(fpi, (long)exehdr.header_size * 16, SEEK_SET) != 0){
		ABORT("couldn't fseek() to data image");
	}
	while((bytes = fread(data,1,32,fpi)) == 32){
		if(fwrite(data,1,32,fpt) != 32){
			ABORT("couldn't write temp file");
		}
		bytes = 0;
	}
	if(bytes > 0){				/* any fraction remaining? */
		if(fwrite(data,1,bytes,fpt) != bytes){
			ABORT("couldn't write temp file");
		}
	}
}

void
relocate_image()
{
	long iptr;			/* image pointer */
	int i;
	unsigned rel_word;
	long long_rel_word;

	for(i=0; i<exehdr.num_rels; i++){
		iptr = (long)reloc_tbl[i].rel_segment*16 + (long)reloc_tbl[i].rel_offset;
		if(fseek(fpt, iptr, SEEK_SET) != 0){
			ABORT("couldn't fseek() to rel item");
		}
		if(fread(&rel_word,sizeof(unsigned),1,fpt) != 1){
			ABORT("couldn't read rel item");
		}
		long_rel_word = (long)rel_word + (long)segment;	/* relocate rel_word */
		if(fseek(fpt, iptr, SEEK_SET) != 0){
			ABORT("couldn't fseek() to rel item");
		}
		rel_word = long_rel_word;
		if(fwrite(&rel_word,sizeof(unsigned),1,fpt) != 1){
			ABORT("couldn't write back rel item");
		}
		if(long_rel_word > 0xffffL){	/* no wrapping allowed! */
			fprintf(stdout,"Segment Wrap! Fileloc=%X:%X, %X+%X\n",reloc_tbl[i].rel_segment,reloc_tbl[i].rel_offset,segment,rel_word);
		}
	}
}

void
write_hex_file()
{
	unsigned char data[256];
	unsigned char bytes;
	unsigned address = offset;

	if(fseek(fpt, (long)0, SEEK_SET) != 0){
		ABORT("couldn't fseek() to data image 0");
	}
	write_extended_address_record(segment -= estart);
	while((bytes = fread(data,1,bytesline,fpt)) == bytesline){
		if(address > address + bytes){	/* segment will overflow */
			write_extended_data_record(address,bytes,data);
			segment += 0x1000;	/* next 64Kb segment */
			address  = 0x0000;
		}
		else{
			write_data_record(address,bytes,data);
			address += bytes;
		}
		bytes = 0;
	}
	if(bytes > 0){				/* any fraction remaining? */
		if(address > address + bytes){	/* segment will overflow */
			write_extended_data_record(address,bytes,data);
		}
		else{
			write_data_record(address,bytes,data);
		}
	}
	if(!no_end_record){
		write_end_record(0);
	}
}

void
write_extended_data_record(address,datalen,data)
unsigned address;
unsigned char datalen;
unsigned char data[];
{
	unsigned excess;

	excess  = address + datalen;	/* bytes overflowing  segment */
	datalen = datalen - excess;	/* bytes fitting into segment */
	write_data_record(address,datalen,data);
	write_extended_address_record(segment + 0x1000);
	if(excess){
		write_data_record(0,excess,&data[datalen]);
	}
}

void
write_data_record(address,datalen,data)
unsigned address;
unsigned char datalen;
unsigned char data[];
{
	build_hex_record(DATAREC,address,datalen,data);
	fputs(intelstring,fpo);
}

void
write_extended_address_record(usba)
unsigned usba;
{
	unsigned char data[2];

	data[0] = hi_byte(usba);
	data[1] = lo_byte(usba);
	build_hex_record(EXADDR,0,2,data);
	fputs(intelstring,fpo);
}

void
write_end_record(start)
unsigned start;
{
	build_hex_record(ENDREC,start,0,(char *)0);
	fputs(intelstring,fpo);
}

void
build_hex_record(type,loadaddress,datalen,data)
unsigned char type;
unsigned loadaddress;
unsigned char datalen;
unsigned char data[];
{
	int checksum = 0;
	int i,j;

	intelstring[0] = ':';
	intelstring[1] = hex_lo_hi(datalen);
	intelstring[2] = hex_lo_lo(datalen);
	checksum += datalen;

	intelstring[3] = hex_hi_hi(loadaddress);
	intelstring[4] = hex_hi_lo(loadaddress);
	intelstring[5] = hex_lo_hi(loadaddress);
	intelstring[6] = hex_lo_lo(loadaddress);
	checksum += hi_byte(loadaddress);
	checksum += lo_byte(loadaddress);

	intelstring[7] = hex_lo_hi(type);
	intelstring[8] = hex_lo_lo(type);
	checksum += type;

	for(i=9,j=0; j<datalen; i+=2,j++){
		intelstring[i]   = hex_lo_hi(data[j]);
		intelstring[i+1] = hex_lo_lo(data[j]);
		checksum += data[j];
	}
	checksum = -checksum;
	intelstring[i+0] = hex_lo_hi(checksum);
	intelstring[i+1] = hex_lo_lo(checksum);
	intelstring[i+2] = '\r';
	intelstring[i+3] = '\n';
	intelstring[i+4] = '\0';
}

static char hextable[] = "0123456789ABCDEF";

unsigned
hex_hi_hi(n)
unsigned n;
{
	return hextable[ (hi_byte(n)>>4)&0xf ];
}

unsigned
hex_hi_lo(n)
unsigned n;
{
	return hextable[ hi_byte(n)&0xf ];
}

unsigned
hex_lo_hi(n)
unsigned n;
{
	return hextable[ (lo_byte(n)>>4)&0xf ];
}

unsigned
hex_lo_lo(n)
unsigned n;
{
	return hextable[ lo_byte(n)&0xf ];
}

unsigned
hi_byte(n)
unsigned n;
{
	return ((n>>8) & 0xff);
}

unsigned
lo_byte(n)
unsigned n;
{
	return (n & 0xff);
}

void
ABORT(char *reason, ...)
{
	va_list argptr;

	va_start(argptr, reason);
	fputs("ERROR: ",stdout);
	vfprintf(stdout, reason, argptr);
	va_end(argptr);

	remove_temp_file();
	how_to_use_message();		/* which also exits program */
}

void
remove_temp_file()
{
	if(tmpfname){
		fclose(fpt);
		remove(tmpfname);
	}
}
	
void
how_to_use_message()
{
	fprintf(stdout,"\n");
	fprintf(stdout,"EXEHEX - This program converts Microsoft .EXE files\n");
	fprintf(stdout,"         into Intel HEX files suitable for burning\n");
	fprintf(stdout,"         into EPROM.\n\n");
	fprintf(stdout,"EXEHEX:  [-D][-Sxxxx][-Oxxxx][-Lxx][-N][-H][-?] infile [outfile]\n");
	fprintf(stdout,"where -Sxxxx sets the start segment to xxxx.\n");
	fprintf(stdout,"      -Oxxxx sets the start offset  to xxxx.\n");
	fprintf(stdout,"      -Lxx   sets number of bytes/line to xx.\n");
	fprintf(stdout,"      -N     prevents the sending of an end record\n");
	fprintf(stdout,"             (useful when making the jump vector)\n");
	fprintf(stdout,"      -D     turns on some debugging messages (should be first).\n");
	fprintf(stdout,"      -Exxxx EPROM starting segment (for dumb PROM burners)\n");
	fprintf(stdout,"      -?     sends this message\n");
	fprintf(stdout,"      -H     sends this message\n\n");
	fprintf(stdout,"the default options are -S0000 -O0000 -L20\n");
	fprintf(stdout,"(All numbers must be in hex, segments in paragraphs)\n");
	exit(1);
}
