/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: memmgr.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for specialised memory management routines.
*				Two systems are provided. The MemStack provides fast
*				and efficient mark and release style memory management
*				(note that destructors are _not_ called by this type of
*				memory management). The MemBlocks is a specialisation of
*				the MemStack class that maintains a free list of all
*				freed blocks for quick reallocation, but requires all
*				objects allocated to be of the same size.
*
* $Id: memmgr.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__MEMMGR_HPP
#define	__MEMMGR_HPP

#ifndef	__TECHLIB_HPP
#include "techlib.hpp"
#endif

#ifndef	__STDDEF_H
#include "stddef.h"
#endif

/*--------------------------- Class Definitions ---------------------------*/

//---------------------------------------------------------------------------
// The following class manages a stack of arbitrary sized blocks, allocating
// memory from a list of large sized blocks for efficiency. Fast mark
// and release memory management can be performed using the MemMarker class.
//
// Note that blocks can only be allocated, not freed using the MemStack. The
// allocate memory will be freed when a MemMarker object goes out of
// scope.
//
// Note that when the memory is freed by the MemMarker, no destructors will
// be called, so use with care.
//---------------------------------------------------------------------------

class MemStack {
protected:
	struct BlockList {
		BlockList	*next;			// Pointer to next block in list
		}		*curBlock;			// List of memory pool blocks
	size_t		blockSize;			// Size of memory pool blocks
	uint		blockCount;			// Number of memory pool blocks allocated
	size_t		curLoc;				// Location within current pool block

			// Return a char * to start of current memory pool
			char *poolStart()	{ return (char*)(curBlock+1); };

			// Allocate a new memory pool block
			bool allocatePoolBlock();

			// Free all pool blocks from 'block' to current maximum
			void freeFrom(uint block);

public:
			// Constructor given size of pool to allocate
			MemStack(size_t size = 8192)
				: blockSize(size), blockCount(0),
				  curBlock(NULL), curLoc(size)
				{ CHECK(size != 0); };

			// Destructor - free's all memory
			~MemStack()	{ freeFrom(0); };

			// Allocate a block from the pool of specified size
			void *allocate(size_t size);

private:
	friend	class MemMarker;
	};

//---------------------------------------------------------------------------
// The following class is used to mark the current position in a MemStack.
// When the MemMarker goes out of scope or is deleted, all memory allocated
// after the MemMarker was constructed will be automatically freed.
//---------------------------------------------------------------------------

class MemMarker {
protected:
	uint		block;			// Number of currently allocated pool block
	size_t		curLoc;			// Location within current pool block
	MemStack&	mem;			// MemStack begin marked

public:
			// Constructor
			MemMarker(MemStack& mem)
				: mem(mem), block(mem.blockCount), curLoc(mem.curLoc) {};

			// Destructor
			~MemMarker()
			{	PRECONDITION(block < mem.blockCount ||
					(block == mem.blockCount && curLoc <= mem.curLoc));
				mem.freeFrom(block);
				mem.curLoc = curLoc;
			};
	};

//---------------------------------------------------------------------------
// The following class manages a number of fixed sized blocks very
// efficiently using a simple free list mechanism. Classes can use this
// class internally to overload the default memory management for speed
// (be careful with derived classes however!).
//---------------------------------------------------------------------------

class MemBlocks : public MemStack {
protected:
	void	*freeList;				// Head of the free list
	uint	size;					// Size of allocated blocks

public:
			// Constructor given the size and number of blocks to allocate
			MemBlocks(uint size,uint count = 100)
				: MemStack(size * count), size(max(size,sizeof(void*))),
				  freeList(NULL)
				{ CHECK(size != 0 && count != 0); };

			// Method to allocate a new block
			void *allocate()
			{	if (freeList == NULL)
					return MemStack::allocate(size);
				else {
					void *temp = freeList;
					freeList = *(void **)temp;
					return temp;
					}
			};

			// Method to free a used block
			void free(void *p)
			{	*(void **)p = freeList;
				freeList = p;
			};
	};

/*--------------------------- Inline Functions ----------------------------*/

//---------------------------------------------------------------------------
// Overloaded operator new to allocate memory from a memory stack
// (rather than having to call MemStack::allocate() directly
//---------------------------------------------------------------------------

inline void *operator new(size_t size,MemStack& m)
{
	return m.allocate(size);
}

#endif	// __MEMMGR_HPP
