/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:     $RCSfile: list.cpp $
* Version:      $Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Member functions for the list class, a class designed to
*				link a series of objects together into a singly linked
*				list. All items placed in the list MUST be derived from
*				the class ListNode.
*
* $Id: list.cpp 1.1 1994/03/09 12:18:03 kjb release $
*
****************************************************************************/

#include "list.hpp"

/*--------------------------- Member functions ----------------------------*/

// We need to instantiate the static member variable's of a class outside
// of their definitions, or they will not be able to link.

int (*GenList::cmp)(ListNode*,ListNode*);

GenList::GenList()
/****************************************************************************
*
* Function:		GenList::GenList
*
* Description:	Constructor for the GenList class. We set the count of items
*				in the GenList to 0, and initialise the head and tail pointers
*				for the GenList. These both point to dummy nodes within the
*				GenList class, so that the dummy tail node always points to
*				itself ensuring we cannot iterate of the end of the GenList.
*				This also simplifies maintaining the head and tail pointers.
*
****************************************************************************/
{
	count = 0;
	head = &hz[0];				// Setup head and tail pointers
	z = &hz[1];
	head->next = z->next = z;
}

GenList::~GenList()
/****************************************************************************
*
* Function:		GenList::~GenList
*
* Description:	Destructor for the GenList class. All we do here is ask the
*				GenList to empty itself.
*
****************************************************************************/
{
	empty();
}

void GenList::empty(void)
/****************************************************************************
*
* Function:		GenList::empty
*
* Description:	Empties the GenList of all elements. We do this by stepping
*				through the GenList deleting all the elements as we go.
*
****************************************************************************/
{
	ListNode *temp;

	while (head->next != z) {
		temp = head->next;
		head->next = head->next->next;
		delete temp;
		}
	count = 0;
}

ListNode* GenList::merge(ListNode *a,ListNode *b,ListNode*& end)
/****************************************************************************
*
* Function:		GenList::merge
* Parameters:	a,b		- Sublist's to merge
*				end		- Pointer to end of merged list
* Returns:		Pointer to the merged sublists.
*
* Description:	Merges two sorted lists of nodes together into a single
*				sorted list, and sets a pointer to the end of the newly
*				merged lists.
*
****************************************************************************/
{
	ListNode	*c;

	// Go through the lists, merging them together in sorted order

	c = z;
	while (a != z && b != z) {
		if (cmp(a,b) <= 0) {
			c->next = a; c = a; a = a->next;
			}
		else {
			c->next = b; c = b; b = b->next;
			}
		};

	// If one of the lists is not exhausted, then re-attach it to the end
	// of the newly merged list

	if (a != z) c->next = a;
	if (b != z) c->next = b;

	// Set end to point to the end of the newly merged list

	while (c->next != z) c = c->next;
	end = c;

	// Determine the start of the merged lists, and reset z to point to
	// itself

	c = z->next; z->next = z;
	return c;
}

void GenList::sort(int (*cmp_func)(ListNode*,ListNode*))
/****************************************************************************
*
* Function:		GenList::sort
* Parameters:	cmp	- Function to compare the contents of two ListNode's.
*
* Description:	Mergesort's all the nodes in the list. 'cmp' must point to
*				a comparison function that can compare the contents of
*				two ListNode's. 'cmp' should work the same as strcmp(), in
*				terms of the values it returns.
*
****************************************************************************/
{
	int			i,N;
	ListNode	*a,*b;		// Pointers to sublists to merge
	ListNode	*c;			// Pointer to end of sorted sublists
	ListNode	*todo;		// Pointer to sublists yet to be sorted
	ListNode	*t;			// Temporary

	// Set up globals required by List::merge().

	GenList::cmp = cmp_func;

	for (N = 1,a = z; a != head->next; N = N + N) {
		todo = head->next; c = head;
		while (todo != z) {

			// Build first sublist to be merged, and splice from main list

			a = t = todo;
			for (i = 1; i < N; i++) t = t->next;
			b = t->next; t->next = z; t = b;

			// Build second sublist to be merged and splice from main list

			for (i = 1; i < N; i++) t = t->next;
			todo = t->next; t->next = z;

			// Merge the two sublists created, and set 'c' to point to the
			// end of the newly merged sublists.

			c->next = merge(a,b,t); c = t;
			}
		}
}

SimpleGenList::~SimpleGenList()
/****************************************************************************
*
* Function:		SimpleGenList::~SimpleGenList
*
* Description:	Destructor for the SimpleList class. All we do here is ask
* 				the SimpleList to empty itself.
*
****************************************************************************/
{
	empty();
}

void SimpleGenList::empty(void)
/****************************************************************************
*
* Function:		SimpleGenList::empty
*
* Description:	Empties the SimpleList of all elements. We do this by
*				stepping through the SimpleList deleting all the elements
*				as we go.
*
****************************************************************************/
{
	ListNode *temp;

	while (head != NULL) {
		temp = head;
		head = head->next;
		delete temp;
		}
}

ulong SimpleGenList::numberOfItems(void) const
/****************************************************************************
*
* Function:		SimpleGenList::numberOfItems
* Returns:		Number of items in the list.
*
* Description:	Since we do not maintain a count for the list, we simply
*				whiz through the list counting the number of items in it.
*
****************************************************************************/
{
	ulong	count = 0;

	for (SimpleGenListIterator i(*this); i; i++)
		count++;

	return count;
}

// Virtual destructor for ListNode's. Does nothing.

ListNode::~ListNode()
{
}
