/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: cstack.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for a creating typed stack classes, with the
*				objects in a contiguous array.
*
* $Id: cstack.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__CSTACK_HPP
#define	__CSTACK_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

#ifndef	__ERROR_HPP
#include "error.hpp"
#endif

/*--------------------------- Class Definition ----------------------------*/

extern void	(*errorHandler)(int err);

//---------------------------------------------------------------------------
// Template class to create a downward growing stack class that maintains
// elements of a particular type (and only this type) in a contiguous array.
// The following operations are supported on the stack:
//
//	StackName(int size)	- Construct a stack of a given size
//	top()				- Examine the top item on the stack
//	element(int)		- Examine the nth item from the top of the stack
//	push(elementType&)	- Push an element onto the stack
//	pushq(elementType&)	- Push an element with no range checking
//	pop()				- Pop the top element from the stack
//	popq()				- Pop the top element with no range checking
//	popn(int)			- Pop n items from the stack, returning top element
//	popqn(int)			- Pop n items from stack with no range checking
//	empty()				- Empty the stack of all elements
//	numberOfItems()		- Return the number of items on the stack
//	isEmpty()			- Returns true if stack is empty
//	isFull()			- Returns true if stack is full
//
//---------------------------------------------------------------------------

template <class T> class CStack {
protected:
	int			size;
	T			*stack;
	T			*p_stack;

public:
			CStack(int size)
			{
				stack = new T[size];
				CStack<T>::size = size;
				if (valid()) empty();
			};
			~CStack()				{ delete stack; };
			T& top() const			{ return *p_stack; };
			T& element(int offset)	{ return *(p_stack + offset - 1); };
			void push(T& node)
				{ isFull() ? errorHandler(STK_OVERFLOW) : pushq(node); };
			void pushq(T& node)		{ *--p_stack = node; };
			T& pop()
			{
				if (isEmpty()) errorHandler(STK_UNDERFLOW);
					return popq();
			};
			T& popq()				{ return *p_stack++; };
			T& popn(int n)
			{
				if (numberOfItems() < n) errorHandler(STK_UNDERFLOW);
					return popqn(n);
			};
			T& popqn(int n)			{ return (p_stack += n)[-n]; };
			void empty()			{ p_stack = stack + size; };
			uint numberOfItems() const
				{ return size - (p_stack - stack); };
			bool isEmpty() const
				{ return p_stack >= (stack + size); };
			bool isFull() const		{ return p_stack <= stack; };
			bool valid() const		{ return p_stack != NULL; };
	};

#endif	// __CSTACK_HPP
