/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: fudge.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for defining a number of macros for performing
*				fuzzy arithmetic on single and double precision
*				floating point values.
*
*				To avoid namespace clashes, all inline functions defined
*				in this header are in uppercase. They can be used to
*				replace previous macro versions of the functions (in fact
*				these are automatically #undef'd to avoid problems).
*
* $Id: fudge.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__FUDGE_HPP
#define	__FUDGE_HPP

/*------------------------- Internal Macro's etc --------------------------*/

// Undefine any possible macros that may already be defined to avoid
// complications.

#undef	FZERO
#undef	FEQ
#undef	FLE
#undef	FGE
#undef	FNE
#undef	FLT
#undef	FGT
#undef	LERP

// Define macros versions of the routines which are quicker when the inline
// function calls get deeply nested (the optimiser does a better job since
// the arguments of an inline function call must be fully evaluated before
// the next call). However the inline function versions are completely
// side effect free.

#define	_FZERO(a,FUDGE)	((a) < FUDGE && (a) > -FUDGE)

#define	_FEQ(a,b,FUDGE)	_FZERO((a)-(b),FUDGE)
#define	_FLE(a,b,FUDGE)	(((a) < (b)) || _FEQ(a,b,FUDGE))
#define	_FGE(a,b,FUDGE)	(((a) > (b)) || _FEQ(a,b,FUDGE))

#define	_FNE(a,b,FUDGE)	(!_FZERO((a)-(b),FUDGE))
#define	_FLT(a,b,FUDGE)	(((a) < (b)) && _FNE(a,b,FUDGE))
#define	_FGT(a,b,FUDGE)	(((a) > (b)) && _FNE(a,b,FUDGE))
#define	_LERP(lo,hi,alpha)	((lo) + ((hi)-(lo))*(alpha))
#define	_SIGN(a)		((a) > 0 ? 1 : -1)

/*--------------------------- Implementation ------------------------------*/

template <class T> inline int FZERO(const T& a,const T& FUDGE)
{
	return _FZERO(a,FUDGE);
}

template <class T> inline int FEQ(const T& a,const T& b,const T& FUDGE)
{
	return _FEQ(a,b,FUDGE);
}

template <class T> inline int FLE(const T& a,const T& b,const T& FUDGE)
{
	return _FLE(a,b,FUDGE);
}

template <class T> inline int FGE(const T& a,const T& b,const T& FUDGE)
{
	return _FGE(a,b,FUDGE);
}

template <class T> inline int FNE(const T& a,const T& b,const T& FUDGE)
{
	return _FNE(a,b,FUDGE);
}

template <class T> inline int FLT(const T& a,const T& b,const T& FUDGE)
{
	return _FLT(a,b,FUDGE);
}

template <class T> inline int FGT(const T& a,const T& b,const T& FUDGE)
{
	return _FGT(a,b,FUDGE);
}

template <class T> inline T LERP(const T& lo,const T& hi,const T& alpha)
{
	return _LERP(lo,hi,alpha);
}

#endif	// __FUDGE_HPP
