/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: str.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for a the String, DynStr and SubStr classes.
*				The DynStr class is a class for representing dynamically
*				sized strings, and the SubStr class is a class for
*				representing finite sub strings that index directly into
*				an allocated dynamic string or a normal C style string.
*
*				To avoid lots of data copying you can set the granularity
*				for a DynStr to a large value, say 1k rather than the
*				default 8 bytes. Since the granularity is global to all
*				string's, you should reset the granularity to the previous
*				value when you are finished. ie:
*
*				myfunc()
*				{
*					uint	old_granularity = String::setGranularity(1024);
*
*					... do stuff with 1k strings ...
*
*					String::setGranularity(old_granularity);
*				}
*
*				Remember that if you convert a SubStr to a (const char *)
*				the string may not necessarily be null terminated, so
*				you need to only look at 'length()' number of characters.
*
* $Id: str.hpp 1.1 1994/03/09 12:17:58 kjb release $
*
****************************************************************************/

#ifndef	__STR_HPP
#define	__STR_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

#ifndef	__IOSTREAM_H
#include "iostream.h"
#endif

#ifndef	__STRING_H
#include <string.h>
#endif

#ifndef	__LIMITS_H
#include <limits.h>
#endif

/*------------------------ Compatability Definitions ----------------------*/

#ifndef	__MSDOS__

void strupr(char *s);
void strlwr(char *s);

#endif

/*--------------------------- Class Definitions ---------------------------*/

class SubStr;
class DynStr;

//---------------------------------------------------------------------------
// The String class is an abstract base class for the dynamically allocated
// string class and substring class.
//
// The only methods and operations available on the base string class
// are ones that do not modify the internal representation.
//---------------------------------------------------------------------------

class String {
protected:
	uint		len;				// Length of the string
	char		*text;				// Pointer to the string data

public:
			// Returns the length of the string
			uint length() const		{ return len-1; };

			// Overloaded cast to a (const char *)
			operator const char * () const	{ return text; };

			// Retrieve the ith character from the string
			char operator [] (uint i) const
				{ return i > len ? '\0' : text[i]; };

			// Equality operator
	friend	int operator == (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) == 0; };
	friend	int operator == (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) == 0; };
	friend	int operator == (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) == 0; };

			// Inequality operator
	friend	int operator != (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) != 0; };
	friend	int operator != (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) != 0; };
	friend	int operator != (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) != 0; };

			// Less than operator
	friend	int operator < (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) < 0; };
	friend	int operator < (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) < 0; };
	friend	int operator < (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) < 0; };

			// Greater than operator
	friend	int operator > (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) > 0; };
	friend	int operator > (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) > 0; };
	friend	int operator > (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) > 0; };

			// Less than or equal to operator
	friend	int operator <= (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) <= 0; };
	friend	int operator <= (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) <= 0; };
	friend	int operator <= (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) <= 0; };

			// Greater than or equal to operator
	friend	int operator >= (const String& s1,const String& s2)
				{ return strncmp(s1,s2,MIN(s1.length(),s2.length())) >= 0; };
	friend	int operator >= (const String& s1,const char* s2)
				{ return strncmp(s1,s2,s1.length()) >= 0; };
	friend	int operator >= (const char *s1,const String& s2)
				{ return strncmp(s1,s2,s2.length()) >= 0; };

			// Convert routines for strings
			int asInt() const;
			short asShort() const	{ return asInt(); };
			long asLong() const;
			uint asUInt() const;
			ushort asUShort() const	{ return asUInt(); };
			ulong asULong() const;
			float asFloat() const;
			double asDouble() const;

			// Split the string into an array of substrings
			uint split(SubStr*& array,const char *fs) const;

			// Compute a hash value for the string
			uint hash() const;

			// Returns true if the string is valid
			bool valid() const		{ return text != NULL; };
	};

//---------------------------------------------------------------------------
// The DynString class is a class for maintaining dynamically sized strings.
// Storage for the string is maintained dynamically on the heap.
//
// Note that the string class is maintained as a 'C' style string
// internally, with an integer count.
//---------------------------------------------------------------------------

class DynStr : public String {
protected:
	uint		size;				// Amount of memory allocated

	static uint	granularity;		// Granularity of memory allocation

			uint computeSize(uint len)
			{
				return ((len + granularity) / granularity) * granularity;
			};

public:
			// Enumerations for the string class
	enum	{ left, center, right };

			// Default constructor
			DynStr();

			// Copy constructor
			DynStr(const DynStr& str);

			// Constructor given another string and counts etc
			DynStr(const String& str,uint pos = 0,uint count = (uint)UINT_MAX);

			// Constructor using a normal C type string
			DynStr(const char *cstr,uint pos = 0,uint count = (uint)UINT_MAX);

			// Constructor given a character and a count value
			DynStr(char c,uint count = 1U);

			// Assignment operator given a String
			DynStr& operator = (const String& str);

			// Assignment operator given a C type string
			DynStr& operator = (const char *cstr);

			// Destructor
			~DynStr();

			// Convert the string to all upper case
			DynStr& toupper()		{ strupr(text); return *this; };

			// Convert the string to all lower case
			DynStr& tolower()		{ strlwr(text); return *this; };

			// Substring deletion method
			DynStr& del(uint pos,uint count = (uint)UINT_MAX,
				bool shrink = true);

			// Substring Insertion method given another string
			DynStr& insert(uint pos,const String& str);

			// Substring Insertion method given a C type string
			DynStr& insert(uint pos,const char *cstr);

			// Character Insertion method
			DynStr& insert(uint pos,const char ch);

			// Substring replacement method given another string
			DynStr& replace(uint pos,const String& str);

			// Substring replacement method given a C type string
			DynStr& replace(uint pos,const char *cstr);

			// Character replacement method
			DynStr& replace(uint pos,const char ch);

			// Justify the string according to the mode
			DynStr& justify(int mode,uint len,bool clip = false,
				bool shrink = true);

			// Trim the string of leading/trailing characters
			DynStr& trim(int mode = center,const char ch = 0,
				bool shrink = true);

			//---------------------------------------------------------------
			// Fast efficient operators (mimimum copying).
			//---------------------------------------------------------------

			// Concatenation operators
			DynStr& operator += (const String& str);
			DynStr& operator += (const char *str);
			DynStr& operator += (const char);

			// Replicate a string a number of times
			DynStr& operator *= (uint count);

			//---------------------------------------------------------------
			// Slower but more flexible operators (lots of construction etc).
			//---------------------------------------------------------------

			// Concatenation operators
	friend	DynStr operator + (const String& s1,const String& s2);
	friend	DynStr operator + (const String& s1,const char *s2);
	friend	DynStr operator + (const char *s1,const String& s2);

			// Replicate a string a number of times
	friend	DynStr operator * (const String& s1,uint count);

			// Shrink the string to the minimum memory allocation
			void shrink();

			// Member functions to set and obtain the granularity
	static	uint setGranularity(uint gran)
				{ uint old = granularity; granularity = gran; return old; };
	static	uint getGranularity()
				{ return granularity; };

			// Method to dump a DynStr to a stream
	friend	ostream& operator << (ostream& o,const DynStr& s)
				{ return o << (const char *)s; };
	};

//---------------------------------------------------------------------------
// The SubStr class is a class for maintaining sub strings that are indexed
// into the storage of an allocated dynamic string, or a c style string.
// No storage is owned by the class, so none is deleted when it is killed.
//
// Thus the class does not allow you to modify the string it represents.
//---------------------------------------------------------------------------

class SubStr : public String {
public:
			// Default constructor
			SubStr();

			// Constructor given another string
			SubStr(const String& str,uint pos = 0,uint count = (uint)UINT_MAX);

			// Constructor using a normal C type string
			SubStr(const char *cstr,uint pos = 0,uint count = (uint)UINT_MAX);

			// Assignment operator given a String
			SubStr& operator = (const String& str);

			// Assignment operator given a C type string
			SubStr& operator = (const char *cstr);

			// Overloaded cast to a DynString
			operator DynStr () const;

			// Convert to leftmost count characters
			SubStr& left(uint count);

			// Convert to middle count characters starting at pos
			SubStr& mid(uint pos,uint count);

			// Convert to rightmost count characters
			SubStr& right(uint count);

			// Method to dump a SubStr to a stream
	friend	ostream& operator << (ostream& o,const SubStr& s);
	};

#endif	// __STR_HPP
