/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: str.cpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Member functions for the string class, an abstract base
*				class for representing strings.
*
* $Id: str.cpp 1.1 1994/03/09 12:18:03 kjb release $
*
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "str.hpp"
#include "techlib.hpp"

//------------------------- Compatability Functions -----------------------//

#ifndef	__MSDOS__

#include <ctype.h>

void strupr(char *s)
{
	for (int i = strlen(s); i; i--) {
		*s = toupper(*s);
		s++;
		}
}

void strlwr(char *s)
{
	for (int i = strlen(s); i; i--) {
		*s = tolower(*s);
		s++;
		}
}

#endif

//----------------------------- Member functions --------------------------//

uint String::split(SubStr*& array,const char *fs) const
/****************************************************************************
*
* Function:		String::split
* Parameters:	array	- Pointer to array of substrings to fill
*				fs		- Set of field separator characters
* Returns:		Number of strings in array.
*
* Description:	Breaks the string up into a number of substrings separated
*				by characters in the field separator string. No memory
*				is actually allocated for the data in the strings, since
*				the substrings will index directly into the original
*				dynamic string (so don't kill the original string before
*				doing something with them).
*
*				If memory allocation fails during the operation, 'array'
*				will be set to NULL.
*
****************************************************************************/
{
	char	*p = text;
	SubStr	*temp;
	int		num,j;
	uint	pos,count;

	// Determine the number of tokens to be parsed

	CHECK(valid());
	num = 0;
	while ((j = strcspn(p,fs)) != 0) {
		p += strspn(p += j,fs);
		num++;
		}

	// Allocate memory for the array

	if ((temp = new SubStr[num]) == NULL)
		goto Done;

	for (j = 0, pos = 0; j < num; j++) {
		count = strcspn(text + pos,fs);
		temp[j] = SubStr(*this,pos,count);
		pos += count;
		pos += strspn(text + pos,fs);
		}

Done:
	array = temp;
	return num;
}

#define	NBITS_IN_UNSIGNED		( NBITS(uint) )
#define SEVENTY_FIVE_PERCENT	((int)(NBITS_IN_UNSIGNED * .75))
#define	TWELVE_PERCENT			((int)(NBITS_IN_UNSIGNED * .125))
#define	HIGH_BITS				( ~( (unsigned)(~0) >> TWELVE_PERCENT) )

uint String::hash() const
/****************************************************************************
*
* Function:		String::hash
* Returns:		Hash value for the String.
*
* Description:	Calculates the hash value for a string. This is done using
*				a shift-and-XOR strategy. The main iteration of the loop
*				shifts the accumulated hash value to the left by a few bits
*				and adds in the current character. When the number gets too
*				large, it is randomized by XORing it with a shifted version
*				of itself. Note that we set up a few macros to determine the
*				number of bits to shift independant of the machine word
*				size.
*
****************************************************************************/
{
	uint	h = 0,g;
	int		i;

	CHECK(valid());
	for (i = 0; i < length(); i++) {
		h = (h << TWELVE_PERCENT) + text[i];
		if ( (g = h & HIGH_BITS) != 0)
			h = (h ^ (g >> SEVENTY_FIVE_PERCENT)) & ~HIGH_BITS;
		}

	return h;
}

/****************************************************************************
*
* The following conversion routines work correctly, given that the strings
* will always be null terminated. If the string is a SubStr, which
* indexes into a DynStr, then we may possibly be saving the null terminator
* and putting it back.
*
****************************************************************************/

int String::asInt() const
/****************************************************************************
*
* Function:		String::asInt
* Returns:		Representation of the string as an integer
*
* Description:	Converts the string to an integer. If the conversion fails,
*				the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	int	result = atoi(text);
	text[length()] = save;
	return result;
}

uint String::asUInt() const
/****************************************************************************
*
* Function:		String::asUInt
* Returns:		Representation of the string as an unsigned integer
*
* Description:	Converts the string to an unsigned integer. If the
*				conversion fails, the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	uint result;
	if (sscanf(text,"%u\n",&result) != 1)
		result = 0;
	text[length()] = save;
	return result;
}

long String::asLong() const
/****************************************************************************
*
* Function:		String::asLong
* Returns:		Representation of the string as a long integer
*
* Description:	Converts the string to an integer. If the conversion fails,
*				the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	long result = atol(text);
	text[length()] = save;
	return result;
}

ulong String::asULong() const
/****************************************************************************
*
* Function:		String::asULong
* Returns:		Representation of the string as an unsigned long
*
* Description:	Converts the string to an unsigned long. If the
*				conversion fails, the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	ulong result;
	if (sscanf(text,"%lu\n",&result) != 1)
		result = 0;
	text[length()] = save;
	return result;
}

float String::asFloat() const
/****************************************************************************
*
* Function:		String::asFloat
* Returns:		Representation of the string as a float
*
* Description:	Converts the string to a floating point number. If the
*				conversion fails, the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	float result = atof(text);
	text[length()] = save;
	return result;
}

double String::asDouble() const
/****************************************************************************
*
* Function:		String::asDouble
* Returns:		Representation of the string as a double
*
* Description:	Converts the string to a floating point number. If the
*				conversion fails, the value will be set to zero.
*
****************************************************************************/
{
	CHECK(valid());
	char save = text[length()];
	double result;
	if (sscanf(text,"%lf\n",&result) != 1)
		result = 0;
	text[length()] = save;
	return result;
}
