/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: array.cpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Implementation for the set of Array clases, implemented
*				using templates. Defines the method implementation following
*				classes:
*
*					Array			- Array of T's
*					SArray			- Sortable array of T's
*					IArray			- Indirect array (array of pointers to T)
*					ISArray			- Indirect sortable array
*
*					ArrayIterator	- Iterator for array's
*					IArrayIterator	- Iterator for indirect array's
*
* $Id: array.cpp 1.1 1994/03/09 12:17:58 kjb release $
*
****************************************************************************/

#include "array.hpp"

/*---------------------------- Implementation -----------------------------*/

template <class T> Array<T>::Array()
	: _size(1), _delta(1), count(0)
/****************************************************************************
*
* Function:		Array<T>::Array
*
* Description:	Default constructor for the array template. This crates
*				an empty array of size 1, with an increment of 1.
*
****************************************************************************/
{
	data = new T[_size];
	CHECK(data != NULL);
}

template <class T> void Array<T>::Array(uint size,uint delta)
	: _delta(delta), count(0)
/****************************************************************************
*
* Function:		Array<T>::Array
* Parameters:	size	- Initial size of the array
*				delta	- Initial increment value for the array
*
* Description:	Constructor for the array template, given the size of the
*				array to create, and the expansion ratio. If the increment
*				is zero, the array is of a fixed size and will not be
*				expanded.
*
*				The array is initially empty containing no valid data.
*
****************************************************************************/
{
	_size = computeSize(size);
	data = new T[_size];
	CHECK(data != NULL);
}

template <class T> Array<T>::Array(const Array<T>& a,uint delta)
	: count(a.count)
/****************************************************************************
*
* Function:		Array<T>::Array
* Parameters:	a	- Array to copy
*
* Description:	Copy constructor for array's. We simply allocate space for
*				the new array, and copy each item individually. We copy
*				the data with a fast memcpy.
*
****************************************************************************/
{
	PRECONDITION(a.data != NULL);
	_delta = (delta == UINT_MAX) ? a._delta : delta;
	_size = computeSize(a._size);
	data = new T[_size];
	CHECK(data != NULL);
	memcpy(data,a.data,count * sizeof(T));
}

template <class T> const Array<T>& Array<T>::operator = (const Array<T>& a)
/****************************************************************************
*
* Function:		Array<T>::operator =
* Parameters:	a	- Array to copy
*
* Description:	Assignment operator for array's. Allocates space for the
*				new array and copies the data with a fast memcpy call.
*
****************************************************************************/
{
	if (data != a.data) {
		_size = a._size;
		_delta = a._delta;
		count = a.count;
		delete [] data;
		data = new T[_size];
		CHECK(data != NULL);
		memcpy(data,a.data,count * sizeof(T));
		}
	return *this;
}

template <class T> bool Array<T>::operator == (const Array<T>& a)
/****************************************************************************
*
* Function:		Array<T>::operator ==
* Parameters:	a	- Array to compare to this array
* Returns:		True if the array elements are all equal, false if not.
*
****************************************************************************/
{
	if (count != a.count)
		return false;
	T *p = data, *pa = a.data;
	for (uint i = 0; i < count; i++)
		if (!(*p++ == *pa++))
			return false;
	return true;
}

template <class T> void Array<T>::insert(T item,uint index,uint aCount)
/****************************************************************************
*
* Function:		Array<T>::insert
* Parameters:	item	- Item to insert into the array
*				index	- Index to insert the item in front of
*				aCount	- Count of elements to insert
*
* Description:	Inserts the specified item into the array at the index'th
*				position. All the elements from [index,count] are moved
*				up one position to make room. The index must be in the
*				range [0,count], and if the value is count it is simply
*				tacked onto the end of the array.
*
****************************************************************************/
{
	PRECONDITION(index <= count);
	if (count+aCount > _size)	expand(count+aCount);

	// Move the items up one position in the array to make room, and insert

	memmove(&data[index+aCount],&data[index],(count-index) * sizeof(T));
	for (uint i = 0; i < aCount; i++)
		data[index+i] = item;
	count += aCount;
}

template <class T> void Array<T>::remove(uint index)
/****************************************************************************
*
* Function:		Array<T>::remove
* Parameters:	index	- Index of element to remove
*
* Description:	Removes an indexed element from the array, by copying all
*				the data down one position. The index must be in the
*				range [0,count).
*
****************************************************************************/
{
	PRECONDITION(index < count);

	// Move the items down one position, and shrink the allocated memory

	count--;
	memmove(&data[index],&data[index+1],(count-index) * sizeof(T));
	shrink();
}

template <class T> uint Array<T>::search(T item,uint first,uint last,
	int direction) const
/****************************************************************************
*
* Function:		Array<T>::search
* Parameters:	item		- Item to search for
*				first		- Index of first element to search
*				last		- Index of last element to search
*				direction	- End to search array from (+1 = start, -1 = end)
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a simple linear search for the item from the
*				specified end of the array.
*
****************************************************************************/
{
	PRECONDITION(first < count && last < count);
	PRECONDITION(direction == +1 || direction == -1);
	if (direction == +1) {
		T *p = &data[first];
		for (uint i = first; i <= last; i++)
			if (*p++ == item)
				return i;
		}
	else {
		T *p = &data[last];
		for (uint i = last; i >= first; i--)
			if (*p-- == item)
				return i;
		}
	return UINT_MAX;
}

template <class T> void Array<T>::expand(uint newSize)
/****************************************************************************
*
* Function:		Array<T>::expand
* Parameters:	newSize	- New size of the array
*
* Description:	Expands the array to be a multiple of 'delta' that includes
*				the specified newSize for the array. The array data is
*				re-allocated and copied to the resized array.
*
*				Note that 'count' must contain the actual number of elements
*				in the array.
*
****************************************************************************/
{
	PRECONDITION(_delta != 0);	// Array is not resizeable when _delta == 0
	PRECONDITION(newSize >= count);

	T *temp = new T[_size = computeSize(newSize)];
	CHECK(temp != NULL);

	// Now copy the data from the old array into the newly resized array.
	// Note that we use a fast memcpy to do this.

	memcpy(temp,data,count * sizeof(T));
	delete [] data;
	data = temp;
}

template <class T> void Array<T>::shrink()
/****************************************************************************
*
* Function:		Array<T>::shrink
*
* Description:	Shrinks the allocated space for the array, if the threshold
*				point has been reached.
*
****************************************************************************/
{
	if (_delta == 0)		// Array is not resizeable when _delta == 0
		return;

	// Only shrink the array when the amount of free space gets smaller
	// than half of the delta value, and it is at least delta in size

	if ((_size - count) > (_delta + _delta/2) && (_size > _delta)) {
		T *temp = new T[_size = computeSize(_size - _delta)];
		CHECK(temp != NULL);
		memcpy(temp,data,count * sizeof(T));
		delete [] data;
		data = temp;
		}
}

template <class T> void Array<T>::setDelta(uint delta)
/****************************************************************************
*
* Function:		Array<T>::setDelta
* Parameters:	delta	- New delta value for the array
*
* Description:	Sets the delta value for the array, expanding or shrinking
*				the array as need be.
*
****************************************************************************/
{
	if (delta >= _delta) {
		_delta = delta;
		expand(_size);
		}
	else {
		_delta = delta;
		shrink();
		}
}

template <class T> void Array<T>::resize(uint newSize)
/****************************************************************************
*
* Function:		Array<T>::resize
* Parameters:	newSize	- New size for the array
*
* Description:	Resizes the array to the new size. If the array is non-
*				resizeable, we bomb out. Note that the array will be empty
*				after this operation.
*
****************************************************************************/
{
	PRECONDITION(_delta != 0);
	empty();
	expand(newSize);
}

template <class T> void Array<T>::empty()
/****************************************************************************
*
* Function:		Array<T>::empty
*
* Description:	Empties the array of all elements.
*
****************************************************************************/
{
	count = 0;
	shrink();
}

template <class T> int SArray<T>::cmp(const void *t1,const void *t2)
/****************************************************************************
*
* Function:		SArray<T>::cmp
* Parameters:	t1,t2	- Elements to compare
* Returns:		Result of comparision:
*
*					t1 < t2,  -1
*					t1 == t2, 0
*					t1 > t2,  1
*
****************************************************************************/
{
	if (*((T*)t1) < *((T*)t2))
		return -1;
	else if (*((T*)t1) > *((T*)t2))
		return 1;
	return 0;
}

template <class T> void SArray<T>::addSorted(T item)
/****************************************************************************
*
* Function:		SArray<T>::addSorted
* Parameters:	item	- Item to add to the array
*
* Description:	Adds the element to the array in sorted order. This
*				function will only work if the elements are already in
*				sorted order, which can be achieved by calling sort().
*
****************************************************************************/
{
	// Search for the spot to put the new item, and insert it into the
	// array.

	T *p = data;
	for (uint i = 0; i < count; i++)
		if (*p++ > item)
			break;
	insert(item,i);
}

template <class T> uint SArray<T>::binarySearch(T item,uint L,uint R) const
/****************************************************************************
*
* Function:		SArray<T>::binarySearch
* Parameters:	item	- Item to search for in the array
*				L		- Index of first element to search
*				R		- Index of last element to search
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a standard binary search on the array looking
*				for the specified item. The elements in the array _must_
*				be in sorted order for this function to work (by calling
*				the sort() member function).
*
****************************************************************************/
{
	PRECONDITION(L < count && R < count);

	while (L < R) {
		uint M = (L+R)/2;
		if (data[M] == item)
			return M;
		if (data[M] < item)
			L = M+1;
		else R = M-1;
		}
	if (data[L] == item)
		return L;
	return UINT_MAX;
}

template <class T> const IArray<T>& IArray<T>::operator = (const IArray<T>& a)
/****************************************************************************
*
* Function:		IArray<T>::operator =
* Parameters:	a	- Array to copy
*
* Description:	Assignment operator for IArray's. First empties the array
*				the copies it.
*
****************************************************************************/
{
	// Check to make sure we are not being assigned to ourselves :-)

	if (data != a.data) {
		empty();
		shouldDelete = false;
		Array<T*>::operator=(a);
		}
	return *this;
}

template <class T> bool IArray<T>::operator == (const IArray<T>& a)
/****************************************************************************
*
* Function:		IArray<T>::operator ==
* Parameters:	a	- Array to compare to this array
* Returns:		True if the array elements are all equal, false if not.
*
****************************************************************************/
{
	if (count != a.count)
		return false;
	T **p = data, **pa = a.data;
	for (uint i = 0; i < count; i++)
		if (!(*(*p++) == *(*pa++)))
			return false;
	return true;
}

template <class T> void IArray<T>::replace(T* item,uint index)
/****************************************************************************
*
* Function:		IArray<T>::replace
* Parameters:	item	- Item to replace in the array
*				index	- Index of the item in the array to replace
*
* Description:	Replaces the item in the array with the new item. The index
*				MUST fall within the current bounds of the array.
*
****************************************************************************/
{
	PRECONDITION(index < count);
	if (shouldDelete)
		delete data[index];
	data[index] = item;
}

template <class T> void IArray<T>::destroy(uint index)
/****************************************************************************
*
* Function:		IArray<T>::destroy
* Parameters:	index	- Index of element to remove
*
* Description:	Removes an indexed element from the array, by copying all
*				the data down one position. The index must be in the
*				range [0,count). If shouldDelete is true, the element is
*				deleted.
*
****************************************************************************/
{
	PRECONDITION(index < count);

	// Move the items down one position, and shrink the allocated memory

	if (shouldDelete)
		delete data[index];
	count--;
	memmove(&data[index],&data[index+1],(count-index) * sizeof(T*));
	shrink();
}

template <class T> uint IArray<T>::search(const T* item,uint first,uint last,
	int direction) const
/****************************************************************************
*
* Function:		IArray<T>::search
* Parameters:	item		- Item to search for
*				first		- Index of first element to search
*				last		- Index of last element to search
*				direction	- End to search array from (+1 = start, -1 = end)
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a simple linear search for the item from the
*				specified end of the array.
*
****************************************************************************/
{
	PRECONDITION(first < count && last < count);
	PRECONDITION(direction == +1 || direction == -1);
	if (direction == +1) {
		T **p = &data[first];
		for (uint i = first; i <= last; i++)
			if (*(*p++) == *item)
				return i;
		}
	else {
		T **p = &data[last];
		for (uint i = last; i >= first; i--)
			if (*(*p--) == *item)
				return i;
		}
	return UINT_MAX;
}

template <class T> void IArray<T>::resize(uint newSize)
/****************************************************************************
*
* Function:		IArray<T>::resize
* Parameters:	newSize	- New size for the array
*
* Description:	Resizes the array to the new size. If the array is non-
*				resizeable, we bomb out. Note that the array will be empty
*				after this operation.
*
****************************************************************************/
{
	PRECONDITION(_delta != 0);
	empty();
	expand(newSize);
}

template <class T> void IArray<T>::empty()
/****************************************************************************
*
* Function:		IArray<T>::empty
*
* Description:	Deletes all of the elements if shouldDelete is set to true
*				(the default).
*
****************************************************************************/
{
	if (shouldDelete) {
		for (uint i = 0; i < count; i++)
			delete (T*)data[i];
		}
	Array<T*>::empty();
}

template <class T> int ISArray<T>::cmp(const void *t1,const void *t2)
/****************************************************************************
*
* Function:		ISArray<T>::cmp
* Parameters:	t1,t2	- Elements to compare
* Returns:		Result of comparision:
*
*					t1 < t2,  -1
*					t1 == t2, 0
*					t1 > t2,  1
*
****************************************************************************/
{
	if (**((T**)t1) < **((T**)t2))
		return -1;
	else if (**((T**)t1) > **((T**)t2))
		return 1;
	return 0;
}

template <class T> void ISArray<T>::addSorted(T* item)
/****************************************************************************
*
* Function:		ISArray<T>::addSorted
* Parameters:	item	- Item to add to the array
*
* Description:	Adds the element to the array in sorted order. This
*				function will only work if the elements are already in
*				sorted order, which can be achieved by calling sort().
*
****************************************************************************/
{
	// Search for the spot to put the new item, and insert it into the
	// array.

	T **p = data;
	for (uint i = 0; i < count; i++)
		if (*(*p++) > *item)
			break;
	insert(item,i);
}

template <class T> uint ISArray<T>::binarySearch(const T* item,uint L,
	uint R) const
/****************************************************************************
*
* Function:		ISArray<T>::binarySearch
* Parameters:	item	- Item to search for in the array
*				L		- Index of first element to search
*				R		- Index of last element to search
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a standard binary search on the array looking
*				for the specified item. The elements in the array _must_
*				be in sorted order for this function to work (by calling
*				the sort() member function).
*
****************************************************************************/
{
	PRECONDITION(L < count && R < count);

	while (L < R) {
		uint M = (L+R)/2;
		if (*data[M] == *item)
			return M;
		if (*data[M] < *item)
			L = M+1;
		else R = M-1;
		}
	if (*data[L] == *item)
		return L;
	return UINT_MAX;
}

