/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#define JED_VERSION "Jed <0.80> "

#include <stdio.h>
#include <string.h>
#ifndef msdos
#ifndef VMS
#include <malloc.h>
#endif
#else
#include <alloc.h>
#include <dos.h>
extern int Cursor_Row;
#endif
#include "buffer.h"
#include "screen.h"
#include "window.h"
#include "paste.h"

#include "ins.h"
#include "ledit.h"
#include "display.h"
#include "sysdep.h"
#include "misc.h"
#include "vterm.h"
#include "slang.h"

Screen_Type Screen[80];

int Screen_Row = 1;
int Screen_Col = 1;
int Cursor_Motion;    /* indicates cursor movement only */
int Scroll_Region_1;
int Scroll_Region_2;
int Scroll_Lines;

int Goal_Column;
int Display_Eight_Bit = 0;

void blank_line(int row)
{
    int i, n;
    char *p;

    n = Window->width;
    p = Screen[row].old;
    i = 0;
    while(i++ < n) *p++ = ' ';
    *p = 0;
}

void update_screen_txt(int row)
{
    char *tmp;
    tmp = Screen[row].new;
    Screen[row].new = Screen[row].old;
    Screen[row].old = tmp;
}

int output(unsigned char *line, int len, int row, int max_col)
{
   int i,ii, visible = 0, ok, quit, tabsize = 8, count, tab, mini_flag,
     mark, ebit;
   unsigned char ch, *out;

   out = (unsigned char *) Screen[row - 1].new;
   tab = User_Vars.tab;
   i = 0;
   ii = 0;
   quit = 0;
   ok = 0;  /* 1 if ok to start filling the out line */

   count = 0;  /* count counts the visible space */

   /* deal with mini_buffer prompt */
   if (IS_MINIBUFFER && Mini_Info.prompt_len)
     {
	if (Screen_Width < Mini_Info.prompt_len) 
	  {
	     len = 0; 
	     count = Screen_Width;
	  }
	else count = Mini_Info.prompt_len;
	mini_flag = 1;
     }
   else mini_flag = 0;

   while (ch = line[i], (i < len) && !quit)  /* && ch != '\n' */
     {
	ebit = 0;
	if (tab && (ch == '\t'))
	  {
	     tabsize = tab * (count / tab + 1) - count;
	     count += tabsize;
	  }
	else if ((ch < ' ') || (ch == 127)) count += 2;
	else if (Display_Eight_Bit || (ch < 127)) count++;
	else if (ch < 160) count += 3;
	else if (ch < 255) count += 2;
	else count += 3;

	if (count >= Window->column) ok = 1;
	if (count > max_col) break;

	i++;
	if (!ok) continue;

	if (mini_flag)
	  {
	     strncpy((char *) out, Mini_Info.prompt, Screen_Width);
	     out[Screen_Width] = 0;
	     visible += (Screen_Width > Mini_Info.prompt_len ? Mini_Info.prompt_len : Screen_Width);
	     ii = visible;
	     mini_flag = 0;
	  }

	if (!Display_Eight_Bit && (ch & 0x80))
	  {
	     out[ii++] = '~';
	     visible += 1;
	     ch = ch & 0x7F;
	     ebit = 1;
	  }

	if (!ebit && (ch == '\t') && tab)
	  {
	     while(tabsize--) out[ii++] = ' ';
	  }

	else if ((ch < ' ') || (ch == 127))
	  {
	     out[ii++] = '^';
	     if (ch == 127) out[ii++] = '?'; else out[ii++] = ch + 'A' - 1;
	     visible += 2;
	  }
	else if (ch == ' ') out[ii++] = ' ';
	else
	  {
	     out[ii++] = ch;
	     visible += 1;
	  }
     }

   /* we have not inserted the prompt */
   if (mini_flag && !len)
     {
	strncpy((char *) out, Mini_Info.prompt, Screen_Width);
	out[Screen_Width] = 0;
	ok = 1;
	ii = visible = (Screen_Width > Mini_Info.prompt_len ? Mini_Info.prompt_len : Screen_Width);
     }
   mark = -1;
   if (ok && count >= max_col)
     {
	out[ii - 1] = '$';
	visible++;
     }

   else
     {
	if (!ii)
	  {
	     out[ii++] = '$';
	     visible += 1;
	  }

	mark = ii;
	while(ii < Window->width) out[ii++] = ' ';
     }

   out[ii] = '\0';

   if (visible)
     {
#ifdef msdos
	(void) mark;
	smart_puts((char *) out,Screen[row-1].old, row);
#else
	if (mark != -1) out[mark] = 0;
	smart_puts((char *)out,Screen[row-1].old, row);
	if (mark != -1) out[mark] = ' ';
#endif
	Screen[row-1].n = visible;
     }

   /* else take care of it in calling routine */
   return(visible);
}

void display_line(Line *line, int row)
{
    int len;

    if (line != NULL)
      {
          len = line->len;
          if (len) if (line->data[len - 1] == '\n') len--;

          if ((len > 0) || (Window->column > 1)
	       || ((row == Screen_Height) && Mini_Info.prompt_len))
            {
	       len = output(line->data, len, row, Window->column + Window->width - 1);
            }
      }
    else len = 0;

    if (len <= 0)
      {
          if (Screen[row-1].n)
            {
                goto_rc(row,1);
                tt_del_eol();
            }
          blank_line(row - 1);
          Screen[row - 1].n = 0;
      }
    Screen[row - 1].line = line;
    Screen[row - 1].flags = 0;
    if (len > 0) update_screen_txt(row-1);
}

void open_scroll(void)
{
    Scroll_Region_1 = 1;
    Scroll_Region_2 = Screen_Height;
    Scroll_Lines = 0;
}

void do_scroll_up(int n)
{
    goto_rc(1,1);
    tt_delete_nlines(n);
}

void do_scroll_down(int n)
{
   goto_rc(1,1);
   reverse_index(n);
}

void execute_scroll(void)
{
   if (Scroll_Lines > 0)
     {
	set_scroll_region(Scroll_Region_1, Scroll_Region_2);
	do_scroll_up(Scroll_Lines);
     }
   else if (Scroll_Lines < 0)
     {
	set_scroll_region(Scroll_Region_1, Scroll_Region_2);
	do_scroll_down(-Scroll_Lines);
     }
   Scroll_Lines = 0;
}

void queue_scroll(int n, int r1, int r2)
{
   if ((r1 != Scroll_Region_1) || (r2 != Scroll_Region_2))
     {
	if ((n == -1) && (Scroll_Region_1 == r1 + Scroll_Lines)
	       && (Scroll_Region_2 == r2))
	Scroll_Lines--;
	else
	  {
	     execute_scroll();
	     Scroll_Region_1 = r1;
	     Scroll_Region_2 = r2;
	     Scroll_Lines = n;
	  }
     }
   else Scroll_Lines += n;
}

void close_scroll(void)
{
    if (Scroll_Lines) execute_scroll();
    if ((Scroll_Region_1 != 1) | (Scroll_Region_2 != Screen_Height))
      reset_scroll_region();
}

/* Here a scrolling region (t,b) is used to scroll line t + n to t.  All
   that is assumed is that  the line at t + n exists! */

int scroll_up(int n, int t, int b)
{
    int i, necess;

    if (n == 0) return(0);

    /* t = Window->top - 1 + t;
       b = Window->top - 1 + b; */

    /* check to see if this is really necessary */
    necess = 0;
    for (i = t - 1; i < b; i++)
      {
          if (Screen[i].n)
            {
                necess = 1;
                break;
            }
      }

    if (!necess) return(0);

    queue_scroll(n, t, b);

    vscroll_up(t,b,n);

    return(necess);
}

/* Here a scrolling region (t,b) is used to scroll line t + n to t. */

void scroll_down(int n, int t, int b)
{
    if (n == 0) return;

    /* t = Window->top - 1 + t;
       b = Window->top - 1 + b; */
    set_scroll_region(t,b);
    reverse_index(n);
    reset_scroll_region();

    vscroll_down(t,b,n);
}

int update_insert_line(int r1, Line *line)
{
    int i, r2, r, necess;
    Line *bot;

    /* normalize r1: */
    /* r1 = Window->top - 1 + r1; */

    /* find the first one that is blank to delimit the region so that the
       loss will be minimal */

    r = r1;   /* not r1 + 1 as obvious (but naive) as it seems */
    r2 = Window->rows + Window->top - 1;
    while (r < r2)
      {
          if (Screen[r - 1].line == NULL) break;
          r++;
      }

    if ((r1 != r2) && (r == r2))
    /* we may have failed so check the bottom up so we don't push linesoff. */
     {
	bot = line;
	for (r = r1 + 1; r <= r2; r++)
	  {
	     bot = bot->next;
	     if (bot == NULL) break;
	     if (Screen[r-1].line == bot) break;
	  }
	if ((bot != NULL) && (r <= r2)) r--;
	if (r > r2) r = r2;
     }
   
   if (r < r1) r = r1;
   r2 = r;

    /* check to see if we gain by doing this */
    necess = 0;
    for (i = r1 - 1; i < r2; i++)
      {
          if (Screen[i].n)
            {
                necess = 1;
                break;
            }
      }
    if (r1 == r2) return(0);
    if (!necess) return(0);

    queue_scroll(-1, r1, r2);

    vscroll_down(r1, r2, 1);
    return(1);
}

Line *find_top()
{
    int n, i;
    Line *line, *next, *prev, *this;

    if (Window->rows == 1) return(CLine);
    /* Check the top window line.  Chances are that one of the lines
     above CLine will match in usual situations */
    line = Screen[Window->top - 1].line;
    prev = CLine;

    if (line != NULL) for (i = 0; i < Window->rows; i++)
      {
          if (prev == line) return(line);
          this = prev->prev;
          if (this == NULL) break;
          else prev = this;
      }

    /*  That was the obvious choice now try some others */
    /* try bottom */
    line = Screen[Window->rows + Window->top - 2].line;
    next = CLine;

    if (line != NULL) for (i = 0; i < Window->rows; i++)
      {
          if (next == line)
            {
                line = CLine;
                while(i < Window->rows - 1)
                  {
                      prev = line->prev;
                      if (prev == NULL) return(line);
                      line = prev;
                      i++;
                  }
                return(line);
            }
          next = next->next;
          if (next == NULL) break;
      }
    /* try to find CLine somewhere in the window */

    next = CLine->next;
    prev = CLine->prev;

    for (i = Window->top - 1; i < Window->top - 1 + Window->rows; i++)
      {
         this = Screen[i].line;
         if (this == CLine)
           {
              line = CLine;
              while(i > Window->top - 1)
                {
		    i--;
		    line = line->prev;
		    if (line == NULL) return(CLine);
                }

              return(line);
           }
         else if ((this == prev) && (this != NULL))
           {
               i--;
               while(i-- > Window->top - 1)
                 {
                     if (prev->prev == NULL) return(prev);
                     prev = prev->prev;
                 }
               return(prev);
           }
         else if ((this == next) && (this != NULL))
           {
               i++;
               while(i-- > Window->top - 1)
                 {
                     if (next->prev == NULL) return(next);
                     next = next->prev;
                 }
               return(next);
           }

      }

    /* not found so check neighbors */
    line = CLine->next;
    if ((line != NULL) && (line == Screen[Window->top - 1].line)) return(CLine);

    n = Window->rows - 1;
    line = CLine->prev;
    if ((line == NULL) || (line != Screen[n + Window->top - 1].line))
      {
         n = n / 2;
      }

    line = prev = CLine;
    while(n--)
      {
          line = line->prev;
          if (line == NULL) return(prev);
          prev = line;
      }

    return(line);
}

void do_scroll(Line *top)
{
    Line *bot, *middle;
    int r1, r, r2, middle_r, match, overwrite = 1, scroll_region_set = 0, srs;

    r2 = Window->top + Window->rows - 1;
    r1 = Window->top;

    open_scroll();
    if (top != NULL) bot = top->next; else bot = NULL;
    if (bot != NULL) for (r = r1 + 1; r <= r2; r++)
      {
          if (bot == Screen[r2 - 1].line)  /* was - 2 */
            {
                overwrite = 0;
                break;
            }
          if (bot == NULL) break;
          bot = bot->next;
      }

    middle = NULL;
    if (overwrite)
      {
          r = 1;
          bot = top;
          middle_r = Window->rows / 2;
          while (r++ < Window->rows)
            {
	       if (r == middle_r) middle = bot;
	       if (bot != NULL) bot = bot -> next;
            }
      }

    /* if the middle or the bottom lines match, we do not overwrite */

    if ((bot != NULL) || (middle != NULL)) for (r = r1; r <= r2; r++)
      {
	 if (((middle == Screen[r - 1].line) && (middle != NULL))
	     || ((bot != NULL) && (bot == Screen[r - 1].line)))
	   {
	      overwrite = 0;
	      break;
	   }
      }

    while(r1 <= r2)
      {
          if (top == NULL) break;
          match = 0;
          for(r = r1; r <= r2; r++)
            {
                if (Screen[r - 1].line == top)
                  {
                      if (r != r1)
                        {
                            scroll_region_set = 0;
                            scroll_up(r - r1, r1, r2);
                        }

                      r1++;
                      match = 1;
                      if (top != NULL) top = top->next;
                      break;
                  }
            }

          if (match) continue;

          /* not found so insert it */
          if ((top != NULL) && (!overwrite))
            {
                srs = update_insert_line(r1, top);
                if (!scroll_region_set) scroll_region_set = srs;
            }
          if (top != NULL) top = top->next;
          r1++;
      }

   /*     if (scroll_region_set) reset_scroll_region(); */
   close_scroll();
   if (overwrite) touch_window();
}

void point_column(int n)
{
    unsigned char *p, *pmax, ch;
    int i, tab;

    if (IS_MINIBUFFER) n -= Mini_Info.prompt_len;

    p = CLine->data;

    pmax = p + (CLine->len - 1);
    if (*pmax != '\n') pmax++;

    tab = User_Vars.tab;
    i = 0;
    n--;   /* start at 0 */
    while(p < pmax)
      {
         if (tab && (*p == '\t'))
	   {
	      i = tab * (i / tab + 1);
	   }
	 else
	   {
	      if (!Display_Eight_Bit && (*p & 0x80)) i++;
	      if (!Display_Eight_Bit) ch = *p & 0x7F; else ch = *p;
	      if ((ch < ' ') || (ch == 127)) i += 2; else i++;
	   }

	 if (i > n) break;
	 p++;
      }
    Point = (int) (p - CLine->data);
}

/* given a position in a line, return apparant distance from bol
   expanding tabs, etc... up to pos */
int calculate_column()
{
    int i, tab;
    unsigned char ch, *pos, *this_pos;

    pos = CLine->data;
    this_pos = pos + Point;
    i = 1;
    tab = User_Vars.tab;
    while(ch = *pos, pos++ < this_pos)
      {
	 if (tab && ch == '\t')
	   {
	      i = tab * ((i - 1)/tab + 1) + 1;  /* tab column tabs */
	   }
	 else
	   {
	      if (!Display_Eight_Bit && (ch & 0x80)) i++;
	      if (!Display_Eight_Bit) ch = ch & 0x7F;
	      if (ch < ' ' || ch == 127) i += 2; else i++;
	   }
      }

    if (IS_MINIBUFFER) i += Mini_Info.prompt_len;

    Screen_Col = i;
    return (i);
}

void point_cursor()
{
    int r, row, c = 1;
    Line *this;

    r = 0;

    for (row = Window->top; row < Window->top + Window->rows; row++)
      {
          this = Screen[row-1].line;
          if (this == NULL) break;
          if (this == CLine)
            {
                r = row;
                break;
            }
      }

    if (Point >= CLine->len)
      {
	  Point = CLine->len - 1;

	  if (Point < 0) Point = 0;
	  else if (*(CLine->data + Point) != '\n') Point++;
      }

    if (r)
      {
          c = calculate_column();
          c = c - Window->column + 1;
          if (c < 1) c = 1; else if (c > Window->width) c = Window->width;
          goto_rc(r , c);
          Screen_Row = r;
          Screen_Col = c;
      }
    else
      msg_error("CLine not found in Window!");

    if (!Cursor_Motion) Goal_Column = c;
}

void make_status_line(Window_Type *w, int do_percent)
{
    int i, p;
    char *s, pstr[10], mrk, flag, spot;
    Buffer *buf;

    if (w->top == Screen_Height) return;   /* minibuffer ? */
    buf = CBuf;
    CBuf = w->buffer;
    if (Screen[w->top - 1].line == CBuf->beg)
      {
	  if (Screen[w->top - 2 + w->rows].line == NULL)
	    strcpy(pstr," (All)");
	  else strcpy(pstr," (Top)");
      }

    else if (Screen[w->top - 2 + w->rows].line == NULL)
      strcpy(pstr, " (Bot)");
    else
      {
	  if (*Error_Buffer || do_percent)
	    p = get_percent();
	  else p = w->percent;
	  sprintf(pstr, " (%d%%)", p);
	  w->percent = p;
      }

   s = Screen[w->rows + w->top - 1].new;
   if (CBuf->marks != NULL) mrk = 'm'; else mrk = '-';
   if (CBuf->flags & FILE_MODIFIED) flag = 'd'; else flag = '-';
   if (CBuf->spots != NULL) spot = 's'; else spot = '-';

   if (CBuf->flags & BUFFER_TRASHED)
      sprintf(s,"-**%c%c%c : %s: %s ", mrk, flag, spot, JED_VERSION, CBuf->name);
   else sprintf(s,"---%c%c%c : %s: %s ", mrk, flag, spot, JED_VERSION, CBuf->name);

   if (CBuf->modes == C_MODE) strcat(s, "=(C)=");
   else if (CBuf->modes == WRAP_MODE) strcat(s, "=(Wrap)=");
   else if (CBuf->modes == LANG_MODE)
     sprintf(s + strlen(s), "=(%s)=", CBuf->mode_str);
   strcat(s, pstr);
   
   if (Defining_Keyboard_Macro) strcat(s, " [Macro]");
   if ((CBuf->beg1 != NULL) || (CBuf->end2 != NULL)) strcat (s, " [Narrow]");

   for (i = strlen(s); i < w->width; i++) s[i] = '-';
   s[w->width] = 0;
   CBuf = buf;
}

int update_status_line(Window_Type *w, int do_percent)
{
    int r;

    if (w->top == Screen_Height) return(0);   /* minibuffer ? */
    r = w->rows + w->top - 1;
    make_status_line(w, do_percent);
    if (!strncmp(Screen[r].old, Screen[r].new, Window->width)) return(0);

    tt_reverse_video();
    goto_rc(r + 1, 1);
    smart_puts(Screen[r].new, Screen[r].old, r + 1);
    tt_normal_video();
    update_screen_txt(r);
    Screen[r].n = 80;
    return(1);
}

/* if force then do update otherwise return 1 if update or 0 if not */
int update_1(Line *top, int force)
{
    int i;
    Line *old_beg, *new_beg;
    Window_Type *w;

    if (!force && (Executing_Keyboard_Macro || (Repeat_Factor != NULL) || (input_pending(2,0))))
      {
	 /* Window->trashed = 1; */
	 /* This messes up blink match and I don not think we need it if all
	    functions return the correct value.
	    */
	 return(0);
      }

    if (!force && !Window->trashed && (Window->next == Window))
      {
	 update_status_line(Window, 0);
	 return(1);
      }

    if (!Window->trashed && Cursor_Motion) return(1);

    w = Window;
    do
      {
	  old_beg = Window->beg.line;
	  if (top == NULL) top = find_top();
	  Window->beg.line = new_beg = top;

          /* scroll the screen to optimal location */
	  do_scroll(top);

	  for (i = Window->top - 1; i < Window->rows + Window->top - 1; i++)
	    {
	       /* the next line is really optional */
	       if (!force && input_pending(2,0)) break;

	       if ((Screen[i].line != top) || (Screen[i].flags))
		 {
		    display_line(top, i + 1);
		 }
	       if (top != NULL) top = top->next;
	    }

	  if (!force && input_pending(2,0))
	    {
	       while(Window != w) other_window();
	       Window->trashed = 1;  /* since cursor not pointed */
	       return(0);
	    }
	  else
	    {
	       if ((w == Window) && (old_beg != new_beg))
	         update_status_line(Window, 1);
	       else update_status_line(Window, 0);
	    }

	  Window->trashed = 0;

	  other_window();
	  top = NULL;
	  /* if (!Window->trashed) top = Window->beg.line; else  top = NULL; */

       } while(Window != w);
     return(1);
}

int update_minibuffer(int ghost)
{
   Window_Type *w;

   if (Executing_Keyboard_Macro) return(ghost);
   if (MiniBuffer != NULL)
     {
	w = Window;
	while (!IS_MINIBUFFER) other_window();
	if (*Message_Buffer) sleep(1);
	display_line(CLine, Screen_Height);
	while (w != Window) other_window();
	return(1);
     }
   else if (ghost && !*Error_Buffer && !*Message_Buffer)
     {
	display_line(NULL, Screen_Height);
	if (!Window->trashed) point_cursor();
	return(0);
     }
   return((*Message_Buffer) || (*Error_Buffer));
}

void do_dialog(char *b)
{
   int len = 0, dout, row = Screen_Height - 1;

   if (! *b)
     {
	if(!KeyBoard_Quit) return;
	b = "Quit!";
     }

   if (MiniBuffer)
     {
	len = Mini_Info.prompt_len;
	Mini_Info.prompt_len = 0;
     }

   dout = output((unsigned char *) b, strlen(b), Screen_Height, Screen_Width);
   if (MiniBuffer) Mini_Info.prompt_len = len;
   if ((b == Error_Buffer) || (KeyBoard_Quit))
     {
	beep();
	fflush(stdout);
	if (!KeyBoard_Quit) sleep(1);
	flush_input();
     }
   else fflush(stdout);
   if (!dout)
     {
	if (Screen[row].n)
	  {
	     goto_rc(Screen_Height,1);
	     tt_del_eol();
	  }
	blank_line(row);
	Screen[row].n = 0;
      }
   else update_screen_txt(row);
}

void update(Line *line)
{
   static int ghost;

   if (*Error_Buffer || KeyBoard_Quit)
     {
	do_dialog(Error_Buffer);
	KeyBoard_Quit = 0;
	Lang_Error = 0;
	ghost = 1;
	(void) update_1(line, 1);
	ghost = update_minibuffer(ghost);
     }
   else if (*Message_Buffer)
     {
	if (!update_1(line, 0))
	  {
	     /* *Message_Buffer = 0; */
	     return;
	  }
	ghost = 1;
	do_dialog(Message_Buffer);
	ghost = update_minibuffer(ghost);
     }
   else
     {
	ghost = update_minibuffer(ghost);
	if (!update_1(line, 0)) return;
     }
   point_cursor();
   *Error_Buffer = *Message_Buffer = 0;
}

/* search for the CLine in the SCreen and flag it as changed */
/* n = 0 means line was changed, n = 1 means it was destroyed */
void register_change(int n)
{
    int i;
    Window_Type *w;

    if (No_Screen_Update)
      {
	  No_Screen_Update = 0;
          if (((n == CINSERT) || (n == CDELETE)) && (Window->next == Window))
	    {
	       /* Since no screen update, we are probably safe to do: */
	       /* Screen[Screen_Row - 1].flags = 1; */
	       return;
	    }
	  w = Window->next;  /* skip this window */
      }
    else w = Window;

    do
      {
	  for (i = w->top - 1; i < w->top + w->rows - 1; i++)
	    {
		if (Screen[i].line == CLine)
		  {
		      Screen[i].flags = 1;
		      if ((n == NLDELETE) || (n == LDELETE)) Screen[i].line = NULL;
		  }
	    }
	  w = w->next;
      }
    while(w != Window);
}

void reset_display()
{
   int i;
#ifdef msdos
   int ab = Attribute_Byte;
   Attribute_Byte = 0x7;
#else
   fputs("\033[?6l",stdout);   /* normal origin mode */
#endif
   goto_rc(Screen_Height,1);
   tt_erase_line();
   fflush(stdout);
   for (i = 0; i < Screen_Height; i++)
     {
	FREE(Screen[i].old);
	FREE(Screen[i].new);
     }
#ifdef msdos
   Attribute_Byte = ab;
#endif
}

void init_display()
{
   int i, j;
   char *old, *new = NULL, ch;

   if (!Screen_Height) get_term_dimensions(&Screen_Width, &Screen_Height);
#ifndef msdos
   fputs("\033[?6h",stdout);   /* relative origin mode */
   reset_scroll_region();
   end_insert();
#endif
   for (i = 0; i < Screen_Height; i++)
     {
	if ((NULL == (old = (char *) MALLOC(Screen_Width + 1)))
	    || (NULL == (new = (char *) MALLOC(Screen_Width + 1))))
	  {
	     exit_error("init_display(): malloc error.");
	  }

	Screen[i].line = NULL;
	Screen[i].flags = 1;
	Screen[i].n = 0;
	Screen[i].old = old;
	Screen[i].new = new;
	ch = ' ';
	/* this is for the status line */
	/* if (i == Window->rows) ch = 1; */
	j = 0; while(j++ < Screen_Width) *old++ = ch;
     }
}

int redraw()
{
    int row, center;
    Window_Type *w;

    cls();
    for (row = 0; row < Screen_Height; row++)
      {
	 Screen[row].n = 0;
	 Screen[row].flags = 1;
	 blank_line(row);
      }
    w = Window;
    center = Window->trashed;
    do
      {
	 w->trashed = 1;
	 w = w->next;
      }
    while(w != Window);
    if (!center) update(Window->beg.line); else update((Line *) NULL);
    return(0);
}

void touch_window()
{
    int i;

    /* if (CBuf != Window->buffer) return; */
    for (i = 0; i < Window->rows; i++)
      {
	 Screen[i + Window->top - 1].flags = 1;
      }

    Window->trashed = 1;
}
