/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <string.h>
#ifndef msdos
#ifndef VMS
#include <malloc.h>
#endif
#else
#include <alloc.h>
#endif
#include "buffer.h"
#include "ins.h"
#include "line.h"
#include "paste.h"
#include "screen.h"
#include "misc.h"
#include "cmds.h"

Buffer *Paste_Buffer;
Buffer *Rectangle_Buffer;

int set_mark_cmd()
{
    Mark *m;

    if ((m = CBuf->marks) == NULL)
      if (NULL == (m = (Mark *) MALLOC(sizeof(Mark))))
        {
	   exit_error("set_mark_cmd: malloc error!");
        }

    m->line = CLine;
    m->point = Point;
    if (m != CBuf->marks) m->next = CBuf->marks;
    CBuf->marks = m;
    message("Mark Set.");
    return(1);
}

int push_spot()
{
    Mark *m;

    if (NULL == (m = (Mark *) MALLOC(sizeof(Mark))))
      {
	  exit_error("push_spot: malloc error!");
      }

    m->line = CLine;
    m->point = Point;
    m->next = CBuf->spots;
    CBuf->spots = m;
    return(1);
}

int push_mark()
{
    Mark *m;

    if (NULL == (m = (Mark *) MALLOC(sizeof(Mark))))
      {
          fprintf(stderr, "push_spot: malloc error!");
          return(0);
      }

    m->line = CLine;
    m->point = Point;
    m->next = CBuf->marks;
    CBuf->marks = m;
    return(0);
}

int pop_mark(int *go)
{
   Mark *m;
   m = CBuf->marks;
   if (m == NULL) return(0);

   if (*go)
     {
	CLine = m->line;
	Point = m->point;
     }

   CBuf->marks = m->next;
   FREE(m);
   return(1);
}

int mark_spot()
{
    push_spot();
    message("Spot Marked.");
    return(1);
}

int pop_spot()
{
    Mark *m;
    m = CBuf->spots;
    if (m == NULL) return(0);

    CLine = m->line;
    Point = m->point;
    CBuf->spots = m->next;
    FREE(m);
    return(1);
}

int exchange_point_mark(void)
{
   Line *tmp;
   int ptmp;

   if (CBuf->marks == NULL) return(0);

   tmp = CLine;
   ptmp = Point;
   CLine = CBuf->marks->line;
   Point = CBuf->marks->point;
   CBuf->marks->line = tmp;
   CBuf->marks->point = ptmp;
   return(1);
}

/* returns 0 if the mark is not set and gives error.  Exchanges point and mark
   to produce valid region.  A valid region is one with mark
   earlier in the buffer than point.  Always call this if using a region
   which reqires point > mark.  Also, push spot first then pop at end. */
int check_region(int *push)
{
   Line *beg;
   int pbeg;

   if (CBuf->marks == NULL)
     {
	msg_error("Set mark first.");
	return(0);
     }

   if (*push) push_spot();
   beg = CBuf->marks->line;
   pbeg = CBuf->marks->point;

   if (beg == CLine)
     {
	if (pbeg <= Point) return(1);
     }

   else
     {
	while((beg != NULL) && (beg != CLine)) beg = beg->next;
	if (beg == CLine) return(1);
     }

   exchange_point_mark();
   return(1);
}

/* char looking_at(void)
{
   unsigned char *p;

   p = CLine->data + Point;
   if (Point < CLine->len) return ((char) *p); else return(0);
}
*/

int widen_buffer(Buffer *b)
{
   Line *beg2, *end1;
   Buffer *c_buf = CBuf;
   
   if ((b->beg1 == NULL) && (b->end2 == NULL)) return(0);
   
   switch_to_buffer(b);
   end1 = CBuf->end1;
   beg2 = CBuf->beg2;

   /* patch up the region to what it was */
   if (end1 != NULL) if (end1 != CBuf->beg)
     {
	end1->next  = CBuf->beg;
	CBuf->beg->prev = end1;
     }
   
   if (beg2 != NULL) if (beg2 != CBuf->end)
     {
	beg2->prev  = CBuf->end;
	CBuf->end->next = beg2;
     }
   
   /* restore the ends */
   if (CBuf->beg1 != NULL) CBuf->beg = CBuf->beg1;
   if (CBuf->end2 != NULL) CBuf->end = CBuf->end2;
   
   /* check to see if a newline was deleted at end of narrowed region */
   if (CLine->next != NULL)
     {
	push_spot();
	eol();
	if (!looking_at("\n")) splice_line();
	pop_spot();
     }
   
   /* NULL the others... */
   CBuf->beg1 = CBuf->end1 = CBuf->beg2 = CBuf->end2 = NULL;
   switch_to_buffer(c_buf);
   return(1);
}
   
int widen()
{
   return widen_buffer(CBuf);
}
   

/* not really a region of points but a region of lines. */
int narrow_to_region()
{
   Line *beg;
   int tmpm;

   if (!check_region(&Number_One)) return(0);       /* spot pushed */

   push_spot();
   tmpm = 1; pop_mark(&tmpm);
   beg = CLine;
   pop_spot();

   /* are we narrow already? */
   if ((CBuf->beg1 != NULL) || (CBuf->end2 != NULL)) widen();

    /* save real buffer limits */
    CBuf->beg1 = CBuf->beg;
    CBuf->end2 = CBuf->end;

    if ((CBuf->end1 = beg->prev) == NULL)   /* beg of buffer */
      {
	 CBuf->end1 = NULL;  /* beg; */
      }

    if (NULL == (CBuf->beg2 = CLine->next))
      {
	 CBuf->beg2 = NULL; /* CLine; */
      }

    /* now the region */
    CBuf->beg = beg;
    CBuf->end = CLine;

    /* for consistency... */
    beg->prev = NULL;
    CLine->next = NULL;
    touch_window();

    /* now do this since it is possible to change these and possible free them
       which will be problem when we widen... */
    if (CBuf->beg1 == CBuf->beg) CBuf->beg1 = NULL;
    if (CBuf->end2 == CBuf->end) CBuf->end2 = NULL;
    pop_spot();
    return(1);
}

int yank()
{
    if (Paste_Buffer == NULL) return(0);
    insert_buffer(Paste_Buffer);
    return(1);
}

int copy_region_to_buffer(Buffer *b)
{
   int first_point, last_point, n, tmpm;
   Line *beg, *end, *new, *old, *first, *last;
   Buffer *save_buffer;

   if (!check_region(&Number_One)) return(0);  /* spot pushed */
   last = CLine;
   last_point = Point;

   tmpm = 1; pop_mark(&tmpm);
   if (b == CBuf)
     {
	msg_error("A buffer cannot be inserted upon itself.");
	pop_spot();
	return(0);
     }

   first = CLine;
   first_point = Point;

   save_buffer = CBuf;
   switch_to_buffer(b);

   old = first;
   end = beg = dup_line(first);
   while (old != last)
     {
	old = old->next;
	new = dup_line(old);
	end->next = new;
	new->prev = end;
	end = new;
     }
   end->next = NULL;

   /* now we have a linked list of new lines between beg and line.  Patch it
      into to buffer b */

   /* beginning line */
   if (beg == end) n = last_point - first_point;
   else n = beg->len - first_point;

   ins_chars(beg->data + first_point, n);

   /* We should be on at thye beginning of line now! */

   if (beg != end)
     {
	beg = beg->next;
	CLine->prev->next = beg;
	beg->prev = CLine->prev;
	end->next = CLine;
	CLine->prev = end;

	end->len = last_point;
	CLine = end;
	Point = 0; eol();
	if (!looking_at("\n")) splice_line();
     }
   switch_to_buffer(save_buffer);
   pop_spot();
   return(1);
}

int copy_to_pastebuffer()
{
   /* delete paste buffer */
   if (Paste_Buffer != NULL) delete_buffer(Paste_Buffer);
   Paste_Buffer = make_buffer();
   strcpy(Paste_Buffer->name, " <paste>");

   copy_region_to_buffer(Paste_Buffer);
   return(0);
}

int delete_region()
{
    int beg_point, tmpm;
    Line *beg, *end, *prev;

    if (!check_region(&Number_Zero)) return(0);

    push_spot();
    tmpm = 1; pop_mark(&tmpm);
    split_line();
    beg = CLine;
    beg_point = Point;

    pop_spot();
    split_line();
    end = CLine->next;

    while(CLine != beg)
      {
	 /* At some point, I want to fix the update to be more efficient */
	 update_marks(LDELETE, 1);
	 prev = CLine->prev;
	 free_line(CLine);
	 CLine = prev;
      }

    CLine = end->prev = beg;
    beg->next = end;
    Point = beg_point;
    splice_line();
    return(1);
}

int copy_region()
{
   copy_to_pastebuffer();
   return(0);
}

int kill_region()
{
   int tmpm = 1;
   /* need two marks for this one */
   push_spot();
   if (!pop_mark(&tmpm))
     {
	check_region(&Number_Zero);
	pop_spot();
	return(0);
     }
   push_mark();
   push_mark();
   pop_spot();

   copy_to_pastebuffer();
   delete_region();
   return(1);
}

static char *Rect_Error = "Rectangle has 0 width.";
int insert_rectangle()
{
   int c1;
   Line *rline;

   if (Rectangle_Buffer == NULL) return(0);

   c1 = calculate_column();
   rline = Rectangle_Buffer->beg;
   while (rline != NULL)
     {
	goto_column(&c1);
	ins_chars(rline->data, rline->len);
	if (CLine->next == NULL)
	  {
	     eol();
	     newline();
	  }
	else CLine = CLine->next;
	rline = rline->next;
     }
   return(1);
}

int open_rectangle()
{
   int c1, n, c2, tmpm;
   Line *save_line;

   if (!check_region(&Number_One)) return(0); /* push_spot(); performed */

   c1 = calculate_column();
   save_line = CLine;
   tmpm = 1; pop_mark(&tmpm);
   c2 = calculate_column();
   n = c2 - c1;
   if (n < 0)
     {
	n = -n;
	c1 = c2;
     }

   while(1)
     {
	goto_column(&c1);
	ins_char_n_times(' ', n);
	if (CLine == save_line) break;
	CLine = CLine->next;
     }
   pop_spot();
   return(1);
}

/* rectangle commands */
int copy_rectangle()
{
    Line *save_line, *line, *beg;
    int c1, c2, dc, tmp, tmpm;
    unsigned char *p1, *p2, *data;

    if (!check_region(&Number_One)) return(0);       /* spot pushed */
    /* delete Rectangle buffer */
    if (Rectangle_Buffer != NULL) delete_buffer(Rectangle_Buffer);

    Rectangle_Buffer = make_buffer();
    strcpy(Rectangle_Buffer->name, " <rect>");
    c2 = calculate_column();
    save_line = CLine;

    tmpm = 1; pop_mark(&tmpm);
    c1 = calculate_column();
    if (c1 == c2)
      {
	 msg_error(Rect_Error);
	 pop_spot();
	 return(0);
      }
    if (c1 > c2)
      {
	 tmp = c1;
	 c1 = c2;
	 c2 = tmp;
	 goto_column(&c1);
      }

    /* go through the region copying rectanglar blocks to Rectanglebuffer */
    dc = c2 - c1;
    line = beg = make_line1(dc);
    beg->prev = NULL;
    while (1)
      {
	 p1 = data = line->data;
	 p2 = data + dc;
	 while (p1 < p2) *p1++ = ' ';
	 line->len = dc;

	 if (c1 == goto_column1(c1))
	   {
	      p1 = CLine->data + Point;
	      (void) goto_column1(c2);
	      p2 = CLine->data + Point;

	      while(p1 < p2) *data++ = *p1++;
	   }
	 if (CLine == save_line) break;
	 CLine = CLine->next;

	 line->next = make_line1(dc);
	 line->next->prev = line;
	 line = line->next;
      }

    line->next = NULL;

    Rectangle_Buffer->line = Rectangle_Buffer->beg = beg;
    Rectangle_Buffer->end = line;
    Rectangle_Buffer->point = 0;

    pop_spot();
    return(0);
}

int kill_rectangle()
{
    Line *save_line, *line, *beg;
   int c1, c2, dc, tmp, n, tmpm;
   unsigned char *p1, *p2, *data;

   if (!check_region(&Number_One)) return(0);

   /* delete Rectangle buffer */
   if (Rectangle_Buffer != NULL) delete_buffer(Rectangle_Buffer);

   Rectangle_Buffer = make_buffer();
   strcpy(Rectangle_Buffer->name, " <rect>");
   c2 = calculate_column();
   save_line = CLine;

   tmpm = 1; pop_mark(&tmpm);
    c1 = calculate_column();
    if (c1 == c2)
      {
	 msg_error(Rect_Error);
	 pop_spot();
	 return(0);
      }
    if (c1 > c2)
      {
	 tmp = c1;
	 c1 = c2;
	 c2 = tmp;
	 goto_column(&c1);
      }

    /* go through the region copying rectanglar blocks to Rectanglebuffer */
    dc = c2 - c1;
    line = beg = make_line1(dc);
    beg->prev = NULL;
    while (1)
      {
	 p1 = data = line->data;
	 p2 = data + dc;
	 while (p1 < p2) *p1++ = ' ';
	 line->len = dc;

	 if (c1 == goto_column1(c1))
	   {
	      p1 = CLine->data + Point;
	      (void) goto_column1(c2);
	      p2 = CLine->data + Point;
	      Point = (int) (p1 - CLine->data);
	      n = (int) (p2 - p1);
	      while(n-- > 0)
	        {
		   *data++ = *p1;
		   del();
		}
	   }
	 if (CLine == save_line) break;
	 CLine = CLine->next;

	 line->next = make_line1(dc);
	 line->next->prev = line;
	 line = line->next;
      }

    line->next = NULL;

    Rectangle_Buffer->line = Rectangle_Buffer->beg = beg;
    Rectangle_Buffer->end = line;
    Rectangle_Buffer->point = 0;

    pop_spot();
    return(0);
}

int blank_rectangle()
{
   int c1, n, c2, pnt, tmpm;
   Line *save_line;
   unsigned char *p, *pmax;

   if (!check_region(&Number_One)) return(0); /* push_spot(); performed */

   c1 = calculate_column();
   save_line = CLine;
   tmpm = 1; pop_mark(&tmpm);
   c2 = calculate_column();
   n = c2 - c1;
   if (n < 0)
     {
	n = -n;
	c1 = c2;
     }

   while(1)
     {
	goto_column(&c1);
	pnt = Point;
	eol();
	if (Point - pnt > n) Point = pnt + n;
	pmax = CLine->data + Point;
	Point = pnt;
	p = CLine->data + Point;
	while (p < pmax) *p++ = ' ';
	register_change(0);
	if (CLine == save_line) break;
	CLine = CLine->next;
     }
   pop_spot();
   mark_buffer_modified(&Number_One);
   return(1);
}

