/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <string.h>
#include <setjmp.h>

#ifndef msdos
#ifndef VMS
#include <malloc.h>
#endif
#else
#include <alloc.h>
#endif
#include "buffer.h"
#include "cmds.h"
#include "paste.h"
#include "screen.h"
#include "window.h"
#include "text.h"
#include "ledit.h"
#include "keymap.h"
#include "sysdep.h"
#include "misc.h"
#include "display.h"
#include "file.h"
#include "slang.h"

jmp_buf Jump_Buffer;

typedef int (*Jed_fp)(void);

VOID *Last_Key_Function;
int *Repeat_Factor;                    /* repeats a key sequence if non null */

typedef struct Jed_Function_Type
  {
      char *name;
      Jed_fp f;
  }
Jed_Function_Type;

Jed_Function_Type Jed_Functions[] =
  {
      {"backward_delete_char", backward_delete_char_cmd},
      {"backward_delete_char_untabify", backward_delete_char_untabify},
      {"beep", beep},
      {"begin_macro", begin_keyboard_macro},
      {"bob", bob},
      {"bol", bol},
      {"brace_bra_cmd", brace_bra_cmd},
      {"brace_ket_cmd", brace_ket_cmd},
      {"capitalize_word", capitalize_word},
      {"center_line", center_line},
      {"copy_region", copy_to_pastebuffer},
      {"delete_char_cmd", delete_char_cmd},
      {"delete_word", delete_word},
      {"digit_arg", digit_arg},
      {"double_line", double_line},
      {"downcase_word", downcase_word},
      {"end_macro", end_keyboard_macro},
      {"enlarge_window", enlarge_window},
      {"eob", eob},
      {"eol_cmd", eol_cmd},
      {"evaluate_cmd", evaluate_cmd},
      {"exchange",exchange_point_mark},
      {"execute_macro", execute_keyboard_macro},
      {"exit_jed", exit_jed},
      {"exit_mini", exit_minibuffer},
      {"find_file", find_file},
      {"format_paragraph", text_format_paragraph},
      {"goto_column", goto_column_cmd},
      {"goto_line", goto_line_cmd},
      {"goto_match", goto_match},
      {"indent_line", indent_line},
      {"insert_file", insert_file_cmd},
      {"kbd_quit", kbd_quit},
      {"kill_buffer", kill_buffer},
      {"kill_line", kill_line},
      {"kill_region", kill_region},
      {"macro_query", macro_query},
      {"mark_spot", mark_spot},
      {"narrow", narrow_to_region},
      {"narrow_paragraph", narrow_paragraph},
      {"newline", newline},
      {"newline_and_indent", newline_and_indent},
      {"next_char_cmd", next_char_cmd},
      {"next_line_cmd", next_line_cmd},
      {"one_window", one_window},
      {"other_window", other_window},
      {"page_down", pagedown_cmd},
      {"page_up", pageup_cmd},
      {"pop_spot", pop_spot},
      {"previous_char_cmd", previous_char_cmd},
      {"previous_line_cmd", previous_line_cmd},
      {"quoted_insert", quoted_insert},
      {"redraw", redraw},
      {"replace", replace},
      {"save_buffers", save_some_buffers},
      {"scroll_left", scroll_left},
      {"scroll_right", scroll_right},
      {"search_backward", search_backward_cmd},
      {"search_forward", search_forward_cmd},
      {"self_insert_cmd", ins_char_cmd},
      {"set_mark_cmd", set_mark_cmd},
      {"split_window", split_window},
      {"switch_to_buffer", get_buffer},
      {"sys_spawn_cmd", sys_spawn_cmd},
      {"text_smart_quote", text_smart_quote},
      {"transpose_lines", transpose_lines},
      {"trim_whitespace", trim_whitespace},
      {"upcase_word", upcase_word},
      {"widen", widen},
      {"write_buffer", write_buffer},
      {"yank", yank},
      {(char *) NULL, NULL}
   };

Key_Type Global_Map[128];
Key_Type *Mini_Map;

#define MAX_KEYMAP_NAME_LEN 8
typedef struct KeyMap_List_Type
{
   char name[MAX_KEYMAP_NAME_LEN + 1];
   Key_Type *keymap;
}
KeyMap_List_Type;

#define MAX_KEYMAPS 10
KeyMap_List_Type KeyMap_List[MAX_KEYMAPS];   /* these better be inited to 0! */

int kbd_quit(void)
{
   KeyBoard_Quit = 1;
   msg_error("Quit!");
   return(0);
}

void add_keymap(char *name, Key_Type *map)
{
   int i;

   for (i = 0; i < MAX_KEYMAPS; i++)
     {
	if (KeyMap_List[i].keymap == NULL)
	  {
	     KeyMap_List[i].keymap = map;
	     strncpy(KeyMap_List[i].name, name, MAX_KEYMAP_NAME_LEN);
	     (KeyMap_List[i].name)[MAX_KEYMAP_NAME_LEN] = 0;
	     return;
	  }
     }
   msg_error("Keymap quota exceeded");
}

VOID *find_function(char *name)
{
   Jed_Function_Type *fp;

   fp = Jed_Functions;

   while((fp != NULL) && (fp->name != NULL))
     {
	if (!strcmp(fp->name, name)) return((VOID *) fp->f);
	fp++;
     }
    return(NULL);
}

#define upcase(ch) (((ch <= 'z') && (ch >= 'a')) ? ch & 0xDF : ch)
#define lowcase(ch) (((ch <= 'Z') && (ch >= 'A')) ? ch | 0x20 : ch)

/* convert things like "^A" to "\001" with ^@ going to 128 */
char *process_keystring(char *s)
{
    static char str[30], ch;
    int i;

    i = 0;
    while (*s != 0)
      {
          ch = *s++;
          if (*s) ch = upcase(ch);
          if (ch == '^')
            {
                ch = *s++;
                if (ch == 0)
                  {
                      str[i++] = '^';
                      break;
                  }
                ch = upcase(ch);
                if (ch == '?') ch = 127;
		else if (ch == '@') ch = 128; else ch = ch - 'A' + 1;
            }

          str[i++] = ch;
      }
    str[i] = 0;
    return(str);
}

Key_Type *malloc_key(unsigned char *str)
{
   Key_Type *new;

   if (NULL == (new = (Key_Type *) MALLOC(sizeof(Key_Type))))
     {
	exit_error("malloc_key: malloc error.");
     }
   strcpy(new->str, (char *) str);
   return(new);
}

static char *Define_Key_Error = "Inconsistency in define key.";

void define_key1(char *s, VOID *f, char type, Key_Type *keymap)
{
    int cmp, i, m, n, len;
    Key_Type *key, *last, *new;
    unsigned char *str;

    str = (unsigned char *) process_keystring(s);

    /* null char gets mapped to 128 */
    /* if (128 == ( i = *str)) i = 0; */
   i = (*str) & 0x7F;

    key = (Key_Type *) &(keymap[i]);
    n = strlen((char *) str);

    if (n < 2)
      {
          if (key->next != NULL)
	    {
	       msg_error(Define_Key_Error);
	       return;
	    }

	  key->f = f;
	  key->type = type;
          if (*str) strcpy(key->str, (char *) str); else *key->str = 0;
          return;
      }

    /* insert the key definition */
    while(1)
      {
          last = key;
          key = key->next;
          if ((key == NULL) || (key->str == NULL)) cmp = -1;
          else
            {
                m = strlen(key->str);
                len = m;
                if (m > n) len = n;
                cmp = strncmp((char *) str, key->str, len);
                if ((m != n) && !cmp)
                  {
		     msg_error(Define_Key_Error);
		     return;
                  }
                if ((m == n) && (!cmp)) cmp = strcmp((char *) str, key->str);
            }

          if (cmp == 0)
            {
	       key->f = f;
	       key->type = type;
	       return;
            }
          else if (cmp < 0)
            {
	       new = malloc_key(str);

	       new -> next = key;
	       last -> next = new;

	       new->f = f;
	       new->type = type;
	       return;
            } /* else cmp > 0 */
      }
}

void define_key(char *s, char *funct, Key_Type *keymap)
{
   VOID *f;
   char type;

   f = find_function(funct);
   if (f == NULL)                      /* assume interpreted */
     {
	if (NULL == (f = (VOID *) MALLOC(strlen(funct) + 1)))
	  {
	     exit_error("Malloc error in define key.");
	  }

	strcpy((char *) f, funct);
	type = F_INTERPRET;
     }
   else type = F_INTRINSIC;

   define_key1(s, f, type, keymap);
}

void init_keymaps(void)
{
    char simple[2], ch;
    simple[1] = 0;

    for (ch = ' '; ch < 127; ch++)
      {
          simple[0] = ch;
          define_key1(simple, (VOID *) ins_char_cmd, F_INTRINSIC, Global_Map);
      }

    define_key1("^[1", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[2", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[3", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[4", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[5", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[6", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[7", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[8", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[9", (VOID *) digit_arg, F_INTRINSIC, Global_Map);
    define_key1("^[0", (VOID *) digit_arg, F_INTRINSIC, Global_Map);

#ifndef msdos
   /* give vtxxx arrow keys */
    define_key1("^[[D", (VOID *) previous_char_cmd,F_INTRINSIC, Global_Map);
    define_key1("^[[A", (VOID *) previous_line_cmd, F_INTRINSIC, Global_Map);
    define_key1("^[[B", (VOID *) next_line_cmd, F_INTRINSIC, Global_Map);
    define_key1("^[[C", (VOID *) next_char_cmd, F_INTRINSIC, Global_Map);
    define_key1("^K^[[C", (VOID *) scroll_left, F_INTRINSIC, Global_Map);
    define_key1("^K^[[D", (VOID *) scroll_right, F_INTRINSIC, Global_Map);
    define_key1("^K^[[A", (VOID *) bob, F_INTRINSIC, Global_Map);
    define_key1("^K^[[B", (VOID *)eob, F_INTRINSIC, Global_Map);
#else
#include "doskeys.c"
#endif

    define_key1("'", (VOID *) text_smart_quote, F_INTRINSIC, Global_Map);
    define_key1("\"", (VOID *) text_smart_quote, F_INTRINSIC, Global_Map);
    define_key1("^?", (VOID *) backward_delete_char_untabify, F_INTRINSIC, Global_Map);
    define_key1("^B", (VOID *) bol, F_INTRINSIC, Global_Map);
    define_key1("^D", (VOID *) pagedown_cmd, F_INTRINSIC, Global_Map);
    define_key1("^E", (VOID *) eol_cmd, F_INTRINSIC, Global_Map);
    define_key1("^FB", (VOID *) search_backward_cmd, F_INTRINSIC, Global_Map);
    define_key1("^FF", (VOID *) search_forward_cmd, F_INTRINSIC, Global_Map);
    define_key1("^G", (VOID *) kbd_quit, F_INTRINSIC, Global_Map);
    define_key1("^I", (VOID *) indent_line, F_INTRINSIC, Global_Map);
    define_key1("^J", (VOID *) delete_word, F_INTRINSIC, Global_Map);
    define_key1("^K(", (VOID *) begin_keyboard_macro, F_INTRINSIC, Global_Map);
    define_key1("^K)", (VOID *) end_keyboard_macro, F_INTRINSIC, Global_Map);
    define_key1("^KC", (VOID *) goto_column_cmd, F_INTRINSIC, Global_Map);
    define_key1("^KD", (VOID *) evaluate_cmd, F_INTRINSIC, Global_Map);
    define_key1("^KE", (VOID *) exit_jed, F_INTRINSIC, Global_Map);
    define_key1("^KG", (VOID *) find_file, F_INTRINSIC, Global_Map);
    define_key1("^KK", (VOID *) copy_to_pastebuffer, F_INTRINSIC, Global_Map);
    define_key1("^KL", (VOID *) goto_line_cmd, F_INTRINSIC, Global_Map);
    define_key1("^KM", (VOID *) mark_spot, F_INTRINSIC, Global_Map);
    define_key1("^KX", (VOID *) execute_keyboard_macro, F_INTRINSIC, Global_Map);
    define_key1("^K^B", (VOID *) set_mark_cmd, F_INTRINSIC, Global_Map);
    define_key1("^K^I", (VOID *) insert_file_cmd, F_INTRINSIC, Global_Map);
    define_key1("^K^L", (VOID *) double_line, F_INTRINSIC, Global_Map);
    define_key1("^K^M", (VOID *) pop_spot, F_INTRINSIC, Global_Map);
    define_key1("^K^P", (VOID *) yank, F_INTRINSIC, Global_Map);
    define_key1("^K^R", (VOID *) replace, F_INTRINSIC, Global_Map);
    define_key1("^K^V", (VOID *) kill_region, F_INTRINSIC, Global_Map);
    define_key1("^K^W", (VOID *) write_buffer, F_INTRINSIC, Global_Map);
    define_key1("^L", (VOID *) kill_line, F_INTRINSIC, Global_Map);
    define_key1("^M", (VOID *) newline_and_indent, F_INTRINSIC, Global_Map);
    define_key1("^R", (VOID *) redraw, F_INTRINSIC, Global_Map);
    define_key1("^U", (VOID *) pageup_cmd, F_INTRINSIC, Global_Map);
    define_key1("^V", (VOID *) delete_char_cmd, F_INTRINSIC, Global_Map);
    define_key1("^W1", (VOID *) one_window, F_INTRINSIC, Global_Map);
    define_key1("^W2", (VOID *) split_window, F_INTRINSIC, Global_Map);
    define_key1("^WO", (VOID *) other_window, F_INTRINSIC, Global_Map);
    define_key1("^XB", (VOID *) get_buffer, F_INTRINSIC, Global_Map);
    define_key1("^XK", (VOID *) kill_buffer, F_INTRINSIC, Global_Map);
    define_key1("^XN", (VOID *) narrow_to_region, F_INTRINSIC, Global_Map);
    define_key1("^XQ", (VOID *) macro_query, F_INTRINSIC, Global_Map);
    define_key1("^XS", (VOID *) save_some_buffers, F_INTRINSIC, Global_Map);
    define_key1("^XW", (VOID *) widen, F_INTRINSIC, Global_Map);
    define_key1("^X^", (VOID *) enlarge_window, F_INTRINSIC, Global_Map);
    define_key1("^X^T", (VOID *) transpose_lines, F_INTRINSIC, Global_Map);
    define_key1("^X^X", (VOID *) exchange_point_mark, F_INTRINSIC, Global_Map);
    define_key1("^Z", (VOID *) sys_spawn_cmd, F_INTRINSIC, Global_Map);
    define_key1("^[<", (VOID *) bob, F_INTRINSIC, Global_Map);
    define_key1("^[>", (VOID *) eob, F_INTRINSIC, Global_Map);
    define_key1("^[C", (VOID *) capitalize_word, F_INTRINSIC, Global_Map);
    define_key1("^[L", (VOID *) downcase_word, F_INTRINSIC, Global_Map);
    define_key1("^[N", (VOID *) narrow_paragraph, F_INTRINSIC, Global_Map);
    define_key1("^[Q", (VOID *) text_format_paragraph, F_INTRINSIC, Global_Map);
    define_key1("^[S", (VOID *) center_line, F_INTRINSIC, Global_Map);
    define_key1("^[U", (VOID *) upcase_word, F_INTRINSIC, Global_Map);
    define_key1("^[X", (VOID *) evaluate_cmd, F_INTRINSIC, Global_Map);
    define_key1("^[\\", (VOID *) trim_whitespace, F_INTRINSIC, Global_Map);
    define_key1("^\\", (VOID *) goto_match, F_INTRINSIC, Global_Map);
    define_key1("`", (VOID *) quoted_insert, F_INTRINSIC, Global_Map);
    define_key1("{", (VOID *) brace_bra_cmd, F_INTRINSIC, Global_Map);
    define_key1("}", (VOID *) brace_ket_cmd, F_INTRINSIC, Global_Map);

    add_keymap("global", Global_Map);
}

int key_interpret(Key_Type *key)
{
   int ret;
   if (key->type == F_INTRINSIC)
     {
	ret = (int) ((Jed_fp ) key->f)();
     }
   else
     {
	(void) interpret((char *) key->f);
	if (Lang_Error) lang_doerror("dokey");
	Lang_Error = 0;
	ret = 1;
     }
   Last_Key_Function = key->f;
   return(ret);
}

#define XKEY(key)  key_interpret((key))

Key_Type *do_key1(char ch)
{
    Key_Type *key, *next;
    int i;
    unsigned char ch1, chup, chlow;

    Last_Command_Char = (int) ch;
    i = ch1 = (unsigned char) ch;

    /* The null char gets mapped to 128 which is at 0 */
    key = (Key_Type *) &((CBuf->keymap)[i]);

    /* if the next one is null, then we know this is it. */
    if (key->next == NULL) return(key);

    key = key->next;
    i = 1;

   Last_Command_Char = jed_getkey();
   ch1 = (char) Last_Command_Char; chup = upcase(ch1); chlow = lowcase(ch1);

   while(key != NULL)
     {
	if (KeyBoard_Quit) break;
	if (ch1 == 0) chlow = chup = ch1 = 128;
	if ((chup == (key->str)[i]) || (chlow == (key->str[i])))
	  {
	     if ((key->str)[i + 1] == 0)
	       {
		  /* look for exact match if possible */
		  if (ch1 != key->str[i])
		    {
		       next = key->next;
		       while (next != NULL)
			 {
			    if (next->str[i] == ch1)
			      {
				 if (next->str[i + 1] == 0) return(next);
				 break;
			      }
			    next = next->next;
			 }
		    }
		  return(key);
	       }

	     /* before reading a new key, check to see if it is lowercase
	        and try to find a match for it.  Note that this works because
	        only the LAST character in a key sequence can be lowercase */

	     else if ((ch1 == chlow) && (ch1 != chup))
	       {
		  next = key->next;
		  /* look for the lowercase one */
		  while (next != NULL)
		    {
		       if (next->str[i] == chlow) return(next);
		       next = next->next;
		    }
	       }

	     Last_Command_Char = ch1 = jed_getkey();
	     chup = upcase(ch1);
	     chlow = lowcase(ch1);
	     i++;
	     continue;
	  }
	else if (chlow < key->str[i]) break;  /* not found */

	/* else */
	key = key->next;
     }
   /* not found */
   return(NULL);
}

int do_key(char ch)
{
   Key_Type *key;
   int repeat;

   key = do_key1(ch);
   if ((key != NULL) && (key->f != NULL))
     {

	if (Repeat_Factor == NULL) return XKEY(key);
	repeat = *Repeat_Factor;

	/* some routines may use the repeat factor as a prefix argument */
	while (repeat-- > 0)
	  {
	     if (KeyBoard_Quit || (*Error_Buffer) ||
		 (!Executing_Keyboard_Macro && (Repeat_Factor == NULL))) break;
	     XKEY(key);
	  }
	Repeat_Factor = NULL;
	return(1);
     }
   else if (!Executing_Keyboard_Macro && !KeyBoard_Quit) 
     {
	beep();
	flush_input();
     }
   
   return(0);
}

void do_jed(void)
{
   char key;

   fflush(stdout);

   key = jed_getkey();
   Last_Command_Char = (int) key;

   Repeat_Factor = NULL;
   if (do_key(key)) Window->trashed = 1;

    /* internal editing commands may have selected a different buffer
       so put it back. */
    if (CBuf != Window->buffer)
      {
	 switch_to_buffer(Window->buffer);
	 window_buffer(CBuf);
      }
}

void jed(void)
{
   if (setjmp(Jump_Buffer) != 0)
     {
	if (CBuf != Window->buffer)
	  {
	     switch_to_buffer(Window->buffer);
	     window_buffer(CBuf);
	  }
	Window->trashed = 1;
	lang_restart();   /* just in case */
	update((Line *) NULL);
     }
   while(1)
     {
	do_jed();
	if (!KeyBoard_Quit && (CBuf->flags & BUFFER_TRASHED)
	    && (!Cursor_Motion)) CBuf->hits += 1;
	if (CBuf->hits > User_Vars.max_hits)
	  {
	     auto_save_buffer(CBuf);
	     check_buffers();   /* check files on disk to see if they are recent */
	  }
	update((Line *) NULL);
     }
}

int digit_arg(void)
{
    static int repeat;
    char buf[20], key;
    int i;

    Repeat_Factor = &repeat;
    i = 0;
    buf[i++] = (char) Last_Command_Char;
    while(1)
      {
	 buf[i] = 0;
	 message(buf);
	 if (!input_pending(2,0) && !Executing_Keyboard_Macro)
	   {
	      message(buf);
	      flush_message();
	   }
	 
	 key = jed_getkey();
	 if ((key < '0') || (key > '9')) break;
	 buf[i++] = key;
      }
   if (!input_pending(2,0) && !Executing_Keyboard_Macro)
     {
	message(buf);
	flush_message();
     }
   buf[i] = 0;
   sscanf(buf, "%d", &repeat);
   do_key(key);
   return(1);
}

void undefine_key(char *s, Key_Type *keymap)
{
   int n, i;
   Key_Type *key, *next, *last, *key_root;
   unsigned char *str;

   str = (unsigned char *) process_keystring(s);

   /* null char gets mapped to 128 */
   if (128 == ( i = *str)) i = 0;

   key = key_root = (Key_Type *) &(keymap[i]);
   if (0 == (n = strlen((char *) str))) return;

   last = key_root;
   key = key_root->next;
   while (key != NULL)
     {
	next = key->next;
	if (!strncmp(key->str, (char *) str, n))
	  {
	     FREE(key);
	     last->next = next;
	  }
	else last = key;
	key = next;
     }

   if (n == 1)
     {
	*key_root->str = 0;
	key_root->f = NULL;
     }
}

Key_Type *copy_keymap(Key_Type *km)
{
   int i;
   Key_Type *new, *old, *new_root;

   if (NULL == (new_root = (Key_Type *) CALLOC(128, sizeof(Key_Type))))
     {
	exit_error("Malloc error in copy keymap.");
     }

   for (i = 0; i < 128; i++)
     {
	old = &(km[i]);
	new = &(new_root[i]);
	new->f = old->f;
	new->type = old->type;
	strcpy(new->str, old->str);

	old = old->next;
	while (old != NULL)
	  {
	     new->next = malloc_key((unsigned char *) old->str);
	     new = new->next;
	     new->f = old->f;
	     new->type = old->type;
	     old = old->next;
	  }
	new->next = NULL;
     }
   return(new_root);
}

char *find_key(int *ret)
{
   char *fstr = NULL;
   Jed_Function_Type *fp;
   Key_Type *key;
   
   *ret = 0;

   if (NULL == (key = do_key1(jed_getkey()))) return(NULL);
   if (key->type == F_INTRINSIC)
     {
	*ret = 1;
	fp = Jed_Functions;

	while((fp != NULL) && (fp->name != NULL))
	  {
	     if ((VOID *) fp->f == key->f)
	       {
		  fstr = fp->name;
		  break;
	       }
	     fp++;
	  }
     }
   else fstr = (char *) key->f;

   return(fstr);
}

Key_Type *create_keymap(char *name)
{
   Key_Type *new;
   if (NULL != (new = copy_keymap(Global_Map))) add_keymap(name, new);
   return new;
}

Key_Type *find_keymap(char *name)
{
   int i;

   for (i = 0; i < MAX_KEYMAPS; i++)
     {
	if (!strcmp(KeyMap_List[i].name, name)) return KeyMap_List[i].keymap;
     }
   return(NULL);
}

void use_keymap(char *name)
{
   Key_Type *map;
   if (NULL == (map = find_keymap(name)))
     {
	msg_error("Unknown keymap.");
     }
   else (CBuf->keymap = map);
}

