/***************************************************************

	sfs_di.c        Multiple Display Routines for
			Space Flight Simulator

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "ctype.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "sfs.h"

#ifdef  __STDC__
#include "malloc.h"
#else
extern  char * malloc();
#define size_t   int
#define time_t   long
#endif

/***************************************************************

	di_init()

***************************************************************/

di_init( sorbit_array, n_orbits, d_array, n_displays, t_array )
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   struct sfs_display ***d_array;
   int *n_displays;
   char ***t_array;
   {
   register int c, d;
   int n;
   int x_foci, x_active, x_new;

   /* first determine the number of active orbits */

   x_active = 0;
   for ( c = 0; c < n_orbits; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
         {
	 ++x_active;
	 }
      }

   /* now determine number of foci */

   x_foci = 0;
   for ( c = 0; c < n_orbits; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
	 {
	 x_new = TRUE;
	 for ( d = 0; d < c; ++d )
	    {
	    if ( sorbit_array[ d ] != NULL )
	       {
	       if ( sorbit_array[ d ]->aorbit->focus ==
		  sorbit_array[ c ]->aorbit->focus )
		  {
		  x_new = FALSE;
		  }
	       }
	    }
	 if ( x_new == TRUE )
	    {
	    ++x_foci;
	    }
	 }
      }

   /* now calculate total number of displays: one visual for each orbit,
      then a ground track and a distant perspective display for each focus */

   *n_displays = x_active + ( 2 * x_foci );

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "Displays: %d total from %d orbits with %d foci",
      *n_displays, x_active, x_foci );
   bw_debug( bw_ebuf );
#endif

   /* allocate memory for d_array */

   if ( ( *d_array = (struct sfs_display **)
      malloc( sizeof( struct sfs_display *) * *n_displays ) ) == NULL )
      {
      bw_error( DI_MEMERR );
      sfs_exit();
      exit( 0 );
      }

   /* allocate memory for each structure within d_array */

   for ( c = 0; c < *n_displays; ++c )
      {
      if ( ( (*d_array)[ c ] = (struct sfs_display *)
	 malloc( sizeof( struct sfs_display ) ) ) == NULL )
	 {
	 bw_error( DI_DSMEMERR );
	 sfs_exit();
	 exit( 0 );
	 }
      }

   /* from now on, use n as the counter for the display being constructed */
   /* fill in the visual simulation displays */

   n = 0;
   for ( c = 0; c < n_orbits; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
	 {
	 (*d_array)[ n ]->type = SFSD_VISUAL;
	 (*d_array)[ n ]->elements
	    = VI_ORBOUTLINE | VI_GRIDFRONT | VI_SURFRONT | VI_STITLE;
	 (*d_array)[ n ]->orbit = c;
	 ++n;
	 }
      }

   /* fill in the ground track displays */

   for ( c = 0; c < n_orbits; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
	 {
	 x_new = TRUE;
	 for ( d = 0; d < c; ++d )
	    {
	    if ( sorbit_array[ d ] != NULL )
	       {
	       if ( sorbit_array[ d ]->aorbit->focus ==
		  sorbit_array[ c ]->aorbit->focus )
		  {
		  x_new = FALSE;
		  }
	       }
	    }
	 if ( x_new == TRUE )
	    {
	    (*d_array)[ n ]->type = SFSD_GROUND;
	    (*d_array)[ n ]->elements
	       = VI_ORBOUTLINE | VI_GRIDFRONT | VI_SURFRONT;
	    (*d_array)[ n ]->orbit = c;
	    (*d_array)[ n ]->focus = sorbit_array[ c ]->aorbit->focus;
	    ++n;
	    }
	 }
      }

   /* fill in the perspective displays */

   for ( c = 0; c < n_orbits; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
	 {
	 x_new = TRUE;
	 for ( d = 0; d < c; ++d )
	    {
	    if ( sorbit_array[ d ] != NULL )
	       {
	       if ( sorbit_array[ d ]->aorbit->focus ==
		  sorbit_array[ c ]->aorbit->focus )
		  {
		  x_new = FALSE;
		  }
	       }
	    }
	 if ( x_new == TRUE )
	    {
	    (*d_array)[ n ]->type = SFSD_PERSP;
	    (*d_array)[ n ]->elements
	       = VI_ORBOUTLINE | VI_GRIDFRONT | VI_SURFRONT;
	    (*d_array)[ n ]->orbit = c;
	    (*d_array)[ n ]->focus = sorbit_array[ c ]->aorbit->focus;
	    ++n;
	    }
	 }
      }

   /* allocate memory for t_array */

   if ( ( *t_array = (char **)
      malloc( sizeof(char *) * *n_displays ) ) == NULL )
      {
      bw_error( DI_DMEMERR );
      sfs_exit();
      exit( 0 );
      }

   /* now allocate memory for and write each title */

   for ( c = 0; c < *n_displays; ++c )
      {
      switch ( (*d_array)[ c ]->type )
	 {
	 case SFSD_VISUAL:
	    sprintf( bw_ebuf, VIS_TITLE,
	       sorbit_array[ (*d_array)[ c ]->orbit ]->aorbit->focus->name,
	       sorbit_array[ (*d_array)[ c ]->orbit ]->name,
	       (*d_array)[ c ]->orbit + 1 );
	    break;
	 case SFSD_PERSP:
	    sprintf( bw_ebuf, PER_TITLE,
	       sorbit_array[ (*d_array)[ c ]->orbit ]->aorbit->focus->adjective );
	    break;
	 case SFSD_GROUND:
	    sprintf( bw_ebuf, GTR_TITLE,
	       sorbit_array[ (*d_array)[ c ]->orbit ]->aorbit->focus->adjective );
	    break;
	 default:
#ifdef  DEBUG
	    sprintf( bw_ebuf, "di_init() received incorrect type %d",
	       (*d_array)[ c ]->type );
	    bw_error( bw_ebuf );
#endif
	    break;
	 }

      if ( ( (*t_array)[ c ] = (char *)
	 malloc( strlen( bw_ebuf ) + 2 ) ) == NULL )
	 {
	 bw_error( DI_DMEMERR );
	 sfs_exit();
	 exit( 0 );
	 }
      strcpy( (*t_array)[ c ], bw_ebuf );
      }

   }

