with Console; use Console;
with Automobile_Interface; use Automobile_Interface;
--***********************************************************
--*															*
--*  Cruise													*  BODY
--*															*
--***********************************************************
package body Cruise is
--|  Purpose
--|  Cruise provides a set of functions and procedures which
--|  implements a cruise control program.  Input function 
--|  Fetch_Key requires the use of the "stty raw" command.  
--|  After this command is entered, the "Return" key must be
--|  implemented using "Cntl-J."  The terminal may be reset 
--|  with the "stty -raw" command.  
--|  This package implements a cruise control program using the
--|  standard engage/disengage, set/resume cruise control 
--|  functions.

--...............................................................
--.																.
--.  Cruise.Decode_Function										.
--.																.
--...............................................................
procedure Decode_Function(func:AUTO_CC_FUNCTION) is
--|  Purpose
--|  Decode input function requested by user.  This procedure
--|  calls the appropriate update procedure.

begin
	if func = 'a' then
		Update_Engine_Status(ENGINE_ON);
	elsif func = 'b' then
		Update_Engine_Status(ENGINE_OFF);
	elsif engine_status = ENGINE_ON then
		case func is
		when 'c' => Update_CC_Engaged_Status(ENGAGE);
		when 'd' => Update_CC_Engaged_Status(DISENGAGE);
		when 'e' => Update_CC_Set_Resume_Status(SET);
		when 'f' => 
			begin
				Update_Pedal_Status(RELEASE_ACCEL);
				Update_Pedal_Status(RELEASE_BRAKE);
				Update_CC_Set_Resume_Status(RESUME);
			end;
		when 'g' => 
			begin
				Update_Pedal_Status(RELEASE_BRAKE);
				Update_Pedal_Status(DEPRESS_ACCEL);
			end;	
		when 'h' => 
			begin
				Update_Pedal_Status(RELEASE_BRAKE);
				Update_Pedal_Status(HOLD_ACCEL);
			end;
		when 'i' => Update_Pedal_Status(RELEASE_ACCEL);
		when 'j' => 
			begin
				Update_Pedal_Status(RELEASE_ACCEL);
				Update_Pedal_Status(DEPRESS_BRAKE);
			end;
		when 'k' => Update_Pedal_Status(RELEASE_BRAKE);
		when others => null;
		end case;
	end if;
end Decode_Function;

--...............................................................
--.																.
--.  Cruise.Update_Pedal_Status									.
--.																.
--...............................................................
procedure Update_Pedal_Status(pedal:PEDAL_ACTION) is
--|  Purpose
--|  Update global variables pedal_operation, brake_status, and
--|  accel_status to the current action requested.

begin
--|  if no change in brake status, do nothing
	if pedal = RELEASE_BRAKE and brake_status = RELEASE_BRAKE then
		null; 
--|  if no change in accelerator status, do nothing
	elsif pedal = RELEASE_ACCEL and accel_status = RELEASE_ACCEL then
		null; 
--|  otherwise, update pedal status and cruise control status accordingly
	else
		pedal_operation := pedal;
		if pedal_operation in BRAKE_ACTION then
			brake_status := pedal_operation;
			if pedal_operation = DEPRESS_BRAKE then
				cc_set_resume_status := UNSET;
				auto_controller := USER;
			end if;
		elsif pedal_operation = DEPRESS_ACCEL and auto_controller = CRUISE_CONTROL and cc_set_resume_status /= RESUME then
			auto_controller := USER_CRUISE;
		elsif pedal_operation = RELEASE_ACCEL and auto_controller = USER_CRUISE then
			auto_controller := CRUISE_CONTROL;
		end if;
		if pedal_operation in ACCEL_ACTION then
			accel_status := pedal_operation;
		end if;
	end if; 
end Update_Pedal_Status;

--...............................................................
--.																.
--.  Cruise.Update_Engine_Status								.
--.																.
--...............................................................
procedure Update_Engine_Status (engine : ENGINE_ACTION) is
--|  Purpose
--|  Update global variable engine_status to the engine action
--|  requested.

begin
	engine_status := engine;
	if engine_status = ENGINE_OFF then 
		desired_speed := 0.0;
		auto_controller := USER;
		cc_engaged_status := DISENGAGE;
		cc_set_resume_status := UNSET;
		brake_status := RELEASE_BRAKE;
		accel_status := RELEASE_ACCEL;
		pedal_operation := RELEASE_PEDALS;
	end if;

end Update_Engine_Status;

--...............................................................
--.																.
--.  Cruise.Update_CC_Set_Resume_Status							.
--.																.
--...............................................................
procedure Update_CC_Set_Resume_Status (CC_set_action : CC_SET_RESUME_ACTION) is
--|  Purpose
--|  Update global variable cc_set_resume_status to the 
--|  set/resume action  

begin
	if cc_engaged_status = ENGAGE and brake_status /= DEPRESS_BRAKE then
		cc_set_resume_status := CC_set_action;
		if cc_set_resume_status in SET_RESUME then
			if cc_set_resume_status = SET then
				desired_speed := Sensed_Speed;
			end if;
			if pedal_operation = DEPRESS_ACCEL then
				auto_controller := USER_CRUISE;
			else
				auto_controller := CRUISE_CONTROL;
			end if;
		end if;
	end if;
end Update_CC_Set_Resume_Status;

--...............................................................
--.																.
--.  Cruise.Update_CC_Engaged_Status	 						.
--.																.
--...............................................................
procedure Update_CC_Engaged_Status (CC_engage_action : CC_ENGAGED_ACTION) is
--|  Purpose
--|  Update global variable cc_engaged_status to the engaged
--|  action.

begin
	cc_engaged_status := CC_engage_action;
	if cc_engaged_status = DISENGAGE then
--|  reset desired_speed, auto_controller, and automobile
--|  functions when cruise control is disengaged 
		desired_speed := 0.0;
		auto_controller := USER;
		if cc_set_resume_status in SET_RESUME then
			cc_set_resume_status := UNSET;
			brake_status := RELEASE_BRAKE;
			accel_status := RELEASE_ACCEL;
			pedal_operation := RELEASE_PEDALS;
		end if;
	end if;
end Update_CC_Engaged_Status;

--...............................................................
--.																.
--.  Cruise.Print_Menu				 							.
--.																.
--...............................................................
procedure Print_Menu is 
--|  Purpose
--|  Print user menu once when program begins.

begin
	Position_Cursor(4,1);
	Put("MENU");
	Position_Cursor(6,1);
	Put("a: Turn Car On");
	Position_Cursor(7,1);
	Put("b: Turn Car Off");
	Position_Cursor(8,1);
	Put("c: Engage Cruise Control");
	Position_Cursor(9,1);
	Put("d: Disenage Cruise Control");
	Position_Cursor(10,1);
	Put("e: Set Cruise Control");
	Position_Cursor(11,1);
	Put("f: Resume Cruise Control");
	Position_Cursor(12,1);
	Put("g: Depress Accelerator");
	Position_Cursor(13,1);
	Put("h: Hold Accelerator");
	Position_Cursor(14,1);
	Put("i: Release Accelerator");
	Position_Cursor(15,1);
	Put("j: Depress Brake");
	Position_Cursor(16,1);
	Put("k: Release Brake");
	Position_Cursor(17,1);
	Put("l: QUIT");

	Position_Cursor(4,35);
	Put("AUTOMOBILE STATUS");
	Position_Cursor(6,35);
	Put("BRAKE ");
	Position_Cursor(7,35);
	Put("ACCELERATOR ");
	Position_Cursor(8,35);
	Put("ENGINE ");
	Position_Cursor(9,35);
	Put("CC ");
	Position_Cursor(10,35);
	Put("CC ");
	Position_Cursor(11,35);
	Put("SPEED ");
	Position_Cursor(11,55);
	Put("MPH");
end Print_Menu;

--...............................................................
--.																.
--.  Cruise.Format_Display										.
--.																.
--...............................................................
procedure Format_Display	(speed_value    : SPEED; 
							engine_text     : in STRING;
							brake_text      : in STRING;
							accel_text      : in STRING;
							engaged_text    : in STRING;
							set_resume_text : in STRING) is
--|  Purpose
--|  Format output for display.  

begin
	Position_Cursor(6,50);
	Put(brake_text);
	Position_Cursor(7,50);
	Put(accel_text);
	Position_Cursor(8,50);
	Put(engine_text);
	Position_Cursor(9,50);
	Put(engaged_text);
	Position_Cursor(10,50);
	Put(set_resume_text);
	Position_Cursor(11,50);
	Put(INTEGER(speed_value),3);
	Position_Cursor(20,4); 
end Format_Display;

--...............................................................
--.																.
--.  Cruise.Determine_Control									.
--.																.
--...............................................................
procedure Determine_Control is
--|  Purpose
--|  Determine if cruise control or user is regulating the
--|  automobile speed

begin
	if auto_controller = CRUISE_CONTROL then
		Maintain_Speed;
	else
		Perform_Auto_Function;
	end if;
end Determine_Control;

--...............................................................
--.																.
--.  Cruise.Fetch_Key											.
--.																.
--...............................................................
function Fetch_Key return BOOLEAN is
--|  Purpose
--|  Fetch user input

begin
	Position_Cursor(20,4);
	Get(key);
	if key /= 'l' then
		Decode_Function(key);
		return FALSE;
	else
		return TRUE;
	end if;
end Fetch_Key;

--...............................................................
--.																.
--.  Cruise.Maintain_Speed										.
--.																.
--...............................................................
procedure Maintain_Speed is
--|  Purpose
--|  Maintain automobile speed at desired speed.

speed_value : SPEED;
 
begin
	speed_value := Sensed_Speed;
	if speed_value < desired_speed then
		Depress_Accelerator_Pedal;
		pedal_operation := DEPRESS_ACCEL;
		accel_status := DEPRESS_ACCEL;
	elsif speed_value > desired_speed then
		Release_Accelerator_Pedal;
		pedal_operation := RELEASE_ACCEL;
		accel_status := RELEASE_ACCEL;
	elsif speed_value = desired_speed then
		Hold_Accelerator_Pedal;
		pedal_operation := HOLD_ACCEL;
		accel_status := HOLD_ACCEL;
		if cc_set_resume_status = RESUME then
			cc_set_resume_status := SET;
		end if;
	end if;
end Maintain_Speed;

--...............................................................
--.																.
--.  Cruise.Perform_Auto_Function								.
--.																.
--...............................................................
procedure Perform_Auto_Function is
--|  Purpose
--|  Apply the desired automobile function using package
--|  Automobile_Interface.

begin
	case pedal_operation is
	when RELEASE_ACCEL  => Release_Accelerator_Pedal;
	when DEPRESS_ACCEL  => Depress_Accelerator_Pedal;
	when HOLD_ACCEL     => Hold_Accelerator_Pedal;
	when DEPRESS_BRAKE  => Depress_Brake_Pedal;
	when RELEASE_BRAKE  => Release_Brake_Pedal;
	when RELEASE_PEDALS => 
		begin
			Release_Accelerator_Pedal;
			Release_Brake_Pedal;
		end;
	end case;

	case engine_status is 
	when ENGINE_ON => Turn_On_Engine;
	when ENGINE_OFF => Turn_Off_Engine;
	end case;

end Perform_Auto_Function;

--...............................................................
--.																.
--.  Cruise.Update_Dashboard									.
--.																.
--...............................................................
procedure Update_Dashboard is
--|  Purpose
--|  Update output text to correspond to current automobile state.

speed_value     : SPEED;
brake_text      : STRING(1..9); 
accel_text      : STRING(1..9);
engine_text     : STRING(1..3);
engaged_text    : STRING(1..10);
set_resume_text : STRING(1..12);

begin
	speed_value := Sensed_Speed;

	case brake_status is
	when DEPRESS_BRAKE => brake_text := "depressed";
	when RELEASE_BRAKE => brake_text := "released "; 
	end case;

	case accel_status is
	when DEPRESS_ACCEL => accel_text := "depressed"; 
	when RELEASE_ACCEL => accel_text := "released "; 
	when HOLD_ACCEL => accel_text    := "held     "; 
	end case;
	
	case pedal_operation is
	when RELEASE_PEDALS => 
		begin
			accel_text := "released ";
			brake_text := "released ";
		end;
	when others => null;
	end case;

	case cc_engaged_status is
	when ENGAGE => engaged_text    := "engaged   ";
	when DISENGAGE => engaged_text := "disengaged";
	end case;

	case cc_set_resume_status is
	when SET => set_resume_text    := "set         ";
	when UNSET => set_resume_text  := "unset       ";
	when RESUME => set_resume_text := "set (resume)";
	end case;

	case engine_status is
	when ENGINE_ON => engine_text  := "on ";
	when ENGINE_OFF => engine_text := "off";
	end case;

	Format_Display(speed_value, engine_text, brake_text, accel_text, engaged_text, set_resume_text);

end Update_Dashboard;

 
end Cruise;
