{ VGABIOS - Interface to VGA BIOS routines (c) Wilbert van Leijen 1990-92 }

Unit VGABios;

Interface

Const
  MinIntensity = 0;
  MaxIntensity = 63;

Type
  ColPageMode  = (_4x64, _16x16);
  DisplayPage  = 0..7;
  FontBlock    = 0..7;
  CharSetType  = (INT1F, INT43, ROM8x14, ROM8x8lo, ROM8x8hi, ROM9x14,
                 ROM8x16, ROM9x16);
  ScanLineType = (CGA200, EGA350, VGA400);
  ColourRange  = MinIntensity..MaxIntensity;
  RGBType      = Record
                   r, g, b   : ColourRange;
                 end;
  RGB64Type    = Array[0..63] of RGBType;
  PaletteType  = Record
                   ColourReg : Array[0..15] of Byte;
                   Border    : Byte;
                 end;
Var
  VGAStatus    : (NotVGA, VGAMono, VGAColour);

Procedure SetVideoMode(mode : Byte);
Procedure GetRGBValue(register : Byte; Var RGB : RGBType);
Procedure SetRGBValue(register : Byte; RGB : RGBType);
Procedure CursorEmulation(emulate : Boolean);
Function GetFontPtr(charset : CharSetType) : Pointer;
Procedure GetFontBlock(Var primary, secondary : FontBlock);
Procedure SetFontBlock(primary, secondary : FontBlock);
Procedure LoadFont(block : FontBlock;
               startchar : Char;
    numofchars, charsize : Integer;
                 charptr : Pointer);

Function GetVideoMode : Byte;

InLine(
  $B4/$0F/               {   MOV    AH, 0Fh    }
  $CD/$10);              {   INT    10h        }

Function GetDisplayPage : Byte;

InLine(
  $B4/$0F/               {   MOV    AH, 0Fh    }
  $CD/$10/               {   INT    10h        }
  $88/$F8);              {   MOV    AL, BH     }

Procedure SetBlink;

InLine(
  $B3/$01/               {   MOV    BL, 1      }
  $B8/$03/$10/           {   MOV    AX, 1003h  }
  $CD/$10);              {   INT    10h        }

Procedure SetIntensity;

InLine(
  $B3/$00/               {   MOV    BL, 0      }
  $B8/$03/$10/           {   MOV    AX, 1003h  }
  $CD/$10);              {   INT    10h        }

Function GetColourPage : Byte;

InLine(
  $B8/$1A/$10/           {   MOV    AX, 101Ah  }
  $CD/$10/               {   INT    10h        }
  $88/$F8);              {   MOV    AL, BH     }

Function GetPageMode : Byte;

InLine(
  $B8/$1A/$10/           {   MOV    AX, 101Ah  }
  $CD/$10/               {   INT    10h        }
  $88/$D8);              {   MOV    AL, BL     }

Procedure GetPalette(Var palette : PaletteType);

InLine(
  $5A/                   {   POP    DX         }
  $07/                   {   POP    ES         }
  $B8/$09/$10/           {   MOV    AX, 1009h  }
  $CD/$10);              {   INT    10h        }

Procedure SetPalette(Var palette : PaletteType);

InLine(
  $5A/                   {   POP    DX         }
  $07/                   {   POP    ES         }
  $B8/$02/$10/           {   MOV    AX, 1002h  }
  $CD/$10);              {   INT    10h        }

Procedure GetColourBlock(Var RGBBlock : RGB64Type);

InLine(
  $5A/                   {   POP    DX         }
  $07/                   {   POP    ES         }
  $B8/$17/$10/           {   MOV    AX, 1017h  }
  $31/$DB/               {   XOR    BX, BX     }
  $B9/$40/$00/           {   MOV    CX, 64     }
  $CD/$10);              {   INT    10h        }

Procedure SetColourBlock(Var RGBBlock : RGB64Type);

InLine(
  $5A/                   {   POP    DX         }
  $07/                   {   POP    ES         }
  $B8/$12/$10/           {   MOV    AX, 1012h  }
  $31/$DB/               {   XOR    BX, BX     }
  $B9/$40/$00/           {   MOV    CX, 64     }
  $CD/$10);              {   INT    10h        }

Procedure SetColourPage(page : Byte);

InLine(
  $58/                   {   POP    AX         }
  $B3/$01/               {   MOV    BL, 1      }
  $88/$C7/               {   MOV    BH, AL     }
  $B8/$13/$10/           {   MOV    AX, 1013h  }
  $CD/$10);              {   INT    10h        }

Procedure SetPageMode(pagemode : ColPageMode);

InLine(
  $58/                   {   POP    AX         }
  $B3/$00/               {   MOV    BL, 0      }
  $88/$C7/               {   MOV    BH, AL     }
  $B8/$13/$10/           {   MOV    AX, 1013h  }
  $CD/$10);              {   INT    10h        }

{ Get the border colour (overscan register) }

Function GetBorderColour : ColourRange;

InLine(
  $B8/$08/$10/           {   MOV    AX, 1008h  }
  $CD/$10/               {   INT    10h        }
  $88/$F8);              {   MOV    AL, BH     }

Procedure SetBorderColour(colour : ColourRange);

InLine(
  $58/                   {   POP    AX         }
  $88/$C7/               {   MOV    BH, AL     }
  $B8/$01/$10/           {   MOV    AX, 1001h  }
  $CD/$10);              {   INT    10h        }

{ Determine whether a call to SetVideoMode should reset the colours to
  their default values or not }

Procedure SaveCurrentPalette(enable : Boolean);

InLine(
  $58/                   {   POP    AX         }
  $B4/$12/               {   MOV    AH, 12h    }
  $B3/$31/               {   MOV    BL, 31h    }
  $CD/$10);              {   INT    10h        }

{ Select the vertical display size.  Select either CGA, EGA or VGA
  resolution }

Procedure SetScanLine(scanlines : ScanLineType);

InLine(
  $58/                   {   POP    AX         }
  $B4/$12/               {   MOV    AH, 12h    }
  $B3/$30/               {   MOV    BL, 30h    }
  $CD/$10);              {   INT    10h        }

{ Switch the default display page }

Procedure SetDisplayPage(page : DisplayPage);

InLine(
  $58/                   {   POP    AX         }
  $B4/$05/               {   MOV    AH, 5      }
  $CD/$10);              {   INT    10h        }

{ Sum all colours to gray scales.  Changes will take effect after
  next call to SetVideoMode }

Procedure SumGrayScale(enable : Boolean);

InLine(
  $58/                   {   POP    AX         }
  $08/$C0/               {   OR     AL, AL     }
  $74/$04/               {   JZ     @1         }
  $48/                   {   DEC    AX         }
  $E9/$01/$00/           {   JMP    @2         }
  $40/                 { @1: INC    AX         }
  $B4/$12/             { @2: MOV    AH, 12h    }
  $B3/$33/               {   MOV    BL, 33h    }
  $CD/$10);              {   INT    10h        }

{ Get colour information from a DAC register }

Function GetRegister(register : Byte) : ColourRange;

InLine(
  $58/                   {   POP    AX         }
  $93/                   {   XCHG   AX, BX     }
  $30/$FF/               {   XOR    BH, BH     }
  $B8/$07/$10/           {   MOV    AX, 1007h  }
  $CD/$10/               {   INT    10h        }
  $88/$F8);              {   MOV    AL, BH     }

{ Store colour information to DAC register }

Procedure SetRegister(register : Byte; colour : ColourRange);

InLine(
  $58/                   {   POP    AX         }
  $88/$C7/               {   MOV    BH, AL     }
  $58/                   {   POP    AX         }
  $88/$C3/               {   MOV    BL, AL     }
  $B8/$00/$10/           {   MOV    AX, 1000h  }
  $CD/$10);              {   INT    10h        }

{ Load the resident 8x8 font }

Procedure LoadFont8x8;

InLine(
  $B8/$12/$11/           {   MOV    AX, 1112h  }
  $B3/$00/               {   MOV    BL, 0      }
  $CD/$10);              {   INT    10h        }

Implementation

{$R-,S- }

{ Switch cursor emulation }

Procedure CursorEmulation(emulate : Boolean); Assembler;

ASM
        MOV     DL, emulate
        XOR     AX, AX
        MOV     ES, AX
        MOV     SI, [0487h]
        OR      DL, DL
        JZ      @1
        AND     Byte Ptr ES:[SI], (not 1)
        JMP     @2
@1:     OR      Byte Ptr ES:[SI], 1
@2:
end;  { CursorEmulation }

{ Set the current video mode.  You must call this procedure to switch
  some VGA features on or off }

Procedure SetVideoMode(mode : Byte); Assembler;

ASM
  { Get current cursor location }

        MOV    AH, 0Fh
        INT    10h
        MOV    AL, BH
        MOV    AH, 3
        INT    10h

        MOV    AL, mode
        XOR    AH, AH
        INT    10h

  { Restore cursor location }

        MOV    AH, 2
        INT    10h

  { Set cursor to underline }

        MOV    CX, 0607h
        MOV    AH, 1
        INT    10h
end;  { SetVideoMode }

{ Get the Red, Green and Blue intensity from a DAC register }

Procedure GetRGBValue(register : Byte; Var RGB : RGBType); Assembler;

ASM
        LES     DI, RGB
        XOR     BX, BX
        MOV     BL, register
        MOV     AX, 1015h
        INT     10h
        MOV     AL, DH
        STOSB
        XCHG    AX, CX
        XCHG    AH, AL
        STOSW
end;  { GetRGBValue }

{ Store the Red, Green and Blue intensity into a DAC register }

Procedure SetRGBValue(register : Byte; RGB : RGBType); Assembler;

ASM
        PUSH    DS
        LDS     SI, RGB
        XOR     BX, BX
        MOV     BL, register
        LODSB
        MOV     DH, AL
        LODSW
        XCHG    CX, AX
        XCHG    CH, CL
        MOV     AX, 1010h
        INT     10h
        POP     DS
end;  { SetRGBValue }

{ Get a pointer to one of the eight resident VGA fonts }

Function GetFontPtr(charset : CharSetType) : Pointer; Assembler;

ASM
        MOV    BH, charset
        MOV    AX, 1130h
        INT    10h
        MOV    DX, ES
        XCHG   AX, BP
end;  { GetFontPtr }

{ Get font block index of current (resident) and alternate character set.
  Up to two fonts can be active at the same time }

Procedure GetFontBlock(Var primary, secondary : FontBlock); Assembler;

ASM
  { Get character map select register:
    (VGA sequencer port 3C4h/3C5h index 3)

    7  6  5  4  3  2  1  0
                    
                     Primary font   (lower 2 bits)
                 Secondary font (lower 2 bits)
               Primary font   (high bit)
             Secondary font (high bit)     }

        MOV     AL, 3
        MOV     DX, 3C4h
        OUT     DX, AL
        INC     DX
        IN      AL, DX
        MOV     BL, AL
        PUSH    AX

  { Get secondary font number: add up bits 5, 3 and 2 }

        SHR     AL, 1
        SHR     AL, 1
        AND     AL, 3
        TEST    BL, 00100000b
        JZ      @1
        ADD     AL, 4
@1:     LES     DI, secondary
        STOSB

  { Get primary font number: add up bits 4, 1 and 0 }

        POP     AX
        AND     AL, 3
        TEST    BL, 00010000b
        JZ      @2
        ADD     AL, 4
@2:     LES     DI, primary
        STOSB
end;  { GetFontBlock }

{ Store the font block index }

Procedure SetFontBlock(primary, secondary : FontBlock); Assembler;

Const
  MapPrimTable : Array[0..7] of Byte = ($00, $01, $02, $03, $10, $11, $12, $13);
  MapSecTable  : Array[0..7] of Byte = ($00, $04, $08, $0C, $20, $24, $28, $2C);

ASM
        MOV     AL, primary
        LEA     BX, MapPrimTable
        XLAT
        MOV     AH, AL
        MOV     AL, secondary
        LEA     BX, MapSecTable
        XLAT
        ADD     AL, AH
        MOV     BL, AL

  { Set block specifier }

        MOV     AX, 1103h
        INT     10h
end;  { SetFontBlock }

{ Load (a part of) a font.  When loaded into the active blocks, changes will
  affect display output }

Procedure LoadFont(block : FontBlock;
               startchar : Char;
    numofchars, charsize : Integer;
                 charptr : Pointer); Assembler;
ASM
        MOV     BL, block
        XOR     DH, DH
        MOV     DL, startchar
        MOV     CX, numofchars
        MOV     BH, Byte Ptr charsize
        LES     BP, charptr
        MOV     AX, 1100h
        INT     10h
end;  { LoadFont }

Begin  { VGABios }
ASM

  { Determine whether active video system is VGA }

        MOV     AX, 1A00h
        INT     10h
        MOV     AH, BL
        CMP     AX, 081Ah
        JE      @1
        MOV     DL, NotVGA
        JMP     @2
@1:     MOV     DL, VGAColour

  { VGA found, determine if registers are mapped to mono }

        MOV     AX, 1200h
        MOV     BL, 10h
        INT     10h
        OR      BH, BH
        JZ      @2
        MOV     DL, VGAMono
@2:     MOV     [VGAStatus], DL
end;
end.  { VGABios }