{ ========================================================================== }
{ Rate57.pas - Rate timer using only low memory clock.     ver 5.7, 01-23-91 }
{                                                                            }
{ This rate timer is a highly accurate timer based on the low memory clock.  }
{ It is device independent by not accessing any special timer hardware and   }
{ does not use any interrupts.  Accuracy is +-1 repetition.                  }
{                                                                            }
{ This timer is great for timing short events (1 microsecond to .1 second)   }
{ to get a Reps-per-second result.  The secret to such great accuracy is     }
{ keying in on the fact that the low memory clock is very accurate and on    }
{ schedule every time it ticks.  So, just make the event work by the clock   }
{ rather than vice-versa.                                                    }
{                                                                            }
{ The timer is set for a 1 second test.  If you can't get enough repetitions }
{ (>100) to get any resolution, you are free to change TestTime up to 3600   }
{ seconds.                                                                   }
{                                                                            }
{ The timer even accounts for the overhead of the timer itself and any other }
{ overhead that is a part of your event.  Just insert your code and go.  You }
{ may experience a little jitter while running in the integrated environment,}
{ but is solid when run as an executable file.                               }
{                                                                            }
{ For further explanation of the high-resolution feature of this timer, see  }
{ comments at the end of this file.                                          }
{                                                                            }
{   Public Domain by James H. LeMay, Eagle Performance Software              }
{ ========================================================================== }
{$A+,D+,E+,L-,N+,R-,S-,V-}

program RateTimer;

uses
  Crt,Strg;

const
  TestTime = 1;  { seconds }
  TicksPerDay = 1573040.0;            { DOS timer ticks/day. }
  TicksPerSec = TicksPerDay/86400.0;

var
  LowClock: word absolute $0000:$046C;
  Tick1,Ticks:           word;
  Reps,OverheadReps:     longint;
  RepsPerSec,TickFactor: real;
  R: real absolute RepsPerSec; { Variable name is easier to Watch }
  { -- Place your variables here: -- }
  S1,S2: string;
  b:     byte;

procedure WaitForTick;
begin
  Tick1 := LowClock;
  repeat
  until LowClock<>Tick1;
  Tick1 := LowClock;
end;

procedure RunTime (Seconds: word);
begin
  if Seconds>3600
    then Seconds:=3600;
  Ticks := trunc(Seconds*TicksPerSec);
  TickFactor := Seconds*TicksPerSec/Ticks; { Accounts for trunc of seconds }
  WaitForTick;
end;

procedure Init;
begin
  { ** Initialize any of your variables here: ** }
  S1 := 'Now is the time to check out Eagle Performance Software Products'+
        'for speed tests';
  S2 := S1;
end;

procedure ResetVariables;
begin
  { ** Place any variables that need to be reset after each test here: ** }
  { StrMove (S2,S1); }
end;

procedure Test;
begin
  Reps := 0;
  RunTime (TestTime);
  repeat
    { ** Insert your test routine here ** }
    b := StrPosL (S1,'speed',1);
    { ** End of test routine ** }
    ResetVariables;
    inc (Reps);
  until LowClock-Tick1>=Ticks;
end;

procedure OverHead;
begin
  OverHeadReps := 0;
  RunTime (TestTime);
  repeat
    ResetVariables;
    inc (OverHeadReps);
  until LowClock-Tick1>=Ticks;
end;

procedure CalcRate;
begin
  RepsPerSec := TickFactor/((TestTime)*(1.0/Reps-1.0/OverHeadReps));
  WriteLn ('Strings/sec = ',RepsPerSec:8:0);
end;

begin
  Init;
  Test;
  OverHead;
  CalcRate;
end.

{ ========================================================================== }
{ The simplicity of the rate timer is deceptive.  Don't let it fool you.  If }
{ you take a just a cursory look, you may think that you can only get 1/18.2 }
{ second resolution out this timer.  But think again.                        }
{                                                                            }
{ There are two clocks in your computer - (1) the 8253/8254 timer chip, and  }
{ (2) the CPU.  (One doesn't usually think of the CPU as a clock!)  Remember,}
{ the low-memory output for the clock comes from the 8253/8254 chip which is }
{ accurate to 1-microsecond, even though it doesn't "tick" but every         }
{ 1/18.2... seconds.                                                         }
{                                                                            }
{ Since we can't get high resolution from the low-memory clock, the program  }
{ does the reverse and gets the high resolution from the CPU instead.  There }
{ are two types of applications for a timer - (1) EVENT TIMER: duration of a }
{ a single event, and (2) RATE TIMER: the number of events in a single       }
{ duration.  Our application in STRG is the latter.  Where resolution (not   }
{ accuracy) is low in the timer, it is gained in the CPU by making sure      }
{ there are a LARGE number of events.                                        }
{                                                                            }
{ Notice that this timer is accurate to +/- 1 repetition as opposed to some  }
{ value in "seconds".  That's because it measures EVENTS, not seconds.  So,  }
{ a test of a given duration producing 100,000 events will certainly have    }
{ higher resolution than the same duration producing only 100 events.  The   }
{ former would have 0.001% resolution while the latter only 1%.  If you      }
{ increase the duration of the test, you can get even higher resolution.     }
{                                                                            }
{ RATE is meant to be a test tool for STRG to give a rough idea of           }
{ comparative performance and works quite well.                              }
{ ========================================================================== }
