/*
	heard.c -- Stations/nodes heard logging routines

  Poor Man's Packet (PMP)
  Copyright (c) 1991 by Andrew C. Payne    All Rights Reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation without fee for NON-COMMERCIAL AMATEUR RADIO USE ONLY is hereby
  granted, provided that the above copyright notice appear in all copies.
  The author makes no representations about the suitability of this software
  for any purpose.  It is provided "as is" without express or implied warranty.

	October, 1989
	Andrew C. Payne
*/
/* ----- Includes ----- */
#include <stdio.h>
#include <alloc.h>
#include <time.h>
#include <mem.h>
#include <string.h>
#include "pmp.h"
#include "keys.h"
#include "heard.h"

static struct mheard *heard;	/* MHEARD list */
int	nheard;                 /* number in mheard list */

extern KEY WaitKey(int prompt);

/* HeardInit()
	Initialize the HEARD list structure.
*/
void HeardInit()
{
	heard = malloc(MAXHEARD * sizeof(struct mheard));
	if(heard == NULL)
		OutOfMemory();
	nheard = 0;
}

/* Heard(l2)
	Heard list upcall for level 2 packets.  This routine is called once
	for all received level 2 packets.
*/
void Heard(struct ax25_packet *p)
{
	int	i;
	time_t	oldest;
	int	noldest;
	int	type;

	if(nheard) {
		oldest = 0x7fffffffL;
		for(i=0; i<nheard; i++) {
			if(!CompAX25Addr(&p->source,&heard[i].call))
				goto update;
			if(heard[i].last < oldest) {
				oldest = heard[i].last;
				noldest = i;
			}
		}
		if(nheard < MAXHEARD)
			i = nheard++;
		else
			i = noldest;	/* use oldest entry */
	} else
		i = nheard++;

	memcpy(&heard[i].call, &p->source, sizeof(struct ax25_addr));
	heard[i].bytes = heard[i].flags = heard[i].count = 0;
	heard[i].id[0] = '\0';

update:
	heard[i].last = time(NULL);
	heard[i].count++;

	type = FrameType(p->cont);
	switch(type) {
		case I:
		case UI:
			heard[i].bytes += p->dlen;
			heard[i].flags |= HEARD_INFO;
			switch(p->pid) {
				case PID_NETROM:
					heard[i].flags |= HEARD_NETROM;
					break;
			}
	}
	if(p->ndigis == 0)
		heard[i].flags |= HEARD_DIRECT;
}

/* timestr(t)
	Given a time in seconds since Jan 1, 1970.
*/
static char *timestr(time_t t)
{
	static char 	s[30];
	struct tm	*timeptr;

	timeptr = localtime(&t);
	sprintf(s,"%02d:%02d:%02d",
		timeptr->tm_hour, timeptr->tm_min, timeptr->tm_sec);

	return s;
}

/* DoHeard()
	Perform the nodes heard command

	(Probably should be in PMP.C)
*/
void DoHeard(void)
{
	int	i,j,t;
	char	s[80];
	int	sort[MAXHEARD];
	struct mheard	*p;

	SaveScreen(TRUE,TRUE);
	CenterTitle(1,"   Nodes Recently Heard   ");

	GotoXY(1,2);
	if(nheard) {
		_uputs(NormalAttr," Node          Time    Count  Bytes  Flags\n\n");

/* sort mheard list by time heard */
		for(i=0; i<nheard; i++)
			sort[i] = i;
		if(nheard > 1) {
			for(i=nheard-1; i>0; i--) {
				for(j=0; j<i; j++) {
					if(heard[sort[j]].last < heard[sort[j+1]].last) {
						t = sort[j];
						sort[j] = sort[j+1];
						sort[j+1] = t;
					}
				}
			}
		}

/* show mheard list */
		for(i=0; i<nheard; i++) {
			p = heard + sort[i];
			sprintf(s," %-9s%c  %s  %4ld  %6ld  ",
				GetAX25Addr(&p->call),
				(p->flags & HEARD_DIRECT) ? ' ' : '*',
				timestr(p->last),
				p->count,
				p->bytes);
			j = p->flags;
			if(j & HEARD_INFO)
				strcat(s,"Text ");
			if(j & HEARD_IP)
				strcat(s,"IP ");
			if(j & HEARD_NETROM)
				strcat(s,"NET/ROM ");
			strcat(s,"\n");
			_uputs(NormalAttr,s);
		}
	} else
		_uputs(NormalAttr,"      ***** None heard *****");

	(void)WaitKey(23);
	RestoreScreen();
}