/*
	PMP.H -- Main header file

  Poor Man's Packet (PMP)
  Copyright (c) 1991 by Andrew C. Payne    All Rights Reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation without fee for NON-COMMERCIAL AMATEUR RADIO USE ONLY is hereby
  granted, provided that the above copyright notice appear in all copies.
  The author makes no representations about the suitability of this software
  for any purpose.  It is provided "as is" without express or implied warranty.

	Andrew C. Payne
	August, 1989
*/

#include "types.h"
#include "dqueue.h"

/* ----- Defines ----- */
#define NETROM			/* include Net/Rom code */
/* #define	TRACE			 include trace/log code */
#define DEBUG			/* include debug code */

#define REMOTE			/* include remote commands */
#define VERSION	"1.1"		/* PMP Version Number */

#define TRUE	1
#define FALSE	0
#define FAIL	-1

#define MAXDIGIS	7	/* max digipeaters in path */
#define MAXCLEN		6	/* # of characters in callsign */

#define malloc(x)	farmalloc(x)

/* ----- Screen colors ----- */
#define normal()	(textattr(NormalAttr))
#define bright()	(textattr(BrightAttr))
#define inverse()	(textattr(InvAttr))

/* ----- Timing Stuff ----- */

#define	BITTIME		1988	/* 1200 baud bit time in Counter #0 ticks */
#define MILLISEC	1193	/* 1 millisecond */
#define BITSPERTICK	66	/* 66 bits per BIOS tick */

/* ----- BIOS Time Stuff ----- */
#define BiosTime()	((long)biostime(0,0))	/* reads the bios timer */
#define BIOSSEC		18			/* tics per second */

/* ----- Control field stuff ----- */
#define I	0x00		/* Information */
#define S	0x01		/* Supervisory */
#define RR	0x01		/* Receiver ready */
#define RNR	0x05		/* Receiver not ready */
#define REJ	0x09		/* Reject */
#define U	0x03		/* unnumbered frames */
#define SABM	0x2f		/* Connect request */
#define DISC	0x43		/* Disconnect */
#define DM	0x0f		/* disconnected mode */
#define UA	0x63		/* unnumbered acknowledge */
#define FRMR	0x87		/* frame reject */
#define UI	0x03		/* unnumbered information */
#define	PF	0x10		/* poll/final bit */

/* ----- Command bit ----- */
#define UNKNOWN	0		/* pre-AX.25 level 2 */
#define COMMAND	1
#define RESPONSE 2

/* ----- Bit masks ----- */
#define	SSIDMASK	0x1e	/* mask for SSID nibble */
#define REPEATED	0x80	/* has been repeated bit */

/* ----- AX.25 Level 3 Protocol ID bytes ----- */
#define PID_NETROM	0xcf		/* NET/ROM */
#define PID_TEXT	0xf0		/* no Level 3 */

/* ----- Global Structures ----- */
struct ax25_level1 {		/* Level 1 packet */
	int	len;		/* number of bytes */
	byte	data[1];	/* zero or more data bytes */
};

struct ax25_addr {		/* AX.25 address */
	byte	call[MAXCLEN];	/* Callsign, shifted left one bit */
	byte	ssid;		/* sub-station ID field */
};

struct ax25_packet {		/* Internal representation of packets */
	struct ax25_addr	dest;		/* destination */
	struct ax25_addr	source;		/* source */
	struct ax25_addr	digis[MAXDIGIS];
	char			repeated[MAXDIGIS];  /* has been repeated */
	int			ndigis;
	byte			cmdresp;	/* command/response bit */
	byte			cont;		/* control byte */
	byte			pid;		/* protocol ID */
	int			dlen;		/* data length */
	byte			data[1];	/* data bytes */
};

struct ax25_con	{		/* link control block */
	struct ax25_packet	header;		/* default packet */
	int			state;		/* state of this link */
	int			vs;		/* send sequence */
	int			vr;		/* receive sequence */
	int			tqs;		/* transmit queue seq */
	long			t1;		/* T1 Timeout */
	long			t2;		/* T2 Timeout */
	long			t3;		/* T3 Timeout */
	int			unack;		/* # of unacked packets */
	int			qsize;		/* # of packets in queue */
	byte			response;	/* enqueued response */
	byte			dreason;	/* reason for disconnect */
	byte			proto;		/* protocol version */
	int			retries;	/* current number of retries */
	int			remotebusy;	/* TRUE of got RNR */
	DQUEUE			dq;		/* outgoing data queue */
	char			type;		/* lins type (binary, text) */
};

struct param_cmd {		/* command dispatch table */
	char	*cmd;			/* command */
	int	(*handler)(char *s);    /* parameter line */
};

/* ----- Connection States ----- */
#define	DISCONNECTED	0	/* disconnected mode */
#define SETUP		1	/* link setup */
#define	DISCONNECTPEND	2	/* pending disconnect */
#define CONNECTED	3	/* link connected */
#define RECOVERY	4	/* link recovery */

/* ----- Protocol Versions ----- */
#define V1	1
#define V2	2

/* ----- Disconnect Reasons ----- */

#define DISC_LOCAL	1	/* local initiated, normal disconnect */
#define DISC_REMOTE	2	/* remote initiated, normal disconnect */
#define DISC_TIMEOUT	3	/* excessive retries */
#define DISC_BUSY	4	/* remote busy (attempted connect) */

/* ----- Line Termination Conventions ----- */
#define EOL_CR		1	/* lines terminate with CR */
#define EOL_LF		2	/* lines terminate with LF */
#define EOL_CRLF	3	/* lines terminate with CR/LF */

/* ----- Link Types ------ */
#define BINARY		1	/* binary stream */
#define TEXT		2	/* text stream */

/* ----- Global Variables ----- */

#ifndef EXTERN
#define EXTERN extern
#endif

#define MAXCTEXTS 5			/* maximum CTEXT lines */
EXTERN int	CallLength;		/* length of callsign string */

/* ----- Timing Parameters ----- */
EXTERN int	TXStartFlags;		/* number of opening SYNCs on each TX */
EXTERN int	TXEndFlags;		/* number of SYNCS at end of packet*/
EXTERN int	Frack;			/* timeout (seconds) */
EXTERN int	Check;			/* keep-alive (seconds) */
EXTERN int	BeaconInt;		/* beacon interval (seconds) */
EXTERN int	Ppersist;		/* p-persistence parameter */
EXTERN int	Slottime;		/* slot wait time */
EXTERN long	Pwait;			/* P-persist timeout */

EXTERN int	Retry;			/* maximum number of retries */

EXTERN struct ax25_addr MyCall;		/* user's callsign */
EXTERN struct ax25_con	AX25_Control;

EXTERN long	SoundEnd;		/* ending time of a sound */
EXTERN long	BeaconEnd;		/* time for next beacon */
EXTERN long	Pwait;			/* P-persist timeout */

/* ----- User Parameters ----- */
EXTERN int	DebugMode;		/* TRUE if in debug mode */
EXTERN int	PassMode;		/* TRUE if in PassAll mode */
EXTERN int 	Sound;			/* TRUE if sound is enabled */

EXTERN int	Pthresh;		/* I-frame threshhold */
EXTERN char	*ctext[MAXCTEXTS];	/* connect message lines */
EXTERN int	nctexts;		/* number of CTEXT lines */
EXTERN char	*btext;			/* beacon/CQ message */
EXTERN char	*baddr;			/* destination/path of beacons */
EXTERN int	MaxFrame;		/* max outstanding frames */
EXTERN int	Paclen;			/* max packet size */
EXTERN int	Ppersist;		/* p-persistence parameter */
EXTERN int	Slottime;		/* slot wait time */

EXTERN int	AutoWrap;		/* word wrap column */

EXTERN char	CaptureFile[40];	/* current capture filename */
EXTERN long	CaptureSize;		/* bytes in current capture file */
EXTERN long	CaptureBuffer;		/* size of capture buffer */

EXTERN KEY	fkeys[10][80];		/* function key definitions */
EXTERN char	ParamFname[40];		/* parameter file name */
EXTERN int	AutoMode;		/* automatic mode */
EXTERN int	ClockSkew;		/* attempt clock adjust */

/* ----- Counters ----- */
EXTERN long	RXCount;		/* number of good frames received */
EXTERN long	RXQOverflow;		/* number of RX Queue overflows */
EXTERN long	RXBOverflow;		/* number of RX Buffer overflows */
EXTERN long	RXFrameErr;		/* number of framing errors received */
EXTERN long	RXCRCErr;		/* number of CRC error received */
EXTERN long	RXFRMR;			/* number of FRMR frames received */
EXTERN long	RXREJ;			/* number of REJ frames received */
EXTERN long	TXCount;		/* number of frames transmitted */
EXTERN long	StartTime;		/* startup time */

/* ----- Screen Parameters ----- */
EXTERN int	monochrome;		/* TRUE if monochrome screen */
EXTERN int	NormalAttr;		/* normal attribute */
EXTERN int	InvAttr;		/* inverse attrubute */
EXTERN int	BrightAttr;		/* bright attribute */
EXTERN int	StatusAttr;		/* status line attribute */
EXTERN int	MsgAttr;		/* message attribute */

/* ----- Function Prototypes ----- */

/* ax25dump.c */
extern void ShowLevel2(struct ax25_packet *p2);
extern char *DumpLevel2(struct ax25_packet *p2);

/* ax25lapb.c */
extern int Connected(void);
extern void AX25_Init(void);
extern void AX25_Flush(void);
extern void AX25_Incoming(struct ax25_packet *p2);
extern void AX25_Open(void);
extern void AX25_Close(void);
extern void AX25_Expire(int n);
extern int LinkSend(byte *p, int l);
extern int AX25QFull(void);

/* ax25subr.c */
extern int SetAX25Addr(struct ax25_addr *a, char *s);
extern char *GetAX25Addr(struct ax25_addr *p);
extern int CompAX25Addr(struct ax25_addr *a1, struct ax25_addr *a2);
extern int FrameType(byte c);
extern int CmdResp(struct ax25_level1 *p);
extern int SetAX25Path(char *s, struct ax25_packet *a);
extern char *GetAX25Path(struct ax25_packet *p);
extern struct ax25_packet *AX25L1toL2(struct ax25_level1 *p);
extern int SendAX25(struct ax25_packet *p);
extern void ReversePath(struct ax25_packet *p1, struct ax25_packet *p2);
extern void SendBeacon(void);
extern void StartBeacon(void);
extern void SendWelcome(void);

/* capture.c */
extern void InitCapture(void);
extern void FlushCapture(void);
extern void Capture(char *s, int len);
extern void OpenCapture(char *fname);
extern void CloseCapture(void);
extern int Capturing(void);

/* console.c */
extern void curon(void);
extern void curoff(void);
extern int cursave(void);
extern void currest(int i);
extern void putstring(int x, int y, int len, byte attr, char *s);
extern void clear_area(int line, int start, int end);
extern void CRTInit(void);
extern void CRTExit(void);
extern int StartScrollback(void);
extern void EndScrollback(void);
extern int MoveScrollback(int n);
extern void WriteScrollback(char *fname);
extern void GotoLeft(void);
extern void ShowData(byte *s, int l);
extern void uprintf(byte attr, char *format, ...);
extern void uputs(byte attr, char *s);
extern void _uputs(byte attr, char *s);
extern void uputtext(byte attr, byte *s, int l);
extern void SaveScreen(int f, int border);
extern void RestoreScreen(void);
extern void CenterTitle(int line, char *t);
extern void GotoXY(int x, int y);
extern void StatusLine(void);
extern void ShowTXRX(int tx, int rx);
extern void NotifyStatus(int old, int new);
extern void Pause(int delay);
extern void Notify(char *s);
extern void SaveMessage(void);
extern void RestoreMessage(void);
extern int GetInput(char *prompt, char *s, int len);

/* heard.c */
extern void HeardInit(void);
extern void Heard(struct ax25_packet *p);
extern void DoHeard(void);

/* keys.c */
extern KEY Nextkey(void);
extern void AddKeystrokes(KEY *s);
extern void InitKeybuffer(void);
extern int convertkey(KEY *dest, char *src);

/* level1rx.c */
extern int RXCarrier(void);
extern int RXLevel1(void);
extern void RXProcess(void);
extern void RXInit(void);

/* level2rx.c */
extern void AX25Level2(struct ax25_level1 *p);

/* level1tx.c */
extern void ClockAdjust(int bitcount);
extern void TXKey(int state);
extern int CRCCheck(struct ax25_level1 *p);
extern void TXQInit(void);
extern int TXQAdd(struct ax25_level1 *packet);
extern void TXQEmpty(void);
extern int TXQBytesInQ(void);

/* level2tx.c */
extern int SendAX25(struct ax25_packet *p);

/* misc.c */
extern char *sob(char *s);
extern char *extract(char *s, char *p);
extern int convert(char *dest, char *src);
extern int dispatch(char *s, struct param_cmd *pcmds);
extern void trim(char *s);
extern void eol_out(int conv, char *src, int len);
extern void eol_in(int conv, char *src, int len);

/* param.c */
extern void InitParameters(void);
extern int ReadParameters(void);
extern char *extract(char *s, char *p);

/* pmp.c */
extern void PeriodicHook(void);
extern void StartSound(int freq, int len);
extern void Pause(int delay);
extern void OutOfMemory(void);

/* trace.c */
extern void LogInit(void);
extern void LogPacket(struct ax25_level1 *p, int dir);
extern void DumpLog(void);

#ifdef REMOTE
/* remote.c */
extern void command(char *s);
#endif

