/*
	keys.c -- Keystroke handling

  Poor Man's Packet (PMP)
  Copyright (c) 1991 by Andrew C. Payne    All Rights Reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation without fee for NON-COMMERCIAL AMATEUR RADIO USE ONLY is hereby
  granted, provided that the above copyright notice appear in all copies.
  The author makes no representations about the suitability of this software
  for any purpose.  It is provided "as is" without express or implied warranty.

	May, 1990
	Andrew C. Payne
*/

#include <stdio.h>
#include <bios.h>
#include <ctype.h>
#include "pmp.h"
#include "keys.h"

/* ----- Local Variables ----- */

static KEY altcodes[] = {		/* ALT-key scan codes A-Z */
	0x1e00, 0x3000, 0x2e00, 0x2000, 0x1200, 0x2100, 0x2200, 0x2300,
	0x1700, 0x2400, 0x2500, 0x2600, 0x3200, 0x3100, 0x1800, 0x1900,
	0x1000, 0x1300, 0x1f00, 0x1400, 0x1600, 0x2f00, 0x1100, 0x2d00,
	0x1500, 0x2c00 };

/* ----- Keystroke buffer handling ----- */
#define KEYBUFLEN 400
static KEY	keybuffer[KEYBUFLEN];		/* keystroke buffer */
static KEY	*keyhead, *keytail;

/* Nextkey()
	Returns the next available keypress code.  Returns 0 if no
	keystrokes available.
*/
KEY Nextkey(void)
{
	KEY	k;

	k = 0;
	if(keyhead != keytail) {
		k = *keyhead++;
		if(keyhead >= keybuffer + KEYBUFLEN)
			keyhead = keybuffer;
	} else if(keypressed())
		k = getkey();

	return k;
}

/* AddKeystrokes(k)
	Given a string of KEYS, adds keystrokes to the keyboard buffer.

	Needs checking for queue overflow.
*/
void AddKeystrokes(KEY *s)
{
	while(*s) {
		*keytail++ = *s++;
		if(keytail >= keybuffer + KEYBUFLEN)
			keytail = keybuffer;
	}
}

/* InitKeybuffer()
	Initializes things in the key buffer
*/
void InitKeybuffer(void)
{
	keytail = keyhead = keybuffer;
}

/* ----- String Conversion ----- */

/* convertkey(dest, src)
	Given a string in the form:  "testing^M", copies the string to the
	destination, converting all control characters to their IBM PC
	scancode representations.

	Returns TRUE if error.
*/
int convertkey(KEY *dest, char *src)
{
	int	c;

	while(*src && *src != '\n') {
		switch(*src) {
		case '^':		/* control character */
			if(src[1] == '^')
				c = '^';
			else {
				c = toupper(src[1]) - '@';
				if(c < 0 || c > ' ')
					return TRUE;
			}
			*dest++ = c;
			src += 2;
			break;
		case '~':		/* ALT- keypress */
			if(src[1] == '~') {
				*dest++ = '~';
			} else {
				c = toupper(src[1]) - 'A';
				if(c < 0 || c > 25)
					return TRUE;		/* error */
				*dest++ = altcodes[c];
			}
			src += 2;
			break;
		default:
			*dest++ = *src++;
			break;
		}
	}
	*dest = '\0';
	return FALSE;
}
