/*
 *      JET PAK - HP DeskJet and LaserJet series printer utilities
 *
 *      JETPRINT program - print a soft font file summary sheet
 *
 *      Version 1.1 (Public Domain)
 */

/* system include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* application include files */
#include "patchlev.h"
#include "jetfont.h"
#include "jetmean.h"
#include "jetutil.h"

/*
 * MODULE GLOBAL DATA
 */

/* buffer of characters to send to printer */
static char pb[BUFSIZ];

/* font command being processed */
static FONT_COMMAND fc;

/* optimum line spacing */
static UNSIGNEDINT font_lpi;

/* choose a lucky number to use for the temporary font */
#define FONTNUMBER 11459

/* these control the layout of the sheet; the idea is that it should
   work regardless of the printer's default font: */
#define MARGIN   200        /* margin used for all lines */
#define RHS      1000       /* position of right hand column in table */
#define DELTA    125        /* offset between chart columns */
#define CPR      16         /* characters per row */

/* line beginning and ending macros */
#define _LB sprintf(pb, "\033*p%dX", MARGIN); os_printstr(pb);
#define _LE sprintf(pb, "\r\n"); os_printstr(pb);

/*
 * LOCAL FUNCTIONS
 */

static void usage_wrong()
{
    /*
     * Print usage message and exit.
     */

    fprintf(stderr, USAGE_HEADER);
    fprintf(stderr, "Usage: JETPRINT [-h] fontfile [fontfile...]\n\n");
    fprintf(stderr, "Print soft font file summary sheets\n\n");
    fprintf(stderr, "  -h            print this usage information\n");
    fprintf(stderr, "  fontfile      soft font file name\n\n");
    exit(1);
}

static char *selection_sequence()
{
    static char retstr[100];

    if (fc.data.font.spacing == PROPORTIONAL)
    {
        sprintf(retstr, "Ec&l%dOEc(%d%cEc(s%du%dp%.1fv%ds%db%dt%dQ",
            fc.data.font.orientation,
            fc.data.font.symbol_set/32,
            '@'+fc.data.font.symbol_set%32,
            fc.data.font.placement,
            fc.data.font.spacing,
            quarter_dots_to_points(fc.data.font.height),
            fc.data.font.style,
            fc.data.font.stroke_weight,
            fc.data.font.typeface,
            fc.data.font.quality);
    }
    else
    {
        sprintf(retstr, "Ec&l%dOEc(%d%cEc(s%du%dp%.1fh%.1fv%ds%db%dt%dQ",
            fc.data.font.orientation,
            fc.data.font.symbol_set/32,
            '@'+fc.data.font.symbol_set%32,
            fc.data.font.placement,
            fc.data.font.spacing,
            quarter_dots_to_lpi(fc.data.font.pitch),
            quarter_dots_to_points(fc.data.font.height),
            fc.data.font.style,
            fc.data.font.stroke_weight,
            fc.data.font.typeface,
            fc.data.font.quality);
    }

    return(retstr);
}
static void headerprint(filename)
char *filename;
{
    /*
     * Print typographical information about the font
     */
    UNSIGNEDINT line_spacing;

    /* the optimum line spacing in the header may be corrupt: only use
       it if it lies within a reasonable range of the rule of thumb
       value (1.2 times the point size) */
    line_spacing = (fc.data.font.height*6)/5;
    if (    (fc.data.font.text_height > line_spacing/2)
         && (fc.data.font.text_height < line_spacing*2) )
        line_spacing = fc.data.font.text_height;

    font_lpi = (UNSIGNEDINT)quarter_dots_to_lpi(line_spacing);

    /* switch to correct page orientation for font first */
    sprintf(pb, "\033&l%dO", fc.data.font.orientation); os_printstr(pb);

    /* print the header information */
         _LB; sprintf(pb, "%s\033*p%dX%s", "File Name:",           RHS, filename); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Orientation:",         RHS, orientation_meaning(fc.data.font.orientation)); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Character Set:",       RHS, symbol_set_meaning(fc.data.font.symbol_set)); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Spacing:",             RHS, spacing_meaning(fc.data.font.spacing)); os_printstr(pb);
    if (fc.data.font.spacing == FIXED)
    {
    _LE; _LB; sprintf(pb, "%s\033*p%dX%.1f Characters per inch", "Pitch:", RHS, quarter_dots_to_lpi(fc.data.font.pitch)); os_printstr(pb);
    }
    _LE; _LB; sprintf(pb, "%s\033*p%dX%.1f Points", "Point Size:", RHS, quarter_dots_to_points(fc.data.font.height)); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Typestyle:",           RHS, style_meaning(fc.data.font.style)); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Stroke Weight:",       RHS, stroke_weight_meaning(fc.data.font.stroke_weight)); os_printstr(pb);
    _LE; _LB; sprintf(pb, "%s\033*p%dX%s", "Selection Sequence:",  RHS, selection_sequence()); os_printstr(pb);
    _LE;
}

static int rowprinted(font_type, row)
UNSIGNEDBYTE font_type; /* HP font type */
int row;                /* row number */
{
    /*
     * Returns whether a (16 character) row in the character chart
     * should be printed at all
     */

    switch (font_type)
    {
    case FONT_TYPE_7BIT:     /* 7-bit font */
        if (row >= 2 && row <= 7)
            return(TRUE);
        break;
    case FONT_TYPE_8BIT:     /* 8-bit font */
        if (row >= 2 && row <= 7)
            return(TRUE);
        if (row >= 10 && row <= 15)
            return(TRUE);
        break;
    case FONT_TYPE_IBM:      /* PC-8 */
        return(TRUE);
    }

    return(FALSE);
}

static int charprinted(font_type, code)
UNSIGNEDBYTE font_type;
int code;
{
    /*
     * Returns whether a character in the chart should be printed
     */
    switch (font_type)
    {
    case FONT_TYPE_7BIT:
        if (code >= 33 && code <= 127)
            return(TRUE);
        break;
    case FONT_TYPE_8BIT:
        if (code >= 33 && code <= 127)
            return(TRUE);
        if (code >= 160 && code <= 255)
            return(TRUE);
        break;
    case FONT_TYPE_IBM:
        if (code != 0x1b && (code < 0x07 || code > 0x0f))
            return(TRUE);
    }

    return(FALSE);
}

static void chartprint(font_type)
UNSIGNEDBYTE font_type;
{
    /*
     * Print a chart of the whole character set
     */
    int i, j;

    if (font_lpi < 6)
    {
        /* large font - set wider line spacing */
        sprintf(pb, "\033&l%dD", font_lpi);
        os_printstr(pb);
    }

    /* print column headers */
    _LE;
    for (i = 0; i < CPR; i++)
    {
        sprintf(pb, "\033*p%dX%02x", MARGIN + DELTA*(i+1), i);
        os_printstr(pb);
    }
    _LE;

    /* print rows */
    for (j = 0; j < 256; j += CPR)
    {
        if (rowprinted(font_type, j/CPR))
        {
            /* print row header */
            sprintf(pb, "\033*p%dX%02x", MARGIN, j);
            os_printstr(pb);

            /* change from default font to the font being sampled */
            sprintf(pb, "\016");
            os_printstr(pb);

            for (i = 0; i < CPR; i++)
            {
                if (charprinted(font_type, j+i))
                {
                    sprintf(pb, "\033*p%dX%c", MARGIN + DELTA*(i+1), j+i);
                    os_printstr(pb);
                }
            }

            /* change back to default font */
            sprintf(pb, "\017");
            os_printstr(pb);

            /* move to next line */
            _LE;
        }
    }

    _LE;
}

static char *poem[] = {
    "In Xanadu did Kubla Khan",
    "A stately pleasure-dome decree:",
    "Where Alph, the sacred river, ran",
    "Through caverns measureless to man",
    "Down to a sunless sea.",
    "So twice five miles of fertile ground",
    "With walls and towers were girdled round.",
};

static void textprint()
{
    int i;

    /* set optimum line spacing for font */
    sprintf(pb, "\033&l%dD", font_lpi);
    os_printstr(pb);

    /* change to the downloaded font */
    sprintf(pb, "\016");
    os_printstr(pb);

    for (i = 0; i < sizeofarray(poem); i++)
    {
        _LB; os_printstr(poem[i]); _LE;
    }
}

static void jetprint()
{
    char inpath[OS_PATH_LEN];
    FILE *infp;
    int r;
    UNSIGNEDBYTE font_type;

    /* build the input file path */
    strcpy(inpath, os_dir);
    strcat(inpath, os_file);

    if (!(infp = fopen(inpath, "rb")))
    {
        fprintf(stderr, ERROR_OPEN_READ_FAILED, os_dir, os_file);
        return;
    }

    /* just read the header information */
    while (    ((r = font_command_read(infp, &fc)) == OK)
            && (fc.command_type != FDC) )
        ;

    if (r != OK)
    {
        /* font header not found - probably not a soft font file */
        fprintf(stderr, ERROR_FILE_READ_FAILED, os_dir, os_file);
        fclose(infp);
        return;
    }

    /* reset printer, set CR and LF to be interpreted as is */
    sprintf(pb, "\033E\033&k0G");
    os_printstr(pb);

    /* download the font as temporary and secondary */
    sprintf(pb, "\033*c%dD\033*c4F", FONTNUMBER);
    os_printstr(pb);

    rewind(infp);
    while ((r = fread(pb, 1, BUFSIZ, infp)) > 0)
        os_printbuf(pb, r);

    /* print information from header */
    headerprint(inpath);
    font_type = fc.data.font.type;

    /* select the downloaded font as the secondary font */
    sprintf(pb, "\033)%dX", FONTNUMBER);
    os_printstr(pb);

    /* print the character set chart */
    chartprint(font_type);

    /* print the sample text */
    textprint();

    /* reset printer */
    sprintf(pb, "\033E");
    os_printstr(pb);
    fclose(infp);

    fprintf(stderr, OK_JETPRINT, os_dir, os_file);
}

main(argc, argv)
int argc;
char *argv[];
{
    char c;

    /* stop getopt() printing errors */
    opterr = FALSE;
    while ((c = getopt(argc, argv, "h")) != EOF)
    {
        switch (c)
        {
        case 'h':
        case '?':
            /* help required, or invalid option specified */
            usage_wrong();
        }
    }

    /* must specify at least one file */
    if (optind >= argc)
        usage_wrong();

    /* process file arguments */
    if (os_findfiles((argc - optind), &argv[optind]) == ERROR)
        fprintf(stderr, ERROR_OUT_OF_HEAP);

    while (os_getfile() != EOF)
        jetprint();

    return(0);
}
