/***********************************************************************/
/* COMM4.C - Commands P-S                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991,1992 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@itc.gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 7877
 * QLD 4121
 * Australia
 */
#include <stdio.h>
#include <stdlib.h>

#include "the.h"

/*#define DEBUG 1*/

/*-------------------------- external data ----------------------------*/
extern LINE *next_line,*curr_line;
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern char display_screens;                      /* number of screens */
extern char number_of_views;                   /* number of open files */
extern bool horizontal;
extern char current_file;         /* pointer to current file */
extern WINDOW *foot,*error_window,*divider;
extern char error_on_screen;
extern unsigned char *rec;
extern unsigned short rec_len;
extern unsigned char *cmd_rec;
extern unsigned short cmd_rec_len;
extern unsigned char mode_insert;        /* defines insert mode toggle */
extern unsigned char in_profile;    /* indicates if processing profile */
extern unsigned char *tempfilename;
extern unsigned short file_start;

extern unsigned char temp_cmd[150];
extern unsigned char dir_filename[80];
extern unsigned char dir_pathname[80];
extern unsigned char sp_path[MAX_FILE_NAME+1] ;
extern unsigned char sp_fname[MAX_FILE_NAME+1] ;
extern unsigned char dir_path[MAX_FILE_NAME+1] ;    /* for dir and ls commands */
extern bool clear_command;
extern short save_coord_x[VIEW_WINDOWS];
extern short save_coord_y[VIEW_WINDOWS];
/*---------------------- function definitions -------------------------*/
#ifdef PROTO
void split_command(unsigned char *,unsigned char *,unsigned char *);
int param_split(unsigned char *,unsigned char *[],int );
long valid_target(unsigned char *);
int get_row_for_focus_line(int,long,long);
unsigned char *get_key_definition(unsigned short key);
void print_line(char,long,short,unsigned char *,unsigned char *);
bool valid_set_command(char *);
unsigned char *get_command(char);
#else
void split_command();
int param_split();
long valid_target();
int get_row_for_focus_line();
unsigned char *get_key_definition();
void print_line();
bool valid_set_command();
unsigned char *get_command();
#endif
/*man-start*********************************************************************
COMMAND
     prefix - set prefix area attributes

SYNTAX
     PREfix ON|OFF [Left|Right]

DESCRIPTION
     The PREFIX set command determines if the prefix area is displayed
     and if so, where it is displayed.
     Executed from within the profile, the only effect is that the
     defaults for all files is changed.
     Executed from the command line, the PREFIX command changes the
     current window displays to reflect the required options.

COMPATIBILITY
     Does not support PREFIX NULLS or PREFIX SYNONYM.

DEFAULT
     PREFIX ON LEFT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Prefix(unsigned char *params)
#else
int Prefix(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern unsigned char PRE_ONx;
extern unsigned char PRE_LEFTx;
/*--------------------------- local data ------------------------------*/
#define PRE_PARAMS  2
 unsigned char *word[PRE_PARAMS+1];
 char parm[PRE_PARAMS];
 register int i;
 unsigned short num_params;
 unsigned short x,y;
 char pre_col;
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Prefix");
#endif
 num_params = param_split(params,word,PRE_PARAMS);
 for (i=0;i<PRE_PARAMS;i++)
     parm[i] = UNDEFINED_OPERAND;
 if (equal("on",word[0],2))
    parm[0] = TRUE;
 if (equal("off",word[0],3))
    parm[0] = FALSE;
 if (parm[0] == UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 switch(parm[0])
    {
     case TRUE:                                           /* prefix on */
         if (equal("left",word[1],1))
            parm[1] = TRUE;
         if (equal("right",word[1],1))
            parm[1] = FALSE;
         if (parm[1] == UNDEFINED_OPERAND)
           {
            display_error(1,word[1]);
#ifdef TRACE
            trace_return();
#endif
            return(OK);
           }
         break;
     case FALSE:
         if (strcmp(word[1],"") != 0)
           {
            display_error(1,word[1]);
#ifdef TRACE
            trace_return();
#endif
            return(OK);
           }
         break;
    }
 PRE_ONx = parm[0];
 PRE_LEFTx = parm[1];
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }

/*---------------------------------------------------------------------*/
/* Now for the real work. Depending on the current state of the prefix */
/* and the intentions of the user, turn on or off the prefix area and  */
/* rearrange the MAIN window.                                          */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->prefix_on)
   {
    case TRUE:                               /* prefix is currently on */
         switch(parm[0])
           {
            case TRUE:
/*---------------------------------------------------------------------*/
/* If the prefix is already on and in the same place as requested,     */
/* ignore the command.                                                 */
/*---------------------------------------------------------------------*/
                 if (CURRENT_VIEW->prefix_left == parm[1])
                    break;
/*---------------------------------------------------------------------*/
/* Now we have to swap the position of the MAIN and PREFIX windows.    */
/* The simplest way is to move the windows to their new positions.     */
/*---------------------------------------------------------------------*/
                 if (CURRENT_VIEW->prefix_left)        /* prefix - left */
                   {
                    CURRENT_SCREEN.top_col -= PREFIX_WIDTH;
                    pre_col = CURRENT_SCREEN.origin_x+CURRENT_SCREEN.screen_cols-PREFIX_WIDTH;
                   }
                 else
                   {
                    CURRENT_SCREEN.top_col += PREFIX_WIDTH;
                    pre_col = CURRENT_SCREEN.origin_x;
                   }
                 rc = mvwin(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.top_row,
                                           CURRENT_SCREEN.top_col);
                 rc = mvwin(CURRENT_WINDOW_PREFIX,CURRENT_SCREEN.top_row,
                                           pre_col);
                 CURRENT_VIEW->prefix_left = parm[1];
                 break;
            case FALSE:                     /* prefix to be turned off */
                 delwin(CURRENT_WINDOW_PREFIX);
                 getyx(CURRENT_WINDOW_MAIN,y,x);
                 delwin(CURRENT_WINDOW_MAIN);
                 CURRENT_SCREEN.cols += PREFIX_WIDTH;
                 CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x;
                 CURRENT_VIEW->prefix_on = FALSE;
                 CURRENT_WINDOW_MAIN = newwin(CURRENT_SCREEN.rows,
                                              CURRENT_SCREEN.cols,
                                              CURRENT_SCREEN.top_row,
                                              CURRENT_SCREEN.top_col);
                 wattrset(CURRENT_WINDOW_MAIN,colour[ATTR_FILEAREA]);
#if !defined(SUN) && !defined(VMS)
                 keypad(CURRENT_WINDOW_MAIN,TRUE);
#endif
                 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
                    CURRENT_VIEW->previous_window = WINDOW_MAIN;
                 else
                   {
                    CURRENT_VIEW->current_window = WINDOW_MAIN;
                    CURRENT_VIEW->previous_window = WINDOW_COMMAND;
                   }
                 show_page();
                 wmove(CURRENT_WINDOW_MAIN,y,x);
                 break;
           }
         break;
    case FALSE:                             /* prefix is currently off */
         switch(parm[0])
           {
            case TRUE:
/*---------------------------------------------------------------------*/
/* Determine where the prefix is to go.                                */
/*---------------------------------------------------------------------*/
                 if (parm[1])                  /* prefix to go on left */
                   {
                    pre_col = CURRENT_SCREEN.origin_x;
                    CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x+PREFIX_WIDTH;
                   }
                 else
                    pre_col = CURRENT_SCREEN.origin_x+CURRENT_SCREEN.screen_cols-PREFIX_WIDTH;
                 CURRENT_SCREEN.cols -= PREFIX_WIDTH;
/*---------------------------------------------------------------------*/
/* Get position of cursor in MAIN window. Delete the existing MAIN     */
/* window and recreate it with a reduced size. Also create the PREFIX  */
/* window at the correct location.                                     */
/*---------------------------------------------------------------------*/
                 getyx(CURRENT_WINDOW_MAIN,y,x);
                 delwin(CURRENT_WINDOW_MAIN);
                 CURRENT_VIEW->prefix_on = TRUE;
                 CURRENT_VIEW->prefix_left = parm[1];
                 CURRENT_WINDOW_MAIN = newwin(CURRENT_SCREEN.rows,
                                              CURRENT_SCREEN.cols,
                                              CURRENT_SCREEN.top_row,
                                              CURRENT_SCREEN.top_col);
                 wattrset(CURRENT_WINDOW_MAIN,colour[ATTR_FILEAREA]);
#if !defined(SUN) && !defined(VMS)
                 keypad(CURRENT_WINDOW_MAIN,TRUE);
#endif
                 CURRENT_WINDOW_PREFIX = newwin(CURRENT_SCREEN.rows,PREFIX_WIDTH,
                                         CURRENT_SCREEN.top_row,pre_col);
                 wattrset(CURRENT_WINDOW_PREFIX,colour[ATTR_PREFIX]);
#if !defined(SUN) && !defined(VMS)
                 keypad(CURRENT_WINDOW_PREFIX,TRUE);
#endif
                 show_page();
/*---------------------------------------------------------------------*/
/* If the previous column position is now no longer valid, set the     */
/* column position to 0.                                               */
/*---------------------------------------------------------------------*/
                 if (x > CURRENT_SCREEN.cols)
                    x = 0;
                 wmove(CURRENT_WINDOW_MAIN,y,x);
                 break;
            case FALSE:
/*---------------------------------------------------------------------*/
/* If the prefix is already off, ignore request to turn it off.        */
/*---------------------------------------------------------------------*/
                 break;
           }
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     print - send text to default printer or print spooler

SYNTAX
     PRint [target] [n]
     PRint LINE [text]
     PRint STRING [text]
     PRint FORMfeed
     PRint CLOSE

DESCRIPTION
     The PRINT command writes a portion of the current file to the default
     printer or print spooler, or text entered on the command line.
     PRINT LINE sends the remainder of the text on the command line to
     the printer followed by a CR/LF (DOS) or LF(UNIX).
     PRINT STRING sends the remainder of the text on the command line to
     the printer without any trailing line terminator.
     PRINT FORMFEED sends a formfeed (^L) character to the printer.
     PRINT CLOSE closes the printer spooler.
     PRINT target sends text from the file contents up to the target to
     the printer followed by a CR/LF (DOS) or LF(UNIX) after each line.
     When an optional [n] isspecified, this sends a formfeed after [n]
     successive lines of text.

COMPATIBILITY
     Does not allow for different printer ports or print spoolers...yet.

SEE ALSO
     printer

STATUS
     Incomplete
**man-end**********************************************************************/
#ifdef PROTO
int Print(unsigned char *params)
#else
int Print(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define PRT_PARAMS  2
 unsigned char *word[PRT_PARAMS+1];
 unsigned short num_params;
 long num_lines;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Print");
#endif
 num_params = param_split(params,word,PRT_PARAMS);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (unsigned char *)"1";
   }
 post_process_line(CURRENT_VIEW->focus_line);
 if (equal("line",word[0],4))
    print_line(NO,0L,0,(unsigned char *)word[1],
#if defined(DOS) || defined(OS2)
              (unsigned char *)"\r\n");
#else
              (unsigned char *)"\r");
#endif
 else
    if (equal("string",word[0],5))
       print_line(NO,0L,0,(unsigned char *)word[1],(unsigned char *)"");
    else
       if (equal("formfeed",word[0],4))
         {
          if (num_params > 1)
            {
             display_error(1,word[1]);
#ifdef TRACE
             trace_return();
#endif
             return(ERROR);
            }
          print_line(NO,0L,0,(unsigned char *)"",(unsigned char *)"\f");
         }
       else
          if (equal("close",word[0],5))
            {
             if (num_params > 1)
               {
                display_error(1,word[1]);
#ifdef TRACE
                trace_return();
#endif
                return(ERROR);
               }
             print_line(YES,0L,0,(unsigned char *)"",(unsigned char *)"");
            }
          else
            {
             if ((num_lines = valid_target(word[0])) == TARGET_ERROR)
               {
                display_error(4,word[0]);
#ifdef TRACE
                trace_return();
#endif
                return(ERROR);
               }
             if (num_params < 2)
                word[1] = (unsigned char *)"0";
             if (valid_positive_integer(word[1]) == NO)
               {
                display_error(4,word[0]);
#ifdef TRACE
                trace_return();
#endif
                return(ERROR);
               }
             print_line(NO,num_lines,atoi(word[1]),(unsigned char *)"",
                        (unsigned char *)"");
            }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     printer - define printer spooler name

SYNTAX
     PRINTER spooler

DESCRIPTION
     The PRINTER command sets up the print spooler name to determine
     where output from the PRINT command goes.

COMPATIBILITY
     Compatible.

SEE ALSO
     print

STATUS
     Incomplete
**man-end**********************************************************************/
#ifdef PROTO
int Printer(unsigned char *params)
#else
int Printer(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
#ifdef UNIX
 extern unsigned char *spooler_name;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Printer");
#endif
 if ((spooler_name = (unsigned char *)memRealloc(spooler_name,
                                                 strlen(params)+1,
                                                 "realloc spooler_name"))
                                            == NULL)
   {
    display_error(30,"");
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 strcpy(spooler_name,params);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     put - write part of a file to another

SYNTAX
     PUT [target] [fileid]

DESCRIPTION
     The PUT command writes a portion of the current file to another
     file, either explicit or temporary.
     When no fileid is supplied the temporary file is overwritten.
     When a fileid is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     Complete.

SEE ALSO
     put,get

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Put(unsigned char *params)
#else
int Put(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define PUT_PARAMS  2
 unsigned char *word[PUT_PARAMS+1];
 unsigned short num_params;
 long num_lines;
 char append;
 unsigned char *filename;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Put");
#endif
 num_params = param_split(params,word,PUT_PARAMS);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (unsigned char *)"1";
   }
 if (num_params > 2)
   {
    display_error(1,word[2]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 if ((num_lines = valid_target(word[0])) == TARGET_ERROR)
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 if (strcmp(word[1],"") == 0)   /* no fileid supplied */
   {
    append = NO;
    filename = tempfilename;
   }
 else
   {
    append = YES;
    filename = word[1];
   }

 post_process_line(CURRENT_VIEW->focus_line);
 if (save_file(filename,YES,num_lines,append) == ERROR)
   {
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     putd - write part of a file to another and delete

SYNTAX
     PUTD [target] [fileid]

DESCRIPTION
     The PUTD command writes a portion of the current file to another
     file, either explicit or temporary and deletes those lines written.
     When no fileid is supplied the temporary file is overwritten.
     When a fileid is supplied the portion of the file written out
     is appended to the specified file.

COMPATIBILITY
     Complete.

SEE ALSO
     put,get

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Putd(unsigned char *params)
#else
int Putd(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define PUTD_PARAMS  2
 unsigned char *word[PUTD_PARAMS+1];
 unsigned short num_params;
 long num_lines;
 char append;
 unsigned char *filename;
 extern unsigned char *tempfilename;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Putd");
#endif
 num_params = param_split(params,word,PUTD_PARAMS);
 if (num_params == 0)
   {
    num_params = 1;
    word[0] = (unsigned char *)"1";
   }
 if (num_params > 2)
   {
    display_error(1,word[2]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 if ((num_lines = valid_target(word[0])) == TARGET_ERROR)
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 if (strcmp(word[1],"") == 0)   /* no fileid supplied */
   {
    append = NO;
    filename = tempfilename;
   }
 else
   {
    append = YES;
    filename = word[1];
   }

 post_process_line(CURRENT_VIEW->focus_line);
 if (save_file(filename,YES,num_lines,append) == ERROR)
   {
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }

 Delete_line(word[0]);

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     quit - exit from the current file

SYNTAX
     QUIT

DESCRIPTION
     The QUIT command exits the user from the current file, provided
     no changes have been made to the file. An error message will be
     displayed if changes have been made.
     The previous file in the ring then becomes the current file.
     If the current file is the only file in the ring, the user is
     returned to the Operating System.

COMPATIBILITY
     Complete.

SEE ALSO
     qquit

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Quit(unsigned char *params)
#else
int Quit(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Quit");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 if (CURRENT_FILE->save_alt > 0)
   {
    display_error(22,"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 free_view_memory();
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     qquit - exit from the current file

SYNTAX
     QQuit

DESCRIPTION
     The QQUIT command exits the user from the current file, whether
     changes have been made to the file or not.
     The previous file in the ring then becomes the current file.
     If the current file is the only file in the ring, the user is
     returned to the Operating System.

COMPATIBILITY
     Complete.

SEE ALSO
     quit

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Qquit(unsigned char *params)
#else
int Qquit(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Qquit");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 free_view_memory();
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     query - display various option settings

SYNTAX
     Query RING|etc

DESCRIPTION
     The QUERY command displays the various settings for options set
     by THE.

COMPATIBILITY
     Shows some options.

STATUS
     Not started.
**man-end**********************************************************************/
#ifdef PROTO
int Query(unsigned char *params)
#else
int Query(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Query");
#endif
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     recover - recover changed or deleted lines

SYNTAX
     RECover [n|*]

DESCRIPTION
     The RECOVER command restores the last n changed or deleted lines
     back into the body of the file.

COMPATIBILITY
     Compatible.

STATUS
     Just started.
**man-end**********************************************************************/
#ifdef PROTO
int Recover(unsigned char *params)
#else
int Recover(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define REC_PARAMS  1
 unsigned char *word[REC_PARAMS+1];
 char parm[REC_PARAMS];
 unsigned short num_params;
 int num;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Recover");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero or '*'.    */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,REC_PARAMS);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (unsigned char *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,word[1]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
 if (strcmp(word[0],"*") == 0)
   {
    num = 99;
   }
 else
   {
    if (!valid_positive_integer(word[0]))
       {
        display_error(4,word[0]);
#ifdef TRACE
        trace_return();
#endif
        return(OK);
       }
    num = atol(word[0]);
   }

 get_from_recovery_list(num);

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     refresh - redraw the current screen

SYNTAX
     REFresh

DESCRIPTION
     The REFRESH command redraws the current contents of the screen.
     This is usually used when some outside influence has stuffed the
     display up.

COMPATIBILITY
     Not applicable.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Refresh(unsigned char *params)
#else
int Refresh(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Refresh");
#endif
 if (strcmp(params,"") != 0)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 wrefresh(curscr);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     reset - cancel the marked block or prefix commands or both

SYNTAX
     RESet ALL|Block|Prefix

DESCRIPTION
     The RESET command unmarks any marked block or outstanding prefix
     commands or both.

COMPATIBILITY
     Only supports BLOCK... at the moment.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Reset(unsigned char *params)
#else
int Reset(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define RES_PARAMS  1
 unsigned char *word[RES_PARAMS+1];
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Reset");
#endif
 num_params = param_split(params,word,RES_PARAMS);
 if (num_params > 1)
   {
    display_error(1,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }

 if (equal("block",word[0],5)
 ||  equal("all",word[0],3)
 ||  num_params == 0)
   {
    if (MARK_VIEW != (VIEW_DETAILS *)NULL)
      {
       MARK_VIEW->mark_start_line = MARK_VIEW->mark_end_line = (-1L);
       MARK_VIEW = (VIEW_DETAILS *)NULL;
       show_page();
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     retrieve - return the next/prior command

SYNTAX
     ? [+]

DESCRIPTION
     The ? command returns the next or prior command from the command
     line stack and displays it on the command line.
     With no parameters, the most recent command entered on the command
     line is retrieved.

COMPATIBILITY
     Does not support multiple '?' as in ??? to retrieve the third last
     command.
     This command is bound to the up and down arrows when on the
     command line.

SEE ALSO
     cmdarrows
STATUS
     Incomplete. Arrows key binding is functional.
**man-end**********************************************************************/
#ifdef PROTO
int Retrieve(unsigned char *params)
#else
int Retrieve(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 char which_way;
 unsigned char *current_command;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Retrieve");
#endif
 if (strcmp(params,"") == 0)
    which_way = DIRECTION_FORWARD;
 else
    if (strcmp(params,"+") == 0)
      which_way = DIRECTION_BACKWARD;
    else
      {
       display_error(1,params);
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }

 wmove(CURRENT_WINDOW_COMMAND,0,0);
 wclrtoeol(CURRENT_WINDOW_COMMAND);
 current_command = get_command(which_way);
 if (current_command != (unsigned char *)NULL)
   {
    waddstr(CURRENT_WINDOW_COMMAND,current_command);
    strcpy(cmd_rec,current_command);
    cmd_rec_len = strlen(cmd_rec);
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     right_arrow - move the cursor to the right

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The right_arrow key causes the cursor to move one column to the
     right. If the cursor is on the right most column of the main
     window, the window will scroll half a window.

COMPATIBILITY
     Does not line wrap when encountering the right edge of the screen.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Right_arrow(unsigned char *params)
#else
int Right_arrow(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Right_arrow");
#endif
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Check for going past end of line - MAX_LINE_LENGTH                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->verify_col+x+1 > MAX_LINE_LENGTH)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* For all windows, if we are not at right column, move 1 pos to right.*/
/*---------------------------------------------------------------------*/
 if (x < CURRENT_SCREEN.cols-1)
   {
    wmove(CURRENT_WINDOW,y,x+1);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* For MAIN window, if we are at right column, shift to right half the */
/* width of the main window.                                           */
/* Check for going past end of line - MAX_LINE_LENGTH                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_MAIN)
    if (x == CURRENT_SCREEN.cols-1)
      {
       x = CURRENT_SCREEN.cols / 2;
       CURRENT_VIEW->verify_col += x;
       show_page();
       wmove(CURRENT_WINDOW,y,x);
      }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     save - save changes to current file

SYNTAX
     SAVE [fileid]

DESCRIPTION
     The save command writes the current file to disk. If a fileid is
     supplied, the current file is saved in that file, unless the file
     already exists when an error is displayed.
     The 'Alterations' counter on the heading line is reset to zero.

COMPATIBILITY
     Complete

SEE ALSO
     ssave, file, ffile

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Save(unsigned char *params)
#else
int Save(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Save");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
 rc =save_file(params,NO,0L,NO);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     schange - selectively change strings

SYNTAX
     Change /string1/string2/ [target] [n] [m]

DESCRIPTION
     The CHANGE command changes one string of text to another.

     The first parameter to the change command is the old and new
     string values, seperated by delimiters.
     The allowable delimiters are '/' '\' and '@'.

     The second parameter is the target; how many lines are to be
     searched for occurrences of the first string to be changed.

     The third parameter determines how many occurrences of 'string1'
     are to be changed on each line.

     The fourth parameter determines at which occurrences of 'string1'
     on the line are changes to commence.

COMPATIBILITY
     Compatible.

DEFAULT
     1 1 1

SEE ALSO
     change

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Schange(unsigned char *params)
#else
int Schange(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Schange");
#endif

 rc = execute_change_command(params,TRUE);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     screen - specifiy number of screens displayed

SYNTAX
     SCReen n [Horizontal|Vertical]

DESCRIPTION
     The SCREEN command specifies the number of views of file(s) to
     display on screen at once. If the number of views specified is 2
     and only one file is currently in the ring, two views of the
     same file are displayed.
     Only 2 views are supported.

COMPATIBILITY
     Only supports above option.

STATUS
     Not complete. (Unavailable)
**man-end**********************************************************************/
#ifdef PROTO
int Screen(unsigned char *params)
#else
int Screen(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SCR_PARAMS  2
 unsigned char *word[SCR_PARAMS+1];
 register int i;
 unsigned short num_params,num_views;
 char save_display_screens;
 bool save_horizontal;
 unsigned short x,y;
 char horiz=(-1);
 VIEW_DETAILS *save_current_view;

#ifndef VMS
 SCREEN_DETAILS *screen0=&screen[0];
 SCREEN_DETAILS *screen1=&screen[1];
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Screen");
#endif

 num_params = param_split(params,word,SCR_PARAMS);
 if (!valid_positive_integer(word[0]))
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if ((num_views = atoi(word[0])) > MAX_SCREENS)
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (num_views == 1)
    horiz = TRUE;
 else
   {
    if (equal("horizontal",word[1],1))
       horiz = TRUE;
    if (equal("vertical",word[1],1))
       horiz = FALSE;
    if (horiz == (-1))
      {
       display_error(1,word[1]);
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
   }
/*---------------------------------------------------------------------*/
/* Set the global variable display_screens to indicate the number of   */
/* screens currently displayed and the orientation of those screens    */
/* Save the old values first so we know how the screens were oriented. */
/*---------------------------------------------------------------------*/
 save_display_screens = display_screens;
 save_horizontal = horizontal;

 display_screens = (char)num_views;
 horizontal=(bool)horiz;

 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* If there is no change to the screens, exit.                         */
/*---------------------------------------------------------------------*/
 if (display_screens == save_display_screens
 &&  horizontal == save_horizontal)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* CURRENTLY ONLY FOR MAKING 2 VIEWS. NOT FOR GOING BACK TO 1 VIEW.....*/
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW->focus_line);

 save_current_view = CURRENT_VIEW;

 set_screen_defaults();

 if (!horizontal && display_screens > 1)
    wnoutrefresh(divider);

 if (display_screens == 1)
   {
    if (CURRENT_SCREEN.screen_view->file_for_view == OTHER_SCREEN.screen_view->file_for_view)
      {
       CURRENT_VIEW = OTHER_SCREEN.screen_view;
       free_view_memory();
       CURRENT_VIEW = save_current_view;
      }
    else
      {
       CURRENT_VIEW = OTHER_SCREEN.screen_view;
       for (i=0;i<VIEW_WINDOWS;i++)
           getyx(CURRENT_VIEW->win[i],save_coord_y[i],save_coord_x[i]);
       getyx(CURRENT_WINDOW,y,x);
       delete_windows();
       set_up_windows();
       for (i=0;i<VIEW_WINDOWS;i++)
           wmove(CURRENT_VIEW->win[i],save_coord_y[i],save_coord_x[i]);
       y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                            CURRENT_VIEW->focus_line,
                            CURRENT_VIEW->current_line);
       if (x > CURRENT_SCREEN.cols)
          x = 0;
       wmove(CURRENT_WINDOW,y,x);
       CURRENT_VIEW = save_current_view;
      }
    current_screen = 0;
    CURRENT_SCREEN.screen_view = CURRENT_VIEW;
   }

 getyx(CURRENT_WINDOW,y,x);
 delete_windows();
 set_up_windows();
 repaint_screen();

 if (display_screens == 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/* now create the new view, if this is the only file */

 if (number_of_views == 1)
   {
    if (defaults_for_other_files() == ERROR)
      {
#ifdef TRACE
       trace_return();
#endif
       return(ERROR);
      }
    CURRENT_FILE = save_current_view->file_for_view;
    CURRENT_FILE->file_views++;
    x = y = 0;
   }
 else
   {
    if (NEXT_VIEW == (VIEW_DETAILS *)NULL)
       CURRENT_VIEW = vd_first;
    else
       CURRENT_VIEW = NEXT_VIEW;
    getyx(CURRENT_WINDOW,y,x);
    delete_windows();
   }
 current_screen = 1;
 set_up_windows();
 CURRENT_SCREEN.screen_view = CURRENT_VIEW;
 repaint_screen();
 if (!horizontal)
   {
    CURRENT_VIEW = save_current_view;
    current_screen = 0;
    pre_process_line(CURRENT_VIEW->focus_line);
   }

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     set - execute various set commands

SYNTAX
     SET set command [set command parameter ...]

DESCRIPTION
     The SET command is a front end to existing 'set' commands. It treats
     the first parameter it receives as a command and executes it.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Set(unsigned char *params)
#else
int Set(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SET_PARAMS  2
 unsigned char *word[SET_PARAMS+1];
 unsigned short num_params;
 unsigned char set_cmd[80];
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Set");
#endif
 strcpy(set_cmd,params);
 num_params = param_split(params,word,SET_PARAMS);
 if (num_params < 1)
   {
    display_error(1,"?????");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (!valid_set_command(word[0]))
    {
     display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
     return(OK);
    }
 rc = command_line(set_cmd);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     showkey - display current key value and command assignation

SYNTAX
     SHOWKey KEY

DESCRIPTION
     The SHOWKEY command prompts the user to enter a key and responds
     with the key name and associated command (if applicable).

COMPATIBILITY
     New feature.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int ShowKey(unsigned char *params)
#else
int ShowKey(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short key;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   ShowKey");
#endif
 display_error(0,"Press the key to be translated...spacebar to exit");
 touchwin(error_window);
 doupdate();
 key = 0;
 while(key != ' ')
   {
    key = my_getch(CURRENT_WINDOW);
    display_error(0,get_key_definition(key));
    wrefresh(error_window);
   }
 error_on_screen = NO;
 touchwin(foot);

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     spltjoin - split/join two lines

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The SPLTJOIN command splits the focus line into two or joins the
     focus line with the next line depending on the position of the
     cursor. If the cursor is after the last column of a line, a 'join'
     is executed, otherwise a 'split' is executed.

COMPATIBILITY
     Compatible.

STATUS
     Incomplete. Does not have [aligned] option.
**man-end**********************************************************************/
#ifdef PROTO
int Spltjoin(unsigned char *params)
#else
int Spltjoin(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short x,y,col;
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Spltjoin");
#endif
/*---------------------------------------------------------------------*/
/* Check here for parameter value of 'Aligned' and process accordingly.*/
/* As you can see, this has not been implemented yet.                  */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window != WINDOW_MAIN)
   {
    display_error(38,"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Reject the command if on the top or bottom line.                    */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->focus_line == 0
 ||  CURRENT_VIEW->focus_line == CURRENT_FILE->number_lines+1)
   {
    display_error(38,"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Copy any changes in the focus line to the linked list.              */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the focus_line.                   */
/*---------------------------------------------------------------------*/
 curr = ll_find(CURRENT_FILE->first_line,CURRENT_VIEW->focus_line);

 getyx(CURRENT_WINDOW,y,x);
 col = (x+CURRENT_VIEW->verify_col-1);
 if (col >= rec_len)
/*---------------------------------------------------------------------*/
/* After the last column of the line, perform a join.                  */
/*---------------------------------------------------------------------*/
   {
    if (curr->next->next == NULL)
      {
/*---------------------------------------------------------------------*/
/* Trying to join with the bottom of file line.                        */
/*---------------------------------------------------------------------*/
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
   meminsmem(rec,curr->next->line,curr->next->length,
             col,MAX_LINE_LENGTH,col);
   rec_len = min(MAX_LINE_LENGTH,col+curr->next->length);
   post_process_line(CURRENT_VIEW->focus_line);
   curr = ll_del(CURRENT_FILE->first_line,curr->next,DIRECTION_BACKWARD);
/*---------------------------------------------------------------------*/
/* Special problem for sos_delline. As the current_line is left where  */
/* it is, when deleting a line using sos_delline and the current_line  */
/* is on the bottom-of-file the current_line MUST be reduced by the    */
/* number of lines deleted. This test can be done here as it is not    */
/* possible to have this situation when the Delete command is issued   */
/* from the command window.                                            */
/*---------------------------------------------------------------------*/
   if (CURRENT_VIEW->current_line == CURRENT_FILE->number_lines+1)
     {
      CURRENT_VIEW->current_line--;
      y++;
     }
/*---------------------------------------------------------------------*/
/* Decrement the number of lines counter for the current file and move */
/* the cursor to the appropriate line.                                 */
/*---------------------------------------------------------------------*/
   CURRENT_FILE->number_lines--;
   wmove(CURRENT_WINDOW,y,x);
   }
 else
/*---------------------------------------------------------------------*/
/* Before the last column of the line, perform a split.                */
/*---------------------------------------------------------------------*/
   {
    add_line(CURRENT_FILE->first_line,curr,(curr->line+col),
               rec_len - col);
    CURRENT_FILE->number_lines++;
    Sos_delend("");
    post_process_line(CURRENT_VIEW->focus_line);
   }
 show_page();
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     ssave - save changes to current file

SYNTAX
     SSave [fileid]

DESCRIPTION
     The ssave command writes the current file to disk. If a fileid is
     supplied, the current file is saved in that file, otherwise the
     current name of the file is used.
     If a fileid is supplied and that fileid already exists, the previous
     contents of that fileid will be replaced with the current file.
     The 'Alterations' counter on the heading line is reset to zero.

COMPATIBILITY
     Complete

SEE ALSO
     save, file, ffile

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Ssave(unsigned char *params)
#else
int Ssave(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Ssave");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
 CURRENT_FILE->autosave_alt = CURRENT_FILE->save_alt = 0;
 rc = save_file(params,YES,0L,NO);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     stay - set condition of cursor position after CHANGE/LOCATE commands

SYNTAX
     STAY ON|OFF

DESCRIPTION
     The STAY set command determines what line is displayed as the current
     line after an unsuccessful LOCATE command or after a CHANGE command.
     With STAY ON, the current line remains where it currently is. With
     STAY OFF, after an unsuccessful LOCATE, the current line is "Bottom
     of File". After a successful CHANGE, the current line is the last
     line affected by the CHANGE command.

COMPATIBILITY
     Complete

DEFAULT
     STAY ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Stay(unsigned char *params)
#else
int Stay(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern unsigned char STAYx;
/*--------------------------- local data ------------------------------*/
#define STA_PARAMS  1
 char parm[STA_PARAMS];
 unsigned char *word[STA_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm4.c:   Stay");
#endif
 num_params = param_split(params,word,STA_PARAMS);

 parm[0] = UNDEFINED_OPERAND;
 if (equal("on",word[0],2))
    parm[0] = ON;
 if (equal("off",word[0],3))
    parm[0] = OFF;
 if (parm[0] == UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (in_profile)
    STAYx = parm[0];
 else
    CURRENT_VIEW->stay = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
