/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"


static BOOL ExternalDestroy = TRUE;

static FILEOPTIONS *clipboard = NULL;
static PARAGRAPH far *curpara;

static int OriginalLineLength;
static KANJI LastDataKanji;
static int CBPos;
static BOOL CurrentlyInKanji;
static long int CurrentBlockSize;
static BYTE far *DataPtr;
static int DataPos;

static FILEPARAHEADER far *fpara = NULL;
static BOOL FormatFirstLastLine = FALSE;
static int NumberOfParagraphs = 0;


/* Variables used for JWP --> Clipboard */

static HANDLE DataHandle;         /* Handle of DataPtr */


/* Variables used for Clipboard --> JWP */

static KANJI far *InsBuffer;        /* Block to hold the insertion buffer */
static int InsPos;



void BlockDelete (FILEOPTIONS *f, POSITION start, POSITION stop)
{
    int len;
    POSITION p;

    if (PARAOF(start) == PARAOF(stop)) {
        DeleteString (f, start, POS2ABS(stop) - POS2ABS(start) + 1,
            OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL);
        return;
    }

    p = start;

    /* First delete the last portions of the first paragraph */

    if (POSOF(p) > 0) {
        len = unitlen(&UNITOF(p, POSOF(p)));
        DeleteString (f, p, len, 0);

        CHAROF(p, POSOF(p)) = '\b';         /* Join the next paragraph */
        CHAROF(p, POSOF(p) + 1) = 0;
        f->nr_bytes++;

        PARAOF(p) = PARAOF(p)->next;
        LINEOF(p) = PARAOF(p)->lines;
        POSOF(p) = 0;
    }

	/* Delete the middle paragraphs */

    for (; PARAOF(p) != PARAOF(stop); ) {
		f->nr_bytes -= (unitlen(PARAOF(p)->text) - 1);
        CHAROF(p, 0) = '\b';
        CHAROF(p, 1) = 0;

        PARAOF(p) = PARAOF(p)->next;
        LINEOF(p) = PARAOF(p)->lines;
        POSOF(p) = 0;
    }

    /* Finally we delete the first portion of the last paragraph */
    /* and update the whole thing                                */

    len = POS2ABS(stop);

    if (len >= 0) {
        if (CHAROF(stop,POSOF(stop))) len++;

        DeleteString (f, p, len, 0);
    }

    /* Set the cursor position */

    f->current = p;

    ReformatParagraph (f, start, PARAOF(start)->next, OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL);
}


void BlockReplace (FILEOPTIONS *f, POSITION start, POSITION stop, KANJI far *buf)
{
    int len;
    POSITION p;

    if (PARAOF(start) == PARAOF(stop)) {
        ReplaceString (f, start, POS2ABS(stop) - POS2ABS(start) + 1, buf,
            OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL);
        return;
    }

    p = start;

    /* First replace the last portions of the first paragraph */

    len = unitlen(&UNITOF(p, POSOF(p)));
    ReplaceString (f, p, len, buf, 0);
    len = unitlen(PARAOF(p)->text);

    CHAROF(p, len) = '\b';         /* Join the next paragraph */
    CHAROF(p, len + 1) = 0;
    f->nr_bytes++;

    PARAOF(p) = PARAOF(p)->next;
    LINEOF(p) = PARAOF(p)->lines;
    POSOF(p) = 0;

	/* Delete the middle paragraphs */

    for (; PARAOF(p) != PARAOF(stop); ) {
		f->nr_bytes -= (unitlen(PARAOF(p)->text) - 1);
        CHAROF(p, 0) = '\b';
        CHAROF(p, 1) = 0;

        PARAOF(p) = PARAOF(p)->next;
        LINEOF(p) = PARAOF(p)->lines;
        POSOF(p) = 0;
    }

    /* Finally we delete the first portion of the last paragraph */
    /* and update the whole thing                                */

    len = POS2ABS(stop);

    if (CHAROF(stop,POSOF(stop))) len++;

    DeleteString (f, p, len, 0);

    f->current = p;                /* Set the cursor position */

    ReformatParagraph (f, start, PARAOF(start)->next, OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MINIMAL);
}



static long int HowMuchData (void)
{
    int i;
    long int len;
    PARAGRAPH far *p;

    /* How much data? */

    for (p = clipboard->paragraph, len = 0L; p != NULL; p = p->next) {
		for (i = 0; p->text[i].kanji; i++) len += (ISKANJI(p->text[i].kanji) ? 2 : 1);
		len += 1L;   /* for \n */
    }

    return (len);
}



static HANDLE PrepareEUCData (void)
{
    int i;
    long int len;
    HANDLE handle;
    PARAGRAPH far *p;
    BYTE far *cp;

    if (clipboard == NULL) {
        ErrorMessage(global.hwnd, "Funny! Clipboard is empty!");
        return (NULL);
    }


    /* Allocate the data block */

    len = HowMuchData() + 5L;

	if (len > C64K) {
        ErrorMessage(global.hwnd, "The clipboard cannot hold so much data!  The limit is 64K!");
        return (NULL);
    }

    handle = GlobalAlloc(GHND, len);
    cp = (BYTE far *) GlobalLock(handle);

    /* Copy data */

    for (p = clipboard->paragraph; p != NULL; p = p->next) {
		for (i = 0; p->text[i].kanji; i++, cp++) {
			if (ISKANJI(p->text[i].kanji)) {
				*cp++ = HIBYTE(p->text[i].kanji) | 0x80;
				*cp = LOBYTE(p->text[i].kanji) | 0x80;
            } else {
				*cp = LOBYTE(p->text[i].kanji) & 0x7f;
            }
        }
        *cp++ = '\n';
    }
    *cp++ = '\0';

    GlobalUnlock(handle);
    return (handle);
}



static int ToClipboardCharIn (void)
{
    KANJI ch;

	if (curpara == NULL) return (EOF);

	ch = curpara->text[CBPos].kanji;

	if (CurrentlyInKanji) {
		CurrentlyInKanji = FALSE;
		CBPos++;
        return (LOBYTE(ch) | 0x80);
    }

    if (!ch) {
        curpara = curpara->next;
        CBPos = 0;
        return ('\n');
    } else if (ISKANJI(ch)) {
        CurrentlyInKanji = TRUE;
        return (HIBYTE(ch) | 0x80);
    } else {
        CBPos++;
        return (LOBYTE(ch) & 0x7f);
    }
}



static void ToClipboardCharOut (int ch)
{
    if (DataPos + 5 >= CurrentBlockSize) {
		/* Reallocate */
        CurrentBlockSize += TEXTBLOCKSIZE;
        if (CurrentBlockSize > C64K) {
            ErrorMessage (global.hwnd, "The clipboard cannot hold so much data!  The limit is 64K!");
			return;
		}
        GlobalUnlock(DataHandle);
        GlobalReAlloc(DataHandle, CurrentBlockSize, GHND);
        DataPtr = (BYTE far *) GlobalLock(DataHandle);
	}

    DataPtr[DataPos++] = ch;
}



static int FromClipboardCharIn (void)
{
	KANJI ch;

    ch = DataPtr[DataPos++];
    if (!ch) return (EOF);

    return (ch);
}



static void FromClipboardCharOut (int ch)
{
    KANJI ch1;

    if (ch == '\r') return;
    if (ch == '\n') ch = '\r';

    if (CurrentlyInKanji) {
        CurrentlyInKanji = FALSE;
        ch1 = ((LastDataKanji << 8) | (ch & 0x7f));
    } else if (ch & 0x80) {
        CurrentlyInKanji = TRUE;
        LastDataKanji = (ch & 0x7f);
        return;
    } else {
        ch1 = ch;
    }

    if (InsPos + 5 >= CurrentBlockSize) {
        /* Reallocate */
        CurrentBlockSize += TEXTBLOCKSIZE;
        if (CurrentBlockSize > (C64K / sizeof(KANJI))) {
            ErrorMessage (global.hwnd, "The clipboard contains too much data!  The limit is 64K!");
            return;
        }
        InsBuffer = BlockRealloc(InsBuffer, CurrentBlockSize * sizeof(KANJI));
    }

    InsBuffer[InsPos++] = ch1;
}



static HANDLE PrepareOtherData (FILEFORMAT format)
{
    int i;
    long int len;

	if (clipboard == NULL) {
        ErrorMessage(global.hwnd, "Funny! Clipboard is empty!");
		return (NULL);
	}

	curpara = clipboard->paragraph;
    CBPos = 0;

    len = HowMuchData() + 5L;

    if (len > C64K) {
        ErrorMessage (global.hwnd, "The clipboard cannot hold so much data!  The limit is 64K!");
        return (NULL);
    }

    i = len / TEXTBLOCKSIZE;
    len = (i + 1) * TEXTBLOCKSIZE;

    DataHandle = GlobalAlloc(GHND, len);
    DataPtr = (BYTE far *) GlobalLock(DataHandle);

    CurrentBlockSize = len;
    DataPos = 0L;

    CurrentlyInKanji = FALSE;

	SetupIO(ToClipboardCharIn, ToClipboardCharOut);
	FileExport(format);

    GlobalUnlock(DataHandle);
    return (DataHandle);
}



static HANDLE PrepareBitmap (void)
{
    long int height, width, max;
	HBITMAP hbitmap;
	HDC hdc, hdcmem;
    HBRUSH hbrush;
	RECT rect;
	PARAGRAPH far *p;
	ONELINE far *lp;
    BOOL OldDraftView;


	/* First, reformat it to the original layout */

	clipboard->width = OriginalLineLength;
	clipboard->height = 0;

    clipboard->linelen = clipboard->width / BASEWIDTH(clipboard);
    if (clipboard->linelen <= 0) clipboard->linelen = 1;

    OldDraftView = global.draftview;
    global.draftview = FALSE;

    ReformatFile(clipboard);


    /* How big must the bitmap be? */

    width = 0;
    height = 0;

    for (p = clipboard->paragraph; p != NULL; p = p->next) {
		for (lp = p->lines; lp != NULL; lp = lp->next) {
            if (lp->length > 0 && ISKANJI(p->text[lp->position + lp->length - 1].kanji)) {
				if (lp->width - clipboard->leading > width) width = lp->width - clipboard->leading;
            } else {
                if (lp->width > width) width = lp->width;
            }
            height += lp->height + clipboard->spacing;
        }
    }


    max = (C64K / width) * 8;

    height -= clipboard->spacing;

    if (height > max) height = max;


	/* Create the bitmap */

	hdc = GetDC(global.hwnd);
	hdcmem = CreateCompatibleDC(hdc);
    SetTextColor(hdcmem, GetSysColor(COLOR_WINDOWTEXT));
    SetBkColor(hdcmem, GetSysColor(COLOR_WINDOW));
    hbitmap = CreateCompatibleBitmap (hdc, width, height);
	SelectObject(hdcmem, hbitmap);

	rect.left = 0;
	rect.right = width;
	rect.top = 0;
	rect.bottom = height;

    hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
    FillRect(hdcmem, &rect, hbrush);
    DeleteObject(hbrush);

    RedrawFile(clipboard, hdcmem, 0, clipboard->spacing - LINEGAP(clipboard), &rect, FALSE);

	DeleteDC(hdcmem);
	ReleaseDC(global.hwnd, hdc);

    global.draftview = OldDraftView;

    return (hbitmap);
}



static HANDLE PrepareInternalData (void)
{
    int i;
    long int len;
    FILEPARAHEADER fh;
	PARAGRAPH far *p;

    DataHandle = GlobalAlloc(GHND, C64K);
    DataPtr = (BYTE far *) GlobalLock(DataHandle);

    /* How many paragraphs? */

    for (i = 0, p = clipboard->paragraph; p != NULL; i++, p = p->next);

    _fmemcpy(DataPtr, &i, sizeof(int));
    DataPtr += sizeof(int);

    for (p = clipboard->paragraph; p != NULL; p = p->next) {
        fh.textsize = unitlen(p->text);
        fh.spacemulti = p->spacemulti;
		fh.firstindent = p->firstindent;
        fh.leftindent = p->leftindent;
        fh.rightindent = p->rightindent;

        _fmemcpy(DataPtr, &fh, sizeof(FILEPARAHEADER));
        DataPtr += sizeof(FILEPARAHEADER);

        len = ((long int) fh.textsize + 1L) * sizeof(UNIT);
        _fmemcpy(DataPtr, p->text, len);
        DataPtr += len;
    }

    GlobalUnlock(DataHandle);
    return (DataHandle);
}



static HANDLE GetCBData (int type)
{
    switch (type) {
        case CF_TEXT:
        case CF_OEMTEXT:
            switch (global.clipboard) {
                case FF_EUC:     return (PrepareEUCData());

                case FF_UNKNOWN: ErrorMessage(global.hwnd, "Bad clipboard format: Unknown");
                                 return (NULL);

                case FF_NORMAL:  ErrorMessage(global.hwnd, "Bad clipboard format: Normal");
                                 return (NULL);

                default:         return (PrepareOtherData(global.clipboard));
            }
			break;

        case CF_BITMAP:
            return (PrepareBitmap());

		case CF_OWNERDISPLAY:
            return (PrepareInternalData());
    }

    ErrorMessage(global.hwnd, "Bad clipboard format type received: %d", type);
	return (NULL);
}



static BOOL ReformatProc (FILEOPTIONS *f, PARAGRAPH far *p, int n)
{
    if (fpara == NULL) return (FALSE);

    if (!FormatFirstLastLine) {
        if (n <= 0) return (FALSE);
        if (n >= NumberOfParagraphs - 1) return (FALSE);
    }

    p->spacemulti = fpara[n].spacemulti;
	p->spacing = (f->basefont->height + f->basefont->spacing) * p->spacemulti / 100;
    p->spacing -= f->basefont->height;
    p->firstindent = fpara[n].firstindent;
    p->leftindent = fpara[n].leftindent;
    p->rightindent = fpara[n].rightindent;

    return (TRUE);
}



static void ImportInternalCBData (void)
{
    int i, j, n;
    long int len;
    BYTE far *cp;
	HANDLE handle;
    UNIT far *up;

    OpenClipboard(global.hwnd);
	handle = GetClipboardData(CF_OWNERDISPLAY);
    cp = DataPtr = (BYTE far *) GlobalLock(handle);

    _fmemcpy(&n, DataPtr, sizeof(int));
	DataPtr += sizeof(int);

	if (fpara != NULL) FreeBlock(fpara);
	fpara = (FILEPARAHEADER far *) BlockAlloc(n * sizeof(FILEPARAHEADER));
    NumberOfParagraphs = n;

    for (i = 0, len = 0L; i < n; i++) {
        _fmemcpy(fpara + i, DataPtr, sizeof(FILEPARAHEADER));
        len += fpara[i].textsize + 1;
        DataPtr += sizeof(FILEPARAHEADER);
		DataPtr += (fpara[i].textsize + 1) * sizeof(UNIT);
    }

    /* Now get the text */

    InsBuffer = (KANJI far *) BlockAlloc((len + 5) * sizeof(KANJI));
    InsPos = 0;

    DataPtr = cp;               /* Rewind to beginning */
    DataPtr += sizeof(int);     /* Skip number of paragraphs */

    for (i = 0; i < n; i++) {
        DataPtr += sizeof(FILEPARAHEADER);
        up = (UNIT far *) DataPtr;
        for (j = 0; j < fpara[i].textsize; j++) InsBuffer[InsPos + j] = up[j].kanji;
        DataPtr += (fpara[i].textsize + 1) * sizeof(UNIT);
        InsBuffer[InsPos + j] = '\r';
        InsPos += j + 1;
    }

    if (InsPos > 0) InsBuffer[InsPos-1] = 0;

    GlobalUnlock(handle);
    CloseClipboard();
}



static KANJI far *ImportOtherCBData (FILEFORMAT format)
{
	HANDLE handle;

    if (!IsClipboardFormatAvailable(CF_TEXT) && !IsClipboardFormatAvailable(CF_OEMTEXT))
        return (NULL);

    OpenClipboard(global.hwnd);
	if (IsClipboardFormatAvailable(CF_TEXT)) {
		handle = GetClipboardData(CF_TEXT);
	} else if (IsClipboardFormatAvailable(CF_OEMTEXT)) {
		handle = GetClipboardData(CF_OEMTEXT);
	}

    DataPtr = (BYTE far *) GlobalLock(handle);
    DataPos = 0;
	CurrentBlockSize = TEXTBLOCKSIZE;

    InsBuffer = (KANJI far *) BlockAlloc(CurrentBlockSize * sizeof(KANJI));
	InsPos = 0;

    CurrentlyInKanji = FALSE;

    SetupIO(FromClipboardCharIn, FromClipboardCharOut);
    FileImport(format);

	GlobalUnlock(handle);
    CloseClipboard();

	return (InsBuffer);
}



static void ClearCBData (void)
{
	if (clipboard == NULL) return;

    CloseFile(clipboard);

	FreeMem(clipboard);

	clipboard = NULL;
}



void CopyToClipboard (FILEOPTIONS *f)
{
    PARAGRAPH far *p, far *cp;

    OpenClipboard(global.hwnd);
    EmptyClipboard();
    ClearCBData();

    if (SELPARA1(f) != NULL) {
        SetClipboardData(CF_OWNERDISPLAY, NULL);
        SetClipboardData(CF_TEXT, NULL);
        SetClipboardData(CF_OEMTEXT, NULL);
        SetClipboardData(CF_BITMAP, NULL);
    } else {
        CloseClipboard();
        return;
    }


    /* Allocate data */

	clipboard = (FILEOPTIONS *) MemAlloc(sizeof(FILEOPTIONS));

    clipboard->type = FN_CLIPBOARD | FN_NORMAL;
	clipboard->relaxmargin = global.relaxmargin;
	clipboard->nr_lines = 0;

	clipboard->basefont = f->basefont;
	clipboard->leading = f->leading;
	clipboard->spacing = f->spacing;
	clipboard->linelen = f->linelen;

	OriginalLineLength = f->linelen * BASEWIDTH(f);


	if (SELPARA1(f) == SELPARA2(f)) {
		cp = clipboard->paragraph = clipboard->eof = StructAlloc(PARAGRAPH);
		_fmemcpy(cp, SELPARA1(f), sizeof(PARAGRAPH));

		cp->lines = cp->lastline = StructAlloc(ONELINE);
        cp->lines->next = cp->lines->prev = NULL;
		cp->next = cp->prev = NULL;
        clipboard->nr_lines = 1;

        cp->lines->length = SELPOS2(f) - SELPOS1(f) + 1;
		cp->textsize = (cp->lines->length + 2) * sizeof(UNIT);
		cp->text = BlockAlloc(cp->textsize);

		_fmemcpy(cp->text, SELPARA1(f)->text + SELPOS1(f),
				 (SELPOS2(f) - SELPOS1(f) + 1) * sizeof(UNIT));
		cp->text[SELPOS2(f) - SELPOS1(f) + 1].kanji = 0;

        TOPPARA(clipboard) = clipboard->paragraph;
        TOPLINE(clipboard) = clipboard->paragraph->lines;

        clipboard->current = clipboard->top;
		POSOF(clipboard->current) = 0;

        clipboard->vscroll = 0;
        CloseClipboard();
		return;
	}

	for (p = SELPARA1(f); ; p = p->next) {
		if (clipboard->paragraph == NULL) {
			clipboard->paragraph = cp = StructAlloc(PARAGRAPH);
			_fmemcpy(cp, p, sizeof(PARAGRAPH));
			cp->prev = NULL;
		} else {
			cp->next = StructAlloc(PARAGRAPH);
			_fmemcpy(cp->next, p, sizeof(PARAGRAPH));
			cp->next->prev = cp;
			cp = cp->next;
		}

		cp->lines = cp->lastline = StructAlloc(ONELINE);
        cp->lines->next = cp->lines->prev = NULL;
        clipboard->nr_lines++;


		if (p == SELPARA1(f)) {
			cp->lines->length = unitlen(p->text) - SELPOS1(f);
			cp->textsize = (cp->lines->length + 1) * sizeof(UNIT);
			cp->text = BlockAlloc(cp->textsize);
			_fmemcpy(cp->text, p->text + SELPOS1(f), cp->textsize);
		} else if (p == SELPARA2(f)) {
			cp->lines->length = SELPOS2(f);
			cp->textsize = (cp->lines->length + 2) * sizeof(UNIT);
			cp->text = BlockAlloc(cp->textsize);
			_fmemcpy(cp->text, p->text, cp->textsize);
			cp->text[SELPOS2(f) + 1].kanji = 0;
			break;
		} else {
			cp->lines->length = unitlen(p->text);
			cp->textsize = (cp->lines->length + 1) * sizeof(UNIT);
			cp->text = BlockAlloc(cp->textsize);
			_fmemcpy(cp->text, p->text, cp->textsize);
		}
	}

	cp->next = NULL;
    clipboard->eof = cp;

    TOPPARA(clipboard) = clipboard->paragraph;
	TOPLINE(clipboard) = clipboard->paragraph->lines;

    clipboard->current = clipboard->top;
	POSOF(clipboard->current) = 0;

	clipboard->vscroll = 0;

    CloseClipboard();
}



BOOL ClipboardFull (void)
{
    HWND hwnd;
    char buffer[MAXLINELEN];

    if (IsClipboardFormatAvailable(CF_OWNERDISPLAY)) {
        hwnd = GetClipboardOwner();
        if (hwnd == NULL) goto Next;

        GetClassName(hwnd, buffer, MAXLINELEN);
        if (strcmp(buffer, "JWP Frame")) goto Next;

        return (TRUE);
    }

Next:

    if (IsClipboardFormatAvailable(CF_TEXT)) return (TRUE);
    if (IsClipboardFormatAvailable(CF_OEMTEXT)) return (TRUE);

    return (FALSE);
}



void InsertFromClipboard (FILEOPTIONS *f)
{
    int i;
    HWND hwnd;
    BOOL Internal;
    char buffer[MAXLINELEN];

    Internal = IsClipboardFormatAvailable(CF_OWNERDISPLAY);

    if (Internal) {
        hwnd = GetClipboardOwner();
        if (hwnd == NULL) return;
        GetClassName(hwnd, buffer, MAXLINELEN);
        if (strcmp(buffer, "JWP Frame")) return;

        ImportInternalCBData();
    } else {
        if (ImportOtherCBData(global.clipboard) == NULL) return;

        if (InsPos > 0) if (InsBuffer[InsPos - 1] == '\r') InsPos--;

        InsBuffer[InsPos] = 0;
    }


    /* If it is to an edit control, no line-breaks */

    if (!(f->type & FN_NORMAL)) {
        for (i = 0; InsBuffer[i]; i++) {
            if (InsBuffer[i] == '\n' || InsBuffer[i] == '\r' || InsBuffer[i] == '\t') {
                if (i > 0) {
                    if (ISKANJI(InsBuffer[i-1])) InsBuffer[i] = 0x2121;
                    else InsBuffer[i] = ' ';
                } else if (InsBuffer[i+1]) {
                    if (ISKANJI(InsBuffer[i+1])) InsBuffer[i] = 0x2121;
                    else InsBuffer[i] = ' ';
                } else {
                    InsBuffer[i] = 0x2121;
                }
            }
        }
    }


    /* Block Replace? */

    if (Internal) SetReformatProc(ReformatProc);

    if (SELPARA1(f) != NULL && SELTYPE(f) == SEL_SELECTION) {
        POSITION abs, start, stop;

        PARAOF(abs) = SELPARA1(f);
		POSOF(abs) = SELPOS1(f);

		AbsoluteToPosition(abs, &start);

		PARAOF(abs) = SELPARA2(f);
		POSOF(abs) = SELPOS2(f);

        AbsoluteToPosition(abs, &stop);

        /* Get rid of the selection */

        SELPARA1(f) = SELPARA2(f) = NULL;   SELPOS1(f) = SELPOS2(f) = 0;
        SELNOW(f) = FALSE;                  SELTYPE(f) = SEL_SELECTION;

        f->current = start;

        if (!FindCaret(f, FALSE)) {
            MoveIntoWindow(f);
			InvalidateRect(f->hwnd, NULL, TRUE);
			UpdateWindow(f->hwnd);
        }

        /* Move the cursor to the end */

        f->current = stop;
        if (CURCHAR(f) >= CURLINE(f)->length) {
            NEXTLINE(f->current);
        } else {
			CURCHAR(f)++;
        }

        FormatFirstLastLine = FALSE;
		UndoAddReplace(f, start, f->current);
        if (SELPARA1(f) == SELPARA2(f)) {
            BlockReplace (f, start, stop, InsBuffer);
        } else {
            BlockDelete(f, start, stop);
            InsertString(f, start, InsBuffer, OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
        }
        UndoFixReplace(f, f->current);
    } else {
        POSITION p;

        FormatFirstLastLine = (CURCHAR(f) <= 0);
        p = f->current;
        InsertString (f, f->current, InsBuffer, OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND | OP_MINIMAL);
        UndoAddInsert(f, p, f->current);
    }

    if (Internal) SetReformatProc(NULL);

    FreeBlock(InsBuffer);
}



void ResetClipboard (void)
{
    if (clipboard == NULL) return;

    OpenClipboard(global.hwnd);
    ExternalDestroy = FALSE;
    EmptyClipboard();
    ExternalDestroy = TRUE;
    SetClipboardData(CF_OWNERDISPLAY, NULL);
    SetClipboardData(CF_TEXT, NULL);
    SetClipboardData(CF_OEMTEXT, NULL);
	SetClipboardData(CF_BITMAP, NULL);
    CloseClipboard();
}



LONG ProcessClipboardMessage (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i, j;
    char buffer[100];
	char far *cp;
    BOOL OldDraftView;
    RECT rect, far *rp;
    PAINTSTRUCT far *ps;
	HANDLE handle;
    POSITION p;


    switch (message) {
        case WM_RENDERFORMAT:
            SetClipboardData(wParam, GetCBData(wParam));
            return (0);

        case WM_DESTROYCLIPBOARD:
            if (ExternalDestroy) ClearCBData();
			return (0);

        case WM_RENDERALLFORMATS:
			OpenClipboard(hwnd);
            ExternalDestroy = FALSE;
            EmptyClipboard();
            ExternalDestroy = TRUE;
            SetClipboardData(CF_TEXT, GetCBData(CF_TEXT));
            SetClipboardData(CF_OEMTEXT, GetCBData(CF_OEMTEXT));
            SetClipboardData(CF_BITMAP, GetCBData(CF_BITMAP));
            CloseClipboard();
            return (0);

        case WM_ASKCBFORMATNAME:
			sprintf(buffer, "%s Formatted Text", PROGNAME);
			cp = (char far *) lParam;

			_fmemcpy(cp, buffer, wParam);
			cp[wParam - 1] = '\0';
			return (0);

        case WM_PAINTCLIPBOARD:
            handle = (HANDLE) LOWORD(lParam);
            ps = (PAINTSTRUCT far *) GlobalLock(handle);
            rp = &(ps->rcPaint);

            /* Translate the coordinates */
            rp->left -= BORDERSPACE;
            rp->top -= BORDERSPACE;
			rp->right -= BORDERSPACE;
            rp->bottom -= BORDERSPACE;

            OldDraftView = global.draftview;
            global.draftview = FALSE;
            RedrawFile(clipboard, ps->hdc, -BORDERSPACE, -BORDERSPACE, rp, FALSE);
            global.draftview = OldDraftView;

            GlobalUnlock(handle);
            return (0);

        case WM_SIZECLIPBOARD:
            handle = (HANDLE) LOWORD(lParam);
            rp = (RECT far *) GlobalLock(handle);
            rect = *rp;
            GlobalUnlock(handle);

            if (rect.right <= 0 || rect.bottom <= 0) return (0);

            GetScrollRange((HWND) wParam, SB_VERT, &i, &j);

            if (j - i > 0) {
                clipboard->width = rect.right - BORDERSPACE;
                clipboard->height = rect.bottom - BORDERSPACE;

                clipboard->linelen = clipboard->width / BASEWIDTH(clipboard);
                if (clipboard->linelen <= 0) clipboard->linelen = 1;

                OldDraftView = global.draftview;
                global.draftview = FALSE;
                ReformatFile(clipboard);
                global.draftview = OldDraftView;

                if (clipboard->nr_lines > 1) {
                    SetScrollRange((HWND) wParam, SB_VERT, 0, clipboard->nr_lines - 1, FALSE);
                } else {
                    SetScrollRange((HWND) wParam, SB_VERT, 0, 1, FALSE);
                }
                SetScrollPos((HWND) wParam, SB_VERT, clipboard->vscroll, TRUE);
            } else {
                /* No scroll bar: turn it on */
                /* Windows will send WM_SIZECLIPBOARD again */

                SetScrollRange((HWND) wParam, SB_VERT, 0, 1, FALSE);
                SetScrollPos((HWND) wParam, SB_VERT, 0, TRUE);
            }
            return (0);

        case WM_VSCROLLCLIPBOARD:
            rect.left = 0;
            rect.top = BORDERSPACE;
            rect.right = clipboard->width + BORDERSPACE;
            rect.bottom = clipboard->height + BORDERSPACE;

            switch (LOWORD(lParam)) {
                case SB_TOP:
                    TOPPARA(clipboard) = clipboard->paragraph;
                    TOPLINE(clipboard) = clipboard->paragraph->lines;
					clipboard->vscroll = 0;
                    InvalidateRect((HWND) wParam, NULL, TRUE);
                    break;

                case SB_BOTTOM:
                    TOPPARA(clipboard) = clipboard->eof;
                    TOPLINE(clipboard) = clipboard->eof->lastline;
                    clipboard->vscroll = clipboard->nr_lines;
                    InvalidateRect((HWND) wParam, NULL, TRUE);
                    break;

                case SB_LINEUP:
					if (!PREVLINE(clipboard->top)) return (0);
                    i = TOPLINE(clipboard)->height + TOPPARA(clipboard)->spacing;
                    clipboard->vscroll--;
					ScrollWindow((HWND) wParam, 0, i, NULL, &rect);
                    break;

				case SB_LINEDOWN:
                    i = TOPLINE(clipboard)->height + TOPPARA(clipboard)->spacing;

					if (!NEXTLINE(clipboard->top)) return (0);
                    clipboard->vscroll++;
					ScrollWindow((HWND) wParam, 0, -i, NULL, &rect);
                    break;

				case SB_PAGEUP:
					i = -(clipboard->spacing - LINEGAP(clipboard));
                    p = clipboard->top;
                    do {
						i += LINEOF(p)->height + clipboard->spacing;
						if (i + LINEGAP(clipboard) > clipboard->height) break;
                        clipboard->vscroll--;
                    } while (PREVLINE(p));

                    clipboard->top = p;
                    InvalidateRect((HWND) wParam, NULL, TRUE);
                    break;

				case SB_PAGEDOWN:
					i = -(clipboard->spacing - LINEGAP(clipboard));
                    p = clipboard->top;
                    do {
						i += LINEOF(p)->height + clipboard->spacing;
						if (i + LINEGAP(clipboard) > clipboard->height) break;
                        clipboard->vscroll++;
                    } while (NEXTLINE(p));

                    clipboard->top = p;
                    InvalidateRect((HWND) wParam, NULL, TRUE);
                    break;

				case SB_THUMBPOSITION:
                    clipboard->vscroll = HIWORD(lParam);
                    i = 0;
                    PARAOF(p) = clipboard->paragraph;
                    LINEOF(p) = clipboard->paragraph->lines;
                    do {
                       if (i >= clipboard->vscroll) break;
                       i++;
                    } while (NEXTLINE(p));

					clipboard->top = p;
                    InvalidateRect((HWND) wParam, NULL, TRUE);
                    return (0);
            }

            SetScrollPos((HWND) wParam, SB_VERT, clipboard->vscroll, TRUE);
            return (0);
    }

	return (DefFrameProc(hwnd, global.clienthwnd, message, wParam, lParam));
}
