//--------------------------------------------------------------------------
//
//      XMSBENCH.CPP: Benchmark program for XMS array class.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the library of which this file
//      is a part freely available.
//
//      This program performs timing tests for XMS arrays of elements
//      of various different sizes, with user-selectable numbers of
//      elements per array, iterations performed, and cache sizes used.
//      It also crashes nicely when calculating timings (domain error
//      occurs if you ask for zero items) and can be terminated ruthlessly
//      using ^C.  The companion utility XMSFREE lets you check how much
//      XMS is free afterwards.
//
//--------------------------------------------------------------------------

#include <iomanip.h>
#include <dos.h>
#include "xms.h"

volatile long far* timer = (volatile long far*) MK_FP(0x40,0x6C);
                                // pointer to BIOS timer count

//--------------------------------------------------------------------------
//
//      Test function.
//
//      This performs a timing test for an XMS array of a specified
//      type with a specified number of elements and cache size for
//      a specified number of iterations.
//
template<class T> void test (T, int elems, int buff, int loops)
{
    long ticks;
    double time;
    T value;
    int i, n;

    //--- allocate the array and report status
    cout << "\nAllocation of " << elems << " items of " << sizeof(T)
         << " bytes each ..." << flush;
    XMSarray<T> a(elems, buff);
    cout << "\b\b\b" << (a.allocated() ? "succeeded" : "failed")
         << "\n" << XMS::available()
         << " bytes of XMS free, buffer size = " << buff << " bytes"
         << endl << endl;

    //--- display headings
    cout << "             Elapsed (ms)  Time per item (s)  Time per byte (s)"
         << endl;

    //--- write test: a[i] = value
    cout << "Writing:     " << flush;
    ticks = *timer;
    for (n = loops; n > 0; n--)
        for (i = 0; i < elems; i++)
            a[i] = value;
    ticks = (*timer - ticks) * 55L;
    cout << setw (7) << ticks;
    time = (ticks * 1000.0) / loops;
    time /= elems;
    cout << setw (18) << time << setw (20) << (time/sizeof(T)) << endl;

    //--- read test: a[i] = value
    //    (NB slower, something always copied?)
    cout << "Reading:     " << flush;
    ticks = *timer;
    for (n = loops; n > 0; n--)
        for (i = 0; i < elems; i++)
            value = a[i];
    ticks = (*timer - ticks) * 55L;
    cout << setw (7) << ticks;
    time = (ticks * 1000.0) / loops;
    time /= elems;
    cout << setw (18) << time << setw (20) << (time/sizeof(T)) << endl;

    //--- copy test 1: a[0] = a[0]
    //    (Item copied from cache; gives raw cache speed)
    cout << "Copying (1): " << flush;
    ticks = *timer;
    for (n = loops; n > 0; n--)
        for (i = 0; i < elems; i++)
            a[0] = a[0];
    ticks = (*timer - ticks) * 55L;
    cout << setw (7) << ticks;
    time = (ticks * 1000.0) / loops;
    time /= elems;
    cout << setw (18) << time << setw (20) << (time/sizeof(T)) << endl;

    //--- copy test 2: a[0] = a[elems-1]
    //    (Cache cruncher: items never cached, gives raw XMS speed)
    cout << "Copying (2): " << flush;
    ticks = *timer;
    for (n = loops; n > 0; n--)
        for (i = 0; i < elems; i++)
            a[0] = a[elems - 1];
    ticks = (*timer - ticks) * 55L;
    cout << setw (7) << ticks;
    time = (ticks * 1000.0) / loops;
    time /= elems;
    cout << setw (18) << time << setw (20) << (time/sizeof(T)) << endl;
}

//--------------------------------------------------------------------------
//
//      Type definitions for array element types.
//
typedef struct { char x[10]; }    T1;
typedef struct { char x[100]; }   T2;
typedef struct { char x[1000]; }  T3;
typedef struct { char x[10000]; } T4;

//--------------------------------------------------------------------------
//
//      Main program.
//
void main ()
{
    cout.setf (ios::fixed, ios::floatfield);
    cout.flags (ios::showpoint | ios::uppercase);
    cout.precision (2);
    cout << "XMS array speed test: "
         << XMS::available() << " bytes of XMS free initially\n\n";
    for (;;)
    {   cout << "Choose size of array items:\n"
                "  1) 10 bytes\n"
                "  2) 100 bytes\n"
                "  3) 1000 bytes\n"
                "  4) 10000 bytes\n"
                "  X) Exit program\n"
                "Enter your choice: ";
        char c;
        cin >> c;
        if (c == 'x' || c == 'X')
            break;
        if (c < '1' || c > '4')
        {   cout << "Invalid choice!\a\n\n";
            continue;
        }

        cout << "Enter number of array elements: ";
        int e;
        cin >> e;
        cout << "Enter number of iterations:     ";
        int i;
        cin >> i;
        cout << "Enter size of internal buffer:  ";
        int b;
        cin >> b;

        if (c == '1')
        {   T1 t;
            test (t, e, b, i);
        }
        if (c == '2')
        {   T2 t;
            test (t, e, b, i);
        }
        if (c == '3')
        {   T3 t;
            test (t, e, b, i);
        }
        if (c == '4')
        {   T4 t;
            test (t, e, b, i);
        }

        cout << "XMS status = " << hex << XMS::status() << dec
             << endl << endl;
    }
}
