/*************************************************************************

	kb_7300.c       Keyboard (KB) Subroutines
			for Bywater Software

			Implementation for AT&T Unix PC (7300)

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

**************************************************************************/

#define	ATT_TAM		TRUE
#define	ATT_MGR		FALSE
#define	WRITE_OUTPUT	FALSE

#include "stdio.h"
#include "fcntl.h"
#include "signal.h"
#include "sys/stat.h"
#include "termio.h"
#include "bw.h"
#include "kb.h"

#if	ATT_TAM
#include "tam.h"
#define	NO_KEY	-1
#else
#define	NO_KEY	FALSE
#endif

#ifndef	TRUE
#define	TRUE	1
#define	FALSE	0
#endif

#ifndef	ESC
#define	ESC	0x1b
#endif

#if ATT_TAM
extern int w;			/* window identifier */
extern int tam_mready;		/* is mouse ready with data ? */
extern int tam_mx, tam_my;	/* mouse position x, y */
int tam_mb, tam_mr;		/* button, reason for mouse */
int tam_kready = FALSE;
int kb_hold = 0;

#if WRITE_OUTPUT
FILE *outfile;
#endif

#else
struct termio old_ttystruct;
int	kb_tty;			/* fd for new terminal file */
#endif


#ifdef	KB_TEST
main()
   {
   register int s;
   static char tbuf[ 3 ];

   kb_init();

   tbuf[ 0 ] = 0;
   while ( tbuf[ 0 ] != 0x1b )
      {
#if ATT_TAM
      s = wgetc( w );
#else
      s = read( kb_tty, tbuf, 1 );
#endif
      printf( "The read() function returned %d \n", s );
      if ( s > 0 )
         {
         printf( "The key is <%c> \n", tbuf[ 0 ] );
         }
      }

   kb_deinit();
   }
#endif

/*************************************************************************

   FUNCTION:       kb_init()

   DESCRIPTION:    This function should perform any initialization
         necessary for the keyboard system.

   INPUT:          none.

   RETURNS:        none.

**************************************************************************/

kb_init()
   {
#if  ATT_TAM

   keypad( 0, 1 );
   wndelay( w, 1 );

#if WRITE_OUTPUT
   if ( ( outfile = fopen( "output", "w" ) ) == NULL )
      {
      bw_error( "Failed to open outfile <output> " );
      }
#endif    
#else
   struct termio new_ttystruct;

   if ( ioctl( 0, TCGETA, &old_ttystruct ) < 0 )  
      {
      bw_error( "Failed to get old tty parameters." );
      }

   if ( ( kb_tty = open( "/dev/tty", O_NDELAY)) == -1 )
      {
      bw_error( "Failed to open /dev/tty" );
      }
#ifdef   DEBUG
   else
      {
      sprintf( bw_ebuf, "/dev/tty returns %d \n", kb_tty );
      bw_debug( bw_ebuf );
      }
#endif

   if ( ioctl( kb_tty, TCGETA, &new_ttystruct ) < 0 )  
      {
      bw_error( "Failed to get new tty parameters." );
      }

   new_ttystruct.c_cc[4] = 0;      /* minimum characters */
   new_ttystruct.c_cc[5] = 0;      /* maximum time */
   new_ttystruct.c_iflag = 0;
   new_ttystruct.c_lflag = ISIG;
   new_ttystruct.c_cflag &= ~CSIZE;
   new_ttystruct.c_cflag |= CS8;
   new_ttystruct.c_cflag &= ~PARENB;
   if ( ioctl( kb_tty, TCSETA, &new_ttystruct ) < 0 )  
      {
      fprintf( stderr, "Failed to set new tty parameters. \n" );
      }

#endif

   }



/*************************************************************************

   FUNCTION:       kb_deinit()

   DESCRIPTION:    This function should perform any necessary
         deinitialization, that is, return the keyboard
         to its default state when a Simple Software
         program is to be exited.

   INPUT:          none.

   RETURNS:        none.

**************************************************************************/

kb_deinit()
   {
#if  ATT_TAM
#else
   if ( ioctl( 0, TCSETA, &old_ttystruct ) < 0 )
      {
      fprintf( stderr, "Failed to restore old tty parameters. \n");
      }
#endif
   }

/*************************************************************************

   FUNCTION:       kb_rxstat()

   DESCRIPTION:    This function determines whether a character is
		   ready from the console.  The function is used
		   especially in telecommunications programs,
		   where it is necessary to poll the keyboard
		   without locking up the program waiting for a
		   response.

   INPUT:          none.

   RETURNS:        The function returns 0 if no character is
		   available and 1 if a character is available.

**************************************************************************/

kb_rxstat()
   {

#if WRITE_OUTPUT
   fprintf( outfile, "kb_rxstat(): enter\n" );
   fflush( outfile );
#endif

   if ( tam_kready == FALSE )
      {
      att_rxstat();
      }

#if WRITE_OUTPUT
   fprintf( outfile, "kb_rxstat(): returns %d\n", tam_kready );
   fflush( outfile );
#endif

   return tam_kready;
   }

att_rxstat()
   {
   int r;
   register int s;
#if ATT_TAM
#else
   static char tbuf[3];
#endif

#if WRITE_OUTPUT
   fprintf( outfile, "att_rxstat(): enter\n" );
   fflush( outfile );
#endif

   /* This should be the only point at which a character
      is read */

#if ATT_TAM
   s = wgetc( w );
#else
   s = read( kb_tty, tbuf, 1 );
#endif

   /* if no character is available, return false */

   if ( s == NO_KEY )
      {
      return FALSE;
      }

   /* we have now read a character */

#if  ATT_TAM
   kb_hold = s;
#else
   kb_hold = tbuf[ 0 ];
#endif

#if WRITE_OUTPUT
   fprintf( outfile, "att_rxstat(): read kb_hold 0x%02x\n", kb_hold );
   fflush( outfile );
#endif

   /* check to see if the available key indicates
      an escape sequence of any sort */

   if ( kb_hold == ESC )
      {
      r = att_esc();            
      }
   else if ( kb_hold > 127 )
      {
      r = att_seq( kb_hold );
      }
   else
      {
      r = kb_hold;
      }

#if WRITE_OUTPUT
   fprintf( outfile, "att_rxstat(): read r 0x%02x\n", r );
   fflush( outfile );
#endif

   /* now check to see if there is a key or mouse input
      available */

   if ( r == 0 )
      {
      if ( ( tam_kready == TRUE ) || ( tam_mready == TRUE ))
         {
         return TRUE;
         }
      else
         {
         return FALSE;
         }
      }

   /* the available key is not 0; thus a character is available
      as "r" */

   kb_hold = r;
   tam_kready = TRUE;
   return TRUE;
   }

/*************************************************************************

   FUNCTION:       kb_rx()

   DESCRIPTION:    This function returns a single character from
		   the keyboard.  If a character is not available
		   it waits.  The function should be able to
		   recognize any special keys, and return the
		   appropriate Simple Software KB conventions
		   designated for them.

   INPUT:          none.

   RETURNS:        The function returns the ASCII code for the
		   key pressed, or the Simple Software KB convention
		   (see kb.h) for a function or other special key.

**************************************************************************/

kb_rx()
   {
#if WRITE_OUTPUT
   int r;
   
   r = att_rx();
   fprintf( outfile, "kb_rx(): 0x%02x\n", r );
   fflush( outfile );
   return r;
#else
   return att_rx();
#endif
   }

att_rx()
   {

   while( tam_kready == FALSE )
      {
      att_rxstat();
      }

   tam_kready = FALSE;
   return kb_hold;
   }

att_esc()
   {
   register int c;
   static char sequence[ 3 ];

#if WRITE_OUTPUT
   fprintf( outfile, "att_esc(): \n" );
   fflush( outfile );
#endif

   if ( kb_rxstat() == TRUE )
      {
      sequence[ 0 ] = kb_hold;
      sequence[ 1 ] = 0;
      tam_kready = 0;
      }
   else
      {
      return ESC;
      }

   if ( kb_rxstat() == TRUE )
      {
      sequence[ 1 ] = kb_hold;
      sequence[ 2 ] = 0;
      tam_kready = FALSE;
      }

   if ( strcmp( sequence, "[A" ) == 0 )
      {
      return KB_UP;
      }
   else if ( strcmp( sequence, "[B" ) == 0 )
      {
      return KB_DOWN;
      }
   else if ( strcmp( sequence, "[D" ) == 0 )
      {
      return KB_LEFT;
      }
   else if ( strcmp( sequence, "[C" ) == 0 )
      {
      return KB_RIGHT;
      }
   else if ( strcmp( sequence, "[?" ) == 0 )
      {
      tam_mready = TRUE;
      wreadmouse( w, &tam_mx, &tam_my, &tam_mb, &tam_mr );
      return 0;
      }
   else if ( strcmp( sequence, "Nh" ) == 0 )
      {
      return KB_P_DOWN;
      }
   else if ( strcmp( sequence, "Ng" ) == 0 )
      {
      return KB_P_UP;
      }
   else if ( strcmp( sequence, "Nf" ) == 0 )
      {
      return KB_DELETE;
      }
   else if ( strcmp( sequence, "Nj" ) == 0 )
      {
      return KB_INSERT;
      }
/*
   else if ( strcmp( sequence, "Oc" ) == 0 )
      {
      return KB_HELP;
      }
   else if ( strcmp( sequence, "NK" ) == 0 )
      {
      return KB_W_LEFT;
      }
   else if ( strcmp( sequence, "NL" ) == 0 )
      {
      return KB_W_RIGHT;
      }
*/
   else if ( strcmp( sequence, "[H" ) == 0 )
      {
      return KB_HOME;
      }
   else if ( strcmp( sequence, "NM" ) == 0 )
      {
      return KB_HOME;
      }
   else if ( strcmp( sequence, "NN" ) == 0 )
      {
      return KB_END;
      }
   else if ( strcmp( sequence, "O" ) == 0 )
      {
      return KB_END;
      }
/*
   else if ( strcmp( sequence, "Ne" ) == 0 )
      {
      return KB_D_WORD;
      }
   else if ( strcmp( sequence, "NE" ) == 0 )
      {
      return KB_D_LINE;
      }
   else if ( strcmp( sequence, "Ox" ) == 0 )
      {
      return KB_FIND;
      }
   else if ( strcmp( sequence, "Oy" ) == 0 )
      {
      return KB_REPLACE;
      }
   else if ( strcmp( sequence, "Ow" ) == 0 )
      {
      return KB_ABANDON;
      }
   else if ( strcmp( sequence, "Oo" ) == 0 )
      {
      return KB_SAVE;
      }
   else if ( strcmp( sequence, "Ok" ) == 0 )
      {
      return KB_QUIT;
      }
*/
   else
      {
      return 0;
      }
   }

att_seq( k )
   int k;
   {
   char *map;

   map = kcodemap( k );

#if WRITE_OUTPUT
   fprintf( outfile, "att_seq(): char 0x%02x, map <%s>\n", k, map + 1 );
   fflush( outfile );
#endif

   if ( strncmp( map + 1, "[A", 2 ) == 0 )
      {
      return KB_UP;
      }
   else if ( strncmp( map + 1, "[B", 2 ) == 0 )
      {
      return KB_DOWN;
      }
   else if ( strncmp( map + 1, "[D", 2 ) == 0 )
      {
      return KB_LEFT;
      }
   else if ( strncmp( map + 1, "[C", 2 ) == 0 )
      {
      return KB_RIGHT;
      }
   else if ( strncmp( map + 1, "[?", 2 ) == 0 )
      {
      tam_mready = TRUE;
      wreadmouse( w, &tam_mx, &tam_my, &tam_mb, &tam_mr );
      return 0;
      }
   else if ( strncmp( map + 1, "Nh", 2 ) == 0 )
      {
      return KB_P_DOWN;
      }
   else if ( strncmp( map + 1, "Ng", 2 ) == 0 )
      {
      return KB_P_UP;
      }
   else if ( strncmp( map + 1, "Nf", 2 ) == 0 )
      {
      return KB_DELETE;
      }
   else if ( strncmp( map + 1, "Nj", 2 ) == 0 )
      {
      return KB_INSERT;
      }
/*
   else if ( strncmp( map + 1, "Oc", 2 ) == 0 )
      {
      return KB_HELP;
      }
   else if ( strncmp( map + 1, "NK", 2 ) == 0 )
      {
      return KB_W_LEFT;
      }
   else if ( strncmp( map + 1, "NL", 2 ) == 0 )
      {
      return KB_W_RIGHT;
      }
*/
   else if ( strncmp( map + 1, "[H", 2 ) == 0 )
      {
      return KB_HOME;
      }
   else if ( strncmp( map + 1, "NM", 2 ) == 0 )
      {
      return KB_HOME;
      }
   else if ( strncmp( map + 1, "NN", 2 ) == 0 )
      {
      return KB_END;
      }
   else if ( strncmp( map + 1, "O", 2 ) == 0 )
      {
      return KB_END;
      }
/*
   else if ( strncmp( map + 1, "Ne", 2 ) == 0 )
      {
      return KB_D_WORD;
      }
   else if ( strncmp( map + 1, "NE", 2 ) == 0 )
      {
      return KB_D_LINE;
      }
   else if ( strncmp( map + 1, "Ox", 2 ) == 0 )
      {
      return KB_FIND;
      }
   else if ( strncmp( map + 1, "Oy", 2 ) == 0 )
      {
      return KB_REPLACE;
      }
   else if ( strncmp( map + 1, "Ow", 2 ) == 0 )
      {
      return KB_ABANDON;
      }
   else if ( strncmp( map + 1, "Oo", 2 ) == 0 )
      {
      return KB_SAVE;
      }
   else if ( strncmp( map + 1, "Ok", 2 ) == 0 )        
      {
      return KB_QUIT;
      }
*/
   else
      {
      return 0;
      }
   }
