/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tlist.hpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Header file for the TList class. This class is used
*				to represent a scrollable list of text strings, that
*				can be used to inplement pick lists etc.
*
* $Id: tlist.hpp 1.2 1994/03/09 11:50:15 kjb Exp $
*
****************************************************************************/

#ifndef	__TLIST_HPP
#define	__TLIST_HPP

#ifndef	__TVIEW_HPP
#include "tview.hpp"
#endif

#ifndef	__TSCROLLB_HPP
#include "tscrollb.hpp"
#endif

#ifndef	__ARRAY_HPP
#include "array.hpp"
#endif

/*------------------------------ Constants --------------------------------*/

ushort const

// Creation flags

	lsMultipleSelect	= 0x0001,	// Multiple cells can be selected
	lsDisjointSelect	= 0x0002,	// Multiple selections can be disjoint
	lsNormal			= 0x0000,	// Normal single selection list

// Flags used when extending selection

	lsExtendLeft		= 0x0010,	// Extending selection left
	lsExtendRight		= 0x0020,	// Extending selection right
	lsExtendUp			= 0x0040,	// Extending selection up
	lsExtendDown		= 0x0080,	// Extending selection down
	lsExtendHoriz		= lsExtendLeft | lsExtendRight,
	lsExtendVert		= lsExtendUp | lsExtendDown,
	lsExtending			= lsExtendHoriz | lsExtendVert,

// Flags used to determine where a mouse click occured

	lsAbove				= 0x0001,	// Mouse click was above
	lsBelow				= 0x0002,	// Mouse click was below
	lsLeft				= 0x0004,	// Mouse click was left
	lsRight				= 0x0008,	// Mouse click was right
	lsInside			= 0x0010,	// Mouse click was inside

// Flags used by each list cell

	lsSelected			= 0x0001,	// The list cell is selected
	lsDirty				= 0x0002;	// The list cell is dirty

/*-------------------------- Class definitions ----------------------------*/

//---------------------------------------------------------------------------
// The TListBase class is an abstract base class used to represent
// scrollable lists of items. The items themselves are not directly
// represented, but will be drawn by the the derived class.
//
// Palette layout:
//		1 = Border
//		2 = Interior
//		3 = Normal cell
//		4 = Selected cell
//		5 = Selected cell interior
//      6 = Cursor color
//---------------------------------------------------------------------------

class TListBase : public TView {
protected:
	ushort		flags;				// Creation flags
	TRect		visible;			// Index's of visible cells
	TRect		range;				// Range of selectable cells
	TRect		selection;			// Selected cells
	TPoint		cellSize;			// Size of each cell
	TPoint		cursor;				// Current position in list
	TScrollBar	*hScroll;			// Horizontal scroll bar (if any)
	TScrollBar	*vScroll;			// Vertical scroll bar (if any)

			// Pure virtual to draw a list cell
	virtual	void drawCell(int i,int j,const TRect& bounds) = 0;

			// Pure virtual to refresh a list cell
	virtual	void refreshCell(int i,int j) = 0;

			// Pure virtual to refresh the list
			void refresh();

			// Method to draw a given a cursor location
			void drawCell(int i,int j);
			void drawCell(const TPoint& cursor)
				{ drawCell(cursor.x,cursor.y); };

			// Methods to scroll to a specified item
			void hScrollTo(int i,bool redraw = true);
			void vScrollTo(int j,bool redraw = true);

			// Method to select a number of cells in specified direction
			void selectNext(ushort direction,short count,ushort modifiers,
				bool toTop = false);

			// Methods to select new cells
			void selectLeft(short count,ushort modifiers,bool toTop = false)
				{ selectNext(lsLeft,count,modifiers,toTop); };
			void selectRight(short count,ushort modifiers,bool toTop = false)
				{ selectNext(lsRight,count,modifiers,toTop); };
			void selectUp(short count,ushort modifiers,bool toTop = false)
				{ selectNext(lsAbove,count,modifiers,toTop); };
			void selectDown(short count,ushort modifiers,bool toTop = false)
				{ selectNext(lsBelow,count,modifiers,toTop); };

			// Method to focus the current cell
			void focusCurrent(bool toTop,bool redraw = true);

			// Method to find a cell from a mouse click
			ushort findCellHit(const TPoint& global,TPoint& loc);

public:
			// Constructor
			TListBase(const TRect& bounds,const TPoint& cellSize,
				TScrollBar *vScroll,TScrollBar *hScroll,
				ushort flags = lsNormal);

			// Virtual to handle an event
	virtual	void handleEvent(TEvent& event,phaseType phase = phFocused);

			// Methods to draw the view
	virtual	void draw(const TRect& clip);

			// Pure virtual to select a list cell
	virtual	void selectCell(int i,int j) = 0;
			void selectCell(const TPoint& p)	{ selectCell(p.x,p.y); };

			// Pure virtal to de-select a list cell
	virtual	void deselectCell(int i,int j) = 0;
			void deselectCell(const TPoint& p)	{ deselectCell(p.x,p.y); };

			// Pure virtual to toggle a list cell
	virtual	void toggleCell(int i,int j) = 0;
			void toggleCell(const TPoint& p)	{ toggleCell(p.x,p.y); };

			// Pure virtual to dirty a cell
	virtual	void dirtyCell(int i,int j) = 0;
			void dirtyCell(const TPoint& p)		{ dirtyCell(p.x,p.y); };

			// Methods to select a range of cells
			void selectRange(int left,int top,int right,int bottom);
			void selectRange(const TRect& r)
				{ selectRange(r.left(),r.top(),r.right(),r.bottom()); };

			// Methods to clear a range of cells
			void clearRange(int left,int top,int right,int bottom);
			void clearRange(const TRect& r)
				{ clearRange(r.left(),r.top(),r.right(),r.bottom()); };

			// Method to clear all currently selected cells
			void clearSelection();

			// Method to set the current cell
			void setCurrentCell(int i,int j);

			// Method to get the current cell
			void getCurrentCell(TPoint& c)	{ c = cursor; };

			// Method to set the horizontal scroll bar
			void setHScroll(TScrollBar *h);

			// Method to set the vertical scroll bar
			void setVScroll(TScrollBar *v);

			// Method to set the horizontal range
			void setHRange(short min,short max);

			// Method to set the vertical range
			void setVRange(short min,short max);

			// Methods to obtain the range values
			ushort minH() const	{ return range.topLeft.x; };
			ushort maxH() const	{ return range.botRight.x-1; };
			ushort minV() const	{ return range.topLeft.y; };
			ushort maxV() const	{ return range.botRight.y-1; };

			// Force a direct update to the list
			void update();

			// Method to return a reference to the view's current palette
	virtual	TPalette& getPalette() const;
	};

//---------------------------------------------------------------------------
// The TList class is a concrete class derived from TListBase that is used
// to manipulate dynamic scrollabe lists of text strings. The string data
// itself is not actually maintained in the list, but simply the pointers
// to the user's data.
//---------------------------------------------------------------------------

class TList : public TListBase {
protected:
	struct CellItem {
		const char	*text;		// Pointer to the cell's text
		uchar		flags;		// Flags for the item

		bool operator == (const CellItem& i)
			{ return text == i.text && flags == i.flags; };
		};

	Array<CellItem>	cells;		// Array of cells;
	TRect			dataBounds;	// Bounds on number of items in the list

			// Method to clear all the items in the list
			void clearItems();

			// Method to find a reference to a cell
			CellItem& findCell(int i,int j)
			{
				return cells[i * (dataBounds.bottom()-dataBounds.top())
							   + j - dataBounds.left()];
			};

			// Method to draw a list cell
	virtual	void drawCell(int i,int j,const TRect& bounds);

			// Method to refresh a cell
	virtual	void refreshCell(int i,int j);

public:
			// Constructor
			TList(const TRect& bounds,const TPoint& cellSize,
				const TRect& dataBounds = TRect(0,0,1,1),
				TScrollBar *vScroll = NULL,TScrollBar *hScroll = NULL,
				ushort flags = lsNormal);

			// Method to set the data bounds for the list
			void setDataBounds(const TRect& bounds);

			// Method to select a list cell
	virtual	void selectCell(int i,int j);

			// Method to de-select a list cell
	virtual	void deselectCell(int i,int j);

			// Method to toggle a list cell
	virtual	void toggleCell(int i,int j);

			// Method to dirty a list cell
	virtual	void dirtyCell(int i,int j);

			// Method to set a specific cell's value
			void setCell(int i,int j,const char *text);

			// Method to get a specific cell's value, and flags
			bool getCell(int i,int j,const char*& text);
	};

#endif	// __TLIST_HPP
