/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: msgbox.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Implementation of the messageBox routine.
*
* $Id: msgbox.cpp 1.2 1994/03/09 11:50:25 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tprogram.hpp"
#include "msgbox.hpp"
#include "tdialog.hpp"
#include "tbutton.hpp"
#include "tsttext.hpp"
#include "tfontmgr.hpp"
#include "techlib.hpp"
#include "tinputl.hpp"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/*----------------------------- Implementation ----------------------------*/

static const char *buttonNames[] = {
	MsgBoxText::yesText,
	MsgBoxText::noText,
	MsgBoxText::okText,
	MsgBoxText::cancelText,
	};

static ushort commands[] = {
	cmYes,
	cmNo,
	cmOk,
	cmCancel
	};

static const char *titles[] = {
	MsgBoxText::warningText,
	MsgBoxText::errorText,
	MsgBoxText::informationText,
	MsgBoxText::confirmText,
	};

ushort messageBox(const char *msg,ushort options)
/****************************************************************************
*
* Function:		messageBox
* Parameters:	msg		- Message text to display in the box
*				options	- Options flags defining the type of box to display
* Returns:		Command code representing the choice made.
*
* Description:	Pops up a standard message box, displays the message and
*				waits for the user to respond. Returns one of the
*				following codes:
*
*					cmOK		- The OK button was hit
*					cmCancel	- The Cancel button was hit
*					cmYes		- The Yes button was hit
*					cmNo		- The No button was hit
*
****************************************************************************/
{
	fontManager.useFont(fmSystemFont);
	int height = MGL_textHeight();
	int buttonHeight = height + 12;
	int msgwidth = MGL_sizex() / 2;

	// Compute the number of lines to display and compute the size of
	// the message box that will hold the text.

	int lines = countLines(msg,msgwidth-1,true);
	TRect bounds(0,0,msgwidth + 30,100);

	TDialog *dialog = new TDialog(bounds,titles[options & 0x03]);
	dialog->setOption(ofCentered,true);

	// Now adjust the size of the dialog box to fit the text that will
	// be displayed in it, and compute the bounding box for the text.

	TRect inner(dialog->innerBounds());
	int	adjustTop = inner.top() - bounds.top();
	int	adjustBottom = bounds.bottom() - inner.bottom();

	inner.bottom() = inner.top() + height * max(3,lines);
	bounds.bottom() = inner.bottom() + adjustBottom + buttonHeight + 20;
	dialog->setBounds(bounds);
	inner.moveTo(0,adjustTop+5);
	inner.right() = msgwidth;

	ushort flags = sfWrap | ((lines == 1) || options & mfCenterText ? 0 : sfLeftJust);
	TMultiLineText *text = new TMultiLineText(inner,msg,flags);
	text->setOption(ofCenterX,true);
	dialog->insert(text);

	// Create the buttons to be inserted into the box, and insert them.
	// Note that the Yes or Ok button is made the default.

	ushort	mask = 0x0080;
	int 	i,count,width = MGL_textWidth(MsgBoxText::cancelText) + 16;
	TButton	*buttons[4];

	for (i = 3,count = 0; i >= 0; i--,mask >>= 1) {
		if (options & mask) {
			buttons[count] = new TButton(TRect(0,0,width,buttonHeight),
				buttonNames[i],commands[i],bfNormal);
			dialog->insert(buttons[count]);
			if (options & (mask << 4)) {
				buttons[count]->setDefault(true);
				dialog->select(buttons[count]);
				}
			count++;
			}
		}

	// Determine where to place each of the buttons that have been created,
	// and move them to the correct locations.

	int 	x,sizex = (bounds.right()-bounds.left()-1+width) / (count+1);

	for (i = count-1, x = sizex - width/2; i >= 0; i--) {
		buttons[i]->moveTo(x - width/2,bounds.bottom() - buttonHeight - 10);
		x += sizex;
		}

	return TProgram::deskTop->execView(dialog);
}

ushort messageBox(ushort options,const char *fmt, ... )
/****************************************************************************
*
* Function:		messageBox
* Parameters:	options	- Options flags defining the type of box to display
*				fmt		- Format string for message text to display
*				...		- Optional printf() style arguments
* Returns:		Command code representing the choice made.
*
* Description:	Same as above, but performs formatting on the string
*				first. The maximum formatted message length is 255
*				characters.
*
****************************************************************************/
{
	char 		msg[256];
	va_list		argptr;

	va_start(argptr,options);
	vsprintf(msg,fmt,argptr);
	return messageBox(msg,options);
}

ushort inputBox(const char *title,const char *label,char *str,short limit)
/****************************************************************************
*
* Function:		inputBox
* Parameters:	title	- Title for the input box
*				label	- Label above the input line
*				str		- Place to store the input data
*				limit	- Maximum number of characters to input
* Returns:		Command that caused the box to be closed.
*
* Description:	Pops up an input box for input, using the data already
*				in 'str' as the default text. Either cmOK or cmCancel will
*				be returned depending on which button was pressed.
*
****************************************************************************/
{
	fontManager.useFont(fmSystemFont);
	int height = MGL_textHeight();
	int buttonHeight = height + 12;
    int msgwidth = MGL_sizex()/2;

	// Compute the number of lines to display and compute the size of
	// the message box that will hold the text.

	TRect bounds(0,0,msgwidth+30,100);

	TDialog *dialog = new TDialog(bounds,title);
	dialog->setOption(ofCentered,true);

	// Now adjust the size of the dialog box to fit the text that will
	// be displayed in it, and compute the bounding box for the text.

	TRect inner(dialog->innerBounds());
	int	adjustBottom = bounds.bottom() - inner.bottom();

	// Create and insert the label

	inner.top() += height;
	inner.bottom() = inner.top() + height;
	inner.left() = 15;
	inner.right() = 15 + MGL_textWidth(label);
	TLabel *l = new TLabel(inner,label,NULL);
	dialog->insert(l);

	// Create and insert the input line

	inner.top() = inner.bottom();
	inner.bottom() = inner.top() + height;
	inner.right() = msgwidth+15;
	TInputLine *input = new TInputLine(inner,limit,str[0] == '\0' ? NULL : str);
	dialog->insert(input);
	dialog->select(input);
	l->setAttached(input);

	// Adjust the bounds of the box to fit, including the buttons

	bounds.bottom() = inner.bottom() + adjustBottom + buttonHeight + 20;
	dialog->setBounds(bounds);

	// Create the buttons to be inserted into the box, and insert them.

	int width = MGL_textWidth(MsgBoxText::cancelText) + 16;
	int sizex = (bounds.right()-bounds.left()-1+width) / 3;
	inner.top() = bounds.bottom() - buttonHeight - 10;
	inner.bottom() = inner.top() + buttonHeight;
	inner.left() = sizex - width;
	inner.right() =  inner.left() + width;
	TButton *okBut = new TButton(inner,MsgBoxText::okText,
		cmOk,bfDefault);

	inner.left() += sizex;
	inner.right() =  inner.left() + width;
	TButton *cancelBut = new TButton(inner,MsgBoxText::cancelText,
		cmCancel,bfNormal);

	dialog->insert(cancelBut);
	dialog->insert(okBut);

	ushort retVal = TProgram::deskTop->execView(dialog);
	strcpy(str,input->getText());
	delete dialog;
	return retVal;
}
