/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tmenubar.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TMenuBar class.
*
* $Id: tmenubar.cpp 1.2 1994/03/09 11:50:43 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tmenubar.hpp"
#include "tmouse.hpp"
#include "tfontmgr.hpp"
#include <stdlib.h>

#define	OFFSETX		2
#define	OFFSETY		4

/*----------------------------- Implementation ----------------------------*/

int TMenuBar::findSelected(TPoint& where)
/****************************************************************************
*
* Function:		TMenuBar::findSelected
* Parameters:	where	- Selection point
* Returns:		Index of item that was selected (or closest one)
*
* Description:	Finds the item closest to the selection point. It is
*				assumed at this stage that the point is know to be contained
*				in the bounding rectangle for the entire menu.
*
****************************************************************************/
{
	for (int i = 0; i < items.numberOfItems(); i++) {
		TRect& b = items[i]->bounds;
		if (b.left() <= where.x && where.x < b.right())
			return items[i]->disabled ? -1 : i;
		}
	if (where.x < items[0]->bounds.left() && !items[0]->disabled)
		return 0;
	i = items.numberOfItems()-1;
	if (where.x >= items[i]->bounds.right() && !items[i]->disabled)
		return i;
	return -1;
}

ushort TMenuBar::execute()
/****************************************************************************
*
* Function:		TMenuBar::execute
* Returns:		Command causing the menu to stop executing.
*
* Description:	Main interaction handling routine for the TMenuBar class.
*				Here we track the mouse in the menu and perform the
*				menu selection etc.
*
****************************************************************************/
{
	TEvent	event;
	bool	autoTracking = true;
	bool 	oldMove = eventQueue.mouseMove(true);
	ushort	result = cmMenu;
	int		selection;

	if (current == -1)
		current = findNextItem(items.numberOfItems()-1);
	selection = current;

	repaintSave = false;			// Nothing saved behind menu's yet
	subMenuID = -1;

	// The first event we recieve here is the original event that caused
	// the menu selection process to begin.

	while (result == cmMenu) {
		getEvent(event);
		switch (event.what) {
			case evMouseDown:
			case evMouseUp:
				autoTracking = true;
				if (includes(event.where)) {
					if ((selection = findSelected(event.where)) != -1) {
						refresh(-1);
						refresh(selection);

						// The menu was selected, so pull it down and
						// execute it.

						result = executeSubMenu(event);
						}
					else
						result = cmValid;
					}
				else
					result = cmValid;
				break;
			case evMouseMove:
				// The mouse was moved, so if we are currently tracking
				// the position of the mouse cursor, check if we need
				// to update the current selection

				if (autoTracking) {
					if (includes(event.where)) {
						selection = findSelected(event.where);
						if (current != selection)
							refresh(selection);
						result = executeSubMenu(event);
						}
					}
				break;
			case evKeyDown:
			case evKeyAuto:
				bool tracking = false;
				switch (event.key.keyCode) {
					case kbUp:
					case kbDown:
					case kbEnter:
					case kbGrayEnter:
						result = executeSubMenu(event);
						break;
					case kbLeft:
						selection = findPrevItem(selection);
						refresh(selection);
						if (subMenuID != -1)
							result = executeSubMenu(event);
						break;
					case kbRight:
						selection = findNextItem(selection);
						refresh(selection);
						if (subMenuID != -1)
							result = executeSubMenu(event);
						break;
					case kbEsc:
						result = cmValid;
						break;
					default:
						if ((selection = findShortCut(event)) != -1) {
							if (!items[selection]->disabled) {
								// Menu was selected, so pull it down
								// and execute it.

								refresh(selection);
								result = executeSubMenu(event);
								}
							}
						else {
							selection = current;
							tracking = autoTracking;
							}
						break;
					}
				autoTracking = tracking;
				break;
			case evCommand:
				// Must have received a cmMenu command to get here

				PRECONDITION(event.message.command == cmMenu);
				refresh(-1);
				refresh(selection);
				autoTracking = false;
				break;
			}

		// Refresh the state of the menu if it has changed

		if (current != selection)
			refresh(selection);
		}

	// Unhilight the menu bar on return, and restore stuff behind menus

	restoreMenu();
	refresh(-1);
	eventQueue.mouseMove(oldMove);
	return result;
}

void TMenuBar::drawItem(int index)
/****************************************************************************
*
* Function:		TMenuBar::drawItem
* Parameters:	index	- Index of the menu item to draw
*
* Description:	Draws the menu item 'index'. If the item is the currently
*				selected item, we draw the item as selected, otherwise
*				we draw it as non-selected.
*
****************************************************************************/
{
	TMenuItem& item = *items[index];

	MGL_pushViewport();
	MGL_setRelViewport(item.bounds);

	int maxx = MGL_maxx();
	int maxy = MGL_maxy();
	int depressed = (index == current);
	int offsetx = OFFSETX + depressed,offsety = OFFSETY + depressed;

	// Highlight the item if it is currently selected

	if (index == current) {
		MGL_setColor(getColor(4));
		MGL_fillRectCoord(0,0,maxx+1,maxy+1);
		MGL_setBorderColors(getColor(1),getColor(2));
		drawBorderCoord(0,0,maxx+1,maxy+1,BDR_INSET,1);
		}

	// Draw the item's text

	MGL_setColor(getColor(item.disabled ? 6 : ((index == current) ? 7 : 5)));
	MGL_setTextJustify(LEFT_TEXT,TOP_TEXT);
	MGL_drawStrXY(offsetx + MGL_charWidth(' '),offsety,item.name);

	// Underline the keyboard shortcut character if present

	if (item.hotChar)
		drawLineCoord(item.hot1.x + depressed,
					  item.hot1.y + depressed,
					  item.hot2.x + depressed - _MVIS_sysLineWidth + 1,
					  item.hot2.y + depressed);

	MGL_popViewport();
}

void TMenuBar::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TMenuBar::draw
* Parameters:	clip	- Clipping rectangle for the view
*
* Description:	Method to draw the menu bar.
*
****************************************************************************/
{
	mouse.obscure();

	MGL_setClipRect(clip);

	// Draw the background for the menu

	MGL_setColor(getColor(3));
	MGL_fillRect(bounds);

	// Draw the border around the menu

	MGL_setBorderColors(getColor(1),getColor(2));
	drawBorder(bounds,BDR_OUTSET,1);

	// Draw the menu items in the menu

	TextJust old;			// Save current text settings
	fontManager.useFont(fmSystemFont);

	for (int i = 0; i < items.numberOfItems(); i++)
		drawItem(i);

	old.use();				// Restore old text settings
	mouse.unobscure();
}

TRect& TMenuBar::setItemBounds(int index,const TPoint& start)
/****************************************************************************
*
* Function:		TMenuBar::setItemBounds
* Parameters:	index	- Index of menu item to compute bounds for
*				start	- Starting point for the item's bounds
* Returns		Reference to the bounding box
*
* Description:	Sets the bounding box for the indexed item.
*
****************************************************************************/
{
	TRect& b = items[index]->bounds;

	b.topLeft = start;
	b.right() = b.left() + MGL_textWidth(items[index]->name)
		+ MGL_textWidth("  ") + 4;
	b.bottom() = b.top() + itemHeight;
	return b;
}

void TMenuBar::doneDefinition()
/****************************************************************************
*
* Function:		TMenuBar::doneDefinition
*
* Description:	Completes the definition of the menu structure. Here we
*				compute the size of the bounding box given the menu items
*				stored in it.
*
****************************************************************************/
{
	int		i;
	metrics	m;

	TextJust old;			// Save current text settings
	fontManager.useFont(fmSystemFont);

	MGL_getFontMetrics(&m);
	itemHeight = m.ascent-m.descent+1+7;

	// Compute the bounding box for each item, and compute the bounding
	// box for the entire menu.

	TPoint start(_MVIS_sysLineWidth,_MVIS_sysLineWidth);
	bounds.bottom() = bounds.top()+1;
	for (i = 0; i < items.numberOfItems(); i++) {
		TRect& b = setItemBounds(i,start);
		bounds += b;
		start.x = b.right();
		}

	bounds.bottom() += _MVIS_sysLineWidth;

	// Now fixup any submenus that are attached to this menu, moving them
	// to the correct location

	for (i = 0; i < items.numberOfItems(); i++) {
		TMenuItem& item = *items[i];
		PRECONDITION(item.isSubMenu());
		item.subMenu->moveTo(item.bounds.left()-_MVIS_sysLineWidth,bounds.bottom()-1);

		// Compute the position of the hot character underscore if
		// present.

		if (item.hotChar) {
			char old = item.name[item.hotIndex];
			item.name[item.hotIndex] = '\0';
			item.hot1.x = OFFSETX;	item.hot1.y = OFFSETY;
			MGL_underScoreLocation(&item.hot1.x,&item.hot1.y,item.name);
			item.hot1.x += MGL_textWidth(item.name) + MGL_charWidth(' ');
			item.hot2 = item.hot1;
			metrics m;
			MGL_getCharMetrics(old,&m);
			item.hot2.x += m.fontWidth-1;
			item.name[item.hotIndex] = old;
			}
		}

	old.use();
}
