
; --------------------------------------------------------------------------
; Procedure   : InitArrays()
; Description : Initialise dynamic arrays for controls, windows
; --------------------------------------------------------------------------

proc InitArrays()

controls["MenuState"] = "None"
controls["Dictionary"] = "None"
controls["Homedir"] = directory()
controls["Printer"] = GetDefaultPrinter()

stdwindow["Height"] = 22
stdwindow["Width"] = 82
stdwindow["Canmove"] = False
stdwindow["Canresize"] = False
stdwindow["Canmaximize"] = False
stdwindow["Origincol"] = 0
stdwindow["Originrow"] = 1
stdwindow["Hasshadow"] = False

helpwindow["Height"] = 24
helpwindow["Width"] = 82
helpwindow["Canmove"] = False
helpwindow["Canresize"] = False
helpwindow["Canmaximize"] = False
helpwindow["Origincol"] = 0
helpwindow["Originrow"] = 1
helpwindow["Hasshadow"] = False

promptwindow["Height"] = 2
promptwindow["Width"] = 80
promptwindow["Style"] = 116
promptwindow["Canmove"] = False
promptwindow["Canresize"] = False
promptwindow["Canmaximize"] = False
promptwindow["Origincol"] = 0
promptwindow["Originrow"] = 23
promptwindow["Hasshadow"] = False
promptwindow["Hasframe"] = False

msgwindow["hasframe"] = False

endproc
WriteLib AppLib InitArrays
Release procs InitArrays

; --------------------------------------------------------------------------
; Procedure   : Warning()
; Description : Displays a warning message, alerts user, then waits
;    Argument - message text
; --------------------------------------------------------------------------

proc warning(mtext)
Private Quarter_Note,Half_Note

Quarter_Note = 337
Half_Note    = 675

message mtext
Sound 783 Quarter_Note                  ;Middle C
Sound 535   Half_Note                   ;G above Middle C
sleep 3000
message ""

endproc
WriteLib AppLib warning
Release procs warning

; --------------------------------------------------------------------------
; Procedure   : SetPrompt()
; Description : Show 2 lines of information at the bottom of the screen
;   Arguments - 2 lines of text
; --------------------------------------------------------------------------

proc SetPrompt(line1, line2)
   private phandle

window create floating attributes promptwindow to phandle
@ 0,0
?? line1
@ 1,0
?? line2
promptvalues["line1"] = line1
promptvalues["line2"] = line2
return phandle

endproc
WriteLib AppLib SetPrompt
Release procs SetPrompt

; --------------------------------------------------------------------------
; Procedure   : RefreshPrompt()
; Description : Refresh 2 lines of information at the bottom of the screen
; --------------------------------------------------------------------------

proc RefreshPrompt()
   private phandle

window create floating attributes promptwindow to phandle
@ 0,0
?? promptvalues["line1"]
@ 1,0
?? promptvalues["line2"]
return phandle

endproc
WriteLib AppLib RefreshPrompt
Release procs RefreshPrompt

; --------------------------------------------------------------------------
; Procedure   : ClearPrompt()
; Description : Remove 2 lines of information from the bottom of the screen
;   Arguments - window handle
; --------------------------------------------------------------------------

proc ClearPrompt(phandle)

window select phandle
window close

endproc
WriteLib AppLib ClearPrompt
Release procs ClearPrompt

; --------------------------------------------------------------------------
; Procedure   : PrinterSetup
; Description : Get name of printer setup to be used and save in controls
; --------------------------------------------------------------------------

proc PrinterSetup()
   private printerlist, reportprinter, spwin

array printerlist[nrecords("Printers")]

if PrinterBox() then
   return
endif

View "Printers"
scan
   printerlist[recno()] = [Printer]
endscan
clearimage

spwin = SetPrompt("Move the highlight to the printer setup you want",
   "Then press Enter or Click the OK button")
showarray printerlist printerlist to reportprinter
ClearPrompt(spwin)

if reportprinter <> "Esc" then
   controls["Printer"] = reportprinter
endif

endproc
WriteLib AppLib PrinterSetup
Release procs PrinterSetup

; --------------------------------------------------------------------------
; Procedure   : GetDefaultPrinter()
; Description : Check for the default printer setup
;     Returns : Printer setup or "None"
; --------------------------------------------------------------------------

proc GetDefaultPrinter()
   private retcode

View "Printers"
moveto [Default]
locate "*"
if retval then
   retcode = [Printer]
else
   retcode = "None"
endif
clearimage
return retcode

endproc
WriteLib AppLib GetDefaultPrinter
Release procs GetDefaultPrinter

; --------------------------------------------------------------------------
; Procedure   : ReportSetup()
; Description : Customise report spec for a particular printer setup
;   Arguments : Table name, report number, setup name
; --------------------------------------------------------------------------

proc ReportSetup(tabname, reptnum, printype)
   private printlen, printwidth, printinit, printreset

if printype = "None" then
   warning("No printer setup defined. Using Paradox defaults")
   return
endif

view "Printers"
if printype = "DEFAULT" then
   moveto [Default]
   locate "*"
else
   moveto [Printer]
   locate printype
endif
if retval = False then
   clearimage
   warning("Unknown printer setup. Using Paradox defaults")
   return
endif

printlen = [Lines]
printwidth = [Width]
printinit = [Setup]
printreset = [Reset]
clearimage

{Report} {Change}
select tabname
select reptnum
Enter
Menu {Setting} {PageLayout} {Length}
select printlen
Menu {Setting} {PageLayout} {Width}
select printwidth
Menu {Setting} {Setup} {Custom} {LPT1}
select printinit
select printreset
Do_It!

endproc
WriteLib AppLib ReportSetup
Release procs ReportSetup

; --------------------------------------------------------------------------
; Procedure   : CheckAccess()
; Description : Check if we have access to each table to be documented
; --------------------------------------------------------------------------

proc CheckAccess()
   private i, workdir, accessok

workdir = controls["Workdir"]
nrecs = nrecords("Formlist")
edit "Formlist"
home
while True
   if tablerights(workdir + [Table], "ReadOnly") then
      accessok = True
   else
      accessok = CheckPassword([Table])
   endif

   if accessok = False then
      del
      loop
   endif

   if atlast() then
      quitloop
   else
      down
   endif
endwhile

Do_It!
clearimage

endproc
WriteLib AppLib CheckAccess
Release procs CheckAccess

; --------------------------------------------------------------------------
; Procedure   : CheckPassword()
; Description : Check if we have access to each table to be documented
;    Argument : Name of the table to check
; --------------------------------------------------------------------------

proc CheckPassword(tabname)
   private i

for i from 1 to 3
   if PasswordBox(tabname) then
      password dbox["value1"]
      if tablerights(workdir + [Table], "ReadOnly") then
         return True
      endif
   else
      warning("No password given. Table will be deleted from the list")
      return False
   endif
   if i < 3 then
      warning("Password incorrect. Try again")
   endif
endfor

warning("Password incorrect. Table will be deleted from the list")
return False

endproc
WriteLib AppLib CheckPassword
Release procs CheckPassword

; --------------------------------------------------------------------------
; Procedure   : CheckTableUse()
; Description : Checks if there is data from a previous dictionary run
;     Returns : Value for controls - None or Usedata
; --------------------------------------------------------------------------

proc CheckTableUse()

if isempty("Formdir") then
   return "None"
else
   view "Formdir"
   controls["Workdir"] = [Directory]
   clearimage
endif
if isempty("Formlist") then
   return "None"
endif
if isempty("Frmimage") then
   return "Analyse"
endif
if isempty("Frmfield") then
   return "Analyse"
endif
return "Usedata"

endproc
WriteLib AppLib CheckTableUse
Release procs CheckTableUse

; --------------------------------------------------------------------------
; Procedure   : ShowHelp
; Description : Display a set (up to 9) help screens
;   Arguments - help set procedure name (no trailing digit), number in set
; --------------------------------------------------------------------------

proc ShowHelp(hprocname, nscreens)
   private screenum, keyup, keydown, nextaction, helparray, reshow,
      helpwin, menusave

dynarray helparray[]
menusave = controls["Menustate"]
reshow = True
window create floating @ 1,0 attributes helpwindow to helpwin
screenum = 1
DisplayMenu("Help")
while true
   if reshow then
      window select helpwin
      clear
      @ 0,0
      execproc hprocname + strval(screenum)
      @ 21,0
      switch
         case nscreens = 1:
            keyup = FALSE keydown = FALSE
         case screenum = 1: ?? "PgDn - Next"
            keyup = FALSE keydown = TRUE
         case screenum = nscreens: ?? "PgUp - Previous"
            keyup = TRUE keydown = FALSE
         otherwise: ?? "PgDn - Next, PgUp - Previous"
            keyup = TRUE keydown = TRUE
      endswitch
      @ 21,40
      ?? "Escape - End help, return to menu"
      if keydown = False then
         menudisable "Next"
      else
         menuenable "Next"
      endif
      if keyup = False then
         menudisable "Previous"
      else
         menuenable "Previous"
      endif
      reshow = False
   endif

   getevent message "Menuselect" key "All" to helparray
   nextaction = HelpWait(helparray)

   switch
      case nextaction = "Quit":
         window select helpwin
         window close
         DisplayMenu(menusave)
         return
      case nextaction = "Up" and keyup = TRUE:
         screenum = screenum - 1
         reshow = True
      case nextaction = "Down" and keydown = TRUE:
         screenum = screenum + 1
         reshow = True
   endswitch
endwhile

endproc
Writelib AppLib ShowHelp
Release Procs ShowHelp

; --------------------------------------------------------------------------
; Procedure   : HelpWait()
; Description : Actions triggered by user when using help
;   Argument  : Dynamic array containing the event data
; --------------------------------------------------------------------------

proc HelpWait(edata)


if edata["type"] = "MESSAGE" then
   switch
      case edata["Menutag"] = "Next" and keydown:
         return "Down"
      case edata["Menutag"] = "Previous" and keyup:
         return "Up"
      case edata["Menutag"] = "Return":
         return "Quit"
      otherwise:
         beep  sleep 100  beep  return "None"
   endswitch
endif
if edata["type"] = "KEY" then
   switch
      case edata["Keycode"] = -73 and keyup:           ;PgUp
         return "Up"
      case edata["Keycode"] = -81 and keydown:         ;PgDn
         return "Down"
      case edata["Keycode"] = 27:                      ;Esc
         return "Quit"
      otherwise:
         beep  sleep 100  beep  return "None"
   endswitch
endif

beep  sleep 100  beep  return "None"

endproc
Writelib AppLib HelpWait
Release Procs HelpWait
