/* petal.h
 * 1997/09/09
 * (c)Cipher AKA Jeff Weeks of Code X Software
 *
 * This header file defines the basic PETAL function structure of PETAL.
 */

#if !defined __petal_graphics__
#define __petal_graphics__

#include <string.h>

// Magic number definitions for drivers
#define  XSHM  0
#define  DGA   1
#define  LFB   2
#define  PMODE 3
#define  BANK  4

// A quick macro to adjust our x and y values by the image's offsets
#define OffsetAdjust(img, x, y) \
  (x) += img->x_off;            \
  (y) += img->y_off;

typedef struct { unsigned char r,g,b;   } rgb;
typedef struct { unsigned char r,g,b,a; } rgba;

// Here's the screen structure.  It's a fairly fundamental object in the PETAL
// graphics library.  This structure contains all the information necessary to
// intensly describe a graphics surface, real or virtual.
typedef struct {
  int size;                // the size in bytes of the whole screen
  int width, height;       // the width and height values
  int x_off, y_off;        // the x and y offsets;  Usually (0,0)
  int bytes_per_line;      // the number of bytes in each scanline
  int bytes_per_pixel;     // the number of bytes in one pixel
  int clipping;            // the clipping flag
  int cx1, cy1, cx2, cy2;  // the clipping rectangle
  int transparent;         // the transparency flag
  int trans_col;           // the transparent colour
  char *address;           // the address of the actual screen
  char bpp;                // the number of bits per pixel
} Image;

// Here's the Driver structure of PETAL.  A very simple, but hightly important,
// object of the graphics library.  Note to self: If this structure is altered,
// then the drivers will have to be recomplied.
typedef struct {
  // Here is some information about the current PETAL driver:
  int magic;               // Contains the driver's magic number
  int OSD1, OSD2, OSD3,    // Operating system dependant information
      OSD4, OSD5, OSD6;    // goes in these 6 variables.
  void *handle;            // the handle of the driver goes here
  Image  display;          // information about the actual display
  Image *default_image;    // the default image that PETAL will write to
  char *window_name;       // the window title for X-Windows
  rgb *pal;

  // Next are pointers to all functions supported by PETAL driver.
  // The driver is the non-portable aspect of PETAL, and therefore,
  // really only includes a blit function.  Every thing else can
  // be done portably
  int  (*detect)(void);
  int  (*set_mode)(char *, int, int, int);
  void (*deinit)(void);
  void (*blit)(Image *, int x, int y);    // a routine to copy an Image to video memory
  void (*get_image)(Image *, int x, int y); // a routine to copy video memory to an Image
  char (*get_pointer)(int *x, int *y);
  char (*get_key)(void);
  void (*set_pal)(rgb *pal);
} Driver;

extern Driver petal;

typedef struct {
  int x, y;
  int colour;
} Point2d;

// The rest of this header file is included in an #if defined statement
// for two reasons.  One being that the drivers don't need to know about
// the petal functions (drivers are C, not C++, and therefore won't
// see what is inside these #if statements) and two, because if the
// drivers were to see some of our C++ definitions, they definitly
// wouldn't like it.
#if defined __cplusplus

// Standard PETAL graphics driver functions
int    init_petal(char *drv_name, char *window_name);
int    init_petal(Driver *driver, char *window_name);
void   deinit_petal(void);
int    set_resolution(int  x, int  y, int  bpp);
int    get_resolution(int *x, int *y, int *bpp);
Image *new_image(void);
Image *new_image(int x, int y);
void   delete_image(Image *img);
void   clip_image(int x, int y, int width, int height, Image *img = petal.default_image);
void   get_image(Image *dest, Image *src, int x=0, int y=0);
void   show_image(Image *img = petal.default_image, int x=0, int y=0);
void   show_image(int x, int y, int width, int height, Image *img = petal.default_image, int tx=0, int ty=0);
void   copy_image(int x, int y, int width, int height, Image *dest, Image *src = petal.default_image, int tx=0, int ty=0);
void   copy_image(Image *dest, Image *src = petal.default_image, int x = 0, int y = 0);
Image *sub_image(int x, int y, int width, int height, Image *img = petal.default_image);
void   transparent(int colour, Image *img = petal.default_image);
void   offset_image(int x, int y, Image *img = petal.default_image);
void   set_bpp32(void);
void   set_bpp24(void);
void   set_bpp16(void);
void   set_bpp15(void);
void   set_bpp8(void);
inline void transparent(bool yes_no, Image *img = petal.default_image) { img->transparent = yes_no; }
inline void clip_image (bool yes_no, Image *img = petal.default_image) { img->clipping = yes_no; }
inline void set_image  (Image *img) { petal.default_image = img; }
inline void get_image  (Image *img = petal.default_image, int x=0, int y=0) { petal.get_image(img, x, y); }
inline void clear_image(Image *img = petal.default_image) { memset(img->address,0,img->size); }
inline void set_pal(rgb *pal) { if(petal.display.bpp == 8) { petal.set_pal(pal); petal.pal = pal;  } }
inline char get_key(void) { return petal.get_key(); }
inline char get_pointer(int *x, int *y) { return petal.get_pointer(x,y); }

// Standard PETAL portable functions...
// I did some research and found pointers to functions to be faster
// than a large switch statement for determining what function to use in what
// colour depth.  For this reason PETAL uses pointers to each function.
// When you set a certain video mode PETAL will setup the pointers for you.

extern unsigned long max_col;

// colour functions
extern int  (*colour)(float fr, float fg, float fb);

// Pixel functions
extern void (*putpixel)(int x, int y, int c, Image *img = petal.default_image);
extern int  (*getpixel)(int x, int y, Image *img = petal.default_image);
extern void (*fputpixel)(char *offset, int c);   // use fput/fgetpixel in custom functions (where you know the offset)
extern int  (*fgetpixel)(char *offset);

// line functions
extern void (*line) (int x1, int y1, int x2, int y2, int c, Image *img = petal.default_image);
extern void (*hline)(int x1, int x2, int y, int c, Image *img = petal.default_image);        // horizontal
extern void (*vline)(int y1, int y2, int x, int c, Image *img = petal.default_image);        // vertical
void rect(int x1, int y1, int x2, int y2, int c, Image *img = petal.default_image);

// circle functions
void circle(int xc, int yc, int r, int c, Image *img = petal.default_image);
void filled_circle(int xc, int yc, int r, int c, Image *img = petal.default_image);

// Filled functions
extern void (*clear)(int c, Image *img = petal.default_image);
extern void (*filled_rect)(int x1, int y1, int x2, int y2, int c, Image *img = petal.default_image);

// polygon functions
void poly(Point2d *points, int num_points, int colour, Image *img = petal.default_image);
#endif

#define MIN(a,b)   ( ((a) < (b)) ? (a) : (b) )
#define MAX(a,b)   ( ((a) > (b)) ? (a) : (b) )
#define MID(a,b,c) (MAX( (a), MIN( (b) , (c) )) )
#define ABS(a)     ( ((a) > 0)   ? (a) : (-(a)) )

int clip(Image *one, Image *two, signed int *x, signed int *y, int *x1, int *y1, int *x2, int *y2);
int get_color(char *text, int bytes);

#endif
