/*
    VGADEMO -- VGA graphics demo.  Pixels, lines, circles, ellipses.
    line1() is the original 'C' line function.
    line2() is an optimized version with assembly language.
    rectangle() is the un-filled rectangle procedure.
    putrow() draws a horizontal line using assembly language.
    fillrect1() is the slow filled rectangle procedure.
    fillrect2() is the fast filled rectangle procedure.
    circle1() uses floating point arithmetic.
    circle2() uses integer arithmetic.
    circle3() also uses the circle's symmetry.
    fillcircle1() draws a line across the circle for each pixel.
    fillcircle2() does not draw duplicate lines, so is faster.
    ellipse() is the un-filled ellipse procedure.
    fillellipse() is the filled ellipse procedure.
    getimage() reads a rectangular image.
    putimage() draws a rectangular image.
    outstr() draws strings.
*/

#include <stdlib.h>
#include <conio.h>

void setmode(int mode);
void cls(void);
void putpixel(int x, int y, int color);
int  getpixel(int x, int y);
void line1(int x1, int x2, int y1, int y2, int color);
void line2(int x1, int x2, int y1, int y2, int color);
void rectangle(int x1, int y1, int x2, int y2, int color);
void fillrect1(int x1, int y1, int x2, int y2, int color);
void putrow(int x1, int x2, int y, int color);
void fillrect2(int x1, int y1, int x2, int y2, int color);
void circle1(int x, int y, int r, int color);
void circle2(int x, int y, int r, int color);
void circle3(int x, int y, int r, int color);
void fillcircle1(int x, int y, int r, int color);
void fillcircle2(int x, int y, int r, int color);
void ellipse(int x, int y, int rx, int ry, int color);
void fillellipse(int cx, int cy, int rx, int ry, int color);
void putimage(void *image, int x1, int y1);
void getimage(void *image, int x1, int y1, int x2, int y2);
void outstr(int x, int y, char *s, int color);

main()
{
    int i;
    int buf[202];

    setmode(0x13);

    while(!kbhit())
    {
        for(i = 0; i < 100; i++)
            putpixel(random(320), random(200), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 40; i++)
            line1(random(320), random(200),
                  random(320), random(200),
                  random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 40; i++)
            line2(random(320), random(200),
                  random(320), random(200),
                  random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 40; i++)
            rectangle(random(320), random(200),
                      random(320), random(200),
                      random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            fillrect1(random(320), random(200),
                      random(320), random(200),
                      random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 40; i++)
            fillrect2(random(320), random(200),
                      random(320), random(200),
                      random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            circle1(random(320), random(200),
                    random(50), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            circle2(random(320), random(200),
                    random(50), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            circle3(random(320), random(200),
                    random(50), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 5; i++)
            fillcircle1(random(320), random(200),
                        random(50), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 5; i++)
            fillcircle2(random(320), random(200),
                        random(50), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            ellipse(random(320), random(200),
                    random(60), random(60), random(256));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 5; i++)
            fillellipse(random(320), random(200),
                        random(60), random(60), random(256));
    }

    while(kbhit()) getch(); cls();

    rectangle(2, 2, 17, 17, 14); rectangle(4, 4, 15, 15, 14);
    rectangle(6, 6, 13, 13, 14); rectangle(8, 8, 11, 11, 14);
    rectangle(1, 1, 18, 18, 12); rectangle(3, 3, 16, 16, 12);
    rectangle(5, 5, 14, 14, 12); rectangle(7, 7, 12, 12, 12);

    getimage(buf, 0, 0, 19, 19);

    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            putimage(buf, random(300), random(180));
    }

    while(kbhit()) getch(); cls();
    while(!kbhit())
    {
        for(i = 0; i < 10; i++)
            outstr(random(192), random(192), "This is a string",
                   random(256));
    }

    while(kbhit()) getch(); cls();
    setmode(0x03);
    return 0;
}

void setmode(int mode)
{
    asm {
        mov ax,mode;
        xor ah,ah;
        int 10h;
    }
}

void cls(void)
{
    asm {

        mov ax,0A000h;      /* ES = video memory */
        mov es,ax;
    
        xor di,di;          /* Set up for clear */
        xor ax,ax;
        mov cx,8000h;
    
        rep stosw;          /* Clear the screen */
    }
}

void putpixel(int x, int y, int color)
{
    asm {
        mov bx,x;           /* BX = X, CX = Y */
        mov cx,y;
        cmp bx,320;         /* Clip to screen */
        jae pp_done;
        cmp cx,200;
        jae pp_done;

        add bh,cl;          /* BX = offset */
        shl cx,6;
        add bx,cx;
    
        mov ax,0A000h;      /* ES = video memory */
        mov es,ax;
        mov ax,color;       /* AX = color */
        mov es:[bx],al;     /* Plot pixel */
    }
pp_done:
}

int getpixel(int x, int y)
{
    asm {
        mov bx,x;           /* BX = X, CX = Y */
        mov cx,y;
        add bh,cl;          /* BX = offset */
        shl cx,6;
        add bx,cx;
    
        mov ax,0A000h;      /* ES = video memory */
        mov es,ax;
        mov al,es:[bx];     /* Read pixel */
    }

    return _AL;
}

void line1(int x1, int y1, int x2, int y2, int color)
{
    int d, x, y, sx, sy, dx, dy;

    dx = x2 - x1; dy = y2 - y1;
    if(dx > 0) sx = 1; else sx = -1;
    if(dy > 0) sy = 1; else sy = -1;
    dx = abs(dx); dy = abs(dy);
    x = x1; y = y1;

    if(dx > dy) {
        d = (dy - dx) / 2;          /* X is major axis */
        while(x != x2) {
            putpixel(x, y, color);  /* Put pixel */
            if(d > 0) {             /* Minor-axis change? */
                d -= dx;            /* Decrease by 1 */
                y += sy;                       
            }
            d += dy;                /* Increase by dy/dx */
            x += sx;
        }
        putpixel(x, y, color);      /* Put last pixel */
    } else {
        d = (dx - dy) / 2;          /* Y is major axis */
        while(y != y2) {
            putpixel(x, y, color);  /* Put pixel */
            if(d > 0) {             /* Minor-axis change? */
                d -= dy;            /* Decrease by 1 */
                x += sx;
            }
            d += dx;                /* Increase by dx/dy */
            y += sy;
        }
        putpixel(x, y, color);      /* Put last pixel */
    }
}

#define PUT_PIXEL(x, y, c, q) \
    asm { mov bx,x; mov cx,y; cmp bx,320; jae d##q;  \
          cmp cx,200; jae d##q; add bh,cl; shl cx,6; \
          add bx,cx; mov ax,color; mov es:[bx],al;   \
        } d##q:

void line2(int x1, int y1, int x2, int y2, int color)
{
    int d, x, y, sx, sy, dx, dy;

    dx = x2 - x1; dy = y2 - y1;
    if(dx > 0) sx = 1; else sx = -1;
    if(dy > 0) sy = 1; else sy = -1;
    dx = abs(dx); dy = abs(dy);
    x = x1; y = y1;

    asm mov ax,0A000h;              /* ES = video memory */
    asm mov es,ax;

    if(dx > dy) {
        d = (dy - dx) / 2;          /* X is major axis */
        while(x != x2) {
            PUT_PIXEL(x, y, color, 0); /* Put pixel */
            if(d > 0) {             /* Minor-axis change? */
                d -= dx;            /* Decrease by 1 */
                y += sy;
            }
            d += dy;                /* Increase by dy/dx */
            x += sx;
        }
        PUT_PIXEL(x, y, color, 1);  /* Put last pixel */
    } else {
        d = (dx - dy) / 2;          /* Y is major axis */
        while(y != y2) {
            PUT_PIXEL(x, y, color, 2); /* Put pixel */
            if(d > 0) {             /* Minor-axis change? */
                d -= dy;            /* Decrease by 1 */
                x += sx;
            }
            d += dx;                /* Increase by dx/dy */
            y += sy;
        }
        PUT_PIXEL(x, y, color, 3);  /* Put last pixel */
    }
}

void rectangle(int x1, int y1, int x2, int y2, int color)
{
    line2(x1, y1, x2, y1, color);
    line2(x1, y2, x2, y2, color);
    line2(x1, y1, x1, y2, color);
    line2(x2, y1, x2, y2, color);
};

void fillrect1(int x1, int y1, int x2, int y2, int color)
{
    int i;

    if(y1 > y2) {           /* Put Y in order */
        i = y1; y1 = y2; y2 = i;
    }

    for(i = y1; i <= y2; i++) line2(x1, i, x2, i, color);
}

void putrow(int x1, int x2, int y, int color)
{
    asm mov ax,0A000h;          /* ES = video memory */
    asm mov es,ax;

    asm mov ax,y;               /* Get parameters */
    asm mov bx,x1;
    asm mov cx,x2;

    asm cmp ax,200;             /* Y out of range? */
    asm jae l4;
    asm cmp bx,cx;              /* Put X in order */
    asm jle l1;
    asm xchg bx,cx;

l1: asm cmp cx,0;
    asm jl l4;                  /* Not visible? */
    asm cmp bx,319;
    asm jg l4;

    asm cmp bx,0;               /* Clip to screen */
    asm jge l2;
    asm mov bx,0;
l2: asm cmp cx,319;
    asm jle l3;
    asm mov cx,319;

l3: asm mov di,bx;              /* DI = offset */
    asm shl ax,6;
    asm add di,ax;
    asm shl ax,2;
    asm add di,ax;

    asm sub cx,bx;              /* CX = length */
    asm inc cx;

    asm mov ax,color;           /* AL, AH = color */
    asm mov ah,al;                

    asm shr cx,1;               /* Store by words */
    asm rep stosw;
    asm adc cx,0;               /* Store possible odd byte */
    asm rep stosb;
l4:
}

void fillrect2(int x1, int y1, int x2, int y2, int color)
{
    int i;

    if(y1 > y2) {           /* Put Y in order */
        i = y1; y1 = y2; y2 = i;
    }

    for(i = y1; i <= y2; i++) putrow(x1, x2, i, color);
}

void circle1(int x, int y, int r, int color)
{
    int i;
    double ix, iy;
    if(r < 1) r = 1;
    ix = 0; iy = r;

    for(i = 0; i < (r * 6.283); i++) {
        putpixel(x + ix, y + iy, color);
        ix = ix + iy / r;
        iy = iy - ix / r;
    }
}

void circle2(int x, int y, int r, int color)
{
    int i, ix, iy;
    if(r < 1) r = 1;
    ix = 0; iy = r * 64;

    for(i = 0; i < ((r * 44) / 7); i++) {
        putpixel(x + ix / 64, y + iy / 64, color);
        ix = ix + iy / r;
        iy = iy - ix / r;
    }
}

void circle3(int x, int y, int r, int color)
{
    int ix, iy, a, b;

    if(r < 1) r = 1;
    ix = 0; iy = r * 64;

    do {
        a = (ix + 32) >> 6; b = (iy + 32) >> 6;
        putpixel(x + a, y + b, color); putpixel(x - a, y + b, color);
        putpixel(x + a, y - b, color); putpixel(x - a, y - b, color);
        putpixel(x + b, y + a, color); putpixel(x - b, y + a, color);
        putpixel(x + b, y - a, color); putpixel(x - b, y - a, color);
        ix = ix + iy / r;
        iy = iy - ix / r;
    } while(b > a);
}

void fillcircle1(int x, int y, int r, int color)
{
    int ix, iy, a, b;

    if(r < 1) r = 1;
    ix = 0; iy = r * 64;

    do {
        a = (ix + 32) >> 6; b = (iy + 32) >> 6;

        /* plot four rows using symmetry */

        putrow(x - a, x + a, y + b, color);
        putrow(x - a, x + a, y - b, color);
        putrow(x - b, x + b, y + a, color);
        putrow(x - b, x + b, y - a, color);

        /* step to next pixel */

        ix = ix + iy / r;
        iy = iy - ix / r;
    } while(b > a);
}

void fillcircle2(int cx, int cy, int r, int color)
{
    int x, y, a, b, na, nb;

    if(r < 1) r = 1;

    x = 0;                      /* calculate original pixel position */
    y = r * 64;
    na = (x + 32) >> 6;
    nb = (y + 32) >> 6;

    do {
        a = na; b = nb;         /* save old pixel position */

        x = x + y / r;          /* step to next pixel */
        y = y - x / r;
        na = (x + 32) >> 6;     /* calculate new pixel position */
        nb = (y + 32) >> 6;

        /* always plot long rows */

        putrow(cx - b, cx + b, cy + a, color);
        putrow(cx - b, cx + b, cy - a, color);

        /* plot short rows only if Y changed */

        if(b != nb) {
            putrow(cx - a, cx + a, cy + b, color);
            putrow(cx - a, cx + a, cy - b, color);
        }
    } while(b > a);
}

void ellipse(int x, int y, int rx, int ry, int color)
{
    int ix, iy, a, b, c, d;

    if(rx < 1) rx = 1; if(ry < 1) ry = 1;

    if(rx > ry) {
        ix = 0; iy = rx * 64;
        do {
            a = (ix + 32) >> 6; b = (iy + 32) >> 6;
            c = (a * ry) / rx; d = (b * ry) / rx;

            /* plot eight pixels using symmetry */
            putpixel(x + a, y + d, color); putpixel(x - a, y + d, color);
            putpixel(x + a, y - d, color); putpixel(x - a, y - d, color);
            putpixel(x + b, y + c, color); putpixel(x - b, y + c, color);
            putpixel(x + b, y - c, color); putpixel(x - b, y - c, color);
            ix = ix + iy / rx;      /* step to next pixel */
            iy = iy - ix / rx;
        } while(b > a);
    } else {
        ix = 0; iy = ry * 64;
        do {
            a = (ix + 32) >> 6; b = (iy + 32) >> 6;
            c = (a * rx) / ry; d = (b * rx) / ry;
    
            /* plot eight pixels using symmetry */
            putpixel(x + c, y + b, color); putpixel(x - c, y + b, color);
            putpixel(x + c, y - b, color); putpixel(x - c, y - b, color);
            putpixel(x + d, y + a, color); putpixel(x - d, y + a, color);
            putpixel(x + d, y - a, color); putpixel(x - d, y - a, color);
            ix = ix + iy / ry;      /* step to next pixel */
            iy = iy - ix / ry;
        } while(b > a);
    }
}

void fillellipse(int cx, int cy, int rx, int ry, int color)
{
    int x, y, a, b, c, d, na, nb, nc, nd;

    if(rx < 1) rx = 1; if(ry < 1) ry = 1;

    if(rx > ry) {
        x = 0;  y = rx * 64;         /* calculate original pixel position */
        na = 0; nb = (y + 32) >> 6;
        nc = 0; nd = (nb * ry) / rx;
    
        do {
            a = na; b = nb; c = nc; d = nd;    /* save old pixel position */
    
            x = x + (y / rx);  /* step to next pixel & get pixel position */
            y = y - (x / rx);
            na = (x + 32) >> 6;  nb = (y + 32) >> 6;
            nc = (na * ry) / rx; nd = (nb * ry) / rx;
                                                  
            putrow(cx-b, cx+b, cy+c, color);     /* always plot long rows */
            putrow(cx-b, cx+b, cy-c, color);
            if(d != nd) {            /* plot short rows only if Y changed */
                putrow(cx-a, cx+a, cy+d, color);
                putrow(cx-a, cx+a, cy-d, color);
            }
        } while(b > a);
    } else {
        x = 0;  y = ry * 64;         /* calculate original pixel position */
        na = 0; nb = (y + 32) >> 6;
        nc = 0; nd = (nb * rx) / ry;

        do {
            a = na; b = nb; c = nc; d = nd;    /* save old pixel position */
    
            x = x + (y / ry);  /* step to next pixel & get pixel position */
            y = y - (x / ry);
            na = (x + 32) >> 6;  nb = (y + 32) >> 6;
            nc = (na * rx) / ry; nd = (nb * rx) / ry;
    
            putrow(cx-d, cx+d, cy+a, color);     /* always plot long rows */
            putrow(cx-d, cx+d, cy-a, color);
            if(b != nb) {            /* plot short rows only if Y changed */
                putrow(cx-c, cx+c, cy+b, color);
                putrow(cx-c, cx+c, cy-b, color);
            }
        } while(b > a);
    }
}

void putimage(void *image, int x1, int y1)
{
    asm pusha;                  /* Save all registers */

    asm mov ax,0A000h;          /* ES = video memory */
    asm mov es,ax;
    
    asm mov dx,y1;              /* DI = offset */
    asm mov ax,x1;
    asm add ah,dl;
    asm shl dx,6;
    asm add ax,dx;
    asm mov di,ax;
    
    asm mov si,image;           /* DS:SI = image */
    asm lodsw;                  /* BX = width */
    asm xchg bx,ax;
    asm lodsw;                  /* DX = height */
    asm xchg dx,ax;
    
    asm mov ax,320;             /* AX = 320 - width */
    asm sub ax,bx;
pi_loop:
    asm mov cx,bx;              /* Copy this line */
    asm rep movsb;
    asm add di,ax;              /* Move to next line */
    asm dec dx;                 /* Loop back */
    asm jnz pi_loop;
    
    asm popa;                   /* Restore registers */
}

void getimage(void *image, int x1, int y1, int x2, int y2)
{
    asm push ds;                /* Save all registers */
    asm pusha;

    asm push ds;                /* ES = DS */
    asm pop es;
    asm mov ax,0A000h;          /* DS = video memory */
    asm mov ds,ax;
    
    asm mov dx,y1;              /* SI = offset */
    asm mov ax,x1;
    asm add ah,dl;
    asm shl dx,6;
    asm add ax,dx;
    asm mov si,ax;
    
    asm mov bx,x2;              /* BX = width */
    asm sub bx,x1;
    asm inc bx;
    asm mov dx,y2;              /* DX = height */
    asm sub dx,y1;
    asm inc dx;
    
    asm mov di,image;           /* ES:DI = image */
    asm mov ax,bx;              /* Store width & height */
    asm stosw;
    asm mov ax,dx;
    asm stosw;
    
    asm mov ax,320;             /* AX = 320 - width */
    asm sub ax,bx;
gi_loop:
    asm mov cx,bx;              /* Copy this line */
    asm rep movsb;
    asm add si,ax;              /* Move to next line */
    asm dec dx;                 /* Loop back */
    asm jnz gi_loop;
    
    asm popa;                   /* Restore registers */
    asm pop ds;
}

void outstr(int x, int y, char *s, int color)
{
    char far *font;
    int a, b, w, d, e;

    asm {
        push bp; mov ax,1130h;      /* Get BIOS font */
        mov bh,3; int 10h;
        mov dx,bp; pop bp;
        mov word ptr font[0], dx;   /* Store in variable */
        mov word ptr font[2], es;
    }

    while(*s) {
        for(a = 0; a < 8; a++) {
            d = font[(*s << 3) + a];
            e = 0x80;
            for(b = 0; b < 8; b++) {
                if(d & e) putpixel(x+b, y+a, color);
                e >>= 1;
            }
        }
        x += 8; s++;
    }
}
