// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: testprog.cpp
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer   
// File Creation Date: 03/25/2000
// Date Last Modified: 08/10/2000
// Copyright (c) 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or 
modify it under the terms of the GNU Lesser General Public 
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. 
  
This library is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of 
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
USA
 
Test program demonstrating how to change thread priorities.
*/
// ----------------------------------------------------------- //   
#include <iostream.h>
#include <stdlib.h>
#include "vbthread.h"
#include "vbcond.h"

// For safe condition variable usage, must use a boolean predicate and 
// a mutex with the condition.
int conditionMet = 0;
vbCondition cond;
vbMutex mutex;

void PrintThreadStatus(vbThread_t *thread)
{
  cout << "Thread OID: " << (int)thread->GetObjectID() << ", "
       << thread->ThreadPriorityMessage() << ", " 
       << thread->ThreadPriorityClassMessage() << '\n';
}

void HandleThreadError(vbThread_t *thread)
{
  cout << "Error setting thread " << thread->GetObjectID()
       << " priority" << endl;
  cout << thread->ThreadExceptionMessage() << endl;
  cout << "Setting priority back to normal" << endl;
  thread->SetThreadPriority(vbTHREAD_PRIORITY_NORMAL);
  thread->SetThreadPriorityClass(vbTHREAD_PRIORITY_CLASS_OTHER);
}

class SimpleThread : public vbThread
{
private: // Base class interface
  void *ThreadEntryRoutine(vbThread_t *thread);
};

void *SimpleThread::ThreadEntryRoutine(vbThread_t *thread)
{
  mutex.MutexLock();
  
  // Block the thread initially
  while (!conditionMet) cond.ConditionWait(&mutex);

  PrintThreadStatus(thread);
  
  mutex.MutexUnlock();

  return 0;
}

// Main thread of execution
int main()
{
  SimpleThread t;

  const int NUM_THREADS = 10;
  vbThread_t *thread[NUM_THREADS];

  int i, rv;
  
  // Create the threads in a blocked state
  for(i = 0; i < NUM_THREADS; i++) {
    thread[i] = t.CreateThread();
    thread[i]->SetObjectID(i); // Assign an arbitary object ID
    
    // Check for any errors created during thread creataion
    if(thread[i]->GetThreadError() != vbTHREAD_NO_ERROR)
      cout << thread[i]->ThreadExceptionMessage() << endl;
  }

  // Change some of the thread priorities and policies before executing them
  rv = t.SetThreadPriority(thread[NUM_THREADS-1], vbTHREAD_PRIORITY_HIGH);
  if(rv != 0) HandleThreadError(thread[NUM_THREADS-1]); 
 
  rv = t.SetThreadPriority(thread[NUM_THREADS-2], vbTHREAD_PRIORITY_HIGH, 
			   vbTHREAD_PRIORITY_CLASS_FIFO);
  if(rv != 0) HandleThreadError(thread[NUM_THREADS-2]); 

  rv = t.SetThreadPriority(thread[NUM_THREADS-3], vbTHREAD_PRIORITY_HIGH, 
			   vbTHREAD_PRIORITY_CLASS_RR);
  if(rv != 0) HandleThreadError(thread[NUM_THREADS-3]); 

  rv = t.SetThreadPriority(thread[0], vbTHREAD_PRIORITY_LOW);
  if(rv != 0) HandleThreadError(thread[0]); 
 
  rv = t.SetThreadPriority(thread[1], vbTHREAD_PRIORITY_LOW, 
			   vbTHREAD_PRIORITY_CLASS_FIFO);
  if(rv != 0) HandleThreadError(thread[1]); 

  rv = t.SetThreadPriority(thread[2], vbTHREAD_PRIORITY_LOW, 
			   vbTHREAD_PRIORITY_CLASS_RR);
  if(rv != 0) HandleThreadError(thread[2]); 

  mutex.MutexLock();
  
  // Set the flag and wake up all the waiters
  conditionMet = 1;
  cond.ConditionBroadcast();
  
  mutex.MutexUnlock();

  // Wait for all the threads to finish before exiting
  for(i = 0; i < NUM_THREADS; i++) t.JoinThread(thread[i]);

  // Cleanup all the vbThread_t pointers
  for(i = 0; i < NUM_THREADS; i++) {
    rv = t.DestroyThread(thread[i]);
    if(rv != 0) {
      cout << "Error destroying thread!" << endl;
    }
  }

  return 0; // Exit the process
}
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //


  

