// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: vbd64.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer  
// File Creation Date: 02/04/1997 
// Date Last Modified: 08/11/2000
// Copyright (c) 1997, 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The vbDatabase64 class defines low-level methods used to build
embedded database systems based on variable block technology.
*/
// ----------------------------------------------------------- // 
#include "vbd64.h"

// Inititalize all static variables and data members
static char vbDefaultFileName[vbMaxNameLength] = "closed64.vbd";

// Current VBD file manager version number.
FAU64 vbDatabase64::VBD64Version = vbDatabaseVersionNumber;

// Initialize the VBD file manager signature and revision letter.
__SBYTE__ vbDatabase64::VBD64Signature[10] = { 'V', 'B', 'D',
					       'B', 'A', 'S', 'E',
					       '6', '4',
					  vbDatabaseRevisionLetter };
// Current 32-bit synchronization word
vbUINT32 vbDatabase64::VBD64InternalCheckWord = vbCheckWord;

vbDatabase64::vbDatabase64()
// Creates a VBD file object.
{
  strcpy(file_name, vbDefaultFileName); // Set the initial file name
  fp = 0;

  // Reset the last reported error and the file status members
  vbd_error = vbDBASE_NO_ERROR;
  is_open = 0;
  is_ok = 0;
  ready_for_reading = 0;
  ready_for_writing = 0;
  
  // Always set the revision letter when the object is constructed
  // so that the size functions work correctly.
  rev_letter = vbDatabaseRevisionLetter;
}

vbDatabase64::~vbDatabase64()
{
  Close();
}

const char *vbDatabase64::DatabaseExceptionMessage()
// Returns a null terminated string that can
// be used to log or print a database exception.
{
  return vbDatabaseExceptionMessage(vbd_error);
}

vbDatabaseError vbDatabase64::Create(const char *fname, FAU64 static_size,
				     __SBYTE__ RevisionLetter)
// Creates a new file and truncate the file if it already exists.
// The "static_size" variable is used to reserve a specified number
// of bytes that will not be affected by the dynamic allocation
// routines. The "RevisionLetter" variable is used to select a
// specific revision letter when the file is created. The revision
// letter is used to enable or disable specific features that will
// determine the amount of overhead per block. Returns a non-zero
// value to indicate an error condition or zero if successful.
{
  // Close any open files 
  if(Close() != vbDBASE_NO_ERROR) return vbd_error;

  // Create and truncate existing files
  fp = vbdFPTR64Create(fname);

  if(fp == 0) {
    is_ok = 0;
    is_open = 0;
    ready_for_writing = 0;
    ready_for_reading = 0;
    vbd_error = vbDBASE_FILE_CREATION_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  else {
    is_open = 1;
    is_ok = 1;
    ready_for_writing = 1;
    ready_for_reading = 1;
    strcpy(file_name, fname);

    // Set the specified revision letter. NOTE This will default to the
    // current revision unless a lower version is specified.
    SetRevisionLetter(RevisionLetter);
    
    file_header.vbd_hs_fptr = static_size + (__LLWORD__)FileHeaderSize();
    last_operation = vbDBASE_READ;
    if(InitFileHdr() != vbDBASE_NO_ERROR) return vbd_error;
  }
  
  // Returns 0 if the file was successfully created and opened.
  return vbd_error = vbDBASE_NO_ERROR;
}

void vbDatabase64::SetRevisionLetter(__SBYTE__ RevisionLetter)
// Set the database revision letter to a specified value. If
// a valid revision is not specified the current revision
// letter will be used. The function is used to detect revision
// letter errors when new database files are created.
{
  // Check the revision letter to ensure a valid revision was specified
  switch(RevisionLetter) {
    case '\0': case ' ': // VBD revision zero
      rev_letter = '\0';
      break;
    case 'A': case 'a':  // Persistent checksum revision 
      rev_letter = 'A';
      break;
    case 'B': case 'b':  // Persistent file lock revision
      rev_letter = 'B';
      break;
    case 'C': case 'c':  // Persistent record lock revision
      rev_letter = 'C';
      break;
    default:
      rev_letter = vbDatabaseRevisionLetter;
      break;
  }
}

vbDatabaseError vbDatabase64::InitFileHdr()
// Initialize the file header with default values and write it
// to disk. Returns a non-zero value to indicate an error
// condition or zero if successful.
{
  file_header.vbd_fs_fptr = (FAU64)0;
  file_header.vbd_eof = file_header.vbd_hs_fptr;
  file_header.vbd_hb_fptr = (FAU64)0; 
  memcpy(file_header.vbd_sig, vbDatabase64::VBD64Signature, 9); 
  file_header.vbd_ver = vbDatabase64::VBD64Version;

  // Set the specified revision letter accroding to current
  // "rev_letter" value;
  file_header.vbd_sig[9] = rev_letter;
  
  // Write the file header and flush the disk buffers to
  // maintain file integrity during multiple file access.
  if(WriteFileHdr()!= vbDBASE_NO_ERROR) return vbd_error;

  // Initialize and write the file lock header for rev B files and higher
  vbFileLockHeader lh;
  InitFileLockHdr(lh);
  if(WriteFileLockHdr(lh) != vbDBASE_NO_ERROR) return vbd_error;
   
  if(file_header.vbd_hs_fptr > (FAU64)FileHeaderSize()) {
     __SBYTE__ zero_byte = 0;
     if(Write(&zero_byte, 1, file_header.vbd_hs_fptr-1) != vbDBASE_NO_ERROR)
       return vbd_error;
  }

  return vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::Open(const char *fname,
				   vbDatabaseAccessMode mode)
// Open an existing file. The "mode" variable determines if the file
// is opened for read only or read/write access. This function will
// check the revision letter when opening an existing file. Returns a
// non-zero value to indicate an error condition or zero if successful.
// NOTE: This version of the open functions will only accept:
// vbDBASE_READONLY and vbDBASE_READWRITE access modes.
{
  // Close any open files
  if(Close() != vbDBASE_NO_ERROR) return vbd_error;

  if(mode == vbDBASE_READONLY) { // Open with read only access
    ready_for_reading = 1;
    ready_for_writing = 0;
    fp = vbdFPTR64Open(fname, vbDBASE_READONLY);
  }
  else { // Open with read/write access
    ready_for_reading = 1;
    ready_for_writing = 1;
    fp = vbdFPTR64Open(fname, vbDBASE_READWRITE);
  }
    
  if(fp == 0) {
    ready_for_reading = 0;
    ready_for_writing = 0;
    vbd_error = vbDBASE_FILE_OPEN_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  else {
    is_open = 1;
    is_ok = 1;
    strcpy(file_name, fname);
    last_operation = vbDBASE_WRITE;
    
    if(ReadFileHdr()!= vbDBASE_NO_ERROR) return vbd_error;

    // Test file type, checking the revision letter
    if(memcmp(file_header.vbd_sig, vbDatabase64::VBD64Signature, 9)) { 
      vbd_error = vbDBASE_WRONG_FILE_TYPE;
#ifdef __CPP_EXCEPTIONS__
       throw vbCDatabaseException();
#else
       return vbd_error;
#endif
    }

    // Set the revision letter according to the file header
    char revision[8];
    memmove(revision, file_header.vbd_sig, 10);
    rev_letter = revision[7];
  }

  // Ensure that true end of file is stored in the file header. 
  vbINT64 filesize = FileSize(fname);
  if(filesize != -1) {
    if(file_header.vbd_eof < filesize) {
      file_header.vbd_eof = filesize;
      if(Flush() != vbDBASE_NO_ERROR) return vbd_error;
    }
  }
  
  return vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::Close()
// Close the open database file. Returns a non-zero value
// to indicate an error condition or zero if successful.
{
  if(IsOpen()) {

    // Write the header if this file was opened with write access
    if(ReadyForWriting()) WriteFileHdr();

    if(vbdFPTR64Close(fp) != 0) {
      vbd_error = vbDBASE_FILE_CLOSE_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
    strcpy(file_name, vbDefaultFileName); // Set the initial file name
  }

  is_open = 0;
  is_ok = 0;
  ready_for_reading = 0;
  ready_for_writing = 0;
  fp = 0; // Set the file pointer to zero after the file is closed

  return vbd_error = vbDBASE_NO_ERROR; 
}

vbDatabaseError vbDatabase64::Flush()
// Flush the file header and any open disk buffers. Returns
// a non-zero value to indicate an error condition or zero
// if successful.
{
  if(ReadyForWriting()) {
    if(WriteFileHdr() != vbDBASE_NO_ERROR) return vbd_error;
    if(vbdFPTR64Flush(fp) != 0) {
      vbd_error = vbDBASE_FILE_WRITE_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
    Seek((FAU64)0, vbDBASE_SEEK_CUR);
  }
  return vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::Seek(FAU64 offset, vbDatabaseSeekMode mode)
// Seek to the specified offset starting at the beginning (SEEK_SET),
// end (SEEK_END) or current offset (SEEK_CUR). Returns a non-zero
// value to indicate an error condition or zero if successful.
{
  if(IsOK()) {
    if(vbdFPTR64Seek(fp, offset, mode)  == (__LLWORD__)-1) {
      vbd_error = vbDBASE_FILE_SEEK_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
    last_operation = vbDBASE_SEEK;
  }
  else {
    vbd_error = vbDBASE_FILE_NOT_READY;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  return vbd_error = vbDBASE_NO_ERROR;
}

FAU64 vbDatabase64::SeekTo(FAU64 file_address)
// Seek to the specified address, optimizing the seek
// operation by moving the file position indicator based
// on the current stream position. Returns the current
// file position after performing the seek operation.
{
  // Get the current stream position
  vbStreamPos64 pos = FilePosition();

  if(file_address == vbCurrAddress64) { // Do not perform a seek operation
    return pos;
  }
  else if(file_address > pos) { // Seek forward to the specified address
    vbStreamPos64 offset = file_address - pos;
    Seek(offset, vbDBASE_SEEK_CUR);
  }
  else if(file_address < pos) { // Seek backward to the specified address
    Seek(file_address, vbDBASE_SEEK_BEG);
  }
  else { // Current file position equals the specified address
    // Find current the position
    Seek((FAU64)0, vbDBASE_SEEK_CUR);
  }
  
  return FilePosition(); // Return current file position after seeking
}

vbDatabaseError vbDatabase64::Read(void *buf, __ULWORD__ bytes,
				   FAU64 file_address)
// Read a specified number of bytes from the specified file offset
// into a memory buffer. Returns a non-zero value to indicate an error
// condition or zero if successful.
{
  if(IsOK()) {
    if(file_address == vbCurrAddress64) {
      if(last_operation == vbDBASE_WRITE) {
	if(Seek(0, vbDBASE_SEEK_CUR) != vbDBASE_NO_ERROR) return vbd_error;
      }
    }
    else {
      if(Seek(file_address, vbDBASE_SEEK_BEG) != vbDBASE_NO_ERROR)
	return vbd_error;
    }

    if(vbdFPTR64Read(fp, buf, bytes) != 0) {
      vbd_error = vbDBASE_FILE_READ_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
    last_operation = vbDBASE_READ;
  }
  else {
    vbd_error = vbDBASE_FILE_NOT_READY;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  return vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::Write(const void *buf, __ULWORD__ bytes, 
				    FAU64 file_address, int flush,
				    int bit_test)
// Write a specific number of bytes from a memory buffer to a
// specified file offset. If the "flush" variable is true, the file
// buffers will be flushed to disk with each write operation. If the
// bit_test variable if true, the CRC of the buffer will be compared
// to the CRC of the actual bytes written to disk. Returns a non-zero
// value to indicate an error condition or zero if successful.
{
  FAU64 buf_address;
  
  if(ReadyForWriting()) {
     if(file_address == vbCurrAddress64) {
       if(last_operation == vbDBASE_READ) {
	 if(Seek(0, vbDBASE_SEEK_CUR) != vbDBASE_NO_ERROR) return vbd_error;
       }
     }
     else {
       if(Seek(file_address, vbDBASE_SEEK_BEG) != vbDBASE_NO_ERROR)
	 return vbd_error;
     }
     
     buf_address = FilePosition(); 
     if(vbd_error != vbDBASE_NO_ERROR) return vbd_error;

     if(vbdFPTR64Write(fp, buf, bytes) != 0) {
       vbd_error = vbDBASE_FILE_WRITE_ERROR;
#ifdef __CPP_EXCEPTIONS__
       throw vbCDatabaseException();
#else
       return vbd_error;
#endif
     }
     last_operation = vbDBASE_WRITE;
  }
  else {
    vbd_error = vbDBASE_FILE_NOT_WRITEABLE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  // Allow application to flush disk buffers after each write
  // operation to ensure the file data stays in sync during multiple
  // file access.
  if(flush) {  
    if(vbdFPTR64Flush(fp) != 0) {
      vbd_error = vbDBASE_FILE_WRITE_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
    if(Seek(0, vbDBASE_SEEK_CUR) != vbDBASE_NO_ERROR) return vbd_error;
  }

  if(bit_test) {
    __ULWORD__ w_csum = calcCRC32((char *)buf, bytes);
    __ULWORD__ r_csum = CalcChecksum(bytes, buf_address);

    // Check for file errors
    if(vbd_error != vbDBASE_NO_ERROR) return vbd_error;

    if(w_csum ^ r_csum) {
      vbd_error = vbDBASE_CHECKSUM_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return vbd_error;
#endif
    }
  }
  return vbd_error = vbDBASE_NO_ERROR;
}

vbUINT32 vbDatabase64::WriteObjectChecksum(FAU64 object_address)
// Used to write a 32-bit checksum for the object at the
// end of a block. The address variable must be set to the
// file address of the block data, not the block header.
// This function assumes that the data has already been
// written to the block. Returns the 32-bit CRC checksum
// value for the object stored in the block or zero if
// this is the wrong revision or an error occurs.
{
  // Perform operation according to the revision letter
  if((rev_letter == '\0') || (rev_letter == ' ')) 
     return (vbUINT32)0; // Always return zero for any file below revision 'A'

  vbUINT32 CRC;
  __ULWORD__ bytes;
  vbBlockHeader64 vb;

  // Calculate the address of the block header
  FAU64 block_address = object_address - VBHeaderSize();
  
  if(IsOK()) {
    // Make sure that the this is a pre-alloacted block.
    if(Read(&vb, sizeof(vbBlockHeader64), block_address) != vbDBASE_NO_ERROR)
      return (vbUINT32)0;
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      vbd_error = vbDBASE_SYNC_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return (vbUINT32)0;
#endif
    }
    // Calculate a checksum based on the block data
    bytes = (vb.block_length - VBHeaderSize()) - sizeof(vbChecksum);
    CRC = CalcChecksum(bytes, object_address);

    // Check for file errors
    if(vbd_error != vbDBASE_NO_ERROR) return (vbUINT32)0;

    // offset address to point to the checksum field located
    // at the end of the block.
    object_address += bytes;

    // Write the CRC for the block header and the block data.
    if(Write(&CRC, sizeof(CRC), object_address) != vbDBASE_NO_ERROR)
      return (vbUINT32)0;
  }

  return CRC;
}

int vbDatabase64::ReadObjectChecksum(FAU64 object_address,
				     __ULWORD__ *object_crc,
				     __ULWORD__ *calc_crc)
// Tests the object's CRC value stored on disk against
// the actual CRC of the bytes stored on disk. The address
// variable must be set to the file address of the block
// data, not the block header. This function assumes that
// the data has already been written to the block. Returns
// true if the object's CRC test good or false if the CRC
// tests bad. Passes back the object's CRC stored on disk
// in the object_crc variable and the calculated CRC value
// in the calc_crc variable. Returns true if the checksum
// value is good or zero if the checksum value is bad or
// a file error occurs.
{
  // Perform operation according to the revision letter
  if((rev_letter == '\0') || (rev_letter == ' ')) {
    *object_crc = 0;
    *calc_crc = 0;
    return 1; // Always return true for any file below revision 'A'
  }
  
  vbUINT32 CRC, objectCRC;
  vbBlockHeader64 vb;
  __ULWORD__ bytes;

  // Calculate the address of the block header
  FAU64 block_address = object_address - VBHeaderSize();
  
  if(IsOK()) {
    // Make sure that the this is a pre-alloacted block.
    if(Read(&vb, sizeof(vbBlockHeader64), block_address) != vbDBASE_NO_ERROR)
      return 0;
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      vbd_error = vbDBASE_SYNC_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#else
      return 0;
#endif
    }

    // Calculate a checksum based on the block data
    bytes = (vb.block_length - VBHeaderSize()) - sizeof(vbChecksum);
    CRC = CalcChecksum(bytes, object_address);

    // Check for file errors
    if(vbd_error != vbDBASE_NO_ERROR) return 0;

    // offset address to point to the checksum field located
    // at the end of the block.
    object_address += bytes;

    // Read the CRC value stored on disk
    if(Read(&objectCRC, sizeof(objectCRC), object_address) != vbDBASE_NO_ERROR)
      return 0;
  }

  if(object_crc) *object_crc = objectCRC;
  if(calc_crc) *calc_crc = CRC;
     
  if(CRC ^ objectCRC) return 0; // Return false if CRC check fails

  return 1; // Return true if the CRC values match
}

__ULWORD__ vbDatabase64::CalcChecksum(__ULWORD__ bytes, FAU64 file_address, 
				 int mem_alloc)
// Calculate a 32-bit CRC checksum for a given number
// of bytes starting at the specified address. Returns a
// 32-bit CRC value. If the mem_alloc variable is true, a
// buffer equal to the specified number of bytes will be
// created in memory. If the mem_alloc variable is false
// or memory allocation fails, the CRC will be calculated
// byte by byte starting at the specified address. NOTE:
// the calling function must check for disk file errors.
// Returns the CRC value.
{
  __ULWORD__ CRC;
  __ULWORD__ len = bytes;
  unsigned char data;
  char *buf = 0;  

  // Create a buffer equal to the object length
  if(mem_alloc) buf = new char[bytes]; 

  if(buf) {
    if(IsOK()) {
      if(Read(buf, bytes, file_address) != vbDBASE_NO_ERROR) return 0;
      CRC = calcCRC32(buf, bytes);
      delete buf;
    }
  }
  else {
    if(IsOK()) {
      // Seek to the specified file address
      SeekTo(file_address);
      if(vbd_error != vbDBASE_NO_ERROR) return 0; 
      CRC = 0xffffffffL;
      while(len--) {
	if(Read(&data, sizeof(data)) != vbDBASE_NO_ERROR) return 0;
	CRC = calcCRC32(data, CRC);
      }
      CRC ^= 0xffffffffL;
    }
  }

  return CRC; 
}

vbDatabaseError vbDatabase64::ReadBlockHdr(vbBlockHeader64 &hdr,
					   FAU64 block_address)
// Reads a block header and tests the block's checkword to ensure
// that this is a valid block. Returns a non-zero value to indicate
// an error condition or zero if successful.
{
  if(Read(&hdr, sizeof(vbBlockHeader64), block_address) != vbDBASE_NO_ERROR)
    return vbd_error;
  if(hdr.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
    vbd_error = vbDBASE_SYNC_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  return vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::WriteBlockHdr(const vbBlockHeader64 &hdr,
                                          FAU64 block_address)
// Writes a block header to the specified block address. Returns
// a non-zero value to indicate an error condition or zero if
// successful.
{
  return Write(&hdr, sizeof(vbBlockHeader64), block_address);
}

vbDatabaseError vbDatabase64::ReadFileHdr()
// Read the database file header. Returns a non-zero value to
// indicate an error condition or zero if successful.
{
  return Read(&file_header, sizeof(vbFileHeader64), vbStartOfFile64);
}

vbDatabaseError vbDatabase64::WriteFileHdr()
// Writes the database file header. Returns a non-zero value to
// indicate an error condition or zero if successful.
{
  return Write(&file_header, sizeof(vbFileHeader64), vbStartOfFile64);
}

FAU64 vbDatabase64::Alloc(__ULWORD__ bytes, vbDatabaseReclaimMethod method)
// Allocates a contiguous number of bytes. The number of bytes allocated
// is adjusted accroding to the revision letter set when the file was
// created or opened. Only the block header and/or record lock header
// is written to the allocated space. Returns the file address of the newly
// allocated block or zero if an error occurs.
{
  FAU64 file_address = (FAU64)0;
  vbBlockHeader64 vb;
  vbRecordLockHeader rlh; // Revision C and higher
  
  if(ReadyForWriting()) {

    // Adjust the number of bytes to allocate space of VB header
    // according to the revision letter.
    if((rev_letter == '\0') || (rev_letter == ' ')) {
      bytes += VBHeaderSize();     // VBD Revision zero
    }
    else {
      bytes += VBHeaderSize();
      bytes += sizeof(vbChecksum); // Rev 'A' and higher
    }
    
    // Try to reclaim a deleted or removed block
    if(file_header.vbd_fs_fptr != (FAU64)0) {
      if(method == vbDBASE_RECLAIM_FIRSTFIT) {
	file_address = ReclaimFirstFit(bytes);
      }
      else if(method == vbDBASE_RECLAIM_BESTFIT) {
	file_address = ReclaimBestFit(bytes);
      }
      else {
	// Extend the file without reclaiming any deleted/removed blocks
        file_address = (FAU64)0;
      }
    }
    
    if(IsOK() && file_address == (FAU64)0) { 
      file_address = file_header.vbd_eof;
      file_header.vbd_eof += bytes;

      // Write to the last byte to avoid possible end of file errors
      __SBYTE__ zero = 0;
      if(Write(&zero, sizeof(__SBYTE__), file_header.vbd_eof-1) !=
	 vbDBASE_NO_ERROR)
        return (FAU64)0;
      file_header.vbd_hb_fptr = file_address;  
    }

    // Mark this newly allocated block as normal
    vb.block_status = vbNormalBlock;  
    vb.block_nd_fptr = 0;

    // Assign the current synchronization word value
    vb.block_check_word = vbDatabase64::VBD64InternalCheckWord; 

    vb.block_length = bytes; // Total number of bytes for this VB
    InitRecordLockHdr(rlh);  // Initialize the lock header

    // Write header for this VB
    if(WriteBlockHdr(vb, file_address) != vbDBASE_NO_ERROR) return (FAU64)0; 

    // Write a new record lock header (Revision C and higher)
    switch(rev_letter) {
      case 'c' : case 'C' :
	if(Write(&rlh, sizeof(vbRecordLockHeader)) != vbDBASE_NO_ERROR)
          return (FAU64)0;
      default:
	break;
    }
  }
  else {
    vbd_error = vbDBASE_FILE_NOT_WRITEABLE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return (FAU64)0;
#endif
  }
  if(file_address > file_header.vbd_hb_fptr)
    file_header.vbd_hb_fptr = file_address;  
  
  // Ensure that the VBD file header stays in sync
  // during multiple file access.
  WriteFileHdr();
  
  return SeekTo(file_address + VBHeaderSize());  
}

int vbDatabase64::Remove(FAU64 object_address)
// Marked the variable block removed indicating that the
// object cannot be undeleted. Returns true if the block
// was or removed or false if the block was not removed.
{
  return Delete(object_address, 1);
}

int vbDatabase64::Delete(FAU64 object_address, int remove)
// Marks the block at object address deleted and leaves the
// object unchanged, allowing it to be undeleted. The deleted
// block is placed on the front of the free space list. If 
// the "remove" variable is true the block is marked removed,
// indicating that the object cannot be undeleted. Returns
// true if the block was deleted/removed or false if the
// block was not deleted/removed.
{
  vbBlockHeader64 vb;
  
  FAU64 block_address = object_address - VBHeaderSize(); 
  if(ReadBlockHdr(vb, block_address) != vbDBASE_NO_ERROR) return 0;

  // Return false if VB is already deleted
  if((vb.block_status & 0xff) != vbNormalBlock) return 0; 
  
  // Mark the block removed and set the status 
  // accounting for all the status bytes
  __ULWORD__ block_status = vb.block_status;
  block_status &= 0xffffff00;
  if(remove)
    block_status = vbRemovedBlock;
  else
    block_status += vbDeletedBlock;
  vb.block_status = block_status;
  
  if(ReadyForWriting()) {
    vb.block_nd_fptr = file_header.vbd_fs_fptr; 
    if(WriteBlockHdr(vb, block_address) != vbDBASE_NO_ERROR) return 0;
    
    // Make sure the free space list is not corrupt
    if(file_header.vbd_fs_fptr != vbFSListCorrupt64) {
      file_header.vbd_fs_fptr = block_address;
      WriteFileHdr();
    }
    return 1; // Return true if successful
  }
  else {
    vbd_error = vbDBASE_FILE_NOT_WRITEABLE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else 
    return 0;
#endif
  }
  return 0; // Ensure that all paths return a value
}

vbStreamPos64 vbDatabase64::FilePosition()
// Returns the current file position or -1 to indicate an error 
// condition.
{
  if(!IsOK()) {
    vbd_error = vbDBASE_FILE_NOT_READY;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return (vbStreamPos64)-1;
#endif
  }

  vbStreamPos64 pos = vbdFPTR64Tell(fp);
  if(pos < 0) {
    vbd_error = vbDBASE_FILE_POSITION_ERROR;
#ifdef __CPP_EXCEPTIONS__
      throw vbCDatabaseException();
#endif
  }
  return pos;
}

const char *vbDatabase64::GetSignature() const
// Return the VBD file signature with no revision letter
{
  size_t len = sizeof(file_header.vbd_sig);
  char *s = new char[len];
  memcpy(s, file_header.vbd_sig, len);
  s[len-1] = 0; 
  return (const char *)s;
}

char *vbDatabase64::GetSignature() 
// Return the VBD file signature with no revision letter
{
  size_t len = sizeof(file_header.vbd_sig);
  char *s = new char[len];
  memcpy(s, file_header.vbd_sig, len);
  s[len-1] = 0;
  return s;
}

__ULWORD__ vbDatabase64::VBTotal()
// Returns the total number of valid blocks in the file.
{
  // Ensure that the VBD file header stays in sync
  // during multiple file access
  TestVBDHeader();

  vbBlockHeader64 vb;
  FAU64 block_address = FindFirstVB(0); // Search the entire file
  __ULWORD__ i = 0;

  while(block_address) { // Until a block of a valid size is found
    if(block_address >= file_header.vbd_eof) break;
    if(Read(&vb, sizeof(vbBlockHeader64)) != vbDBASE_NO_ERROR)
      break; // Return the number counted if an error occurs;

    if(!IsOK()) break;

    // If this is not a valid block, find the next one
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      block_address = FindFirstVB(block_address);
    }
    else {
      block_address += vb.block_length;
      SeekTo(block_address);
      i++;
    }
  }
  return i;
}

__ULWORD__ vbDatabase64::VBDeleted(__ULWORD__ *d, __ULWORD__ *r)
// Returns the total number of removed and deleted blocks.
{
  // Ensure that the VBD file header stays in sync
  // during multiple file access
  TestVBDHeader();

  vbBlockHeader64 vb;
  FAU64 block_address = file_header.vbd_fs_fptr; 
  __ULWORD__ i = 0;

  // Set the deleted and removed pointers to zero
  if(d) *d = 0; if(r) *r = 0;
  
  if(block_address == vbFSListCorrupt64) return vbFSListCorrupt64;
  SeekTo(block_address);

  while(block_address) { // Until a block of a valid size is found
    if(Read(&vb, sizeof(vbBlockHeader64)) != vbDBASE_NO_ERROR)
      break; // Return the number counted if an error occurs

    if(!IsOK()) break;

    // If this is not a valid block, the free space list is corrupt
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      file_header.vbd_fs_fptr = vbFSListCorrupt64;
      WriteFileHdr(); 
      return vbFSListCorrupt64;
    }

    // Added to prevent an infinite loop. 
    // If the block is not marked deleted or removed, the
    // Next deleted VBD pointer is bad. This will cause
    // an infinite loop if the end of the free space
    // list is pointing to valid block.
    switch((__SBYTE__)(vb.block_status & 0xff)) {
      case vbDeletedBlock :
	// Make sure the block is not pointing to itself
	if(block_address == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
	  return 0;
	}
	break;

      case vbRemovedBlock :
	// Make sure the block is not pointing to itself
	if(block_address == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
	  return 0;
	}
	break;

      default :
	file_header.vbd_fs_fptr = vbFSListCorrupt64;
	WriteFileHdr(); 
	return 0;
    }

    block_address = vb.block_nd_fptr;
    if(block_address == vbFSListCorrupt64) return vbFSListCorrupt64;
    SeekTo(block_address);

    if(d) {
      if((vb.block_status & 0xff) == vbDeletedBlock)
	*d = *d+1;
    }

    if(r) {
      if((vb.block_status & 0xff) == vbRemovedBlock)
	*r = *r+1;
    }

    i++;
  }
  return i;
}

__ULWORD__ vbDatabase64::ObjectLength(FAU64 object_address)
// Returns the object length in bytes at the specified object address.
{
  vbBlockHeader64 vb;
  FAU64 block_address;

  // Calculate the address of the block header
  if(object_address == vbCurrAddress64)
    block_address = FilePosition() - VBHeaderSize(); 
  else
    block_address = object_address - VBHeaderSize(); 
  
  if(ReadBlockHdr(vb, block_address) != vbDBASE_NO_ERROR)
    return 0;

  __ULWORD__ len;

  // Adjust the number of bytes to allocate space of VB header
  // according to the revision letter.
  if((rev_letter == '\0') || (rev_letter == ' ')) {
    len = vb.block_length - VBHeaderSize(); // VBD Revision zero
  }
  else { // Rev 'A' and higher
    len = vb.block_length - VBHeaderSize();
    len -= sizeof(vbChecksum);
  }

  return len;
}

__ULWORD__ vbDatabase64::VBLength(FAU64 object_address)
// Returns the total block length in bytes at the specified
// object address.
{
  vbBlockHeader64 vb;
  FAU64 block_address;
  
  // Calculate the address of the block header
  if(object_address == vbCurrAddress64)
    block_address = FilePosition() - VBHeaderSize(); 
  else
    block_address = object_address - VBHeaderSize(); 

  if(ReadBlockHdr(vb, block_address) != vbDBASE_NO_ERROR)
    return 0;

  return vb.block_length;
}

int vbDatabase64::UnDelete(FAU64 object_address)
// Undeletes a block if it has not been removed or reclaimed.
{
  vbBlockHeader64 vb, prev_VB; 
  FAU64 addr, block_address, prev_addr;
  addr = file_header.vbd_fs_fptr; 

  block_address = object_address - VBHeaderSize(); // Address of VB header
  if(Read(&vb, sizeof(vbBlockHeader64), block_address) != vbDBASE_NO_ERROR)
    return 0;
  
  // Return false if VB is not marked deleted 
  if((vb.block_status & 0xff) != vbDeletedBlock) return 0; 

  // Loop until the block is found in the free space list
  while(addr) { 
    if(Read(&vb, sizeof(vbBlockHeader64), addr) != vbDBASE_NO_ERROR)
      return 0;

    if(!IsOK()) break;
    
    // Signal not to use the free space list.
    // If this is not a valid block, the free space list is corrupt
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      file_header.vbd_fs_fptr = vbFSListCorrupt64;
      WriteFileHdr(); 
      return 0;
    }
    
    // If the block is not marked deleted or removed, the
    // Next deleted VBD pointer is bad. This will cause
    // an infinite loop if the end of the free space
    // list is pointing to valid block.
    switch((__SBYTE__)(vb.block_status & 0xff)) {
      case vbDeletedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
	  return 0;
	}
	break;

      case vbRemovedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
	  return 0;
	}
	break;

      default :
	file_header.vbd_fs_fptr = vbFSListCorrupt64;
	WriteFileHdr(); 
	return 0;
    }

    // Found the block in the free space list
    if(addr == block_address) { 
      if(prev_addr == (FAU64)0) { // Adjust the free space list
	// At the head of freespace list, so make a new head
	file_header.vbd_fs_fptr = vb.block_nd_fptr;
	if(WriteFileHdr() != vbDBASE_NO_ERROR) return 0; 
      }
      else {
	// In the middle of free space, so link prev to Next
	prev_VB.block_nd_fptr = vb.block_nd_fptr;
	if(WriteBlockHdr(prev_VB, prev_addr)!= vbDBASE_NO_ERROR) return 0;
      }

      // Undelete the specified block
      // Mark the block with a normal attribute
      __ULWORD__ block_status = vb.block_status;
      block_status &= 0xffffff00;
      block_status += vbNormalBlock;
      vb.block_status = block_status;
      
      vb.block_nd_fptr = 0;  // Always 0 unless VB is marked deleted

      // Write header for this VB
      if(WriteBlockHdr(vb, addr)!= vbDBASE_NO_ERROR) return 0;  

      return 1; // Return true if successful
    }
    
    // Keep looping through the free space list
    prev_addr = addr;
    prev_VB = vb;
    addr = vb.block_nd_fptr;
  } 

  return 0; // Return false if block was not undeleted
}

FAU64 vbDatabase64::GetVBDFreeSpace() 
{
  vbFileHeader64 fh;
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  // Ensure the in memory copy and the disk copy are the same
  if(fh.vbd_fs_fptr != file_header.vbd_fs_fptr) { 
    ReadFileHdr();
  }

  return file_header.vbd_fs_fptr;
}

FAU64 vbDatabase64::GetEOF() 
{
  vbFileHeader64 fh;
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  // Ensure the in memory copy and the disk copy are the same
  if(fh.vbd_eof != file_header.vbd_eof) { 
    ReadFileHdr();
  }

  return file_header.vbd_eof;
}

FAU64 vbDatabase64::GetHeapStart() 
{
  vbFileHeader64 fh;
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  // Ensure the in memory copy and the disk copy are the same
  if(fh.vbd_hs_fptr != file_header.vbd_hs_fptr) { 
    ReadFileHdr();
  }

  return file_header.vbd_hs_fptr;
}

FAU64 vbDatabase64::GetHighestVB()
{
  vbFileHeader64 fh;
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  // Ensure the in memory copy and the disk copy are the same
  if(fh.vbd_hb_fptr != file_header.vbd_hb_fptr) { 
    ReadFileHdr();
  }

  return file_header.vbd_hb_fptr;
}

FAU64 vbDatabase64::StaticArea()
{
  vbFileHeader64 fh;
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  // Ensure the in memory copy and the disk copy are the same
  if(fh.vbd_hs_fptr != file_header.vbd_hs_fptr) { 
    ReadFileHdr();
  }

  return (FAU64)(file_header.vbd_hs_fptr - FileHeaderSize());
}   

int vbDatabase64::TestVBDHeader()
// This function is used to ensure that the in memory copy
// of the VBD file header and the disk copy stay in sync
// during multiple file access.
{
  vbFileHeader64 fh;
  int errors = 0;
  
  Read(&fh, sizeof(vbFileHeader64), vbStartOfFile64);

  if(fh.vbd_fs_fptr != file_header.vbd_fs_fptr) { 
    ReadFileHdr();
    errors++;
  }

  if(fh.vbd_eof != file_header.vbd_eof) { 
    ReadFileHdr();
    errors++;
  }

  if(fh.vbd_hs_fptr != file_header.vbd_hs_fptr) { 
    ReadFileHdr();
    errors++;
  }

  if(fh.vbd_hb_fptr != file_header.vbd_hb_fptr) { 
    ReadFileHdr();
    errors++;
  }

  if(fh.vbd_hs_fptr != file_header.vbd_hs_fptr) { 
    ReadFileHdr();
    errors++;
  }

  return errors;
}

// ==============================================================
// General purpose file utilites (BEGIN HERE)
// ==============================================================
int vbDatabase64::Exists(const char *fname)
// Returns true if the file exists
{
  return vbdFPTR64Exists(fname);
}

FAU64 vbDatabase64::FileSize(const char *fname)
// Returns the file size. Use after file has been closed
// and re-opened to ensure that all the buffers are flushed
// to disk. Returns -1 to indicate an error condition.
{
  return (FAU64)vbdFPTR64FileSize(fname);
}
// ==============================================================
// General purpose file utilites (END HERE)
// ==============================================================

// ==============================================================
// Reclaim functions. (BEGIN HERE)
// ==============================================================
FAU64 vbDatabase64::ReclaimBestFit(__ULWORD__ bytes)
// Searches the free space list for a block that can be reused.
// This function will search the free space list for an "exact- 
// fit" first and then try to find the "best-fit" for the number
// of bytes requested. NOTE: The byte size is adjusted by the
// Alloc() function to allocate space for the block header plus
// the object. Returns address of the reclaimed space, or zero
// if a deleted or removed block of the appropriate size is not
// found. An exact-fit is a block that matches the exact number
// of bytes requested. If an exact-fit cannot be found, the next
// block big enough to hold number of bytes requested plus the
// a block header with overhead and least one byte left over becomes
// a best-fit block. The search continues until a best-fit
// block with the least number of unused bytes is found. The
// used bytes in the best-fit block are used to create a new
// block that will be put back on the free space list. This
// will keep the gaps between the blocks as small as possible,
// with the smallest gap being as large as a single block header
// plus any block overhead plus one byte.
{
  // Cannot reuse any blocks if the free space list is corrupt
  if(file_header.vbd_fs_fptr == vbFSListCorrupt64) return (FAU64)0;

  vbBlockHeader64 vb, prev_VB, new_VB;
  vbBlockHeader64 best_fit_prev_VB, best_fit_VB;
  FAU64 addr, prev_addr, new_addr;
  FAU64 best_fit_addr, best_fit_prev_addr;
  __ULWORD__ avail_len, unused_len, best_fit_unused_len = 0;
  
  // Constants for the best-fit criteria. NOTE: The maximum length
  // of a block to reuse equals: (max_limit * byte_multiple) * bytes 
  const unsigned max_limit = 10;    // Maximum number of byte multiples
  const double byte_multiple = .25; // Byte multiples  

  double best_byte_len, byte_percent = 0;
  double bytes_requested = (double)bytes;
  unsigned i;
  unsigned best_length[max_limit];
  
  // Calculate the best-fit byte values
  for(i = 0; i < max_limit; i++) {
    byte_percent += byte_multiple;
    best_byte_len = bytes_requested * byte_percent;
    best_length[i] = (unsigned)best_byte_len;
  }

  addr = file_header.vbd_fs_fptr;
  SeekTo(addr);
  prev_addr = best_fit_addr = (FAU64)0;
  
  // Search the entire free space list until an exact-fit 
  // or a best-fit block is found.
  while(addr) { 
    if(Read(&vb, sizeof(vbBlockHeader64)) != vbDBASE_NO_ERROR) 
      return (FAU64)0;
    if(!IsOK()) break;
    
    // Signal not to use the free space list.
    // If this is not a valid block, the free space list is corrupt
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      file_header.vbd_fs_fptr = vbFSListCorrupt64;
      WriteFileHdr(); 
      return (FAU64)0;
    }

    // If the block is not marked deleted or removed, the
    // Next deleted VBD pointer is bad. This will cause
    // an infinite loop if the end of the free space
    // list is pointing to valid block.
    switch((__SBYTE__)(vb.block_status & 0xff)) {
      case vbDeletedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
          return (FAU64)0;
	}
	break;

      case vbRemovedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
          return (FAU64)0;
	}
	break;

      default :
	file_header.vbd_fs_fptr = vbFSListCorrupt64;
	WriteFileHdr(); 
        return (FAU64)0;
    }

    avail_len = vb.block_length; // Length of object plus sizeof block header

    // Unused length must be big enough to hold a two block headers
    // plus the block overhead plus the object.
    __ULWORD__ total_len = bytes;
    if((rev_letter == '\0') || (rev_letter == ' ')) {
      total_len += VBHeaderSize(); // VBD Revision zero
    }
    else {
      total_len += VBHeaderSize();
      total_len += sizeof(vbChecksum); // Rev 'A' and higher
    }
    if(avail_len > total_len) {
      unused_len = avail_len - bytes;
    }
    else {
      unused_len = 0;
    }
    
    if(avail_len == bytes) {
      // Block is an exact fit
      if(prev_addr == (FAU64)0) {
	// At the head of freespace list
	file_header.vbd_fs_fptr = vb.block_nd_fptr;
        if(WriteFileHdr() != vbDBASE_NO_ERROR) return (FAU64)0; 
      }
      else {
	// In the middle of free space
	prev_VB.block_nd_fptr = vb.block_nd_fptr;
	if(WriteBlockHdr(prev_VB, prev_addr) != vbDBASE_NO_ERROR)
          return (FAU64)0;
      }
      return (__LLWORD__)IsOK() ? (__LLWORD__)addr : (__LLWORD__)0;
    }

    if(unused_len > 0) { // Found bigger block with room for header
      for(i = 0; i < max_limit; i++) {
	if(unused_len <= best_length[i]) { 
	  // Use the block matching the best-fit criteria
	  if(best_fit_unused_len > best_length[i]) { 
	    // Use the block if it is a better then the current one
	    best_fit_addr = addr;
	    best_fit_prev_addr = prev_addr;
	    best_fit_VB = vb;
	    best_fit_prev_VB = prev_VB;
	    best_fit_unused_len = unused_len;
	  }
	}
      }
    }
    
    // Block is not big enough
    prev_addr = addr;
    prev_VB = vb;
    addr = vb.block_nd_fptr;
    SeekTo(addr);

  } // End of block search

  // Could not find a best fit
  if(best_fit_addr == (FAU64)0) return (FAU64)0; 

  // Reuse the block and any remaining bytes
  new_addr = best_fit_addr + bytes;
  new_VB.block_check_word = vbDatabase64::VBD64InternalCheckWord;

  // Mark the block with a removed attribute 
  new_VB.block_status = vbRemovedBlock;
  
  new_VB.block_nd_fptr = best_fit_VB.block_nd_fptr;
  new_VB.block_length = best_fit_unused_len;
  if(WriteBlockHdr(new_VB, new_addr) != vbDBASE_NO_ERROR) return 0;

  // Adjust the free space list
  if(best_fit_prev_addr == (FAU64)0) {
    // At the head of freespace
    file_header.vbd_fs_fptr = new_addr;
    if(WriteFileHdr() != vbDBASE_NO_ERROR) return 0; 
  }
  else { // In the middle of freespace
    best_fit_prev_VB.block_nd_fptr = new_addr;
    if(WriteBlockHdr(best_fit_prev_VB, best_fit_prev_addr) != vbDBASE_NO_ERROR)
      return 0;
  }
  
  return (__LLWORD__)IsOK() ? (__LLWORD__)best_fit_addr : (__LLWORD__)0;
}

FAU64 vbDatabase64::ReclaimFirstFit(__ULWORD__ bytes)
// Searchs the free space list for the first block that can
// be reused. This function will search the free space list
// for a "first-fit" big enough to hold the number of bytes
// requested. NOTE: The byte size is adjusted by the Alloc()
// function to allocate space for the block header plus the object.
// Returns address of the reclaimed space, or zero if a deleted
// or removed block of the appropriate size is not found. If an
// "exact-fit" is found (a block that matches the exact number
// of bytes requested) the address of that block is returned.
// Otherwise the address of the first block big enough to hold
// number of bytes requested plus the size a block header with 
// overhead and at least one byte left over is returned. The used
// bytes in the first-fit block are used to create a new block that
// will be put back on the free space list. 
{
  // Cannot reuse any blocks if the free space list is corrupt
  if(file_header.vbd_fs_fptr == vbFSListCorrupt64) return (FAU64)0;

  vbBlockHeader64 vb, prev_VB, new_VB;
  FAU64 addr, prev_addr, new_addr;
  __ULWORD__ avail_len, unused_len;

  addr = file_header.vbd_fs_fptr; prev_addr = (FAU64)0;
  SeekTo(addr);

  // Search the free space list until a first-fit is found
  while(addr) { 
    if(Read(&vb, sizeof(vbBlockHeader64)) != vbDBASE_NO_ERROR)
      return (FAU64)0;

    if(!IsOK()) break;
    
    // Signal not to use the free space list.
    // If this is not a valid block, the free space list is corrupt
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord) {
      file_header.vbd_fs_fptr = vbFSListCorrupt64;
      WriteFileHdr(); 
      return (FAU64)0;
    }

    // If the block is not marked deleted or removed, the
    // Next deleted VBD pointer is bad. This will cause
    // an infinite loop if the end of the free space
    // list is pointing to valid block.
    switch((__SBYTE__)(vb.block_status & 0xff)) {
      case vbDeletedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
          return (FAU64)0;
	}
	break;

      case vbRemovedBlock :
	// Make sure the block is not pointing to itself 
	if(addr == vb.block_nd_fptr) { 
	  file_header.vbd_fs_fptr = vbFSListCorrupt64;
	  WriteFileHdr(); 
          return (FAU64)0;
	}
	break;

      default :
	file_header.vbd_fs_fptr = vbFSListCorrupt64;
	WriteFileHdr(); 
        return (FAU64)0;
    }

    avail_len = vb.block_length; // Length of object plus sizeof block header

    // Unused length must be big enough to hold a two block headers
    // plus the block overhead plus the object.
    __ULWORD__ total_len = bytes;
    if((rev_letter == '\0') || (rev_letter == ' ')) {
      total_len += VBHeaderSize(); // VBD Revision zero
    }
    else {
      total_len += VBHeaderSize();
      total_len += sizeof(vbChecksum); // Rev 'A' and higher
    }
    if(avail_len > total_len) {
      unused_len = avail_len - bytes;
    }
    else {
      unused_len = 0;
    }

    if(avail_len == bytes) {
      // Block is an exact fit
      if(prev_addr == (FAU64)0) {
	// At the head of freespace list
	file_header.vbd_fs_fptr = vb.block_nd_fptr;
        if(WriteFileHdr() != vbDBASE_NO_ERROR) return (FAU64)0; 
      }
      else {
	// In the middle of free space
	prev_VB.block_nd_fptr = vb.block_nd_fptr;
	if(WriteBlockHdr(prev_VB, prev_addr)!= vbDBASE_NO_ERROR) 
          return (FAU64)0;
      }
      break;
    }

    if(unused_len > 0) {
      // Reuse any remaining bytes.
      new_addr = addr + bytes;
      new_VB.block_check_word = vbDatabase64::VBD64InternalCheckWord;

      // Mark the block with a removed attribute 
      new_VB.block_status = vbRemovedBlock;

      new_VB.block_nd_fptr = vb.block_nd_fptr;
      new_VB.block_length = unused_len;
      if(WriteBlockHdr(new_VB, new_addr) != vbDBASE_NO_ERROR) return (FAU64)0;
      if(prev_addr == (FAU64)0) {
	// At the head of freespace
	file_header.vbd_fs_fptr = new_addr;
        if(WriteFileHdr() != vbDBASE_NO_ERROR) return (FAU64)0;
      }
      else { // In the middle of freespace
	prev_VB.block_nd_fptr = new_addr;
	if(WriteBlockHdr(prev_VB, prev_addr) != vbDBASE_NO_ERROR)
          return (FAU64)0;
      }
      break;
    }

    // Block is not big enough
    prev_addr = addr;
    prev_VB = vb;
    addr = vb.block_nd_fptr;
    SeekTo(addr);
  } // End of block search
  
  return (__LLWORD__)IsOK() ? (__LLWORD__)addr : (__LLWORD__)0;
}
// ==============================================================
// Reclaim functions. (END HERE)
// ==============================================================

// ==============================================================
// Header Size Functions. (BEGIN HERE)
// ==============================================================
size_t vbDatabase64::FileHeaderSize() const
{
  size_t fh = sizeof(vbFileHeader64);
  size_t flh = sizeof(vbFileLockHeader);
  
  // Perform operation according to the revision letter
  switch(rev_letter) {
    case 'b' : case 'B' :
      return fh + flh;
    case 'c' : case 'C' :
      return fh + flh;
    default:
      break;
  }
  return fh;
}

size_t vbDatabase64::FileHeaderSize()
{
  size_t fh = sizeof(vbFileHeader64);
  size_t flh = sizeof(vbFileLockHeader);
  
  // Perform operation according to the revision letter
  switch(rev_letter) {
    case 'b' : case 'B' :
      return fh + flh;
    case 'c' : case 'C' :
      return fh + flh;
    default:
      break;
  }
  return fh;
}

size_t vbDatabase64::VBHeaderSize() const 
{
  size_t bh = sizeof(vbBlockHeader64);
  size_t blh = sizeof(vbRecordLockHeader);
  
  // Perform operation according to the revision letter
  switch(rev_letter) {
    case 'c' : case 'C' :
      return bh + blh;
    default:
      break;
  }
  return bh;
}

size_t vbDatabase64::VBHeaderSize()
{
  size_t bh = sizeof(vbBlockHeader64);
  size_t blh = sizeof(vbRecordLockHeader);
  
  // Perform operation according to the revision letter
  switch(rev_letter) {
    case 'c' : case 'C' :
      return bh + blh;
    default:
      break;
  }
  return bh;
}
// ==============================================================
// Header Size Functions. (END HERE)
// ==============================================================

// ==============================================================
// VBD Linear Navigation Functions (BEGIN HERE)
// ==============================================================
FAU64 vbDatabase64::FindFirstVB(FAU64 offset)
// Search through the database until a valid block is found.
// The search starts at the heap start or the offset value.
// Returns 0 if no valid block is found in the file or the end
// of file is reached.
{
  vbBlockHeader64 vb;
  FAU64 file_address = (FAU64)0;

  // Ensure that the database header stays in sync
  // during multiple file access
  TestVBDHeader();

  // No VBs have been allocated yet
  if(file_header.vbd_hs_fptr == file_header.vbd_eof) return (FAU64)0;
     
  if(!offset)
    file_address = file_header.vbd_hs_fptr; // If no offset, start at heap
  else {
    file_address = file_address + offset; // offset the starting address 

    if(file_address >= file_header.vbd_eof) // Prevent offsetting past EOF
      return 0; // Invalid address
  }
  
  while(1) {
    if((FAU64)(file_address + VBHeaderSize()) >= file_header.vbd_eof ||
       !IsOK()) return (FAU64)0;

    if(Read(&vb, sizeof(vbBlockHeader64), file_address) != vbDBASE_NO_ERROR)
      return (FAU64)0;
    if(vb.block_check_word != vbDatabase64::VBD64InternalCheckWord)
      file_address++; // Loop through the file byte by byte
    else
      break; // Found valid block
  }
  return SeekTo(file_address);
}

FAU64 vbDatabase64::FindFirstObject(FAU64 offset)
// Search through the database until a valid block is found and
// then return the object's address. If the block is marked deleted
// continue searching until the first normal block is found. The
// search starts at the heap start or the offset. Returns 0 if
// no valid block is found in the file or the end of file is reached.
{
  vbBlockHeader64 vb;
  FAU64 file_address = FindFirstVB(offset);
  if(!file_address) return 0;

  while(1) { // Loop until a normal block status is found
    if(ReadBlockHdr(vb, file_address) != vbDBASE_NO_ERROR) return (FAU64)0;
    if((vb.block_status & 0xff) == vbNormalBlock) break;
    file_address = FindFirstVB(file_address+vb.block_length);
    if(!file_address) return (FAU64)0;
  }

  return file_address + VBHeaderSize();
}

FAU64 vbDatabase64::FindNextVB(FAU64 offset)
// Search through the database until the next valid block after the
// first valid block is found. The search starts at the heap start
// or the offset value. Returns 0 if no valid block is found or the
// end of file is reached.
{
  vbBlockHeader64 vb;

  // No VBs have been allocated yet
  if(file_header.vbd_hs_fptr == file_header.vbd_eof) return (FAU64)0;

  FAU64 file_address = FindFirstVB(offset);

  if(!file_address) return (FAU64)0; // No Vaild block found

  if(ReadBlockHdr(vb, file_address) != vbDBASE_NO_ERROR) return (FAU64)0;
  FAU64 NextVB = file_address + vb.block_length;

  // This is last the blocks
  if(NextVB >= file_header.vbd_eof) return file_address; 

  // Ensure block header is valid
  if(ReadBlockHdr(vb, NextVB) != vbDBASE_NO_ERROR) return (FAU64)0;

  return NextVB;
}

FAU64 vbDatabase64::FindNextObject(FAU64 offset)
// Search through the database until the next valid block after the
// first valid block is found and then return the object's address.
// If the block is marked deleted continue searching until the next
// normal block is found. The search starts at the heap start or the
// offset value. Returns 0 if no valid block is found or the end of
// file is reached.
{
  vbBlockHeader64 vb;
  FAU64 file_address = FindNextVB(offset);
  if(!file_address) return 0;
  
  while(1) { // Loop until a normal block status is found
    if(ReadBlockHdr(vb, file_address) != vbDBASE_NO_ERROR) return (FAU64)0;
    if((vb.block_status & 0xff) == vbNormalBlock) break;
    file_address = FindNextVB(file_address+vb.block_length);
    if(!file_address) return (FAU64)0;
  }

  return file_address + VBHeaderSize();
}

FAU64 vbDatabase64::FindPrevVB(FAU64 offset)
// Search backwards through the database until a valid block is found and
// then return the block's address. If the block is marked deleted
// continue searching until the first normal block is found. The
// search starts at the offset. Returns 0 if no valid block is found in
// the file or the start of file is reached.
{
  const __LLWORD__ DEF_BUF_SIZE = 1000 * sizeof(vbINT64);
  vbBlockHeader64 vb;
  FAU64 file_address = (FAU64)0;
  vbINT64 bufsize, rdsize;
  char* buf;

  // Ensure that the database header stays in sync
  // during multiple file access.
  TestVBDHeader();
 
  // No blocks have been allocated yet
  if(file_header.vbd_hs_fptr == file_header.vbd_eof) return (FAU64)0;
  
  if(!offset) return (FAU64)0;
  else {
    // offset the starting address 
    file_address = file_address + offset - VBHeaderSize(); 
    
    // Prevent offsetting past EOF
    if(file_address <= file_header.vbd_hs_fptr ||
       file_address > file_header.vbd_eof) 
      return 0; // Invalid address
  }
  bufsize = file_address - file_header.vbd_hs_fptr;
  if (bufsize == 0) {
    file_address = file_header.vbd_hs_fptr + VBHeaderSize();
    return file_address;
  }
  if (bufsize > DEF_BUF_SIZE) bufsize = DEF_BUF_SIZE;

  buf = new char[bufsize];

  while (1) {
    if (!IsOK()) return (FAU64)0;
    rdsize = bufsize;
    if (file_address - rdsize < file_header.vbd_hs_fptr) {
      rdsize = file_address - file_header.vbd_hs_fptr;
    }
    if (rdsize < sizeof(vbINT32)) return (FAU64)0;

    file_address -= rdsize;
    if(Read(buf, rdsize, file_address) != vbDBASE_NO_ERROR) return (FAU64)0;
    for(vbINT64 i = rdsize - sizeof(vbINT64); i >= 0; i--) {
      if((file_address+i == file_header.vbd_hs_fptr) ||
	 (*((vbUINT32*)(buf+(__LLWORD__)i)) == \
vbDatabase64::VBD64InternalCheckWord)) {
        Read(&vb, sizeof(vbBlockHeader64), file_address+i);
	if((vb.block_status & 0xff) == vbNormalBlock) {
	  return file_address+i;
	}
      }
    }
  }
}

FAU64 vbDatabase64::FindPrevObject(FAU64 offset)
// Search backwards through the database until a valid block is found
// and then return the object's address. If the block is marked deleted
// continue searching until the first normal block is found. The
// search starts at the offset. Returns 0 if no valid block is found in
// the file or the start of file is reached.
{
  vbBlockHeader64 vb;
  FAU64 file_address = FindPrevVB(offset);
  if(!file_address) return (FAU64)0;

  while(1) { // Loop until a normal block status is found
    if(ReadBlockHdr(vb, file_address) != vbDBASE_NO_ERROR) return 0;
    if((vb.block_status & 0xff) == vbNormalBlock) break;
    file_address = FindPrevVB(file_address+vb.block_length);
    if(!file_address) return (FAU64)0;
  }

  return file_address + VBHeaderSize();
}  
// ==============================================================
// VBD Linear Navigation Functions (END HERE)
// ==============================================================

// ==============================================================
// VBD File Lock Functions (BEGIN HERE)
// ==============================================================
void vbDatabase64::InitFileLockHdr(vbFileLockHeader &hdr)
// Initialize a the lock header for a newly 
// constructed file lock header.
{
  hdr.file_lock_protect = (vbUINT32)0;
  hdr.file_read_lock = (vbUINT32)0;
  hdr.file_write_lock = (vbUINT32)0;
}

vbDatabaseError vbDatabase64::WriteFileLockHdr(const vbFileLockHeader &hdr)
// Write the lock header to the file. Returns a non-zero value
// to indicate an error condition or zero if successful.
{
  switch(rev_letter) {
    case 'b' : case 'B' :
      break;
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  return Write(&hdr, sizeof(vbFileLockHeader), sizeof(vbFileHeader64));
}

vbDatabaseError vbDatabase64::ReadFileLockHdr(vbFileLockHeader &hdr)
// Read the lock header from the file. Returns a non-zero value to
// indicate an error condition or zero if successful.
{
  switch(rev_letter) {
    case 'b' : case 'B' :
      break;
    case 'c' : case 'C' :
      break;
    default:
      InitFileLockHdr(hdr);
      return vbd_error = vbDBASE_NO_ERROR;
  }

  return Read(&hdr, sizeof(vbFileLockHeader), sizeof(vbFileHeader64));
}

int vbDatabase64::LockFile(vbDatabaseLockType l_type) 
// Lock the file and set the lock access mode to shared or exclusive.
// Returns a non-zero value if the file cannot be locked or the lock 
// variable cannot be changed because it is exclusive or another thread 
// is currently updating it. 
{
  switch(rev_letter) {
    case 'b' : case 'B' :
      break;
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  // Read the record lock to ensure that this record has a lock header
  // and to keep the header in sync during multiple file access.
  vbFileLockHeader lock_header;
  if(ReadFileLockHdr(lock_header) != vbDBASE_NO_ERROR) return vbd_error;

  // Cannot modifiy this lock until the thread holding it releases the
  // lock header.
  if(lock_header.file_lock_protect != (vbUINT32)0) {
    vbd_error = vbDBASE_FILELOCK_ACCESS_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  
  switch(l_type) {
    case vbDBASE_WRITELOCK :
      // Cannot obtain an exclusive lock until all the threads in the
      // read lock queue have finished
      if((lock_header.file_read_lock != (vbUINT32)0) || 
	 (lock_header.file_write_lock != (vbUINT32)0)) {
	vbd_error = vbDBASE_FILELOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
	throw vbCDatabaseException();
#else
	return vbd_error;
#endif
      }
      else {
	lock_header.file_write_lock = 1;
      }

      lock_header.file_lock_protect = 1;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;

      // Skip over the read lock member
      SeekTo(sizeof(vbFileHeader64)+ (sizeof(vbUINT32) * 2));
      if(Write(&lock_header.file_write_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error;

      lock_header.file_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    case vbDBASE_READLOCK :
      // This lock is exclusively owned and cannot be read locked
      // until the thread holding this lock release the exclusive lock.
      if(lock_header.file_write_lock == 1) {
	vbd_error = vbDBASE_FILELOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }

      if(lock_header.file_read_lock < 0xFFFFFFFF) {
	lock_header.file_read_lock++;
      }
      else {
	vbd_error = vbDBASE_FILELOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }
      
      lock_header.file_lock_protect = 1;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;

      if(Write(&lock_header.file_read_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error; 

      lock_header.file_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    default: // Invalid lock type specified
      vbd_error =  vbDBASE_INVALID_LOCK_TYPE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  return vbd_error = vbd_error = vbDBASE_NO_ERROR;
}

int vbDatabase64::UnlockFile(vbDatabaseLockType l_type)
// Unlock the file. Returns a non-zero value if the file cannot be 
// unlocked or the lock variable cannot be changed because it is 
// exclusive or another thread is currently updating it. 
{
  switch(rev_letter) {
    case 'b' : case 'B' :
      break;
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }
  
  // Read the record lock to ensure that this record has a lock header
  // and to keep the header in sync during multiple file access.
  vbFileLockHeader lock_header;
  if(ReadFileLockHdr(lock_header) != vbDBASE_NO_ERROR) return vbd_error;

  // Cannot modifiy this lock until the thread holding it releases the
  // lock header.
  if(lock_header.file_lock_protect != (vbUINT32)0) {
    vbd_error = vbDBASE_FILELOCK_ACCESS_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  
  switch(l_type) {
    case vbDBASE_WRITELOCK :
      if(lock_header.file_write_lock == (vbUINT32)0)
	return vbDBASE_NO_ERROR;
      else
	lock_header.file_write_lock = (vbUINT32)0;
      
      lock_header.file_lock_protect = 1;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;

      // Skip over the read lock member
      SeekTo(sizeof(vbFileHeader64)+ (sizeof(vbUINT32) * 2));
      if(Write(&lock_header.file_write_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error;

      lock_header.file_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    case vbDBASE_READLOCK :
      if(lock_header.file_read_lock == (vbUINT32)0)// This record is not locked
	return vbd_error = vbDBASE_NO_ERROR;
      else if(lock_header.file_read_lock  > (vbUINT32)0) // Prevent rollover
      	lock_header.file_read_lock--;
      else
	lock_header.file_read_lock = (vbUINT32)0;
      
      lock_header.file_lock_protect = 1;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;

      if(Write(&lock_header.file_read_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error; 

      lock_header.file_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.file_lock_protect, sizeof(vbUINT32),
               sizeof(vbFileHeader64)) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    default: // Invalid lock type specified
      vbd_error =  vbDBASE_INVALID_LOCK_TYPE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  return vbd_error = vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::ResetFileLock()
// Reset the file's lock header. This function will clear
// all the file lock fields without testing the lock or the
// lock protect. Returns a non-zero value to indicate an error
// condition or zero if successful.
{
  switch(rev_letter) {
    case 'b' : case 'B' :
      break;
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }
  vbFileLockHeader lock_header;
  InitFileLockHdr(lock_header);
  return WriteFileLockHdr(lock_header);
}
// ==============================================================
// VBD File Lock Functions (END HERE)
// ==============================================================

// ==============================================================
// VBD Record Lock Functions (BEGIN HERE)
// ==============================================================
void vbDatabase64::InitRecordLockHdr(vbRecordLockHeader &hdr)
// Initialize a the lock header for a newly constructed 
// record lock header.
{
  hdr.record_lock_protect = (vbUINT32)0;
  hdr.record_read_lock = (vbUINT32)0;
  hdr.record_write_lock = (vbUINT32)0;
}

vbDatabaseError vbDatabase64::ReadRecordLockHdr(vbRecordLockHeader &hdr, 
						FAU64 block_address)
// Read the block's record lock header. Returns a non-zero value to
// indicate an error condition or zero if successful.
{
  switch(rev_letter) {
    case 'c' : case 'C' :
      break;
    default:
      InitRecordLockHdr(hdr);
      return vbd_error = vbDBASE_NO_ERROR;
  }

  // Ensure that this is a valid variable block
  vbBlockHeader64 block_header;
  if(ReadBlockHdr(block_header, block_address) != vbDBASE_NO_ERROR)
    return vbd_error;

  // Read the record lock directly following the block header 
  return Read(&hdr, sizeof(vbRecordLockHeader));
}

vbDatabaseError vbDatabase64::WriteRecordLockHdr(const vbRecordLockHeader &hdr,
						 FAU64 block_address)
// Write the block's record lock header. Returns a non-zero value to
// indicate an error condition or zero if successful.
{
  switch(rev_letter) {
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  // Ensure that this is a valid variable block
  vbBlockHeader64 block_header;
  if(ReadBlockHdr(block_header, block_address) != vbDBASE_NO_ERROR)
    return vbd_error;

  return Write(&hdr, sizeof(vbRecordLockHeader));
}

int vbDatabase64::LockRecord(vbDatabaseLockType l_type, FAU64 block_address)
// Lock block at the specified address. Returns a non-zero value
// if the record cannot be locked or the lock variable cannot be
// changed because it is exclusive or another thread is currently 
// updating it.
{
  switch(rev_letter) {
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  // Read the record lock to ensure that this record has a lock header
  // and to keep the header in sync during multiple file access.
  vbRecordLockHeader lock_header;
  if(ReadRecordLockHdr(lock_header, block_address) != vbDBASE_NO_ERROR) 
    return vbd_error;

  // Cannot modifiy this lock until the thread holding it releases the
  // lock header.
  if(lock_header.record_lock_protect != (vbUINT32)0) {
    vbd_error = vbDBASE_RECORDLOCK_ACCESS_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  
  if(block_address == vbCurrAddress64) block_address = FilePosition();
  FAU64 header_address = block_address+sizeof(vbBlockHeader64);

  switch(l_type) {
    case vbDBASE_WRITELOCK :
      // Cannot obtain an exclusive lock until all the threads in the
      // read lock queue have finished
      if(lock_header.record_read_lock != (vbUINT32)0) {
	vbd_error = vbDBASE_RECORDLOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }

      if(lock_header.record_write_lock == (vbUINT32)0) {
	lock_header.record_write_lock = 1;
      }
      else { 
	vbd_error = vbDBASE_RECORDLOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }
      
      lock_header.record_lock_protect = (vbUINT32)1;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;

      // Skip over the read lock member
      SeekTo(header_address + (sizeof(vbUINT32) * 2));
      if(Write(&lock_header.record_write_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error;

      lock_header.record_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    case vbDBASE_READLOCK :
      // This lock is exclusively owned and cannot be read locked
      // until the thread holding this lock release the exclusive lock.
      if(lock_header.record_write_lock == 1) {
	vbd_error = vbDBASE_RECORDLOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }

      if(lock_header.record_read_lock < 0xFFFFFFFF) {
	lock_header.record_read_lock++;
      }
      else {
	vbd_error = vbDBASE_RECORDLOCK_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
      }
      
      lock_header.record_lock_protect = 1;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;

      if(Write(&lock_header.record_read_lock, sizeof(vbUINT32)) != 
	 vbDBASE_NO_ERROR)
	return vbd_error; 

      lock_header.record_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    default: // Invalid lock type specified
      vbd_error =  vbDBASE_INVALID_LOCK_TYPE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  return vbd_error = vbd_error = vbDBASE_NO_ERROR;
}

int vbDatabase64::UnlockRecord(vbDatabaseLockType l_type, FAU64 block_address)
// Unlock the record and reset the lock protect value. Returns a non-zero
// value if the record cannot be unlocked or the lock variable cannot 
// be changed because it is exclusive or another thread is currently 
// updating it.
{
  switch(rev_letter) {
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  // Read the record lock to ensure that this record has a lock header
  // and to keep the header in sync during multiple file access.
  vbRecordLockHeader lock_header;
  if(ReadRecordLockHdr(lock_header, block_address) != vbDBASE_NO_ERROR) 
    return vbd_error;

  // Cannot modifiy this lock until the thread holding it releases the
  // lock header.
  if(lock_header.record_lock_protect != (vbUINT32)0) {
    vbd_error = vbDBASE_RECORDLOCK_ACCESS_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }
  
  if(block_address == vbCurrAddress64) block_address = FilePosition();
  FAU64 header_address = block_address+sizeof(vbBlockHeader64);

  switch(l_type) {
    case vbDBASE_WRITELOCK :
      if(lock_header.record_write_lock == (vbUINT32)0)
	return vbDBASE_NO_ERROR;
      else
	lock_header.record_write_lock = (vbUINT32)0;
      
      lock_header.record_lock_protect = 1;   
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;

      // Skip over the read lock member
      SeekTo(header_address + (sizeof(vbUINT32) * 2));
      if(Write(&lock_header.record_write_lock, sizeof(vbUINT32)) !=
	 vbDBASE_NO_ERROR)
	return vbd_error;

      lock_header.record_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    case vbDBASE_READLOCK :
      if(lock_header.record_read_lock == (vbUINT32)0) // Record is not locked
	return vbd_error = vbDBASE_NO_ERROR;
      else if(lock_header.record_read_lock  > (vbUINT32)0) // Prevent rollover
      	lock_header.record_read_lock--;
      else
	lock_header.record_read_lock = (vbUINT32)0;
      
      lock_header.record_lock_protect = 1;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;

      if(Write(&lock_header.record_read_lock, sizeof(vbUINT32)) != 
	 vbDBASE_NO_ERROR)
	return vbd_error; 

      lock_header.record_lock_protect = (vbUINT32)0;
      if(Write(&lock_header.record_lock_protect, sizeof(vbUINT32),
	       header_address) != vbDBASE_NO_ERROR)
	return vbd_error;
      break;

    default: // Invalid lock type specified
      vbd_error =  vbDBASE_INVALID_LOCK_TYPE;
#ifdef __CPP_EXCEPTIONS__
    throw vbCDatabaseException();
#else
    return vbd_error;
#endif
  }

  return vbd_error = vbd_error = vbDBASE_NO_ERROR;
}

vbDatabaseError vbDatabase64::ResetRecordLock(FAU64 block_address)
// Reset the block's record lock. This function will clear
// all the record lock fields without testing the lock or the
// lock protect. Returns a non-zero value to indicate an error
// condition or zero if successful.
{
  switch(rev_letter) {
    case 'c' : case 'C' :
      break;
    default:
      return vbd_error = vbDBASE_NO_ERROR;
  }

  vbRecordLockHeader lock_header;
  InitRecordLockHdr(lock_header);
  return WriteRecordLockHdr(lock_header, block_address);
}
// ==============================================================
// VBD Record Lock Functions (END HERE)
// ==============================================================
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
