// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- //
// C++ Header File Name: thelpers.h
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer   
// File Creation Date: 03/25/2000
// Date Last Modified: 08/11/2000
// Copyright (c) 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ---------- Include File Description and Details  ---------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The VB thread helpers are a collection of classes and standalone 
functions used to emulate uniform thread synchronization functions
on multiple target platforms. NOTE: None of the data structures and 
functions defined are intended to be used directly. They are used 
by the vbThread synchronization classes to call the correct native 
thread API functions for each supported platform.    
*/
// ----------------------------------------------------------- //   
#ifndef __VBTHREAD_HELPER_FUNCTIONS_HPP__
#define __VBTHREAD_HELPER_FUNCTIONS_HPP__

#include "thrtypes.h"

// --------------------------------------------------------------
// Type definitions
// --------------------------------------------------------------
#if defined (__WIN32__)
typedef LPSECURITY_ATTRIBUTES vbMutexAttribute; 
typedef LPSECURITY_ATTRIBUTES vbConditionAttribute;

#elif defined (__HPUX10__) && defined (__POSIX__)
// HPUX 10.20 Common Multithread Architecture (CMA) services
// POSIX 1003.4 interface
typedef cma_t_attr vbMutexAttribute;
typedef cma_t_attr vbConditionAttribute;

#elif defined (__POSIX__)
// POSIX 1003.1c 1995 thread standard
typedef pthread_mutexattr_t vbMutexAttribute;
typedef pthread_condattr_t vbConditionAttribute;

#else
#error You must define an API: __WIN32__ or __POSIX___
#endif
// --------------------------------------------------------------

// --------------------------------------------------------------
// Enumerations
// --------------------------------------------------------------
enum vbMutexError { // Mutex error codes
  vbMUTEX_NO_ERROR = 0, // No errors reported
  vbMUTEX_INVALID_CODE, // Invalid error code

  vbMUTEX_ATTR_DESTROY_ERROR, // Error destroying attribute
  vbMUTEX_ATTR_INIT_ERROR,    // Error initializing the attribute
  vbMUTEX_DESTROY_ERROR,      // Error destroying mutex
  vbMUTEX_INIT_ERROR,         // Error initializing the mutex
  vbMUTEX_LOCK_ERROR,         // Error locking the mutex
  vbMUTEX_SET_SHARE_ERROR,    // Error setting shared attribute
  vbMUTEX_TRY_LOCK_ERROR,     // Error trying to lock the mutex
  vbMUTEX_UNLOCK_ERROR        // Error unlocking the mutex
};

enum vbConditionError { // Condition variables error codes
  vbCONDITION_NO_ERROR = 0, // No errors reported
  vbCONDITION_INVALID_CODE, // Invalid error code
  
  vbCONDITION_ATTR_DESTROY_ERROR, // Error destroying attribute
  vbCONDITION_ATTR_INIT_ERROR,    // Error initializing the attribute
  vbCONDITION_BROADCAST_ERROR,    // Error broadcasting
  vbCONDITION_DESTROY_ERROR,      // Error destroying condition
  vbCONDITION_EXTERNAL_ERROR,     // External mutex error
  vbCONDITION_INIT_ERROR,         // Error initializing condition
  vbCONDITION_INTERNAL_ERROR,     // Internal condition error
  vbCONDITION_SET_SHARE_ERROR,    // Error setting shared attribute
  vbCONDITION_SIGNAL_ERROR,       // Error signaling
  vbCONDITION_TIMED_WAIT_ERROR,   // Error during a timed waiting
  vbCONDITION_WAIT_ERROR          // Error waiting
};

enum vbSemaphoreError { // Semaphore error codes
  vbSEMAPHORE_NO_ERROR = 0,   // No errors reported
  vbSEMAPHORE_INVALID_CODE,   // Invalid error code
  vbSEMAPHORE_DESTROY_ERROR,  // Error destroying semaphore
  vbSEMAPHORE_INIT_ERROR,     // Error initializing the semaphore
  vbSEMAPHORE_INTERNAL_ERROR, // Internal mutex error
  vbSEMAPHORE_POST_ERROR,     // Error posting
  vbSEMAPHORE_WAIT_ERROR      // Error waiting
};
// --------------------------------------------------------------

// --------------------------------------------------------------
// Thread synchronization helper classes
// --------------------------------------------------------------
// Data structure used to create shared and single process mutexs
struct vbMutex_t
{
public:
  vbMutex_t() {
    is_initialized = 0; process_type = vbPROCESS_PRIVATE;
    mutex_error = vbMUTEX_NO_ERROR;
  }
  ~vbMutex_t() { }

private: // Disallow copying and assignment
  vbMutex_t(const vbMutex_t &ob) { }
  void operator=(const vbMutex_t &ob) { } 

public:
  vbProcessType process_type;       // Shared or private access
  vbMutexError mutex_error;         // Last error reported
  vbMutexAttribute mutex_attribute; // Mutex attribute
  int is_initialized;               // True if initialized
  
#if defined (__WIN32__)
  // Mutex types that can be shared across multiple processes
  HANDLE shared_mutex; 

  // Mutex type that can only be shared by threads of the same process
  CRITICAL_SECTION mutex;

#elif defined (__POSIX__)
  pthread_mutex_t mutex;

#else
#error You must define an API: __WIN32__ or __POSIX___
#endif
};

// Data structure used to create shared and single process condition variables
struct vbCondition_t
{
public:
  vbCondition_t() {
    is_initialized = 0; process_type = vbPROCESS_PRIVATE;
    condition_error = vbCONDITION_NO_ERROR;
  }
  ~vbCondition_t() { }

private: // Disallow copying and assignment
  vbCondition_t(const vbCondition_t &ob) { }
  void operator=(const vbCondition_t &ob) { }

public:
  vbProcessType process_type;               // Shared or private access
  vbConditionError condition_error;         // Last error reported
  vbConditionAttribute condition_attribute; // Condition attribute
  int is_initialized;                       // True if initialized
  
#if defined (__WIN32__)
  // Members needed to emulate condition variables under WIN32
  int waiters_count;            // Number of threads waiting
  int release_count;            // Number of threads to release
  int wait_generation_count;    // Current generation count
  vbMutex_t waiters_count_lock; // Serialize access to generation count
  HANDLE event; // Event used to block and release waiting threads

#elif defined (__POSIX__)
  pthread_cond_t condition;

#else
#error You must define an API: __WIN32__ or __POSIX___
#endif
};

// Data structure used to create shared and single process semphores
struct vbSemaphore_t
{
public:
  vbSemaphore_t() {
    is_initialized = 0; process_type = vbPROCESS_PRIVATE;
    semaphore_error = vbSEMAPHORE_NO_ERROR;
  }
  ~vbSemaphore_t() { }

private: // Disallow copying and assignment
  vbSemaphore_t(const vbSemaphore_t &ob) { }
  void operator=(const vbSemaphore_t &ob) { }

public:
  vbProcessType process_type;       // Shared or private access
  vbSemaphoreError semaphore_error; // Last error reported
  int is_initialized;               // True if initialized
  int value;                        // Value during operation
  int value_after_operation;        // Value after an operation
  vbMutex_t mutex;                  // Lock for semaphore variables
  vbCondition_t semaphore;          // Condition used for wait operations
};
// --------------------------------------------------------------

// --------------------------------------------------------------
// Cross-platform standalone synchronization functions
// --------------------------------------------------------------
int vbThreadMutexInit(vbMutex_t *m, vbProcessType type = vbPROCESS_PRIVATE);
int vbThreadMutexDestroy(vbMutex_t *m);
int vbThreadMutexAttributeInit(vbMutexAttribute *attr);
int vbThreadMutexAttributeDestroy(vbMutexAttribute *attr);
int vbThreadMutexLock(vbMutex_t *m);
int vbThreadMutexUnlock(vbMutex_t *m);
int vbThreadMutexTryLock(vbMutex_t *m);
int vbThreadConditionInit(vbCondition_t *c,
			  vbProcessType type = vbPROCESS_PRIVATE);
int vbThreadConditionDestroy(vbCondition_t *c);
int vbThreadConditionAttributeInit(vbConditionAttribute *attr);
int vbThreadConditionAttributeDestroy(vbConditionAttribute *attr);
int vbThreadConditionSignal(vbCondition_t *c);
int vbThreadConditionBroadcast(vbCondition_t *c);
int vbThreadConditionWait(vbCondition_t *c, vbMutex_t *external_mutex);
int vbThreadConditionTimedWait(vbCondition_t *c, vbMutex_t *external_mutex,
			       unsigned long sec, unsigned long nsec);
int vbThreadSemaphoreInit(vbSemaphore_t *s,
			  vbProcessType type = vbPROCESS_PRIVATE);
int vbThreadSemaphoreDestroy(vbSemaphore_t *s);
int vbThreadSemaphoreValue(vbSemaphore_t *s);
int vbThreadSemaphoreWait(vbSemaphore_t *s);
int vbThreadSemaphorePost(vbSemaphore_t *s);
int vbThreadSemaphoreDecrement(vbSemaphore_t *s);
int vbThreadSemaphoreIncrement(vbSemaphore_t *s);
// --------------------------------------------------------------

// --------------------------------------------------------------
// Synchronization debug functions
// --------------------------------------------------------------
const char *vbThreadMutexExceptionMessage(vbMutex_t *m);
const char *vbThreadConditionExceptionMessage(vbCondition_t *c);
const char *vbThreadSemaphoreExceptionMessage(vbSemaphore_t *s);
// --------------------------------------------------------------

#endif // __VBTHREAD_HELPER_FUNCTIONS_HPP__
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

