/*************************************************************************
** interpcom-1.1  (command interpreter)                                  **
** complex.c : Complex numbers                                           **
**                                                                       **
** Copyright (C) 1998  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**									 **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details. 			 **
**									 **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free		 **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France								 **
**                                                                       **
 *************************************************************************/

#include "interp.h"



/*----------------------------------------------------------------------
    Addition de nombres complexes en simple precision
----------------------------------------------------------------------*/
fcomplex 
Cadd(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r + b.r;
    c.i = a.i + b.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Soustraction de nombres complexes en simple precision
----------------------------------------------------------------------*/
fcomplex 
Csub(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r - b.r;
    c.i = a.i - b.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne l'oppose d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Cnegat(fcomplex a)
{
    fcomplex        c;

    c.r = -a.r;
    c.i = -a.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de nombres complexes en simple precision
----------------------------------------------------------------------*/
fcomplex 
Cmul(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r * b.r - a.i * b.i;
    c.i = a.r * b.i + a.i * b.r;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Definition d'un nombre complexe en simple precision
    a partir de sa partie reelle et de sa partie imaginaire
----------------------------------------------------------------------*/
fcomplex 
Complex(float re, float im)
{
    fcomplex        c;

    c.r = re;
    c.i = im;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Definition d'un nombre complexe en notation polaire en simple 
    precision a partir de son module et de son argument
----------------------------------------------------------------------*/
fpolaire 
Polaire(float r, float t)
{
    fpolaire        c;

    c.rm = r;
    c.th = t;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne le conjugue d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Conjg(fcomplex z)
{
    fcomplex        c;

    c.r = z.r;
    c.i = -z.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division de deux nombres complexes en simple precision
----------------------------------------------------------------------*/
fcomplex 
Cdiv(fcomplex a, fcomplex b)
{
    fcomplex        c;
    float           r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
	r = b.i / b.r;
	x = b.r + r * b.i;
	c.r = (a.r + r * a.i) / x;
	c.i = (a.i - r * a.r) / x;
    } else {
	r = b.r / b.i;
	x = b.i + r * b.r;
	c.r = (a.r * r + a.i) / x;
	c.i = (a.i * r - a.r) / x;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Module d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
float 
Cabs(fcomplex z)
{
    float           x,
                    y,
                    module,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0)
	module = y;
    else if (y == 0.0)
	module = x;
    else if (x > y) {
	t = y / x;
	module = x * sqrt(1.0 + t * t);
    } else {
	t = x / y;
	module = y * sqrt(1.0 + t * t);
    }
    return module;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Racine carree d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Csqrt(fcomplex z)
{
    fcomplex        c;
    float           x,
                    y,
                    w,
                    r;

    if ((z.r == 0.0) && (z.i == 0.0)) {
	c.r = 0.0;
	c.i = 0.0;
	return c;
    } else {
	x = fabs(z.r);
	y = fabs(z.i);
	if (x >= y) {
	    r = y / x;
	    w = sqrt(x) * sqrt(0.5 * (1.0 + sqrt(1.0 + r * r)));
	} else {
	    r = x / y;
	    w = sqrt(y) * sqrt(0.5 * (r + sqrt(1.0 + r * r)));
	}
	if (z.r >= 0.0) {
	    c.r = w;
	    c.i = z.i / (2.0 * w);
	} else {
	    c.i = (z.i >= 0.0) ? w : -w;
	    c.r = z.i / (2.0 * c.i);
	}
	return c;
    }
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication d'un nombre reel et d'un nombre complexe 
    en simple precision
----------------------------------------------------------------------*/
fcomplex 
RCmul(float x, fcomplex a)
{
    fcomplex        c;

    c.r = x * a.r;
    c.i = x * a.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division d'un nombre complexe par un nombre reel
    en simple precision
----------------------------------------------------------------------*/
fcomplex 
RCdiv(fcomplex a, float x)
{
    fcomplex        c;

    c.r = a.r / x;
    c.i = a.i / x;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne l'inverse d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Cinv(fcomplex b)
{
    fcomplex        c;
    float           r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
	r = b.i / b.r;
	x = b.r + r * b.i;
	c.r = 1.0 / x;
	c.i = -r / x;
    } else {
	r = b.r / b.i;
	x = b.i + r * b.r;
	c.r = r / x;
	c.i = -1.0 / x;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Addition de deux nombres complexes en double precision
----------------------------------------------------------------------*/
dcomplex 
dCadd(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r + b.r;
    c.i = a.i + b.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Soustraction de nombres complexes en double precision
----------------------------------------------------------------------*/
dcomplex 
dCsub(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r - b.r;
    c.i = a.i - b.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne l'oppose d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
dCnegat(dcomplex a)
{
    dcomplex        c;

    c.r = -a.r;
    c.i = -a.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de nombres complexes en double precision
----------------------------------------------------------------------*/
dcomplex 
dCmul(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r * b.r - a.i * b.i;
    c.i = a.r * b.i + a.i * b.r;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Definition d'un nombre complexe en double precision
    a partir de sa partie reelle et de sa partie imaginaire
----------------------------------------------------------------------*/
dcomplex 
dComplex(double re, double im)
{
    dcomplex        c;

    c.r = re;
    c.i = im;
    return c;
}
/*--------------------------------------------------------------------*/





/*----------------------------------------------------------------------
    Definition d'un nombre complexe en notation polaire en double 
    precision a partir de son module et de son argument
----------------------------------------------------------------------*/
dpolaire 
dPolaire(double r, double t)
{
    dpolaire        c;

    c.rm = r;
    c.th = t;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne le conjugue d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
dConjg(dcomplex z)
{
    dcomplex        c;
    c.r = z.r;
    c.i = -z.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division de deux nombres complexes en double precision
----------------------------------------------------------------------*/
dcomplex 
dCdiv(dcomplex a, dcomplex b)
{
    dcomplex        c;
    double          r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
	r = b.i / b.r;
	x = b.r + r * b.i;
	c.r = (a.r + r * a.i) / x;
	c.i = (a.i - r * a.r) / x;
    } else {
	r = b.r / b.i;
	x = b.i + r * b.r;
	c.r = (a.r * r + a.i) / x;
	c.i = (a.i * r - a.r) / x;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Module d'un nombre complexe en double precision
----------------------------------------------------------------------*/
double 
dCabs(dcomplex z)
{
    double          x,
                    y,
                    module,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0)
	module = y;
    else if (y == 0.0)
	module = x;
    else if (x > y) {
	t = y / x;
	module = x * sqrt(1.0 + t * t);
    } else {
	t = x / y;
	module = y * sqrt(1.0 + t * t);
    }
    return module;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Racine carree d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
xdCsqrt(dcomplex z)
{
    dcomplex        c;
    double          x,
                    y,
                    w,
                    r;

    if ((z.r == 0.0) && (z.i == 0.0)) {
	c.r = 0.0;
	c.i = 0.0;
	return c;
    } else {
	x = fabs(z.r);
	y = fabs(z.i);
	if (x >= y) {
	    r = y / x;
	    w = sqrt(x) * sqrt(0.5 * (1.0 + sqrt(1.0 + r * r)));
	} else {
	    r = x / y;
	    w = sqrt(y) * sqrt(0.5 * (r + sqrt(1.0 + r * r)));
	}
	if (z.r >= 0.0) {
	    c.r = w;
	    c.i = z.i / (2.0 * w);
	} else {
	    c.i = (z.i >= 0.0) ? w : -w;
	    c.r = z.i / (2.0 * c.i);
	}
	return c;
    }
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication d'un nombre reel et d'un nombre complexe en double 
    precision
----------------------------------------------------------------------*/
dcomplex 
dRCmul(double x, dcomplex a)
{
    dcomplex        c;

    c.r = x * a.r;
    c.i = x * a.i;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Retourne l'inverse d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
xdCinv(dcomplex b)
{
    dcomplex        c;
    double          r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
	r = b.i / b.r;
	x = b.r + r * b.i;
	c.r = 1.0 / x;
	c.i = -r / x;
    } else {
	r = b.r / b.i;
	x = b.i + r * b.r;
	c.r = r / x;
	c.i = -1.0 / x;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division d'un nombre complexe par un nombre reel en double precision
----------------------------------------------------------------------*/
dcomplex 
dRCdiv(dcomplex a, double x)
{
    dcomplex        c;

    c.r = a.r / x;
    c.i = a.i / x;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conversion d'un nombre complexe de la forme (r, theta)
    a la forme (reel,imag) en simple precision
----------------------------------------------------------------------*/
fcomplex 
Conv_ptc(fpolaire a)
{
    fcomplex        z;

    z.r = a.rm * cos(a.th);
    z.i = a.rm * sin(a.th);
    return z;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conversion d'un nombre complexe de la forme (reel,imag)
    a la forme (r, theta) en simple precision
----------------------------------------------------------------------*/
fpolaire 
Conv_ctp(fcomplex z)
{
    fpolaire        a;
    float           x,
                    y,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0 && y == 0.0) {
	a.rm = 0.;
        a.th = 0.;
	return a;
    } 
    if (x < y) {
	t = x / y;
	a.rm = y * sqrt(1.0 + t * t);
    } else {
	t = y / x;
	a.rm = x * sqrt(1.0 + t * t);
    }

    x = z.r / a.rm;
    y = z.i / a.rm;
    if (x == 1.0) {
	a.th = 0.0;
	return a;
    }
    if (x == -1.0) {
	a.th = 3.141593;
	return a;
    }
    a.th = acos(x);
    if (y < 0.0)
        a.th = 6.2831853 - a.th;
    return a;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conversion d'un nombre complexe de la forme (r, theta)
    a la forme (reel,imag) en double precision
----------------------------------------------------------------------*/
dcomplex 
dConv_ptc(dpolaire a)
{
    dcomplex        z;

    z.r = a.rm * cos(a.th);
    z.i = a.rm * sin(a.th);
    return z;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conversion d'un nombre complexe de la forme (reel,imag)
    a la forme (r, theta) en double precision
----------------------------------------------------------------------*/
dpolaire 
dConv_ctp(dcomplex z)
{
    dpolaire        a;
    double          x,
                    y,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0 && y == 0.0) {
	a.rm = 0.;
        a.th = 0.;
	return a;
    } 
    if (fabs(x) < fabs(y)) {
	t = x / y;
	a.rm = y * sqrt(1.0 + t * t);
    } else {
	t = y / x;
	a.rm = x * sqrt(1.0 + t * t);
    }

    x = z.r / a.rm;
    y = z.i / a.rm;
    if (x == 1.0) {
	a.th = 0.0;
	return a;
    }
    if (x == -1.0) {
	a.th = 3.141592653589793;
	return a;
    }
    a.th = acos(x);
    if (y < 0.0)
        a.th = 6.2831853071795864 - a.th;
    return a;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (r, theta) 
    en simple precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
fcomplex 
Clog_p(fpolaire a)
{
    fcomplex        z;

    z.r = log(a.rm);
    z.i = a.th;
    return z;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (r, theta) 
    en simple precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
fpolaire
Plog_p(fpolaire a)
{
    return Conv_ctp(Clog_p(a));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (reel, imag) 
    en simple precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
fcomplex
Clog_c(fcomplex z)
{
    return Clog_p(Conv_ctp(z));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Sinus d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Csin_c(fcomplex z)
{
    fcomplex        u;

    u.r = sin(z.r) * cosh(z.i);
    u.i = cos(z.r) * sinh(z.i);
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Cosinus d'un nombre complexe en simple precision
----------------------------------------------------------------------*/
fcomplex 
Ccos_c(fcomplex z)
{
    fcomplex        u;

    u.r = cos(z.r) * cosh(z.i);
    u.i = -sin(z.r) * sinh(z.i);
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (reel, imag) 
    en simple precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
fpolaire 
Pexp_c(fcomplex z)
{
    fpolaire        u;
    float           x;

    u.rm = exp(z.r);
    x = floor(z.i / 6.283185);
    u.th = z.i - 6.283185 * x;
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (reel, imag) 
    en simple precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
fcomplex 
Cexp_c(fcomplex z)
{
    return Conv_ptc(Pexp_c(z));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (r, theta) 
    en simple precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
fpolaire
Pexp_p(fpolaire a) 
{
    return Pexp_c(Conv_ptc(a));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (r, theta) 
    en double precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
dcomplex 
dClog_p(dpolaire a)
{
    dcomplex        z;

    z.r = log(a.rm);
    z.i = a.th;
    return z;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (r, theta) 
    en double precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
dpolaire
dPlog_p(dpolaire a)
{
    return dConv_ctp(dClog_p(a));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Logarithme d'un nombre complexe sous la forme (reel, imag) 
    en double precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
dcomplex
dClog_c(dcomplex z)
{
    return dClog_p(dConv_ctp(z));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Sinus d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
dCsin_c(dcomplex z)
{
    dcomplex        u;

    u.r = sin(z.r) * cosh(z.i);
    u.i = cos(z.r) * sinh(z.i);
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Cosinus d'un nombre complexe en double precision
----------------------------------------------------------------------*/
dcomplex 
dCcos_c(dcomplex z)
{
    dcomplex        u;

    u.r = cos(z.r) * cosh(z.i);
    u.i = -sin(z.r) * sinh(z.i);
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (reel, imag) 
    en double precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
dpolaire 
dPexp_c(dcomplex z)
{
    dpolaire        u;
    double          x;

    u.rm = exp(z.r);
    x = floor(z.i / 6.283185307179586);
    u.th = z.i - 6.283185307179586 * x;
    return u;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (reel, imag) 
    en double precision. Le resultat est sous la forme (reel, imag)
----------------------------------------------------------------------*/
dcomplex 
dCexp_c(dcomplex z)
{
    return dConv_ptc(dPexp_c(z));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Exponentielle d'un nombre complexe sous la forme (r, theta) 
    en double precision. Le resultat est sous la forme (r, theta)
----------------------------------------------------------------------*/
dpolaire
dPexp_p(dpolaire a) 
{
    return dPexp_c(dConv_ptc(a));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Addition de deux nombres complexes sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Padd(fpolaire a, fpolaire b)
{
    return Conv_ctp(Cadd(Conv_ptc(a), Conv_ptc(b)));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Soustraction de deux nombres complexes sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Psub(fpolaire a, fpolaire b)
{
    return Conv_ctp(Csub(Conv_ptc(a), Conv_ptc(b)));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de deux nombres complexes sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Pmul(fpolaire a, fpolaire b)
{
    fpolaire 	    c;

    c.rm = a.rm * b.rm;
    c.th = a.th + b.th;
    if (c.th > 6.283185)
	c.th -= 6.283185;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de deux nombres complexes sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Pdiv(fpolaire a, fpolaire b)
{
    fpolaire 	    c;

    c.rm = a.rm / b.rm;
    c.th = a.th - b.th;
    if (c.th < 0.)
	c.th += 6.283185;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conjugaison d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Pconjg(fpolaire a)
{
    fpolaire	    c;
   
    c.rm = a.rm;
    c.th = -a.th;
    if (c.th < 0.)
	c.th += 6.283185;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Racine carree d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Psqrt(fpolaire a)
{
    fpolaire	    c;

    if (a.rm <= 0.0)
	c.rm = 0.0;
    else
	c.rm = sqrt(a.rm);
    c.th = a.th / 2.;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Puissance reelle d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Ppuis(fpolaire a, float x)
{
    fpolaire	    c;

    if (a.rm <= 0.0)
	c.rm = 0.0;
    else
	c.rm = pow(a.rm, x);
    c.th = a.th * x;
    c.th = c.th - floor(c.th / 6.283185) * 6.283185;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Puissance reelle d'un nombre complexe sous la forme (reel,imag)
    (simple precision)
----------------------------------------------------------------------*/
fcomplex
Cpuis(fcomplex a, float x)
{
    return Conv_ptc(Ppuis(Conv_ctp(a), x));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication d'un nombre reel et d'un nombre complexe sous la 
    forme (r, theta) (simple precision)
----------------------------------------------------------------------*/
fpolaire
RPmul(float x, fpolaire a)
{
    fpolaire	    c;

    c.rm = fabs(x * a.rm);
    if (x >= 0)
	c.th = a.th;
    else {
	c.th = a.th + 3.141593;
        if (c.th > 6.283185)
	    c.th -= 6.283185;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division d'un nombre reel par un nombre complexe sous la forme
    (r, theta) (simple precision)
----------------------------------------------------------------------*/
fpolaire
RPdiv(fpolaire a, float x)
{
    fpolaire	    c;

    c.rm = fabs(a.rm / x);
    if (x >= 0)
	c.th = a.th;
    else {
	c.th = a.th + 3.141593;
        if (c.th > 6.283185)
	    c.th -= 6.283185;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Inversion d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
fpolaire
Pinv(fpolaire a)
{
    fpolaire	    c;

    c.rm = 1. / a.rm;
    c.th = - a.th;
    if (c.th < 0.0)
	c.th += 6.283185;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Addition de deux nombres complexes sous la forme (r, theta)
    (double precision)
----------------------------------------------------------------------*/
dpolaire
dPadd(dpolaire a, dpolaire b)
{
    return dConv_ctp(dCadd(dConv_ptc(a), dConv_ptc(b)));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Soustraction de deux nombres complexes sous la forme (r, theta)
    (double precision)
----------------------------------------------------------------------*/
dpolaire
dPsub(dpolaire a, dpolaire b)
{
    return dConv_ctp(dCsub(dConv_ptc(a), dConv_ptc(b)));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de deux nombres complexes sous la forme (r, theta)    
    (double precision)
----------------------------------------------------------------------*/
dpolaire
dPmul(dpolaire a, dpolaire b)
{
    dpolaire 	    c;

    c.rm = a.rm * b.rm;
    c.th = a.th + b.th;
    if (c.th > 6.283185307179586)
	c.th -= 6.283185307179586;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication de deux nombres complexes sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
dpolaire
dPdiv(dpolaire a, dpolaire b)
{
    dpolaire 	    c;

    c.rm = a.rm / b.rm;
    c.th = a.th - b.th;
    if (c.th < 0.)
	c.th += 6.283185307179586;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Conjugaison d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
dpolaire
dPconjg(dpolaire a)
{
    dpolaire	    c;
   
    c.rm = a.rm;
    c.th = -a.th;
    if (c.th < 0.)
	c.th += 6.283185307179586;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Racine carree d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
dpolaire
dPsqrt(dpolaire a)
{
    dpolaire	    c;

    if (a.rm <= 0.0)
	c.rm = 0.0;
    else
	c.rm = sqrt(a.rm);
    c.th = a.th / 2.;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Puissance reelle d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
dpolaire
dPpuis(dpolaire a, double x)
{
    dpolaire	    c;

    if (a.rm <= 0.0)
	c.rm = 0.0;
    else
	c.rm = pow(a.rm, x);
    c.th = a.th * x;
    c.th = c.th - floor(c.th / 6.283185307179586) * 6.283185307179586;
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Puissance reelle d'un nombre complexe sous la forme (reel,imag)
    (simple precision)
----------------------------------------------------------------------*/
dcomplex
dCpuis(dcomplex a, double x)
{
    return dConv_ptc(dPpuis(dConv_ctp(a), x));
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Multiplication d'un nombre reel et d'un nombre complexe sous la
    forme (r, theta) (simple precision)
----------------------------------------------------------------------*/
dpolaire
dRPmul(double x, dpolaire a)
{
    dpolaire	    c;

    c.rm = fabs(x * a.rm);
    if (x >= 0)
	c.th = a.th;
    else {
	c.th = a.th + 3.141592653589793;
        if (c.th > 6.283185307179586)
	    c.th -= 6.283185307179586;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Division d'un nombre reel par un nombre complexe sous la forme
    (r, theta) (simple precision)
----------------------------------------------------------------------*/
dpolaire
dRPdiv(dpolaire a, double x)
{
    dpolaire	    c;

    c.rm = fabs(a.rm / x);
    if (x >= 0)
	c.th = a.th;
    else {
	c.th = a.th + 3.141592653589793;
        if (c.th > 6.283185307179586)
	    c.th -= 6.283185307179586;
    }
    return c;
}
/*--------------------------------------------------------------------*/




/*----------------------------------------------------------------------
    Inversion d'un nombre complexe sous la forme (r, theta)
    (simple precision)
----------------------------------------------------------------------*/
dpolaire
dPinv(dpolaire a)
{
    dpolaire	    c;

    c.rm = 1. / a.rm;
    c.th = - a.th;
    if (c.th < 0.0)
	c.th += 6.283185307179586;
    return c;
}
/*--------------------------------------------------------------------*/
