	title	TRYBCD -- Copyright 1997, Morten Elling
	subttl	Demonstrate BCDASM routines (DOS/Win32)

	;
	; Build : run maketry.bat
	; Usage : trybcd [> trybcd.txt]
	;

; ///// INTERFACE /////////////////////////////////////////////////////

	include model.inc	; Get memory model
	include modelt.inc	; Model macros
	include bcd.ash 	; BCDASM prototypes
	include bin2asc.ash	; Number conversion prototypes
	include consolio.ash	; Console I/O prototypes
	include consolio.inc	; DOS/Win32 definitions

	if (@Model gt 1) and (@WordSize eq 2)
	.stack	200h		; Define a stack unless tiny/flat
	endif
	locals			; Local label string prefix is @@

	if ((@Interface and 7fh)-1 ge 3)
	%out	This demo assembles in any 16-bit memory model (tiny
	%out	thru tchuge) and 32-bit model flat, but the language
	%out	specifier must be C, STDCALL, or SYSCALL to preserve
	%out	SI/ESI and DI/EDI across calls to BCDASM procedures.
	.err
	endif

; ///// MACROS ////////////////////////////////////////////////////////
;
; Write string to std. output
;
@show	MACRO string
	local id
	.const
	id db string,00h
	align @WordSize
	.code
	call  WriteZStr, @dsaddr(id)
	endm
;
; Write string and new-line to std. output
;
@shownl MACRO string
	ifnb  <string>
	@show <string,0dh,0ah>
	 else
	call  WriteNL
	endif
	endm
;
; Conditionally write string to std. output
;
@showif MACRO cc,string,op
	local c1,c2,around
	c1    substr <cc>,1,1
	ifdifi <n>,c1
	jn&cc short around
	 else
	c2 substr <cc>,2	;; Not 'jnz' is 'jz'
	c2 catstr <j>,c2
	c2 short around
	endif
	@show <string>
	ifnb  <op>		;; Typically: jmp @@label
	op
	endif
	around:
	endm

;
; Define tbyte-sized signed BCD. This macro remedies
; TASM32's inability to initialize a negative tbyte.
; Example: @dt <-123>,<87>,<?>,<+5>
;
@dt	MACRO b0,b1,b2,b3,b4
	local sign,bcdn
	.errnz (BCDSZ - 10)
	sign  substr <b0>,1,1
	ifidn sign,<->
	bcdn  substr <b0>,2	;; Strip sign character
	dt bcdn
	ORG   ($-1)		;; Back up location counter by one
	db 80h			;; Sign byte goes here
	 else
	dt b0
	endif
	ifb   <b1>
	exitm
	endif
	@dt b1,b2,b3,b4 	;; Recurse
	endm

;
; Dump register contents in hex
;
@dumpreg MACRO reg, caption
	mov   rdx, reg
	sub   rsp, 16
	mov   rax, rsp
	call  HexN, @ssr(rax), rdx
	ifnb  <caption>
	@show <caption>
	endif
        mov   rax, rsp
	call  WriteZStr, @ssr(rax)
	add   rsp, 16
	endm


; ///// FORWARDS //////////////////////////////////////////////////////

	.code
	; Procedure types for forward-referenced internal proc.s
@ptype	fac	near :@uint
@ptype	showres	near :dataptr, :dataptr, :@uint, :@uint
@ptype	main	near


; ///// GLOBALS ///////////////////////////////////////////////////////

	BCDSZ	= 10		; Use standard tbyte BCDs
	BCDSZx2 = BCDSZ*2	; Double-precision BCD size
	BCDSZx3 = BCDSZ*3	; Triple-precision BCD size
	SCRATCHSZ = 256+BCDSZ*4 ; Size of scratch buffer

	.data?
scratch db SCRATCHSZ dup (?)	; Scratch buffer
isRedir db ?			; Flags output redirected


; ///// STARTUP CODE //////////////////////////////////////////////////

	.code
	; Minimal don't-do-this-at-home startup code

	ife @isWin32
	pushstate
	.8086
	.startup
	popstate		; Restore selected processor
	if @Model eq 7		; 7 = model tchuge
	mov   ax, @data
	mov   ds, ax
	assume ds:@data
	endif
	cld
	mov   cx, 0121h
	shl   ch, cl
	test  ch, ch
	jnz   run
	.const
	msgreq db "Requires 80186 CPU or later$"
	; Not entirely true: TASM will fake <push constant>
	; for the 8086 processor but the code will be bloated.
	.code
	lea   dx, [msgreq]
	mov   ah, 09h
	int   21h
	mov   ax, 4c01h
	int   21h
	 else ; Win32
	@Startup:
	endif

run:	mov   rax, rsp
	push  rax
	call  main
	pop   rax
	cmp   rax, rsp
	@showif <nz>,<"*** Stack imbalanced ***">

die:	ife @isWin32
	mov   ax, 4c00h
	int   21h
	 else
	call  ExitProcess, 0
	endif


; /////////////////////////////////////////////////////////////////////
;	Wait for user key (ignored if output redirected), abort if Esc

pause	proc
	test  [isRedir], 0ffh
	jnz sh @@ret
	;
	@show <"<Press Space to continue, Esc to exit>",0dh>
@@p1:	call  GetKey
	cmp   al, 1bh
	jz sh @@p2
	cmp   al, ' '
	jne   @@p1
@@p2:	push  rax
	@show <"                                      ",0dh,0ah>
	pop   rax
	cmp   al, 1bh		; Halt program if Esc pressed
	je    die
@@ret:	RET
pause	endp


; /////////////////////////////////////////////////////////////////////
;	Display normal-size BCD with thousands separators, no decimals

displays proc
arg	@@pBCD:dataptr
	call  bcdFmt, @dsaddr(scratch), SCRATCHSZ, [@@pBCD], BCDSZ, \
		3*BCDSZ+4, 0, 0, -1, \	; width, numDec, prec, rtJust
		-1, -1, '.', ','	; signCh, fillCh, sepCh, sepMCh
	call  WriteZStr, @dsaddr(scratch)
	RET
displays endp


; /////////////////////////////////////////////////////////////////////
;	Display double-size BCD with thousands separators, no decimals

displaysx proc
arg	@@pBCD:dataptr
	call  bcdFmt, @dsaddr(scratch), SCRATCHSZ, [@@pBCD], BCDSZx2, \
		3*BCDSZx2+4, 0, 0, -1, -1, -1, '.', ','
	call  WriteZStr, @dsaddr(scratch)
	RET
displaysx endp


; /////////////////////////////////////////////////////////////////////
; 	Test addition and subtraction

try_add_sub proc
local	@@one	:tbyte, \
	@@diff	:tbyte

	.data
	__sum label tbyte
	@dt <999999999999999996>

	.code
	@shownl <"7 additions by one, will wrap at zero">
	lea   rsi, [@@one]
	lea   rdi, [__sum]
	call  bcdLd1, @ssr(rsi), BCDSZ
	mov   rcx, 7
@@addl: push  rcx
	call  bcdAdd, @dsr(rdi), @ssr(rsi), BCDSZ
	call  displays, @dsr(rdi)
	@shownl
	pop   rcx
	loop  @@addl
	@shownl


	@shownl <"7 subtractions by one, will go minus">
	lea   rsi, [@@one]
	lea   rdi, [@@diff]
	call  bcdMov, @ssr(rdi), @dsaddr(__sum), BCDSZ
	mov   rcx, 7
@@subl: push  rcx
	call  bcdSub, @ssr(rdi), @ssr(rsi), BCDSZ
	call  displays, @ssr(rdi)
	@shownl
	pop   rcx
	loop  @@subl

	@shownl
	@shownl
	RET
try_add_sub endp


; /////////////////////////////////////////////////////////////////////
; 	Shift BCD number X nibbles left or right

try_shift proc
local	@@nibbls:@uint, \
	@@Snum	:tbyte

	.const
	__Pnum label tbyte
	@dt <-34218626481299>

	.code
        @shownl <"Shifting left and right by X digits">
	mov   [@@nibbls], 0
	lea   rsi, [__Pnum]
	lea   rdi, [@@Snum]

@@resh: call  bcdMov, @ssr(rdi), @dsr(rsi), BCDSZ	; Copy P to S
	call  bcdShl, @ssr(rdi), BCDSZ, [@@nibbls]	; Shift left
	call  displays, @ssr(rdi)

	call  bcdMov, @ssr(rdi), @dsr(rsi), BCDSZ	; Copy P to S
	call  bcdShr, @ssr(rdi), BCDSZ, [@@nibbls]	; Shift right
	call  displays, @ssr(rdi)		; Display side-by-side
	@shownl

	inc   [@@nibbls]
	cmp   [@@nibbls], 2*(BCDSZ-1)
	jbe   @@resh


	@shownl
	@shownl
	@shownl
	RET
try_shift endp


; /////////////////////////////////////////////////////////////////////
;	Number conversions

try_cvt proc
local	@@bina	:qword, \
	@@binb	:qword, \
	@@cvtd	:tbyte:(BCDSZx2/BCDSZ)
	BINASZ = 8
	BINBSZ = 8

	.const
	__Bnum label tbyte
	@dt <-876543210987654321>

	.code
        @shownl
	@shownl <"Number conversions:">
	@show <"Convert packed signed BCD     ">
	lea   rsi, [__Bnum]
	call  displays, @dsr(rsi)
	@shownl


	@show <"to binary                     ">
	lea   rdi, [@@bina]
	call  bcdP2b, @ssr(rdi), BINASZ, @dsr(rsi), BCDSZ
	@shownl <"   (cannot display)">


	@show <"then binary to ASCII          ">
	mov   rsi, rdi
	lea   rdi, [scratch]
	call  bin2asc, @dsr(rdi), @ssr(rsi), BINASZ
	call  WriteZStr, @dsr(rdi)
	@shownl


	@show <"then ASCII to packed BCD      ">
	mov   rsi, rdi
	lea   rdi, [@@cvtd]
	call  bcdA2p, @ssr(rdi), BCDSZ, @dsr(rsi)
	test  rax, rax
	@showif <z>,<"bcdA2p failed">,<jmp @@end>
	call  displays, @ssr(rdi)
	@shownl


	@show <"then again to binary          ">
	mov   rsi, rdi
	lea   rdi, [@@binb]
	call  bcdP2b, @ssr(rdi), BINBSZ, @ssr(rsi), BCDSZ
	@shownl <"   (cannot display)">


	@show <"then binary to packed BCD     ">
	mov   rsi, rdi
	lea   rdi, [@@cvtd]
	call  bcdB2p, @ssr(rdi), BCDSZ, @ssr(rsi), BINBSZ
	test  rax, rax
	@showif <z>,<"bcdB2p failed">,<jmp @@end>
	call  displays, @ssr(rdi)
	@shownl


	@show <"then to big-endian            ">
	mov   rsi, rdi
	lea   rdi, [scratch]
	call  bcdLe2be, @dsr(rdi), @ssr(rsi), BCDSZ
	@shownl <"   (cannot display)">


	@show <"and back to little-endian BCD ">
	mov   rsi, rdi
	lea   rdi, [@@cvtd]
	call  bcdBe2le, @ssr(rdi), @dsr(rsi), BCDSZ
	call  displays, @ssr(rdi)
	@shownl


	@show <"then to un-packed signed BCD  ">
	mov   rsi, rdi
	lea   rdi, [scratch]
	call  bcdP2u, @dsr(rdi), @ssr(rsi), BCDSZ
	@shownl <"   (cannot display)">


	@show <"and back to packed signed BCD ">
	mov   rsi, rdi
	lea   rdi, [@@cvtd]
	call  bcdU2p, @ssr(rdi), @dsr(rsi), BCDSZx2
	test  rax, rax
	@showif <z>,<"bcdU2p failed">,<jmp @@end>
	call  displays, @ssr(rdi)
	@shownl


	@show <"finally BCD to raw ASCII      ">
	mov   rsi, rdi
	lea   rdi, [scratch]
	call  bcdP2a, @dsr(rdi), @ssr(rsi), BCDSZ
	call  WriteZStr, @dsr(rdi)

@@end:	@shownl
	@shownl
	@shownl
	RET
try_cvt endp


; /////////////////////////////////////////////////////////////////////
;	Various BCD output formats

try_fmt proc
	.const
	__Fnum label tbyte
	@dt <-156876645616892333>

	.code
	@shownl
        @show <"Here's the number -156876645616892333 in various ">
	@shownl <"output formats:">
	@shownl

	lea   rsi, [__Fnum]
	lea   rdi, [scratch]

	@show <"- Default format (width = length)       ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		-1, -1, -1, -1, \	; width, numDec, prec, rtJust
		-1, -1, -1, -1		; signCh, fillCh, sepCh, sepMCh
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32                              ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+2+1, 0, 0, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, left justified              ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 0, 0, 0,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, precision 2, no 1000's sep  ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 0, 2, -1,  -1, -1, -1, 0
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, precision 2, '_' fill       ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 0, 2, -1,  -1, '_', -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 2, precision 2     ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 2, 2, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- As above, but custom separators       ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 2, 2, -1,  -1, -1, ',', '.'
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 9, precision 0     ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 9, 0, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 12, precision 2    ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 12, 2, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 15, precision 19   ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 15, 19, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 20, precision 5    ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 20, 5, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- Width 32, decimals 20, precision 23   ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		3*BCDSZ+3, 20, 23, -1, 	-1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl



	.const
	__Gnum label tbyte
	@dt <+995>

	.code
	@shownl
	@shownl <"The number +995 in various formats:">
	@shownl
	lea   rsi, [__Gnum]

	@show <"- Width 6, decimals 2, precision 2      ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		6, 2, 2, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- As above, but precision 1             ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		6, 2, 1, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- As above, but precision 0             ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		6, 2, 0, -1,  -1, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- As 2nd format, but no sign            ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		6, 2, 1, -1,  0, -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@show <"- As 2nd format, but sign ' '           ">
	call  bcdFmt, @dsr(rdi), SCRATCHSZ, @dsr(rsi), BCDSZ, \
		6, 2, 1, -1,  ' ', -1, -1, -1
	call  WriteZStr, @dsr(rdi)
	@shownl

	@shownl
	RET
try_fmt endp


; /////////////////////////////////////////////////////////////////////
;	Double-size signed multiplication and division

try_mul_div proc
	.data
	__mproduct label byte
if 1
	__mplic label tbyte
	@dt <-12356738473699812>,<?>	; Result is double-size
	__mplyr label tbyte
	@dt   <+928341627328234>,<?>
 else
	__mplic label tbyte
	@dt <-99999999999999999>,<?>
	__mplyr label tbyte
	@dt <+99999999999999999>,<?>
endif	; 2----+----1----+----0 	; Digits 0..17

	.code
        @shownl
	@shownl <"Double-size signed multiplication">

	@show <"Multiplier is:                ">
	lea   rsi, [__mplyr]
	call  displays, @dsr(rsi)
	@shownl

	@show <"Multiplicand is:              ">
	lea   rdi, [__mplic]
	call  displays, @dsr(rdi)
	@shownl

	@shownl <"Double-precision product is: ">
	call  bcdImul, @dsr(rdi), @dsr(rsi), BCDSZx2
	call  displaysx, @dsr(rdi)
	@shownl
	@shownl
	@shownl


	@shownl <"Signed division of product by multiplier">
	@show <"Quotient is:                  ">
	call  bcdSx, \		; Sign-extend divisor
		@dsr(rsi), @dsr(rsi), BCDSZ
	call  bcdIdiv, \	; Divide product by multiplier
		@dsr(rdi), @dsr(rsi), BCDSZx2
	test  rax, rax
	@showif <z>,<"*Overflow*">,<jmp sh @@ret>
	call  displays, @dsr(rdi)
	@shownl

	@show <"Remainder is:                 ">
	add   rdi, BCDSZ	; Remainder is in upper half
	call  displays, @dsr(rdi)
@@ret:	@shownl
	@shownl
	@shownl
	RET
try_mul_div endp


; /////////////////////////////////////////////////////////////////////
;	Compute and output power-of-two table

	MAX_POW = 129

try_pow2 proc
local	@@pown	:tbyte:(BCDSZx2/BCDSZ), \
	@@exp	:@uint

	@shownl <"Double-precision power-of-two table (2^n):">
	call  pause

	mov   [@@exp], 0	; 2^0 = 1
	lea   rdi, [@@pown]
	call  bcdLd1, @ssr(rdi), BCDSZx2
	jmp sh @@pow0

@@powl: inc   [@@exp]
	call  bcdAdd, @ssr(rdi), @ssr(rdi), BCDSZx2
	test  rax, rax		; Double @@pown
	jnz sh @@end		; Stop when carry

				; Shew double-precision result
				; and exponent (e.g. "4,096  2^12")
@@pow0: call  displaysx, @ssr(rdi)
	lea   rax, [@@exp]
	lea   rsi, [scratch+3]
	call  bin2asc, @dsr(rsi), @ssr(rax), @WordSize
	sub   rsi, 3
	mov   @wptr [rsi], '  '
	mov   @wptr [rsi+2], '^2' ; Replace '+' with '^'
	call  WriteZStr, @dsr(rsi)
	@shownl

	cmp   [@@exp], MAX_POW
	jb    @@powl

@@end:	@shownl
	@shownl
	RET
try_pow2 endp


; /////////////////////////////////////////////////////////////////////
;	Compute and output first N factorials

	MAX_FAC = 50

try_fac	proc
local	@@facn	:tbyte:(BCDSZx3/BCDSZ), \
	@@work	:tbyte:(BCDSZx3/BCDSZ), \
	@@n	:@uint

	@shownl <"Triple-precision table of factorials (n!):">
	call  pause

	mov   [@@n], 0
	lea   rsi, [@@work]
	lea   rdi, [@@facn]	; Destination
@@facl: call  fac, [@@n]        ; Compute n!
	test  rax, rax		; Exit loop if overflow
	@showif <nz>,<"*Overflow*">,<jmp @@end>
				; Format multiple-precision result
	call  bcdFmt, @dsaddr(scratch), SCRATCHSZ, @ssr(rdi), \
		BCDSZx3, 3*BCDSZx3-7, 0, 0, -1, -1, -1, -1, -1

				; Append "  n!"
	lea   rbx, [scratch]
	add   rbx, rax		; End of string
	dec   rbx		; Overwrite '+'
	mov   rax, [@@n]
	aam
	xchg  al, ah
	or    rax, '00'
	mov   @wptr [rbx], '  '
	mov   [rbx+2], rax
	mov   @wptr [rbx+4], '!'

	call  WriteZStr, @dsaddr(scratch)
	@shownl

	inc   [@@n]
	cmp   [@@n], MAX_FAC
	ja sh @@end
	jmp   @@facl

@@end:	@shownl
	@shownl
	@shownl
	RET
try_fac	endp


; ----- Compute n!
; Entry @@n n, ss:rdi destination, ss:rsi work space
; Exit	rax = overflow flag

fac	proc @Type_fac
arg	@@n   :@uint

	mov   rax, [@@n]
	sub   rax, 1		; Get (n-1)! into destination
	jbe sh @@lt2
	call  fac, rax		; Recurse
	test  rax, rax		; Exit if overflow
	jnz sh @@ret
				; Duplicate destination
	call  bcdMov, @ssr(rsi), @ssr(rdi), BCDSZx3
				; Multiply destination by n
@@addn: dec   [@@n]
	jz sh @@ok
	call  bcdAdd, @ssr(rdi), @ssr(rsi), BCDSZx3
	test  rax, rax
	jnz sh @@ret		; Exit if overflow
	jmp   @@addn

@@lt2:	; 0! = 1! = 1
	call  bcdLd1, @ssr(rdi), BCDSZx3
@@ok:	sub   rax, rax
@@ret:	RET
fac	endp


; /////////////////////////////////////////////////////////////////////
;	Addition of many (cached) numbers

	NMULT = 800
	NNBRS = 250

	.const
nbrs	label tbyte
 REPT NNBRS / 10
 @dt <48371123723>,<12357212812>,<9102345312312>,<82233718231>,<-921872>
 @dt <671273378777>,<9834547654321>,<-123123>,<12389898989>,<8323476611>
 ENDM
nbrsend label byte

	.code
try_addmany proc
local	@@sum	:tbyte

	@show <"Addition of 200,000 18-digit signed numbers...">
				; Loop count = NMULT * NNBRS
	lea   rdi, [@@sum]	; Reset sum to zero
	call  bcdLdz, @ssr(rdi), BCDSZ

	mov   rcx, NMULT
@@outr: push  rcx		; Loop, ignoring overflow
	lea   rsi, [nbrs]
@@innr: call  bcdAdd, @ssr(rdi), @dsr(rsi), BCDSZ
	add   rsi, BCDSZ
	cmp   rsi, OFFSET nbrsend
	jb    @@innr
	pop   rcx
	dec   rcx
	jnz   @@outr

	@shownl <"done.">
	@show <"The sum is:      ">
	call  displays, @ssr(rdi)
	@shownl
	@shownl
	@shownl
	RET
try_addmany endp

	
; /////////////////////////////////////////////////////////////////////
;	Addition of huge numbers

	HUGELOOP  = 1000
	HUGEBCDSZ = 5000

try_addhuge proc
	; Note: In .model tchuge .data? shares segment with .data
	;	and .const (bloated .exe); better to use .fardata?
	.data?
	__hugebcd db HUGEBCDSZ dup (?)

	.code
        @shownl <"Addition of one thousand 10,000-digit numbers: ">
	call  pause

	@show <"Initializing...">
	; Grab whatever is in memory and limit to BCD range
	lea   rsi, [__hugebcd]
	mov   rdi, rsi
	mov   rcx, HUGEBCDSZ
	mov   al, 77h		; Let's pretend 8 fingers only
@@lim:	and   [rdi], al
	inc   rdi
	dec   rcx
	jnz   @@lim
	mov   [rdi-1], cl	; Sign


	@show <"adding...">
	mov   rdi, HUGELOOP
@@dbl:	mov   rax, HUGEBCDSZ	; Add to self n times
	call  bcdAdd, @dsr(rsi), @dsr(rsi), rax
	dec   rdi
	jnz   @@dbl


	@shownl <"done. The sum won't be displayed.">
	@shownl
	RET
try_addhuge endp


; /////////////////////////////////////////////////////////////////////
;	Test return values and flags of miscellaneous
;	arithmetic routines performed on simple data.

try_misc proc
local	@@pPairs:@uint, \
	@@cPairs:@uint, \
	@@nbrA	:tbyte, \
	@@nbrB	:tbyte

	.const
	__nA label tbyte
	@dt <+2>		; Max. 3 digits (see proc showres)
	__nB label tbyte
	@dt <-3>
	__nC label tbyte
	@dt <0>
	__pairs @dui __nA,__nB, __nA,__nC, __nB,__nC
		@dui __nB,__nA, __nC,__nA, __nC,__nB
		@dui __nA,__nA, __nB,__nB, __nC,__nC
	NPAIRS = ($-__pairs) / (@WordSize*2)

	.code
        @shownl
	@shownl <"Various arithmetic and return values">
	@shownl

	mov   [@@pPairs], OFFSET __pairs
	mov   [@@cPairs], NPAIRS
	lea   rdi, [@@nbrA]
	lea   rsi, [@@nbrB]

@@nxtp: ; Copy A and B to local storage to allow alterations
	mov   rbx, [@@pPairs]
	mov   rax, [rbx]
	push  @uiptr [rbx+@WordSize]
	call  bcdMov, @ssr(rdi), @dsr(rax), BCDSZ
	pop   rax
	call  bcdMov, @ssr(rsi), @dsr(rax), BCDSZ
	add   [@@pPairs], (@WordSize*2)


	@show <"------------------------------------">
	@shownl <"------------------------------------">
	@show <"Compare A and B         (bcdCmp A, B)">
	call  bcdCmp, @ssr(rdi), @ssr(rsi), BCDSZ
	pushf
	pop   rcx
	call  showres, @ssr(rdi), @ssr(rsi), rax, rcx

	@show <"Subtract B from A       (bcdSub A, B)">
	call  bcdSub, @ssr(rdi), @ssr(rsi), BCDSZ
	pushf
	pop   rcx
	call  showres, @ssr(rdi), @nullptr, rax, rcx

	@show <"Add B to A              (bcdAdd A, B)">
	call  bcdAdd, @ssr(rdi), @ssr(rsi), BCDSZ
	pushf
	pop   rcx
	call  showres, @ssr(rdi), @nullptr, rax, rcx

	@show <"Compare A against zero  (bcdCmpz A)">
	call  bcdCmpz, @ssr(rdi), BCDSZ
	pushf
	pop   rcx
	call  showres, @ssr(rdi), @nullptr, rax, rcx

	@show <"Reverse sign of A       (bcdNeg A)">
	call  bcdNeg, @ssr(rdi), BCDSZ
	pushf
	pop   rcx
	call  showres, @ssr(rdi), @nullptr, rax, rcx

	@show <"Absolute value of A     (bcdAbs A)">
	call  bcdAbs, @ssr(rdi), BCDSZ
	call  showres, @ssr(rdi), @nullptr, rax, -1

	@show <"Swap A and B            (bcdSwap A, B)">
	call  bcdSwap, @ssr(rdi), @ssr(rsi), BCDSZ
	call  showres, @ssr(rdi), @ssr(rsi), rax, -1

	call  pause
	dec   [@@cPairs]
	jz sh @@ret
	jmp   @@nxtp

@@ret:	RET
try_misc endp


; ----- Display one or two BCDs along with accumulator and flags
showres proc @Type_showres
arg	@@pA	:dataptr, \
	@@pB	:dataptr, \
	@@acc	:@uint, \
	@@flax	:@uint

; ----- Display A (and B) value in a 4-byte field
	@shownl
	@show <"A =">
	call  bcdFmt, @dsaddr(scratch), SCRATCHSZ, [@@pA], BCDSZ,\
		4, 0, 0, -1, -1, -1, '.', ','
	call  WriteZStr, @dsaddr(scratch)

	cmp   @uiptr [@@pB], 0	; Is it a @nullptr ?
	@showif <z>,<"            ">,<jmp sh @@sr1>
	@show <"     B =">
	call  bcdFmt, @dsaddr(scratch), SCRATCHSZ, [@@pB], BCDSZ,\
		4, 0, 0, -1, -1, -1, '.', ','
	call  WriteZStr, @dsaddr(scratch)


; ----- Display accumulator relative to 0
@@sr1:	@show <"     Return value ">
	cmp   [@@flax], -1
	@showif <z>,<"undefined">,<jmp sh @@end>
	cmp   [@@acc], 0
	@showif <g>,<"> 0">,<jmp sh @@sr2>
	@showif <l>,<"< 0">,<jmp sh @@sr2>
	@show <"= 0">
@@sr2:	;


; ----- Display CPU flags if model language is STDCALL
;	(C, CPP, SYSCALL destroy flags after stack cleanup)

	if (@Interface and 7Fh) eq 3
	.data
	__flstr db "CPU flags: sf=?, zf=?, cf=?",0
	.code
	mov   ah, @bptr [@@flax]
	lea   rbx, [__flstr+14]
	mov   ch, 11000001b	; Display mask
@@fl1:	shl   ah, 1
	mov   al, '0'
	adc   al, 0
	shl   ch, 1
	jnc sh @@fl2
	mov   [rbx], al
	add   rbx, 6
@@fl2:	jnz   @@fl1
	@show <"     ">
	call  WriteZStr, @dsaddr(__flstr)
	endif


@@end:	@shownl
	@shownl
	RET
showres endp


; ///// MAIN //////////////////////////////////////////////////////////

main	proc @Type_main
	mov   rsi, es		; Save original ES

; ----- Get redirection state of std. output
	ife @isWin32
	call  IsDevRedir, STDOUT
	 else
	call  IsDevRedir, STD_OUTPUT_HANDLE
	endif
	mov   [isRedir], al

; ----- Say hello
	@shownl
	@shownl
	@shownl <"This is a short demo of the BCDASM routines that ">
	@shownl <"operate on packed, signed Binary-Coded Decimals.">
	ife @isWin32
	@shownl <"(The output of this program may be redirected.)">
	endif	; The Win32 IsDevRedir is on the ToDo list...
	@shownl
	call  pause

; ----- Display model-related information
	@shownl <"Segment registers and stack pointer:">
	@dumpreg CS, <"CS ">
	@dumpreg DS, <"   DS ">
	@dumpreg SS, <"   SS ">
	@dumpreg rsi,<"   ES ">		; ES on entry
	ife @isUse32
	@dumpreg SP, <"   SP ">
	else
	@dumpreg ESP, <"   ESP ">
	endif
        @shownl
	.const
	% __inf db "(Symbol MDL = &MDL, symbol CPU = &CPU)",0
	.code
	call  WriteZStr, @dsaddr(__inf)
	@shownl
	@shownl
	call  pause

; -----	Run demo
	call  try_add_sub
	call  pause
	call  try_shift
	call  pause
	call  try_cvt
	call  pause
	call  try_fmt
	call  pause
	call  try_mul_div
	;call pause
	call  try_pow2
	;call pause
	call  try_fac
	call  pause
	call  try_addmany
	;call pause
	call  try_addhuge
	call  pause
	call  try_misc
	@shownl <"End of TRYBCD.">
	RET
main	endp

	END   @Startup