/*
 * a print utility, more UNIX compatible than others
 *
 * pr [ option ] ... [ file ] ...
 *
 */

#include <stdio.h>

/* Useful Defines */

#define FF '\014'
#define MAXFILES 15

/* forward function references */

char *malloc();
char *tod();

/* global definitions */

int lines = 66;
int width = 80;
int tab = 8;
int columns = 1;
int isheader = FALSE;
char header[72];
int printheader = TRUE;
int useff = FALSE;
int startpage = 0;
int multiple = FALSE;
char separator = ' ';		/* character to use instead of white space */
char opts[] = "123456789p:h:w:fl:ts:mT:";
int page;
char crlf[] = "\r\n";
char **pgarray;
int usestdin = TRUE;		/* set false when a file argument appears */
int endpage;
int files = 0;			/* index into fp list */
FILE *fp[MAXFILES];		/* array of files for multiple printing */

/* declarations */

extern int optopt;
extern char *optarg;
extern int optind;
extern char timestr[];

/* ------------------------------------------------------------------------ */

/* main -- process arguments and cycle through list of files */

main(argc,argv)
int argc;
char *argv[];
{
	int switchar;

	switchar = swchar();
	while (optind < argc) {
		if (argv[optind][0] != switchar) {
			if (!isheader) strcpy(header,argv[optind]);
			usestdin = FALSE;
			fp[files] = fopen(argv[optind],"r");
			if (fp[files] == NULL) {
				fprintf(stderr,"pr: can't open %s\r\n",argv[optind]);
			}
			else {
				if (!multiple) {
					pr(fp);
				}
				else ++files;
			}
			++optind;
		}
		else if (getopt(argc,argv,opts) == '?') {
			continue;
		}
		else if (optopt == '1') columns = 1;
		else if (optopt == '2') columns = 2;
		else if (optopt == '3') columns = 3;
		else if (optopt == '4') columns = 4;
		else if (optopt == '5') columns = 5;
		else if (optopt == '6') columns = 6;
		else if (optopt == '7') columns = 7;
		else if (optopt == '8') columns = 8;
		else if (optopt == '9') columns = 9;
		else if (optopt == 'p') {
			startpage = atoi(optarg);
		}
		else if (optopt == 'h') {
			isheader = TRUE;
			strcpy(header,optarg);
		}
		else if (optopt == 'w') {
			width = atoi(optarg);
			if (width == 0) width = 72;
		}
		else if (optopt == 'l') {
			lines = atoi(optarg);
			if (lines == 0) lines = 66;
		}
		else if (optopt == 't') {
			printheader = FALSE;
			useff = FALSE;		/* automatically */
		}
		else if (optopt == 'T') {
			tab = atoi(optarg);
			if (tab == 0) tab = 8;
		}
		else if (optopt == 'f') {
			useff = TRUE;
		}
		else if (optopt == 's') {
			separator = optarg[0];
		}
		else if (optopt == 'm') {
			multiple = TRUE;
		}
	} /* end argument processing */
	if (usestdin) {
		fp[files] = stdin;
		pr(fp);
	}
	else if (multiple) {
		if (!isheader) header[0] = 0;
		pr(fp);
	}
}

/* ------------------------------------------------------------------------ */

pr(fp)
FILE *fp[];
{
	page = 1;
	if (printheader) {
		if (useff) endpage = lines - 7;
		else endpage = lines - 5;
	}
	else {
		useff = FALSE;
		endpage = lines;
	}

	while (prpage(fp) != EOF) {
		++page;
	}
}

/* ------------------------------------------------------------------------ */

allocpage()
{
	int i;

	if (pgarray != NULL) return;
	if ((pgarray = malloc(lines * sizeof(char *))) == NULL)
		perror("out of memory");
	setmem(pgarray,(lines * sizeof(char *)),0);

	for (i=0;i<lines;++i) {
		if ((pgarray[i]=malloc(width+2)) == NULL)
			perror("out of memory");
		setmem(pgarray[i],width+2,0);
	}
}

/* ------------------------------------------------------------------------ */

initpage()
{
	int i;

	for (i=0;i<lines;++i) {
		if (pgarray[i] != NULL) {
			setmem(pgarray[i],width+2,0);
		}
	}
}

/* ------------------------------------------------------------------------ */

cleanup()
{
	int i;

	for (i=0;i<lines;++i) {
		if (pgarray[i] != 0) free(pgarray[i]);
	}
	free(pgarray);
	pgarray = NULL;
}

/* ------------------------------------------------------------------------ */

prpage(fp)
FILE *fp[];
{
	int i;
	char *p;
	int iseof;

	iseof = buildpage(fp);
	if (page < startpage) return 0;

	for (i=0;i<endpage;++i) {
		p = pgarray[i];
		while (*p) putchar(*p++);
		putchar('\r');
		putchar('\n');
	}
	if (useff) putchar('\f');	
	else if (printheader) {
		for (i=0;i<5;++i) {
			putchar('\r');
			putchar('\n');
		}
	}
	return iseof;
}

/* ------------------------------------------------------------------------ */

buildpage(fp)
FILE *fp[];
{
	int i;
	int vpos;
	int colwid;
	int start;
	int insidecolumn;
	int retval;
	int startline;

	if (pgarray == NULL) allocpage();
	else initpage();

	vpos = 0;
	if (printheader && !useff) {
		vpos += 2;
	}
	if (printheader) {
		sprintf(pgarray[vpos++],"%s  %s Page %d",tod(),header,page);
		vpos += 2;
	}

/* Variables */

	if (columns > 1) colwid = (width/columns) -1;
	else if (multiple) colwid = (width/files) -1;
	start = 0;

/* Single column case */

	if (!multiple && columns == 1) {
		for (;vpos<endpage;++vpos) {
			retval = buildline(pgarray[vpos],width,fp[files]);
			if (retval == EOF) {
				fclose(fp[files]);
				return EOF;
			}
			else if (retval == FF) return 0;
		}
	}

/* Multiple column case */

	if (!multiple && columns > 1) {
		startline = vpos;
		for (i=0;i<columns;++i) {
			insidecolumn = (i < columns-1);	/* true or false */
			for (;vpos<endpage;++vpos) {
				if (insidecolumn) 
				    setmem(pgarray[vpos]+start,colwid,' ');
				else setmem(pgarray[vpos]+start,colwid,0);
				if (buildline(pgarray[vpos]+start,colwid,fp[files])
				    == EOF) {
					fclose(fp[files]);
					return EOF;
				}
				if (insidecolumn)
					charcat(pgarray[vpos],separator);
			} /* end for each line */
			start += (colwid+1);
			vpos = startline;
		} /* end for each column */
	} /* end column mode */

/* Multiple file case */

	if (multiple) {
		startline = vpos;
		for (i=0;i<files;++i) {
			insidecolumn = (i < files-1);	/* true or false */
			for (;vpos<endpage;++vpos) {
				if (insidecolumn) 
				    setmem(pgarray[vpos]+start,colwid,' ');
				else setmem(pgarray[vpos]+start,colwid,0);
				if (
				   fp[i] != NULL && 
				   buildline(pgarray[vpos]+start,colwid,fp[i])
				    == EOF
				) {
					fclose(fp[i]);
					fp[i] = NULL;
				}
				if (insidecolumn)
					charcat(pgarray[vpos],separator);
			} /* end for each line */
			start += (colwid+1);
			vpos = startline;
		} /* end for each column */
		for (i=0;i<files;++i) if (fp[i] != NULL) return 0;
		return EOF;
	} /* end multiple */

	return 0;
}

/* ------------------------------------------------------------------------ */

buildline(s,w,fp)
char *s;
int w;
FILE *fp;
{
	int hpos;
	int c;

	hpos = 0;
	while ((c = getc(fp)) != EOF) {
		switch (c) {
		case '\n':
			return 0;
			break;
		case '\r':
			break;
		case '\f':
			if (!multiple) return FF;
			break;
		case '\t':
			if (hpos < w) *s++ = ' ';
			++hpos;
			while ((hpos % tab) && (hpos < w)) {
				*s++ = ' ';
				++hpos;
			}
			break;
		default:
			if (isprint(c) && hpos < w) *s++ = c;
			++hpos;
			break;
		}
	}
	if (c == EOF) return EOF;
	return 0;
}

/* ------------------------------------------------------------------------ */

perror(string)
char *string;
{
	cleanup();
	fprintf(stderr,"pr: %s\r\n",string);
	exit(1);
}

/* ------------------------------------------------------------------------ */

charcat(s,c)
char *s;
int c;
{
	char *p;
	p = s + strlen(s);
	*p++ = c;
	*p = 0;
	return;
}
