/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tmenu.hpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Header file for the TMenu class. This class is the class
*				of menu object used to represent a vertical list of
*				items in a box.
*
*				The items in the menu are always drawn in the current
*				system font.
*
* $Id: tmenu.hpp 1.2 1994/03/09 11:50:15 kjb Exp $
*
****************************************************************************/

#ifndef	__TMENU_HPP
#define	__TMENU_HPP

#ifndef	__TVIEW_HPP
#include "tview.hpp"
#endif

#ifndef	__TKEYS_HPP
#include "tkeys.hpp"
#endif

#ifndef	__ARRAY_HPP
#include "array.hpp"
#endif

/*-------------------------- Class definitions ----------------------------*/

//---------------------------------------------------------------------------
// The TMenuItem class is used to represent a single menu item stored in
// a menu. The TMenu class is used to collect together TMenuItems into
// the actual menu structure.
//---------------------------------------------------------------------------

class TMenu;

class TMenuItem {
public:
	char		*name;			// Name of the menu item
	ushort		command;		// Command to post when selected
	bool		disabled;		// True if item is disabled
	HotKey		hotKey;			// Menu item hot key value
	char		hotChar;		// Character code for hot character
	ushort		hotIndex;		// Index of hot character
	ushort		helpCtx;		// Help context number for the menu item
	TPoint		hot1,hot2;		// Coordinates of hot character underscore
	TRect		bounds;			// Bounding rectangle for the item
	union {
		const char *hotKeyText;	// Text to display for hot key
		TMenu	*subMenu;		// Pointer to submenu
		};

public:
			// Constructors
			TMenuItem(const char *name,
					  ushort command,
					  const HotKey& hotKey = HotKey(kbNoKey,0),
					  ushort helpCtx = hcNoContext,
					  const char *hotKeyText = NULL);

			TMenuItem(const char *name,
					  TMenu *subMenu,
					  ushort helpCtx = hcNoContext);

			// Destructor
			~TMenuItem();

			// Overloaded operator equals - we never compare these
			bool operator == (const TMenuItem& i)	{ return false; };

			// Returns true if item is a sub menu
			bool isSubMenu()	{ return command == 0; };
	};

//---------------------------------------------------------------------------
// The TSubMenuItem class is used to represent the submenu item entry in a
// menu, and exists simply to clarify the code used to create the menu
// items.
//---------------------------------------------------------------------------

class TSubMenuItem : public TMenuItem {
public:
			// Constructor
			TSubMenuItem(const char *name,
					 TMenu *subMenu,
					 ushort helpCtx = hcNoContext)
				: TMenuItem(name,subMenu,helpCtx) {};
	};

//---------------------------------------------------------------------------
// The TMenuItemSeparator class is used to represent the sepatatore menu
// item in the menus. Separators are always disabled.
//---------------------------------------------------------------------------

class TMenuItemSeparator : public TMenuItem {
public:
			// Constructor
			TMenuItemSeparator() : TMenuItem("-",cmError)
				{ disabled = true; };
	};

//---------------------------------------------------------------------------
// The TMenu class is used to represent a vertical list of menu items in
// a box.
//
// Note that not all of the text settings passed to the TMenu constructor
// are recognised. The text is always drawn in the current system font.
//
// Palette layout:
//		1 = Light border
//		2 = Dark border
//		3 = Normal interior
//		4 = Selected interior
//		5 = Normal text
//		6 = Disabled text
//		7 = Selected text
//---------------------------------------------------------------------------

class TMenu : public TView {
protected:
	IArray<TMenuItem>	items;			// Array of items for the menu
	int					current;		// Index of the current item
	int					itemWidth;		// Maximum menu item width
	int					itemHeight;		// Height of each item
	TMenu				*parent;		// Pointer to parent menu
	void				*save;			// Save area behind submenu's
	bool				repaintSave;	// Repaint the saved area
	int					subMenuID;		// Id of displayed submenu
	static TPoint		arrow[7];		// Arrow used for submenus

			// Method to draw a menu item in the menu
	virtual	void drawItem(int index);

			// Method to compute the bounding rectangle for an item
	virtual	TRect& setItemBounds(int index,const TPoint& start);

			// Method to handle a selection from the menu
			void doSelection(TEvent& event);

			// Method to find the menuitem with a specified shortcut
			int findShortCut(TEvent& event);

			// Method to check for and handle keyboard hot keys
			bool checkHotKey(TEvent& event);

			// Method to find the next valid menu item
			int findNextItem(int index);

			// Method to find the previous valid menu item
			int findPrevItem(int index);

			// Method to find which menu item was hit
			int findSelected(TPoint& where);

			// Method to begin executing a subMenu
			ushort executeSubMenu(TEvent& event);

			// Method to display a menu and restore it
			void displayMenu(int selection);
			void restoreMenu();

			// Method to check for total inclusion of a point (all parents)
			bool totalIncludes(TPoint& where);

public:
			// Constructor
			TMenu(const TRect& bounds = TRect(0,0,0,0));

			// Destructor
	virtual	~TMenu();

			// Virtual to handle an event
	virtual	void handleEvent(TEvent& event,phaseType phase = phFocused);

			// Virtual to execute the menu tracking the mouse etc
	virtual	ushort execute();

			// Methods to retrieve and post events - eventually calls
			// eventQueue.xxx() if not overridden
	virtual	void putEvent(TEvent& event);
	virtual	bool getEvent(TEvent& event,ushort mask = evAll);
	virtual	bool peekEvent(TEvent& event,ushort mask = evAll);

			// Virtual to draw the menu
	virtual	void draw(const TRect& clip);

			// Method to refresh the menu with a newly selected item
			void refresh(int selection);

			// Virtual to move view to another location
	virtual	void moveTo(int x,int y);

			// Methods to invalidate the specified rectangle
	virtual	void invalidRect(TRect& rect,bool global);

			// Method to return a reference to the view's current palette
	virtual	TPalette& getPalette() const;

			// Method to obtain the help context number for the menu
	virtual	ushort getHelpCtx() const;

			// Overloaded operator + to add a menu item
			TMenu& operator + (TMenuItem* item);

			// Method to complete the menu definition
	virtual	void doneDefinition();

			// Method to map a color index to a real color value
			color_t getColor(uchar index);

			// Method to set the current item
			void setCurrent(int index)	{ current = index; };

			// Method to swap a menu item - old item is _not_ deleted
			void swapItem(int itemNo,TMenuItem *item)
				{ items[itemNo] = item; };

			// Method to disabled a menu item
			void disableItem(int itemNo)
				{ items[itemNo]->disabled = true; };

			// Method to enable a menu item
			void enableItem(int itemNo)
				{ items[itemNo]->disabled = false; };
	};

#endif	// __TMENU_HPP
