program prtfile ;
  { Prints a text file on the list device, formatted with various
    user-supplied options.  Turbo Pascal, MS/PC-DOS.  Public Domain.

    Bill Meacham
    1004 Elm Street, Austin, Tx  78703

    This revision picks up the DOS date and time and puts it into the header.

    To quit, enter a blank file name when it asks you for one.
    To quit prematurely, type any letter.  It will ask if you want to
    quit.

    Last modified: 11/12/87 }

{$V-}  { Turn off strict type-checking for strings }

label            99 ;               { for premature exit }

const
    formfeed   = ^L ;
    bell       = ^G ;
    linelength = 255 ;              { max length of text file lines }

type
    st_typ  = string[linelength] ;
    regpack = record case integer of
                1: (AX,BX,CX,DX,BP,DI,SI,DS,ES,Flags : integer) ;
                2: (AL,AH,BL,BH,CL,CH,DL,DH : byte)
               end ;
    str14   = string[14] ;
    str66   = string[66] ;

var
    registers                  : regpack ;
    line, header               : st_typ ;      { print lines }
    blank_line                 : st_typ ;      { to add indentation }
    page_num,line_cnt, i, n, p : integer ;     { counters }
    indent, spacing, max_lines : integer ;     { user-supplied }
    first_page, last_page      : integer ;     { user_supplied }
    fname                      : string[66] ;  { file name }
    ipt_file                   : text ;        { input file }
    ok                         : boolean ;     { whether file exists }
    reply                      : char ;        { to get user response }
    quit                       : boolean ;     { to flag when last page printed }

{ ----------------------------------------------------------------- }

function date_and_time : str14 ;
  { get DOS system date and time }

var
  year,
  month,day,
  hour,min  : string[2] ;

begin
  with registers do
    begin
      AX := $2A00 ;
      msdos(registers) ;
      str(CX-1900,year) ;
      str(DH,month) ;
      str(DL,day) ;
      AX := $2C00 ;
      msdos (registers) ;
      str(CH:2,hour) ;
      str(CL:2,min) ;
    end ;
  if  min[1] = ' ' then  min[1] := '0' ;
  if  (hour[1] = ' ')
  and (hour[2] = '0') then
      hour := '00' ;
  date_and_time := concat (month,'/',day,'/',year,' ',hour,':',min) ;
end ; { function getdate }

{ ----------------------------------------------------------------- }

procedure print_page_header ;
  { prints header line at top of each page -- revised, 11/17/84 }
    var
        i : integer ;
    begin
        page_num := page_num + 1 ;
        if page_num > last_page then
            quit := true
        else
          begin
            if page_num >= first_page then
              begin
                if page_num > first_page then
                    write (lst, formfeed) ;
                writeln (lst) ;
                write (lst, header) ;
                writeln (lst, page_num) ;
                writeln (lst) ;
                for i := 1 to spacing do
                    writeln (lst)
              end ;
            line_cnt := 3 + spacing
          end
    end ;  { proc print_page_header }

{ ----------------------------------------------------------------- }

procedure print (line : st_typ ; num_newlines : integer) ;
  { prints a line and the number of newlines indicated }
    var
        i : integer ;
    begin
        if line_cnt > max_lines then
            print_page_header ;
        if  (page_num >= first_page)
        and (page_num <= last_page) then
          begin
            write (lst,line) ;
            for i := 1 to num_newlines do
                writeln (lst)
          end ;
        line_cnt := line_cnt + num_newlines
    end ;  { proc print }
 
{ ----------------------------------------------------------------- }

procedure add_blanks (var st : st_typ ; num_blanks : integer) ;
  { appends the number of blanks indicated to the string }
    var
        i : integer ;
    begin
        for i := 1 to num_blanks do
            st := concat (st,' ')
    end ;  { proc add_blanks }

{ ----------------------------------------------------------------- }

function adjust_line (line : st_typ) : st_typ ;
  { Converts tabs to spaces and adds indentation by moving characters
    one by one from the input string to a work string.  If it encounters
    a tab character it expands the tab to the proper number of spaces.
    Finally, the indentation string is inserted in front of all the
    characters and the function returns the work string. }
    
    const
        tab = ^I ;
    var
        i            : integer ;    { loop counter }
        next_char    : integer ;    { where the next character goes
                                      in the work string }
        work_str     : st_typ ;     { work string to build adjusted line }
    begin
        work_str := '' ;
        next_char := 1 ;
        for i := 1 to length(line) do
            if not (line[i] = tab) then
              begin
                work_str := concat(work_str,line[i]) ;
                next_char := next_char + 1
              end
            else         { character is a tab -- convert to spaces }
                repeat
                    work_str := concat(work_str,' ') ;
                    next_char := next_char + 1
                until (next_char > 8) and ((next_char mod 8) = 1) ;
        insert (blank_line,work_str,1) ;
        adjust_line := work_str
    end ;  { --- proc adjust_line --- }

{ ----------------------------------------------------------------- }

begin { --- MAIN --- }
    while true do                            { endless loop }
      begin
        writeln ;
        writeln ('This prints a text file, paginated with header and DOS date & time.') ;
        writeln ('Please specify options --  <cr> on file name to cancel.') ;
        writeln ('Defaults are no indent, single spacing, 58 lines per page,') ;
        writeln ('start at first page, stop after last.') ;
        writeln ;

        repeat
            fname := '' ;                    { get file name }
            write   ('File name? ') ;
            readln  (fname) ;
            for n := 1 to length(fname) do
                fname[n] := upcase(fname[n]) ;
            if fname = '' then
                halt                         { --- Exit loop here --- }
            else
              begin
                assign (ipt_file,fname) ;
                {$i-}
                reset (ipt_file) ;
                {$i+}
                ok := (ioresult = 0) ;
                if not ok then
                  begin
                    writeln (bell,'File ',fname,' not found.') ;
                    fname := ''
                  end
              end
        until ok ;

        indent := 0 ;                        { get indentation }
        write   ('Number of spaces to indent? ') ;
        readln  (indent) ;
        if indent < 0 then indent := 0 ;
        blank_line := '' ;
        if not (indent = 0 ) then
            for i := 1 to indent do
                blank_line := concat (' ',blank_line) ;

        spacing := 0 ;                       { get spacing }
        write   ('Line spacing? ') ;
        readln  (spacing) ;
        if spacing < 1 then spacing := 1 ;

        max_lines := 0 ;                     { get page length }
        write   ('Max lines per page? ') ;
        readln  (max_lines) ;
        if max_lines < 1 then
            max_lines := 58 ;

        line := '' ;                         { get header }
        write  ('Header? ') ;
        readln (line) ;

        first_page := 0 ;                    { get first page to print }
        write ('Start at what page? ') ;
        readln (first_page) ;
        if first_page < 1 then
            first_page := 1 ;

        last_page := 0 ;                     { get last page to print }
        write ('Quit after what page? ') ;
        readln (last_page) ;
        if last_page < 1 then
            last_page := maxint ;

        header := blank_line ;               { build header line }
        header := concat(header,fname,'  ',line) ;
        if length(header) < 57 then
            add_blanks (header, 57 - length(header))
        else
            add_blanks (header,2) ;
        header := concat (header,date_and_time,' Page ') ;
        page_num := 0 ;
        line_cnt := maxint ;                 { force first page header }

        quit := false ;
        writeln ('Printing ',fname) ;
        while not (eof(ipt_file)) do         { print the text file }
          begin
            readln (ipt_file,line) ;
            if not (indent = 0) then         { add identation }
                line := adjust_line (line) ;
            repeat
                n := pos(formfeed,line) ;    { handle embedded formfeeds }
                if not (n = 0) then
                  begin
                    print (copy(line,1,n-1),spacing) ;
                    print_page_header ;
                    if quit then
                        goto 99 ;
                    delete (line,1,n) ;
                    for i := 1 to indent do
                        line := concat(' ',line) ;
                  end
            until n = 0 ;
            print  (line,spacing) ;

            if keypressed then               { check for premature exit }
              begin
                writeln ;
                write  ('+++ Quit now? (Y/N): ') ;
                readln (reply) ;
                if upcase(reply) = 'Y' then
                    goto 99
              end ;
            if quit then
                goto 99
          end ;

99:         write (lst,formfeed) ;
        writeln (bell,'Done!')
      end
end.
