;****************************************************************************
;*
;*                         SuperVGA Test Library
;*
;*                  Copyright (C) 1994 SciTech Software
;*                          All rights reserved.
;*
;* Filename:    $RCSfile: svga.asm $
;* Version:     $Revision: 1.3 $
;*
;* Language:    80386 Assembler
;* Environment: IBM PC Real Mode and 16/32 bit Protected Mode.
;*
;* Description: Assembly language support routines for the SuperVGA test
;*              library.
;*
;* $Id: svga.asm 1.3 1994/03/20 07:44:20 kjb release $
;*
;****************************************************************************

		IDEAL

INCLUDE "model.mac"             ; Memory model macros

header  svga                    ; Set up memory model

CRTC    EQU 3D4h        		; Port of CRTC registers

if flatmodel
		ES_BX	EQU EBX			; Near pointer is in EBX
else
		ES_BX	EQU	es:bx		; Far pointer is in ES:BX
endif

begdataseg	svga

		$EXTRN  _maxx,INTEGER
		$EXTRN  _maxy,INTEGER
		$EXTRN  _maxcolor,DWORD
		$EXTRN  _bytesperline,WORD
		$EXTRN  _pagesize,DWORD
		$EXTRN  _curBank,INTEGER
		$EXTRN  _bankSwitch,DWORD
		$EXTRN  _extendedflipping,BOOL
		$EXTRN  _bankAdjust,INTEGER
		$EXTRN  _writeBank,DWORD	; Relocated write bank routine
		$EXTRN  _readBank,DWORD 	; Relocated read bank routine
		$EXTRN	_screenPtr,DWORD	; Start of display memory

OriginOffset	dw	0			; Offset within video buffer
BankOffset      dw  0           ; Starting bank in video memory

enddataseg	svga

begcodeseg	svga

;----------------------------------------------------------------------------
; void putPixel16(int x,int y,long color)
;----------------------------------------------------------------------------
; Routine sets the value of a pixel in native VGA graphics modes.
;
; Entry:        x       -   X coordinate of pixel to draw
;               y       -   Y coordinate of pixel to draw
;               color   -   Color of pixel to draw
;
;----------------------------------------------------------------------------
procstart   __putPixel16

		ARG     x:INTEGER, y:INTEGER, color:DWORD

		enter_c	0

; Compute the pixel's address in video buffer

		mov     _ax,[y]
		mov     _bx,[x]
		mul     [_bytesperline]     ; DX:AX := y * BytesPerLine

		mov     cl,bl               ; CL := low-order byte of x

		shr     _bx,3               ; _BX := x/8
		add     bx,ax
		adc     dx,0                ; DX:BX := y*BytesPerLine + x/8
		add     bx,[OriginOffset]   ; DX:BX := byte offset in video buffer
		adc     dx,[BankOffset]

		cmp     dl,[BYTE _curBank]
		je      @@NoChange

		mov     _ax,_dx
		call    setBank

@@NoChange:
if flatmodel
		and		ebx,0FFFFh			; Mask out top 16 bits
		add		ebx,[_screenPtr]	; EBX -> start of video memory
else
		mov		es,[WORD _screenPtr+2]
endif
		mov     ah,1                ; AH := unshifted bit mask
        and     cl,7                ; CL := x & 7
        xor     cl,7                ; CL := # bits to shift left

; set Graphics Controller Bit Mask register

        shl     ah,cl               ; AH := bit mask in proper postion
        mov     dx,3CEh             ; GC address register port
        mov     al,8                ; AL := Bit Mask Register number
        out     dx,ax

; set Graphics Controller Mode register

        mov     ax,0205h            ; AL := Mode register number
                                    ; AH := Write mode 2 (bits 0,1)
                                    ;   Read mode 0 (bit 3)
        out     dx,ax

; set data rotate/Function Select register

        mov     ax,3                ; AL := Data Rotate/Func select reg #
        out     dx,ax

; set the pixel value

		mov     al,[ES_BX]          ; latch one byte from each bit plane
		mov     al,[BYTE color]     ; AL := pixel value
		mov     [ES_BX],al          ; update all bit planes

; restore default Graphics Controller registers

        mov     ax,0FF08h           ; default bit mask
        out     dx,ax

        mov     ax,0005             ; default mode register
        out     dx,ax

        mov     ax,0003             ; default function select
        out     dx,ax

		leave_c
		ret

procend     __putPixel16

;----------------------------------------------------------------------------
; void putPixel256(int x,int y,long color)
;----------------------------------------------------------------------------
; Routine sets the value of a pixel in native VGA graphics modes.
;
; Entry:        x       -   X coordinate of pixel to draw
;               y       -   Y coordinate of pixel to draw
;               color   -   Color of pixel to draw
;
;----------------------------------------------------------------------------
procstart   __putPixel256

		ARG     x:INTEGER, y:INTEGER, color:DWORD

		enter_c	0

		mov     _ax,[y]
        mul     [_bytesperline]
        add     ax,[WORD x]
        adc     dx,0                ; DX:AX := y * BytesPerLine + x
        add     ax,[OriginOffset]
        adc     dl,[BYTE BankOffset]; DL := bank number
        mov     bx,ax               ; BX := Offset in buffer
        cmp     dl,[BYTE _curBank]
        je      @@NoChange

		mov     _ax,_dx
        call    setBank

@@NoChange:
if flatmodel
		and		ebx,0FFFFh			; Mask out top 16 bits
		add		ebx,[_screenPtr]	; EBX -> start of video memory
else
		mov		es,[WORD _screenPtr+2]
endif
		mov     al,[BYTE color]
		mov     [ES_BX],al          ; Replace the pixel

		leave_c
		ret

procend     __putPixel256

;----------------------------------------------------------------------------
; void putPixel32k(int x,int y,long color)
;----------------------------------------------------------------------------
; Routine sets the value of a pixel in native VGA graphics modes.
;
; Entry:        x       -   X coordinate of pixel to draw
;               y       -   Y coordinate of pixel to draw
;               color   -   Color of pixel to draw
;
;----------------------------------------------------------------------------
procstart   __putPixel32k

		ARG     x:INTEGER, y:INTEGER, color:DWORD

		enter_c	0

		mov     _ax,[y]
        mul     [_bytesperline]
		mov     _bx,[x]
		shl     _bx,1
        add     ax,bx
        adc     dx,0                ; DX:AX := y * BytesPerLine + x * 2
        add     ax,[OriginOffset]
        adc     dl,[BYTE BankOffset]; DL := bank number
        mov     bx,ax               ; BX := Offset in buffer
        cmp     dl,[BYTE _curBank]
        je      @@NoChange

        mov     al,dl
        call    setBank

@@NoChange:
if flatmodel
		and		ebx,0FFFFh			; Mask out top 16 bits
		add		ebx,[_screenPtr]	; EBX -> start of video memory
else
		mov		es,[WORD _screenPtr+2]
endif
        mov     ax,[WORD color]
		mov     [ES_BX],ax          ; Replace the pixel

		leave_c
		ret

procend     __putPixel32k

;----------------------------------------------------------------------------
; void putPixel16m(int x,int y,long color)
;----------------------------------------------------------------------------
; Routine sets the value of a pixel in native VGA graphics modes.
;
; Entry:        x       -   X coordinate of pixel to draw
;               y       -   Y coordinate of pixel to draw
;               color   -   Color of pixel to draw
;
;----------------------------------------------------------------------------
procstart   __putPixel16m

		ARG     x:INTEGER, y:INTEGER, color:DWORD

		enter_c	0

		mov     _ax,[y]
        mul     [_bytesperline]
		mov     _bx,[x]
        add     ax,bx
        adc     dx,0
        shl     bx,1
        add     ax,bx
        adc     dx,0                ; DX:AX := y * BytesPerLine + x * 3
        add     ax,[OriginOffset]
        adc     dl,[BYTE BankOffset]; DL := bank number
        mov     bx,ax               ; BX := Offset in buffer
        cmp     dl,[BYTE _curBank]
        je      @@NoChange

        mov     al,dl
        call    setBank

@@NoChange:
if flatmodel
		and		ebx,0FFFFh			; Mask out top 16 bits
		add		ebx,[_screenPtr]	; EBX -> start of video memory
else
		mov		es,[WORD _screenPtr+2]
endif
		mov     al,[BYTE color]
		mov     [ES_BX],al          ; Replace the first byte
        cmp     bx,0FFFFh
        jne     @@NotSplit1

; Arrghh!! We have a case where a single pixel can be split across a
; bank boundary, if the bytes per line value is 1920. This can cause the
; machine to hang (and produce strange pixels).

        inc     dl
        mov     al,dl
        call    setBank         ; Change video banks

@@NotSplit1:
        mov     al,[BYTE color+1]
		mov     [ES_BX+1],al        ; Replace the middle
        cmp     bx,0FFFEh
        jne     @@NotSplit2

        inc     dl
        mov     al,dl
        call    setBank     ; Change video banks

@@NotSplit2:
        mov     al,[BYTE color+2]
		mov     [ES_BX+2],al        ; Replace the last byte

		leave_c
		ret

procend     __putPixel16m

;----------------------------------------------------------------------------
; void clear16(void)
;----------------------------------------------------------------------------
; Routine to clear the screen. Works even if the display contains more than
; one bank, so will work for 1024x768 and 1280x1024 video modes.
;----------------------------------------------------------------------------
procstart   __clear16

		enter_c	0

; Setup graphics controller

        mov     dx,3CEh             ; DX := Graphics Controller I/O port

        mov     ah,0                ; AH := Background color
        xor     al,al               ; AL := 0 (Set/Reset register number)
        out     dx,ax               ; load set/reset register

        mov     ax,0F01h            ; AH := 1111b (mask for Enable set/reset)
                                    ; AL := 1 (enable Set/reset reg number)
        out     dx,ax               ; load enable set/reset reg

		mov     _ax,[_maxy]
		inc     _ax
        mul     [_bytesperline]     ; DX:AX := number of bytes to fill
        mov     bx,ax               ; BX := bytes in last bank to fill
        mov     dh,dl               ; DH := number of full banks to fill

if flatmodel
		mov		edi,[_screenPtr]	; ES:EDI -> start of video buffer
else
		les		di,[_screenPtr]		; ES:DI -> start of video buffer
endif
		add		di,[OriginOffset]
		xor		_ax,_ax
		mov     ax,[BankOffset]     ; AX := starting bank number
        cld                         ; Moves go up in memory

        or      dh,dh               ; More than one bank to fill?
        jz      @@SingleBank        ; No, only fill a single bank

; Fill all of the full 64k banks first

@@OuterLoop:
		push	_di
        call    setBank
		mov     _cx,4000h            ; Need to set 4000h double words per bank
	rep stosd
		pop		_di
        inc     al
        dec     dh
        jnz     @@OuterLoop

; Now fill the last partial bank

@@SingleBank:
		call    setBank
		xor		_cx,_cx
		mov     cx,bx
		shr     _cx,2                ; _CX := number of double words to set
    rep stosd

; Restore graphics controller

        mov     dx,3CEh             ; DX := Graphics Controller I/O port
        xor     ax,ax               ; AH := 0, AL := 0
        out     dx,ax               ; Restore default Set/Reset register

        inc     ax                  ; AH := 0, AL := 1
        out     dx,ax               ; Restore enable Set/Reset register

		leave_c
		ret

procend     __clear16

;----------------------------------------------------------------------------
; void clear256(void)
;----------------------------------------------------------------------------
; Routine to clear the screen. Assumes pages begin on bank boundaries
; for simplicity of coding.
;----------------------------------------------------------------------------
procstart   __clear256

		enter_c	0

		mov     _ax,[_maxy]
		inc     _ax
        mul     [_bytesperline]     ; DX:AX := number of bytes to fill
        mov     bx,ax               ; BX := bytes in last bank to fill
        mov     dh,dl               ; DH := number of full banks to fill

if flatmodel
		mov		edi,[_screenPtr]	; EDI -> start of video memory
else
		les		di,[_screenPtr]		; ES:DI -> video buffer
endif
		add		di,[OriginOffset]
		mov     dl,[BYTE BankOffset]; DL := starting bank number
        cld                         ; Moves go up in memory

        or      dh,dh               ; More than one bank to fill?
        jz      @@SingleBank        ; No, only fill a single bank

; Fill all of the full 64k banks first

@@OuterLoop:
		push	_di
		mov     al,dl
        call    setBank
        xor     eax,eax             ; Clear to black
		mov     _cx,4000h           ; Need to set 4000h double words per bank
	rep stosd
		pop		_di
        inc     dl
        dec     dh
        jnz     @@OuterLoop

; Now fill the last partial bank

@@SingleBank:
        mov     al,dl
        call    setBank
		xor     eax,eax             ; Clear to black
		xor		_cx,_cx
        mov     cx,bx
		shr     _cx,2               ; _CX := number of double words to set
    rep stosd

		leave_c
		ret

procend     __clear256

;----------------------------------------------------------------------------
; void clear32k(void)
;----------------------------------------------------------------------------
; Routine to clear the screen. Assumes pages begin on bank boundaries
; for simplicity of coding.
;----------------------------------------------------------------------------
procstart   __clear32k

		enter_c	0

		mov     _ax,[_maxy]
		inc     _ax
        mul     [_bytesperline]     ; DX:AX := number of bytes to fill
        mov     bx,ax               ; BX := bytes in last bank to fill
        mov     dh,dl               ; DH := number of full banks to fill

if flatmodel
		mov		edi,[_screenPtr]	; EDI -> start of video memory
else
		les		di,[_screenPtr]		; ES:DI -> video buffer
endif
		add		di,[OriginOffset]
		mov     dl,[BYTE BankOffset]; DL := starting bank number
        cld                         ; Moves go up in memory

; Fill all of the full 64k banks first

@@OuterLoop:
		push	_di
        mov     al,dl
        call    setBank
        xor     eax,eax             ; Clear to black
		mov     _cx,4000h           ; Need to set 4000h double words per bank
	rep stosd
		pop		_di
		inc     dl
		dec     dh
		jnz     @@OuterLoop

; Now fill the last partial bank

		mov     al,dl
        call    setBank
		xor     eax,eax             ; Clear to black
		xor		_cx,_cx
        mov     cx,bx
		shr     _cx,2               ; _CX := number of double words to set
    rep stosd

		leave_c
		ret

procend     __clear32k

;----------------------------------------------------------------------------
; void clear16m(void)
;----------------------------------------------------------------------------
; Routine to clear the screen. Assumes pages begin on bank boundaries
; for simplicity of coding.
;----------------------------------------------------------------------------
procstart   __clear16m

		enter_c	0

		mov     _ax,[_maxy]
		inc     _ax
        mul     [_bytesperline]     ; DX:AX := number of bytes to fill
        mov     bx,ax               ; BX := bytes in last bank to fill
        mov     dh,dl               ; DH := number of full banks to fill

if flatmodel
		mov		edi,[_screenPtr]	; EBX -> start of video memory
else
		les		di,[_screenPtr]		; ES:DI -> video buffer
endif
		add		di,[OriginOffset]
		mov     dl,[BYTE BankOffset]; DL := starting bank number
        cld                         ; Moves go up in memory

; Fill all of the full 64k banks first

@@OuterLoop:
		push	_di
        mov     al,dl
        call    setBank
        xor     eax,eax             ; Clear to black
		mov     _cx,4000h           ; Need to set 4000h double words per bank
	rep stosd
		pop		_di
        inc     dl
        dec     dh
        jnz     @@OuterLoop

; Now fill the last partial bank

        mov     al,dl
        call    setBank
		xor     eax,eax             ; Clear to black
		xor		_cx,_cx
		mov     cx,bx
		shr     _cx,2               ; _CX := number of double words to set
    rep stosd

		leave_c
		ret

procend     __clear16m

ife flatmodel

;----------------------------------------------------------------------------
; void _copyTest16(void)
;----------------------------------------------------------------------------
; Routine to copy the top half of video memory to the bottom half of
; video memory. To ensure that we a moving across a bank boundary in
; 16 color modes, we copy the data to the second video page.
;----------------------------------------------------------------------------
procstart   __copyTest16

		enter_c	0
		push    ds					; Save registers

        mov     ax,[_maxy]
        inc     ax
        shr     ax,1                ; AX := (Yres+1) / 2
        mul     [_bytesperline]
        mov     cx,ax               ; CX := Number of bytes to move

; Set up graphics controller

        mov     dx,3CEh             ; DX := Graphics Controller address port
        mov     ax,0105h            ; AH := 1 (read mode 0, write mode 1)
                                    ; AL := 5 (Mode register number)
        out     dx,ax               ; Set up mode

        mov     di,[WORD _pagesize] ; ES:DI := offset into destination buffer
        mov     ax,[WORD _pagesize+2]
        add     di,cx
        adc     al,0
        call    setBank             ; Set the read/write bank number

        xor     si,si               ; DS:SI := offset into source buffer
        xor     ax,ax
        call    setReadBank         ; Set the read bank number

		mov		ax,[WORD _screenPtr+2]
		mov     ds,ax               ; DS:SI -> source buffer
        mov     es,ax               ; ES:DI -> destination buffer
        cld                         ; Moves go up in memory

    rep movsb                       ; Move all data in bank FAST!

; Restore default graphics controller state

        mov     ax,0005h            ; default mode register value
        out     dx,ax

		pop     ds
		leave_c
		ret

procend     __copyTest16

;----------------------------------------------------------------------------
; void _copyTest256(void)
;----------------------------------------------------------------------------
; Routine to copy the top half of video memory to the bottom half of
; video memory, to test moving data across bank boundaries using separate
; read/write banks. To simplify the coding we move the first 100 scan
; lines down to start at scanline 205. This ensure allows us to move data
; from bank 0 to bank 2 in 640x??? display modes.
;----------------------------------------------------------------------------
procstart   __copyTest256

		enter_c	0
		push    ds					; Save registers

        mov     ax,100
        mul     [_bytesperline]
        mov     cx,ax               ; CX := Number of bytes to move
        shr     cx,1                ; CX := Number of words to move

        mov     ax,205
        mul     [_bytesperline]
        mov     di,ax               ; DI := offset into destination bank
        mov     al,dl
        call    setBank             ; Set the read/write bank number

        xor     si,si               ; DS:SI := offset into source buffer
        xor     al,al
        call    setReadBank         ; Set the read bank number

		mov		ax,[WORD _screenPtr+2]
		mov     ds,ax               ; DS:SI -> source buffer
        mov     es,ax               ; ES:DI -> destination buffer
        cld                         ; Moves go up in memory

    rep movsw                       ; Move all data in bank FAST!

		pop     ds
		leave_c
		ret

procend     __copyTest256

endif

;----------------------------------------------------------------------------
; void setActivePage(int which)
;----------------------------------------------------------------------------
; Routine to set the video page for active output.
;
; Entry:        page    - Page number of page to use
;
;----------------------------------------------------------------------------
procstart   _setActivePage

		ARG     which:INTEGER

		enter_c	0

; Calculate 18 bit address of page in video memory

		xor     eax,eax
		mov     _ax,[which]         ; EAX := page number
		mul     [_pagesize]         ; EDX:EAX := result
		mov		[OriginOffset],ax	; Save video buffer offset
		shr     eax,16
		mov     [BankOffset],ax     ; Save video bank offset

		leave_c
		ret

procend     _setActivePage

;----------------------------------------------------------------------------
; void setVisualPage(int which)
;----------------------------------------------------------------------------
; Routine to set the visible video page.
;
; Entry:        page    - Page number of page to use
;
;----------------------------------------------------------------------------
procstart       _setVisualPage

		ARG     which:INTEGER

		enter_c	0

; Calculate 18 bit address of page in video memory

		xor     eax,eax
		mov     _ax,[which]         ; EAX := page number
		mul     [_pagesize]         ; EAX := starting address in memory
		mov     edx,eax
		shr     edx,16              ; DX:AX := starting address in memory

		cmp     [_extendedflipping],0
		je      @@VGAFlip           ; We have no extended page flipping

		div     [_bytesperline]     ; AX := starting scanline,
									; DX := starting byte
		mov     cx,dx
        cmp     [WORD _maxcolor],0Fh
        je      @@16Color
        cmp     [WORD _maxcolor],0FFh
        je      @@SetIt
        cmp     [WORD _maxcolor+2],0FFh
        je      @@16MColor

		shr     cx,1                ; CX := starting pixel in buffer
		jmp     @@SetIt

@@16Color:
		shl     cx,3                ; CX := starting pixel in buffer
		jmp     @@SetIt

@@16MColor:
		mov		bx,ax				; Preserve AX
		xor     dx,dx
		mov     ax,cx
		mov     cx,3
		div     cx
		mov     cx,ax               ; CX := starting pixel in buffer
		mov		ax,bx				; Restore AX

@@SetIt:
		mov     bx,ax               ; BX := starting scanline in buffer

; Wait for display enable to be active (active low), to be sure that
; both halves of the start address will take place in one frame.

		mov     dx,03DAh            ; DX := video status port

@@WaitDEVBE:
		in      al,dx
		test    al,1
		jnz     @@WaitDEVBE         ; Wait for Display Enable

		mov     ax,04F07h
		mov     dx,bx               ; DX := starting scanline number
		xor     bx,bx               ; BX := 0 - set display start
		int     10h                 ; Set the display start address
		jmp     @@Done

@@VGAFlip:
		mov     bx,ax               ; BX := bottom 16 bits of address


; Wait for display enable to be active (active low), to be sure that
; both halves of the start address will take place in one frame. We
; preload a few values here to save time after the DE has been
; detected.

		mov     cl,0Ch              ; CL := Start Address High register
		mov     ch,bh               ; CH := high byte of new address
		mov     bh,bl               ; BH := low byte of new address
		mov     bl,0Dh              ; BL := Start Address Low register
		mov     dx,03DAh            ; DX := video status port

@@WaitDEVGA:
		in      al,dx
		test    al,1
		jnz     @@WaitDEVGA         ; Wait for Display Enable

		cli
		mov     dx,03D4h            ; DX := CRTC I/O port (3D4h)
		mov     ax,bx
		out     dx,ax
		mov     ax,cx
		out     dx,ax
		sti

; Now wait for the start of the vertical sync, to ensure that the old
; page will be invisible before anything is drawn on it.

@@Done:
		mov     dx,03DAh            ; DX := video status port
@@WaitStartVert:
		in      al,dx               ; Wait for start of vertical retrace
		test    al,8
		jz      @@WaitStartVert

@@Exit:
		leave_c
		ret

procend     _setVisualPage

;----------------------------------------------------------------------------
; setBank   Sets the read/write bank from assembly language
;----------------------------------------------------------------------------
;
; Entry:        _AX  - New read/write bank number
;
; Exit:         _AX  - New read/write bank number
;
; Registers:    All preserved!
;
; Note that some VESA BIOSes and TSR's set the first window to be
; write only and the second window to be read only, so we need to set both
; windows for most common operations to the same value. The Universal
; VESA VBE sets both the read and write banks to the same value for
; Window A, and changed the read bank only for Window B, hence the second
; call is _not_ required when the Universal VESA VBE is installed. You can
; determine what the window does by looking at the WindowAAttributes in
; the SuperVGAInfo block returned by function 00h. You could use this
; information to optimise bank switching when using faster VBE's like
; the Universal VESA VBE (but I have no bothered to do that here).
;----------------------------------------------------------------------------
procstart   setBank

        mov     [_curBank],_ax      ; Save current write bank number
		cmp     [_writeBank],0
		je      @@VESABank
		call    [_writeBank]        ; Call relocated version
		ret

@@VESABank:
        push    _ax
        push    _bx
        push    _dx
		mul     [_bankAdjust]       ; Adjust to VESA granularity
        push    _ax
        mov     _dx,_ax             ; DX := bank number
        xor     _bx,_bx             ; BX := select window A
		cmp		[_bankSwitch],0
		je		@@UseInt10

		call    [_bankSwitch]       ; Set write window
        pop     _dx
        inc     _bx
		call    [_bankSwitch]       ; Set read window
        pop     _dx
        pop     _bx
        pop     _ax
		ret

; Use the int 10h interface if the bankSwitch routine is NULL, which will
; be the case in protected mode until VBE 2.0

@@UseInt10:
        mov     _ax,04F05h
		int		10h
        pop     _dx
        inc     _bx
        mov     _ax,04F05h
        pop     _dx
        pop     _bx
        pop     _ax
		ret

procend     setBank

;----------------------------------------------------------------------------
; setReadBank   Sets the read bank from assembly language
;----------------------------------------------------------------------------
;
; Entry:        AX  - New read bank number
;
; Exit:         AX  - New read bank number
;
; Registers:    All preserved!
;
;----------------------------------------------------------------------------
procstart   setReadBank

		mov     [_curBank],-1       ; Ensure banking will be re-loaded
		cmp     [_readBank],0
		je      @@VESABank
		call    [_readBank]         ; Call relocated version
		ret

@@VESABank:
        push    _ax
        push    _bx
        push    _dx
		mul     [_bankAdjust]       ; Adjust to VESA granularity
        mov     _dx,_ax             ; DX := bank number
        mov     _bx,1               ; BX := select window B
		cmp		[_bankSwitch],0
		je		@@UseInt10

		call    [_bankSwitch]       ; Set read window
        pop     _dx
        pop     _bx
        pop     _ax
		ret

; Use the int 10h interface if the bankSwitch routine is NULL, which will
; be the case in protected mode until VBE 2.0

@@UseInt10:
        mov     _ax,04F05h
		int		10h
        pop     _dx
        pop     _bx
        pop     _ax
        ret

procend     setReadBank

;----------------------------------------------------------------------------
; void setBank(int bank)
;----------------------------------------------------------------------------
; Sets the new read/write bank number from C
;----------------------------------------------------------------------------
procstart   _setBank

        ARG     bank:INTEGER

		push    _bp
		mov     _bp,_sp
        mov     _ax,[bank]
		call    setBank
		pop     _bp
        ret

procend     _setBank

;----------------------------------------------------------------------------
; void setReadBank(int bank)
;----------------------------------------------------------------------------
; Sets the new reading bank number from C
;----------------------------------------------------------------------------
procstart   _setReadBank

        ARG     bank:INTEGER

		push    _bp
		mov     _bp,_sp
        mov     _ax,[bank]
		call    setReadBank
		pop     _bp
        ret

procend     _setReadBank

endcodeseg	svga

		END
