/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tevent.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Module to implement the mouse and keyboard event collection
*				queue routines for the toolkit.
*
* $Id: tevent.cpp 1.2 1994/03/09 11:50:31 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tevent.hpp"
#include <string.h>

/*--------------------------- Global Variables ----------------------------*/

TEventQueue	eventQueue;

/*----------------------------- Implementation ----------------------------*/

TEventQueue::TEventQueue()
/****************************************************************************
*
* Function:		TEventQueue::TEventQueue
*
* Description:	Constructor for the event queue. We simply initialise the
*				event gathering subsystem, and set a few internal flags
*				to zero.
*
****************************************************************************/
{
	EVT_init();

	// Default values for double click speed and mouse auto repeat rate

	doubleClick = 8;
	autoRepeat = 0;
	autoDelay = 6;
	doubleClickThresh = 5;
	firstAuto = true;
	memset(&downMouse,0,sizeof(downMouse));
}

bool TEventQueue::getNext(TEvent& evt,ushort mask)
/****************************************************************************
*
* Function:		TEventQueue::getNext
* Parameters:	evt		- Place to store the event
*				mask	- Event mask
* Returns:		True if an event was found, false if no events pending.
*
* Description:	Returns the next instance of the event from the event
*				queue, removing it from the queue. In here we handle the
*				double click and mouse auto repeat events.
*
****************************************************************************/
{
	event	ev;

	if (EVT_getNext(&ev,mask)) {
		// An event was pending in the event queue, so repackage it and
		// return it.

		evt.what = ev.what;
		if (evt.what & evKeyboard) {
			evt.key.keyCode = ev.message;
			evt.key.modifiers = ev.modifiers;
			evt.key.when = ev.when;
			return true;
			}
		if (evt.what & evMouse) {
			evt.mouse.buttons = ev.message;
			evt.mouse.when = ev.when;
			evt.mouse.modifiers = ev.modifiers;
			evt.mouse.doubleClick = false;
			evt.where.x = ev.where_x;
			evt.where.y = ev.where_y;

			if (evt.what & evMouseMove) {
				downMouse.where = evt.where;
				return true;
				}

			if ((evt.what & evMouseDown) == 0) {
				downMouse.what = evNothing;
				firstAuto = true;
				return true;
				}

			// Determine if the last mouse event was a double click event

			TPoint diff(evt.where - downMouse.where);
			diff.x = ABS(diff.x);	diff.y = ABS(diff.y);

			if (!downMouse.mouse.doubleClick
				&& (evt.mouse.buttons == downMouse.mouse.buttons)
				&& (evt.mouse.when - downMouse.mouse.when) <= doubleClick
				&& (diff.x <= doubleClickThresh)
				&& (diff.y <= doubleClickThresh)) {
				evt.mouse.doubleClick = true;
				}
			downMouse = evt;
			autoTicks = evt.mouse.when;
			return true;
			}
		if (evt.what & evTimerTick) {
			evt.timer.when = ev.when;
			return true;
			}
		if (evt.what & (evCommand | evBroadcast)) {
			evt.message.command = ev.modifiers;
			evt.message.infoLong = ev.message;
			return true;
			}
		if (evt.what & evRepaint)
			return true;
		}

	if ((mask & evMouseAuto) && (downMouse.what & evMouseDown)) {
		// No events are pending and the mouse is currently down, so
		// check to see if we should generate an auto mouse down event

		long ticks = (*((long*)FP(0x40,0x6C)));
		if (ticks - autoTicks >= autoRepeat + (firstAuto ? autoDelay : 0)) {
			evt = downMouse;
			evt.what = evMouseAuto;
			autoTicks = evt.mouse.when = ticks;
			firstAuto = false;
			return true;
			}
		}

	evt.what = evNothing;
	return false;
}

bool TEventQueue::peekNext(TEvent& evt,ushort mask)
/****************************************************************************
*
* Function:		TEventQueue::peekNext
* Parameters:	evt		- Place to store the event
*				mask	- Event mask
* Returns:		True if an event was pending, false if not.
*
* Description:	Returns a copy of the next event from the event queue, but
*				does not remove the event. Double click and auto mouse down
*				events are not returned.
*
****************************************************************************/
{
	event	ev;

	if (EVT_peekNext(&ev,mask)) {
		// An event was pending in the event queue, so repackage it and
		// return it.

		evt.what = ev.what;
		if (evt.what & evKeyboard) {
			evt.key.keyCode = ev.message;
			evt.key.modifiers = ev.modifiers;
			evt.key.when = ev.when;
			return true;
			}
		if (evt.what & evMouse) {
			evt.mouse.buttons = ev.message;
			evt.mouse.when = ev.when;
			evt.mouse.modifiers = ev.modifiers;
			evt.mouse.doubleClick = false;
			evt.where.x = ev.where_x;
			evt.where.y = ev.where_y;
			return true;
			}
		if (evt.what & evTimerTick) {
			evt.timer.when = ev.when;
			return true;
			}
		if (evt.what & (evCommand | evBroadcast)) {
			evt.message.command = ev.modifiers;
			evt.message.infoLong = ev.message;
			return true;
			}
		}

	evt.what = evNothing;
	return false;
}

bool TEventQueue::post(const TEvent& evt)
/****************************************************************************
*
* Function:		TEventQueue::post
* Parameters:	evt	- Event to post to the event queue
* Returns:		True if the event was posted, false if queue is full.
*
* Description:	Posts an event to the event queue. If the event is a double
*				click event, we simulate this by duplicating a mouse down
*				event.
*
****************************************************************************/
{
	ms_status	ms;

	memset(&ms,0,sizeof(ms));

	if (evt.what & evKeyboard)
		return EVT_post(evt.what,evt.key.keyCode,&ms,evt.key.modifiers);

	if (evt.what & evMouse) {
		ms.x = evt.where.x;
		ms.y = evt.where.y;
		if (!EVT_post(evt.what,evt.mouse.buttons,&ms,evt.mouse.modifiers))
			return false;
		if (evt.mouse.doubleClick)
			return EVT_post(evt.what,evt.mouse.buttons,&ms,evt.mouse.modifiers);
		}

	// To handle the posting of command and broadcast events to the
	// event queue, we store the info stuff as a long in the message
	// field, and the command for the message in the modifiers field.

	ms.x = ms.y = 0;
	ms.but_status = 0;
	return EVT_post(evt.what,evt.message.infoLong,&ms,
		evt.message.command);
}
