/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tview.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TView class.
*
* $Id: tview.cpp 1.2 1994/03/09 11:50:49 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tview.hpp"
#include "tgroup.hpp"

/*----------------------------- Implementation ----------------------------*/

TView::TView(const TRect& bounds)
	: owner(NULL), state(sfVisible), helpCtx(hcNoContext),
	  options(0), dragMode(0), growMode(0)
/****************************************************************************
*
* Function:		TView::TView
* Parameters:	bounds	- Bounding box for the view
*
* Description:	Constructor for the view, given a bounding box.
*
****************************************************************************/
{
	setBounds(bounds);
}

TView::~TView()
/****************************************************************************
*
* Function:		TView::~TView
*
* Description:	Destructor for a view. If the view has an owner, it
*				removes the view from the owner's group.
*
****************************************************************************/
{
	if (owner != NULL)
		owner->remove(this);
}

void TView::drawRectCoord(int left,int top,int right,int bottom, int width)
/****************************************************************************
*
* Function:		TView::drawRectCoord
* Parameters:	left	- Left coordinate of rectangle to draw
*				top		- Top coordinate of rectangle to draw
*				right	- Right coordinate of rectangle to draw
*				bottom	- Bottom coordinate of rectangle to draw
*				width	- Width of the rectangle (in sysLineWidth dimensions)
*
* Description:	Draws a rectangle using the current system line width. The
*				rectangle is completely contained within the bounding
*				rectangle, which is different to the way that the MGL draws
*				rectangles with large line widths.
*
****************************************************************************/
{
	int		w,h;

	MGL_getPenSize(&w,&h);
	width *= _MVIS_sysLineWidth;			// Adjust to system line width
	MGL_setPenSize(width,width);			// Adjust pen size

	bottom -= width-1;						// Compensate rectangle size
	right -= width-1;
	MGL_rectCoord(left,top,right,bottom);	// Draw the frame

	MGL_setPenSize(w,h);
}

void TView::drawLineCoord(int left,int top,int right,int bottom,int width)
/****************************************************************************
*
* Function:		TView::drawLineCoord
* Parameters:	left	- Left coordinate of rectangle to draw
*				top		- Top coordinate of rectangle to draw
*				right	- Right coordinate of rectangle to draw
*				bottom	- Bottom coordinate of rectangle to draw
*				width	- Width of the rectangle (in sysLineWidth dimensions)
*
* Description:	Draws a line with the specified with adjusted to the current
*				system line width units. The line will be placed below and
*				to the right of the coordinates which is the default for
*				the MGL.
*
****************************************************************************/
{
	int		w,h;

	MGL_getPenSize(&w,&h);
	width *= _MVIS_sysLineWidth;			// Adjust to system line width
	MGL_setPenSize(width,width);			// Adjust pen size
	MGL_lineCoord(left,top,right,bottom);	// Draw the line
	MGL_setPenSize(w,h);
}

void TView::setupViewport() const
/****************************************************************************
*
* Function:		TView::setupViewport
*
* Description:	Sets up the viewport to that of the inner part of the
*				view, for drawing objects within the view and for
*				simplified event processing.
*
****************************************************************************/
{
	MGL_pushViewport();
	MGL_setRelViewport(bounds);
}

void TView::setupOwnerViewport() const
/****************************************************************************
*
* Function:		TView::setupOwnerViewport
*
* Description:	Sets the viewport to the owner's viewport, regardless of
*				what the current viewport is already set to. This requires
*				traversing the view tree, so is slower than the above
*				routine but can be used to draw something in a view at
*				any time (and hence must be used with care).
*
****************************************************************************/
{
	MGL_pushViewport();

	if (owner) {
		TRect	view;
		TPoint	d;

		owner->findGlobalDelta(d);
		owner->getExtent(view);
		view.offset(d);
		MGL_setViewport(view);
		}
	else
		MGL_setViewport(bounds);
}

void TView::findGlobalDelta(TPoint& d)
/****************************************************************************
*
* Function:		TView::findGlobalDelta
* Parameters:   d	- Place to store the delta coordinate
*
* Description:	Computes the delta value to convert a global coordinate
*				to a local coordinate (or vice versa), by traversing
*				the currently active view tree. This is different
*				to using the globalToLocal()/localToGlobal() calls, as
*				these calls assume that the current viewport is set to
*				that containing the view.
*
****************************************************************************/
{
	if (owner) {
		TPoint	newd;
		owner->findGlobalDelta(newd);
		d += newd;
		}
	d += bounds.topLeft;
}

TView *TView::findModalView()
/****************************************************************************
*
* Function:		TView::findModalView
* Returns:		Pointer to the currently active modal view (NULL if none).
*
* Description:	By default TView's cannot be a modal view.
*
****************************************************************************/
{
	return NULL;
}

void TView::idle()
/****************************************************************************
*
* Function:		TView::idle
*
* Description:	Routine called when nothing is happening. By default we
*				do nothing.
*
****************************************************************************/
{
}

cursor *TView::getCursor(const TPoint&)
/****************************************************************************
*
* Function:		TView::getCursor
* Returns:		Pointer to the cursor definition for the view.
*
* Description:	We return NULL, which means use the normal cursor defintion
*				by default. If this is overloaded to return a pointer
*				to another cursor definition, the cursor will be
*				changed to this when the cursor is within the bounds
*				of the view.
*
****************************************************************************/
{
	return NULL;
}

bool TView::valid(ushort)
/****************************************************************************
*
* Function:		TView::valid
* Parameters:	command	- cmValid, or command ending the modal view
* Returns:		True if the view is valid
*
****************************************************************************/
{
	return true;
}

void TView::handleEvent(TEvent& event,phaseType)
/****************************************************************************
*
* Function:		TView::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Default view event handling mechanism. If the event is
*				a mouse down event with the left button, and the view
*				is selectable, then the view is selected.
*
****************************************************************************/
{
	if (event.what == evMouseDown && (event.mouse.buttons & mbLeftButton)
			&& !(state & (sfFocused | sfDisabled))
			&& (options & ofSelectable)) {
		// The view is selectable but not currently selected or disabled,
		// so select the view.

		select();

		// If the ofFirstClick flag is not set, then we clear the event
		// so that it will not be handled by the view as a mouse down.

		if (!(options & ofFirstClick))
			clearEvent(event);
		}
}

void TView::putEvent(TEvent& event)
/****************************************************************************
*
* Function:		TView::putEvent
* Parameters:	event	- Event to post
*
* Description:	Simply calls the owing viewing to store the event. If this
*				is not overridden by any class along the way, it eventually
*				calls TProgram::putEvent() which will simply store it as
*				the next pending event. It will _not_ be stored in the
*				event queue proper, you will need to call
*				eventQueue.post() to do this (for macro expansion etc).
*
****************************************************************************/
{
	if (owner)
		owner->putEvent(event);
}

bool TView::getEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TView::getEvent
* Parameters:	event	- Place to store the event
*				mask	- Event mask for obtaining events
* Returns:		True if an event was pending, false if not.
*
* Description:	Simply calls the owner to get the event. If this is not
*				overridden by any class along the way, it eventually gets
*				back to TProgram::getEvent which will return the next
*				pending event (it will always look for the event posted
*				with putEvent() first, then looks in the event queue).
*
****************************************************************************/
{
	if (owner)
		return owner->getEvent(event,mask);
	return false;
}

bool TView::peekEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TView::peekEvent
* Parameters:	event	- Place to store the event
*				mask	- Event mask for obtaining events
* Returns:		True if an event is pending, false if not.
*
* Description:	Simply calls the owner to peek the event. If this is not
*				overridden by any class along the way, it eventually gets
*				back to TProgram::getEvent which will return the next
*				pending event (it will always look for the event posted
*				with putEvent() first, then looks in the event queue).
*
*				The event is not removed if it was pending.
*
****************************************************************************/
{
	if (owner)
		return owner->peekEvent(event,mask);
	return false;
}

void TView::show()
/****************************************************************************
*
* Function:		TView::show
*
* Description:	Makes the view visible, by setting the visible flag.
*
****************************************************************************/
{
	if (!(state & sfVisible))
		setState(sfVisible,true);
}

void TView::hide()
/****************************************************************************
*
* Function:		TView::hide
*
* Description:	Makes the view not visible, by resetting the visible flag.
*
****************************************************************************/
{
	if (state & sfVisible)
		setState(sfVisible,false);
}

void TView::enable()
/****************************************************************************
*
* Function:		TView::enable
*
* Description:	Enables the view to accept events.
*
****************************************************************************/
{
	if (state & sfDisabled)
		setState(sfDisabled,false);
}

void TView::disable()
/****************************************************************************
*
* Function:		TView::disable
*
* Description:	Disables the view from accepting events, and will cause the
*				visual style of it to be updated.
*
****************************************************************************/
{
	if (!(state & sfDisabled))
		setState(sfDisabled,true);
}

void TView::redraw()
/****************************************************************************
*
* Function:		TView::redraw
*
* Description:	Performs a redraw operation for the view. We do this
*				by obtaining the current clipping rectangle for the
*				view, and request a redraw for the view with that
*				clipping rectangle.
*
*				If the region is non-rectangular, then we break up the
*				redraw into a number of draw operations with the smallest
*				possible clipping rectangles.
*
****************************************************************************/
{
	TRect	clip(bounds);

	if (getClipRect(clip)) {
		draw(clip);
		}
}

void TView::setBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TView::setBounds
* Parameters:	bounds	- New bounding rectangle for the view
*
* Description:	Sets the bounding rectangle for the view. This may need to
*				be overridden by derived classes to handle the case when
*				the view is moved when inserted into a group.
*
****************************************************************************/
{
	TView::bounds = bounds;
	size = bounds.botRight - bounds.topLeft;
}

void TView::moveTo(int x,int y)
/****************************************************************************
*
* Function:		TView::moveTo
* Parameters:	x,y	- New position to move view to
*
****************************************************************************/
{
	bounds.moveTo(x,y);
}

bool TView::getClipRect(TRect& c) const
/****************************************************************************
*
* Function:		TView::getClipRect
* Parameters:	c	- Place to store the clip rectangle
* Returns:		True if the clip rectangle is non-empty.
*
* Description:	Obtains the current clip rectangle for the view. This
*				clip rectangle will be relative to the owner's viewport.
*
****************************************************************************/
{
	if ((state & sfVisible) && (state & sfExposed)) {
		TRect	ownerClip(bounds);

		if (owner) {
			// Obtain the owner's clip rectangle relative to the owners'
			// owner's view, and make relative to the owner's view.

			ownerClip.offset(owner->getBounds().topLeft);
			if (!owner->getClipRect(ownerClip))
				return false;
			ownerClip.offset(-owner->getBounds().left(),
							 -owner->getBounds().top());
			}

        c &= ownerClip;
		return !c.isEmpty();
		}
	else
		return false;
}

void TView::invalidRect(TRect& rect,bool global)
/****************************************************************************
*
* Function:		TView::invalidRect
* Parameters:	rect	- Rectangle to invalidate.
*				global	- True if repaint event should be performed globally
*
* Description:	Invalidates the specified rectangle, requiring it to be
*				refreshed. If the view has an owner, we do a recursive
*				call to the owner with the rectangle moved relative to the
*				owner's viewport. Eventually this will get to the
*				TProgram view (hopefully), which will save the invalid
*				rectangle for the pending repaint event (posted by
*				the TProgram object).
*
****************************************************************************/
{
	if (owner && !(state & sfLockRepaint) && (state & sfVisible)) {
		rect.offset(bounds.topLeft);
		owner->invalidRect(rect,global);
		}
}

void TView::repaint(const TRect& rect)
/****************************************************************************
*
* Function:		TView::repaint
* Parameters:	rect	- Rectangle to repaint
*
* Description:	Simple invalidates the specified rectangle. Note that the
*				invalidRect routine must be able to modify the rectangle
*				so we create a local copy.
*
****************************************************************************/
{
	TRect r(rect);
	invalidRect(r,false);
}

void TView::globalRepaint(const TRect& rect)
/****************************************************************************
*
* Function:		TView::globalRepaint
* Parameters:	rect	- Rectangle to repaint
*
* Description:	Simple invalidates the specified rectangle. Note that the
*				invalidRect routine must be able to modify the rectangle
*				so we create a local copy.
*
****************************************************************************/
{
	TRect r(rect);
	invalidRect(r,true);
}

bool TView::includes(const TPoint& p) const
/****************************************************************************
*
* Function:		TView::includes
* Parameters:	p	- Point to check for inclusion
* Returns:		True if the point is within the view's bounds
*
* Description:	Determines if the 'global' coordinate p is within the
*				bounds of the view. It is assume by this stage that the
*				current viewport is set to that of the owner.
*
****************************************************************************/
{
	Point temp(p);
	globalToLocal(temp);
	bool result = bounds.includes(temp);
	return result;
}

void TView::setState(ushort aState,bool set)
/****************************************************************************
*
* Function:		TView::setState
* Parameters:	aState	- State flag/s to set
*				set		- True if flag should be set, false if cleared
*
****************************************************************************/
{
	// Disabled views cannot become active

	if ((aState & sfActive) && (state & sfDisabled))
		return;

	if (set)
		state |= aState;
	else
		state &= ~aState;

	if (owner) {
		if (aState & sfVisible) {
			if (owner->state & sfExposed)
				setState(sfExposed,set);
			}
		if (aState & sfFocused) {
			message(owner,evBroadcast,
				(set ? cmReceivedFocus : cmReleasedFocus), this);
			}
		if (aState & sfDisabled)
			repaint();
		}
}

void TView::setOption(ushort aOption,bool set)
/****************************************************************************
*
* Function:		TView::setOption
* Parameters:	aOption	- Option flag/s to set
*				set		- True if flag should be set, false if cleared
*
****************************************************************************/
{
	if (set)
		options |= aOption;
	else
		options &= ~aOption;
}

void TView::select()
/****************************************************************************
*
* Function:		TView::select
*
* Description:	Selects the view by asking the views owner to select it.
*
****************************************************************************/
{
	if (owner != NULL)
		owner->select(this);
}

TPalette& TView::getPalette() const
/****************************************************************************
*
* Function:		TView::getPalette
* Returns:		Reference to the default palette for views.
*
****************************************************************************/
{
	static char		ch = 0;
	static TPalette	palette(&ch,0);
	return palette;
}

color_t TView::getColor(uchar index)
/****************************************************************************
*
* Function:		TView::getColor
* Parameters:	index	- Index of the color to map
* Returns:		Actual color value for the current video mode
*
* Description:	Attempts to map a color index value for the current view
*				to a real color value supported by the current video mode.
*				If the color is out of range, we return 0 (generally black).
*
****************************************************************************/
{
	if (index == 0)
		return BLACK;

	// Continually map the color value from the current view up through
	// all the owners until we have no more owner. When we get there,
	// we look up the color value in the global SystemPalette, which is
	// contains color values specific to the current video mode.

	TView	*view = this;
	do {
		TPalette& p = view->getPalette();
		if (p[0] != 0) {
			if (index > p[0])
				return BLACK;
			index = p[index];
			if (index == 0)
				return BLACK;
			}
		view = view->owner;
		} while (view != NULL);

    return MGL_realColor(index);
}

ushort TView::getHelpCtx() const
/****************************************************************************
*
* Function:		TView::getHelpCtx
*
* Description:	Returns the help context number for the view.
*
****************************************************************************/
{
	return helpCtx;
}

