/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tbutton.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TButtonBase and TButton classes.
*
* $Id: tbutton.cpp 1.2 1994/03/09 11:50:31 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tbutton.hpp"
#include "tgroup.hpp"
#include "tmouse.hpp"
#include "tfontmgr.hpp"
#include "techlib.hpp"

/*----------------------------- Implementation ----------------------------*/

TButtonBase::TButtonBase(const TRect& bounds,ushort command,ushort flags,
	ushort borderWidth)
	: TView(bounds), command(command), flags(flags),
	  borderWidth(borderWidth * _MVIS_sysLineWidth)
/****************************************************************************
*
* Function:		TButtonBase::TButtonBase
* Parameters:	bounds		- Bounding rectangle for the button
*				command		- Command code to send when activated
*				flags		- Flags for the button
*				borderWidth	- Width of the button border
*
* Description:	Constructor for the TButtonBase class.
*
****************************************************************************/
{
	amDefault = ((flags & bfDefault) != 0);
	options |= ofSelectable | ofFirstClick | ofPreProcess | ofPostProcess;

	// *** Disable the button if command is not enabled???
}

void TButtonBase::activate()
/****************************************************************************
*
* Function:		TButtonBase::activate
*
* Description:	This routine gets called when the button press was accepted,
*				so we broadcast the command for the button.
*
****************************************************************************/
{
	// Signal all views in the current owner's view to record their
	// history configurations.

	message(owner,evBroadcast,cmRecordHistory);

	if (flags & bfBroadcast)
		message(owner,evBroadcast,command,this);
	else {
		// Post the event to the event queue to be handled by the
		// focused view.

		TEvent event;
		event.what = evCommand;
		event.message.command = command;
		event.message.infoPtr = this;
		putEvent(event);
		}
}

void TButtonBase::handleEvent(TEvent& event,phaseType)
/****************************************************************************
*
* Function:		TButtonBase::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for buttons. Buttons can only be
*				pressed by the logical left mouse button.
*
****************************************************************************/
{
	switch (event.what) {
		case evMouseDown:
			if (!(event.mouse.buttons & mbLeftButton))
				break;

			// We have a left mouse down within the button, so track the
			// mouse waiting for the pending mouse up event. We turn
			// mouse movement event collection on while doing this.

			bool 	oldMove = eventQueue.mouseMove(true);
			bool 	down = true,done = false;
			TRect	clip;

			drawBody(bounds,down);
			do {
				getEvent(event);
				switch (event.what) {
					case evMouseMove:
						if (down != includes(event.where))
							drawBody(bounds,down = !down);
						break;
					case evMouseUp:
						if (event.mouse.buttons & mbLeftButton)
							done = true;
						break;
					}
				} while (!done);

			// Pending mouse up has been caught, so if the button was
			// still down, activate it and pop it back up again.

			if (down) {
				activate();
				drawBody(bounds,false);
				}
			clearEvent(event);
			eventQueue.mouseMove(oldMove);
			break;
		case evBroadcast:
			// We have a broadcast event,

			switch (event.message.command) {
				case cmDefault:
					// The default message is broadcast. If this button
					// is the default, then activate it.

					if (amDefault) {
						activate();
						clearEvent(event);
						}
					break;
				case cmGrabDefault:
				case cmReleaseDefault:
					// The default view for the group has changed. If it
					// was released and we are the normal default button,
					// reset ourselves as the default. If we are normally
					// the default and the default is being grabbed,
					// relinquish the default for the moment.

					if (flags & bfDefault || amDefault) {
						amDefault = (event.message.command == cmReleaseDefault);
						PRECONDITION(owner != NULL);
						TRect r(bounds);
						r.inset(-_MVIS_sysLineWidth,-_MVIS_sysLineWidth);
						owner->repaint(r);
						}
					break;

				// Need to check for the button becoming disabled if the
				// command it must post gets disabled!!
				}
			break;
		}}

void TButtonBase::drawBody(const TRect& clip,bool depressed)
/****************************************************************************
*
* Function:		TButtonBase::drawBody
* Parameters:	clip		- Clipping rectangle to draw the button with
*				depressed	- True if button is depressed
*
* Description:	Internal routine for the button class to draw the button
*				body in the current state. This can be called by all
*				subclasses of button to draw a default button body.
*
*				Note that this routine draws the button in the context
*				of the owner's viewport.
*
****************************************************************************/
{
	mouse.obscure();

	MGL_setClipRect(clip);

	// Clear the button background

	MGL_setColor(getColor(depressed ? 3 : 4));
	MGL_fillRect(bounds);

	// Draw the border around the button

	MGL_setBorderColors(getColor(1),getColor(2));
	MGL_drawBorderCoord(bounds.left()+_MVIS_sysLineWidth,
		bounds.top()+_MVIS_sysLineWidth,
		bounds.right()-_MVIS_sysLineWidth,bounds.bottom()-_MVIS_sysLineWidth,
		(depressed ? BDR_INSET : BDR_OUTSET),borderWidth-_MVIS_sysLineWidth);

	if (borderWidth > _MVIS_sysLineWidth) {
		MGL_setColor(getColor(8));
		if (amDefault) {
			drawRect(bounds);
			drawLineCoord(bounds.left(),bounds.top()-_MVIS_sysLineWidth,
				bounds.right()-_MVIS_sysLineWidth,
				bounds.top()-_MVIS_sysLineWidth);
			drawLineCoord(bounds.right(),bounds.top(),bounds.right(),
				bounds.bottom()-_MVIS_sysLineWidth);
			drawLineCoord(bounds.left(),bounds.bottom(),
				bounds.right()-_MVIS_sysLineWidth,bounds.bottom());
			drawLineCoord(bounds.left()-_MVIS_sysLineWidth,bounds.top(),
				bounds.left()-_MVIS_sysLineWidth,
				bounds.bottom()-_MVIS_sysLineWidth);
			}
		else {
			drawLineCoord(bounds.left()+_MVIS_sysLineWidth,bounds.top(),
				bounds.right()-1-_MVIS_sysLineWidth,bounds.top());
			drawLineCoord(bounds.right()-_MVIS_sysLineWidth,
				bounds.top()+_MVIS_sysLineWidth,
				bounds.right()-_MVIS_sysLineWidth,
				bounds.bottom()-1-_MVIS_sysLineWidth);
			drawLineCoord(bounds.left()+_MVIS_sysLineWidth,
				bounds.bottom()-_MVIS_sysLineWidth,
				bounds.right()-1-_MVIS_sysLineWidth,
				bounds.bottom()-_MVIS_sysLineWidth);
			drawLineCoord(bounds.left(),bounds.top()+_MVIS_sysLineWidth,
				bounds.left(),bounds.bottom()-1-_MVIS_sysLineWidth);
			}
		}

	mouse.unobscure();
}

void TButtonBase::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TButtonBase::draw
*
* Description:	Default routine to draw the button. Simply calls drawBody(),
*				so should be overridden by any subclasses.
*
****************************************************************************/
{
	drawBody(clip,false);
}

void TButtonBase::redraw()
/****************************************************************************
*
* Function:		TButtonBase::redraw
*
* Description:	Performs a redraw operation for the button. We need to
*				modify this to allow for drawing the default button
*				border around the outside of the default button.
*
****************************************************************************/
{
	bounds.inset(-_MVIS_sysLineWidth,-_MVIS_sysLineWidth);
	TRect clip(bounds);

	if (getClipRect(clip)) {
		bounds.inset(_MVIS_sysLineWidth,_MVIS_sysLineWidth);
		draw(clip);
		}
	else
		bounds.inset(_MVIS_sysLineWidth,_MVIS_sysLineWidth);
}

TPalette& TButtonBase::getPalette() const
/****************************************************************************
*
* Function:		TButtonBase::getPalette
* Returns:		Pointer to the standard palette for buttons
*
****************************************************************************/
{
	static char cpButton[] = {1,2,17,18,19,20,21,22};
	static TPalette palette(cpButton,sizeof(cpButton));
	return palette;
}

void TButtonBase::setDefault(bool enable)
/****************************************************************************
*
* Function:		TButtonBase::setDefault
* Parameters:	enable	- True if button should be enabled as default
*
* Description:	Makes the button the current default button for the
*				view. Allows buttons that are not normally the default
*				to grab the default behaviour or release it.
*
****************************************************************************/
{
	if (!(flags & bfDefault)) {
		// We are not already the standard default, so change states
		// and notify all other buttons of the change

		message(owner,evBroadcast,
			(enable ? cmGrabDefault : cmReleaseDefault),this);
		amDefault = enable;

		// Note that when changing the default button, we must also
		// update the default button outline which lies outside of the
		// normal button bounds, so we request a repaint event that
		// will repaint the entire button.

		PRECONDITION(owner != NULL);
		TRect r(bounds);
		r.inset(-_MVIS_sysLineWidth,-_MVIS_sysLineWidth);
		owner->repaint(r);
		}

	//**** Need to override setstate to handle the view being disabled
	// enabled etc.
}

void TButtonBase::setState(ushort aState,bool set)
/****************************************************************************
*
* Function:		TButtonBase::setState
* Parameters:	aState	- State flag to set
*				set		- True if flag should be set, false if cleared
*
****************************************************************************/
{
	TView::setState(aState,set);

	if (aState & sfFocused) {
		// The button was just focussed or unfocused, so change the
		// default status for the button (focussed button's automatically
		// become the temprary default).

		setDefault(set);
		}
}

TButton::TButton(const TRect& bounds,const char *title,ushort command,
	ushort flags,ushort borderWidth)
	: TButtonBase(bounds,command,flags,borderWidth),
	  title(newHotStr(title,hotChar,hotIndex))
/****************************************************************************
*
* Function:		TButton::TButton
* Parameters:	bounds		- Bounding rectangle for the button
*				command		- Command code to send when activated
*				flags		- Flags for the button
*				borderWidth	- Width of the button border
*
* Description:
*
****************************************************************************/
{
	setBounds(bounds);				// Calculate the bounds again
}

TButton::~TButton()
/****************************************************************************
*
* Function:		TButton::~TButton
*
* Description:	Destructor for the TButton class. Deletes the button's
*				title.
*
****************************************************************************/
{
	delete title;
}

void TButton::setBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TButton::setBounds
* Parameters:	bound	- New bounding box for the button
*
* Description:	Sets the bounding box for the button. We pre-calculate
*				the position of the text within the button here.
*
****************************************************************************/
{
	TButtonBase::setBounds(bounds);

	// Set the current font and size, and obtain the fonts metrics

	metrics	m;
	fontManager.useFont(fmSystemFont);
	MGL_getFontMetrics(&m);

	TextJust	old;
	tjust.use();

	// Compute the location to draw the text at

	int hjust = (flags & bfLeftJust ? LEFT_TEXT :
				(flags & bfRightJust ? RIGHT_TEXT : CENTER_TEXT));
	int vjust = (flags & bfTopJust ? TOP_TEXT :
				(flags & bfBottomJust ? BOTTOM_TEXT : CENTER_TEXT));
	tjust.setJustification(hjust,vjust);
	tjust.use();

	TRect r(bounds);
	r.moveTo(0,0);
	r.inset(borderWidth+_MVIS_sysLineWidth,borderWidth+_MVIS_sysLineWidth);

	int adjust = (fontManager.getCurrentFont() == fmFont8x8);
	start.x = 	(hjust == LEFT_TEXT ? 	 r.left() :
				(hjust == CENTER_TEXT ? (r.left() + r.right())/2 :
										 r.right()-1));
	start.y =	(vjust == TOP_TEXT ? 	 r.top() :
				(vjust == CENTER_TEXT ? (r.top() + r.bottom()-adjust)/2 :
										 r.bottom()-1 + m.descent));

	// Compute the location of the hot character underscore, if present

	if (hotChar) {
		hot1 = start;
		MGL_underScoreLocation(&hot1.x,&hot1.y,title);
		char oldChar = title[hotIndex];
		((char*)title)[hotIndex] = '\0';
		hot1.x += MGL_textWidth(title);
		hot2 = hot1;
		MGL_getCharMetrics(oldChar,&m);
		hot2.x += m.fontWidth-1;
		((char*)title)[hotIndex] = oldChar;
		}

	old.use();
}

void TButton::drawBody(const TRect& clip,bool depressed)
/****************************************************************************
*
* Function:		TButton::draw
* Parameters:	clip		- Clipping rectangle for drawing the view
*				depressed	- True if button is depressed
*
* Description:	Draws the body of the button with the textual caption in
*				the correct place.
*
****************************************************************************/
{
	mouse.obscure();
	TButtonBase::drawBody(clip,depressed);	// Draw the blank body

	MGL_setColor(getColor(state & sfDisabled ? 7 :
		(state & sfFocused ? 6 : 5)));

	TRect	c(bounds);
	c.inset(borderWidth,borderWidth);
    c &= clip;
	MGL_setClipRect(c);						// Clip text to clip rect

	TextJust old;
	tjust.use();
	fontManager.useFont(fmSystemFont);
	MGL_drawStrXY(start.x + bounds.left() + depressed,
				  start.y + bounds.top() + depressed,title);

	if (hotChar)
		drawLineCoord(hot1.x + bounds.left() + depressed,
					  hot1.y + bounds.top() + depressed,
					  hot2.x + bounds.left() + depressed - _MVIS_sysLineWidth + 1,
					  hot2.y + bounds.top() + depressed);

	old.use();
	mouse.unobscure();
}

void TButton::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TButton::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for buttons. Buttons have associated
*				hot key characters.
*
****************************************************************************/
{
	TButtonBase::handleEvent(event,phase);

	if (phase != phFocused && event.what == evKeyDown) {
		// Check for hot key's here.

		if (phase == phPreProcess && !(event.key.modifiers & mdAlt))
			return;

		if (hotChar && event.key.charScan.charCode == hotChar) {
			activate();
			clearEvent(event);
			}
		}
}

void TButton::setTitle(const char *newTitle)
/****************************************************************************
*
* Function:		TButton::setTitle
* Parameters:	newTitle	- New title for the button
*
* Description:	Changes the title text for the button. We also repaint the
*				entire button to show the changed button text if the
*				button is currently visible.
*
****************************************************************************/
{
	delete title;
	title = newStr(newTitle);
	setBounds(bounds);
	repaint();
}
