/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tmenu.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TMenu class.
*
* $Id: tmenu.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tmenu.hpp"
#include "tgroup.hpp"
#include "tmouse.hpp"
#include "tfontmgr.hpp"
#include <alloc.h>

/*---------------------------- Global Variables ---------------------------*/

TPoint TMenu::arrow[7] = {TPoint(0,2),TPoint(3,2),TPoint(3,-1),TPoint(7,3),
						  TPoint(3,7),TPoint(3,5),TPoint(0,5)};

/*----------------------------- Implementation ----------------------------*/

TMenu::TMenu(const TRect& bounds)
	: TView(bounds), parent(NULL)
/****************************************************************************
*
* Function:		TMenu::TMenu
* Parameters:	bounds			- bounding box for the menu
*
* Description:	Constructor for the menu.
*
****************************************************************************/
{
	items.setDelta(10);			// Use a reasonable delta value
	current = -1;				// Default to first item in menu

	options |= ofPreProcess;	// Enable pre-processing of events
}

TMenu::~TMenu()
{
}

TMenu& TMenu::operator + (TMenuItem* item)
/****************************************************************************
*
* Function:		TMenu::operator +
* Parameters:	item	- Pointer to the item to insert
* Returns:		Reference to the menu the item was inserted into
*
****************************************************************************/
{
	items.add(item);
	if (item->isSubMenu())
		item->subMenu->parent = this;
	return *this;
}

void TMenu::putEvent(TEvent& event)
/****************************************************************************
*
* Function:		TMenu::putEvent
* Parameters:	event	- Event to post
*
* Description:	Climbs up the parent tree first, before calling the owner's
*				event routine (SubMenu's have no owners).
*
****************************************************************************/
{
	if (parent)
		parent->putEvent(event);
	else if (owner)
		owner->putEvent(event);
}

bool TMenu::getEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TMenu::getEvent
* Parameters:	event	- Place to store the event
* Returns:		True if an event was pending, false if not.
*
* Description:	Climbs up the parent tree first, before calling the owner's
*				event routine (SubMenu's have no owners).
*
****************************************************************************/
{
	if (parent)
		return parent->getEvent(event,mask);
	else if (owner)
		return owner->getEvent(event,mask);
	return false;
}

bool TMenu::peekEvent(TEvent& event,ushort mask)
/****************************************************************************
*
* Function:		TMenu::peekEvent
* Parameters:	event	- Place to store the event
* Returns:		True if an event is pending, false if not.
*
* Description:	Climbs up the parent tree first, before calling the owner's
*				event routine (SubMenu's have no owners).
*
****************************************************************************/
{
	if (parent)
		return parent->getEvent(event,mask);
	else if (owner)
		return owner->peekEvent(event,mask);
	return false;
}

void TMenu::invalidRect(TRect& rect,bool global)
/****************************************************************************
*
* Function:		TMenu::invalidRect
* Parameters:	rect	- Rectangle to invalidate.
*				global	- True if repaint event should be performed globally
*
* Description:	Overloaded to call the paren't routine if the parent exists,
*				otherwise we call the standard handling routine.
*
****************************************************************************/
{
	if (parent)
		parent->invalidRect(rect,global);
	else
		TView::invalidRect(rect,global);
}

int TMenu::findSelected(TPoint& where)
/****************************************************************************
*
* Function:		TMenu::findSelected
* Parameters:	where	- Selection point
* Returns:		Index of item that was selected (or closest one)
*
* Description:	Finds the item closest to the selection point. It is
*				assumed at this stage that the point is know to be contained
*				in the bounding rectangle for the entire menu.
*
****************************************************************************/
{
	for (int i = 0; i < items.numberOfItems(); i++) {
		TRect& b = items[i]->bounds;
		if (b.top() <= where.y && where.y < b.bottom())
			return items[i]->disabled ? -1 : i;
		}
	if (where.y < items[0]->bounds.top() && !items[0]->disabled)
		return 0;
	i = items.numberOfItems()-1;
	if (where.y >= items[i]->bounds.bottom() && !items[i]->disabled)
		return i;
	return -1;
}

int TMenu::findNextItem(int index)
/****************************************************************************
*
* Function:		TMenu::findNextItem
* Parameters:	index	- Menu item index to start search with
* Returns:		Index of the next non-disabled menu item
*
****************************************************************************/
{
	bool onceAround = false;
	do {
		if (++index >= items.numberOfItems()) {
			if (onceAround) {
				index = -1;
				break;
				}
			index = 0;
			onceAround = true;
			}
		} while (items[index]->disabled);
	return index;
}

int TMenu::findPrevItem(int index)
/****************************************************************************
*
* Function:		TMenu::findPrevItem
* Parameters:	index	- Menu item index to start search with
* Returns:		Index of the previous non-disabled menu item
*
****************************************************************************/
{
	bool onceAround = false;
	do {
		if (--index < 0) {
			if (onceAround) {
				index = -1;
				break;
				}
			index = items.numberOfItems()-1;
			onceAround = true;
			}
		} while (items[index]->disabled);
	return index;
}

bool TMenu::totalIncludes(TPoint& where)
/****************************************************************************
*
* Function:		TMenu::totalIncludes
* Parameters:	where	- Point to check for inclusion
* Returns:		True if point is included
*
* Description:	Checks to see if the point is within the bounds of the
*				current menu, or any of it's parents.
*
****************************************************************************/
{
	TMenu *menu = this;
	bool inside = false;
	while (menu) {
		if (menu->includes(where)) {
			inside = true;
			break;
			}
		menu = menu->parent;
		}
	return inside;
}

ushort TMenu::executeSubMenu(TEvent& event)
/****************************************************************************
*
* Function:		TMenu::executeSubMenu
* Parameters:	subMenu	- Pointer to the submenu to execute
* Returns:		Command causing menu to stop executing.
*
****************************************************************************/
{
	// Draw the menu and execute it.

	switch (event.what) {
		case evMouseUp:
			event.what = evKeyDown;
			event.key.keyCode = kbHome;
			putEvent(event);
			break;
		case evMouseDown:
		case evMouseMove:
			if (subMenuID == current)
				items[subMenuID]->subMenu->refresh(-1);
			items[current]->subMenu->setCurrent(-1);
			break;
		case evKeyDown:
			event.what = evCommand;
			event.message.command = cmMenu;
			putEvent(event);
			break;
		}

	displayMenu(current);
	return items[current]->subMenu->execute();
}

void TMenu::displayMenu(int selection)
/****************************************************************************
*
* Function:		TMenu::displayMenu
*
* Description:	Saves the area behind the current pull down menu and
*				display's it.
*
****************************************************************************/
{
	if (repaintSave) {
		if (subMenuID == selection)
			return;
		restoreMenu();
		}

	mouse.obscure();
	TMenu *subMenu = items[subMenuID = selection]->subMenu;
	long size = MGL_divotSize(subMenu->getBounds());
	save = NULL;
	if ((save = farmalloc(size)) != NULL)
		MGL_getDivot(subMenu->getBounds(),save);

	subMenu->draw(TRect(0,0,MGL_sizex()+1,MGL_sizey()+1));
	repaintSave = true;
	mouse.unobscure();
}

void TMenu::restoreMenu()
/****************************************************************************
*
* Function:		TMenu::restoreMenu
*
* Description:	Restore saved area behind pull down menu.
*
****************************************************************************/
{
	if (repaintSave) {
		if (save) {
			mouse.obscure();
			MGL_putDivot(save);
			mouse.unobscure();
			free((void*)save);
			}
		else {
			repaint(items[subMenuID]->subMenu->getBounds());
			TEvent event;
			getEvent(event);
			}
		}
	save = NULL;
	repaintSave = false;
	subMenuID = -1;
}

ushort TMenu::execute()
/****************************************************************************
*
* Function:		TMenu::execute
* Returns:		Command causing the menu to stop executing.
*
* Description:	Main interaction handling routine for the TMenu class. Here
*				we track the mouse in the menu and perform the selection
*				etc.
*
*				Is it assumed that the menu is visible when this routine
*				is called, and may only need refreshing.
*
****************************************************************************/
{
	TEvent	event;
	bool	done = false;
	bool	autoTracking = true;
	bool 	oldMove = eventQueue.mouseMove(true);
	ushort	result = cmValid;
	int		selection;

	selection = current;
	repaintSave = false;			// Nothing saved behind subMenu's yet
	subMenuID = -1;

	// The first event we recieve here is the original event that caused
	// the menu selection process to begin.

	while (!done) {
		getEvent(event);
		switch (event.what) {
			case evMouseDown:
			case evMouseUp:
				autoTracking = true;
				if (includes(event.where)) {
					selection = findSelected(event.where);
					if (selection != -1) {
						if (items[selection]->isSubMenu()) {
							refresh(selection);
							result = executeSubMenu(event);
							if (result != cmMenu)
								done = true;
							}
						else if (event.what == evMouseUp) {
							// The menu item was selected, so handle the
							// selection here.

							result = items[selection]->command;
							done = true;
							}
						else
							restoreMenu();
						}
					else {
						if (event.what == evMouseUp)
							done = true;
						restoreMenu();
						}
					}
				else {
					if (parent)
						putEvent(event);
					result = cmMenu;
					done = true;
					}
				break;
			case evMouseMove:
				// The mouse was moved, so if we are currently tracking
				// the position of the mouse cursor, check if we need
				// to update the current selection

				if (autoTracking) {
					if (includes(event.where)) {
						selection = findSelected(event.where);
						if (selection != -1 && items[selection]->isSubMenu()) {
							if (current != selection) {
								restoreMenu();
								refresh(selection);
								}
							result = executeSubMenu(event);
							if (result != cmMenu)
								done = true;
							}
						else
							restoreMenu();
						}
					else if (parent && parent->totalIncludes(event.where)) {
						putEvent(event);
						result = cmMenu;
						done = true;
						}
					else
						selection = -1;
					}
				break;
			case evKeyDown:
			case evKeyAuto:
				bool tracking = false;
				switch (event.key.keyCode) {
					case kbUp:
						selection = findPrevItem(selection);
						break;
					case kbDown:
						selection = findNextItem(selection);
						break;
					case kbLeft:
						if (items[selection]->isSubMenu() && subMenuID != -1)
							restoreMenu();
						else if (parent) {
							putEvent(event);
							result = cmMenu;
							done = true;
							}
						break;
					case kbRight:
						if (items[selection]->isSubMenu() && subMenuID == -1) {
							if ((result = executeSubMenu(event)) != cmMenu)
								done = true;
							}
						else if (parent) {
							putEvent(event);
							result = cmMenu;
							done = true;
							}
						break;
					case kbHome:
						selection = findNextItem(items.numberOfItems()-1);
						break;
					case kbEnd:
						selection = findPrevItem(0);
						break;
					case kbEnter:
					case kbGrayEnter:
						if (items[selection]->isSubMenu()) {
							if ((result = executeSubMenu(event)) != cmMenu)
								done = true;
							}
						else {
							result = items[selection]->command;
							done = true;
							}
						break;
					case kbEsc:
						done = true;
						break;
					default:
						// Check for global hotkey values here. We do this
						// by finding the granddaddy menu of all and check
						// for hotkeys from there.

						TMenu *menu = this;
						while (menu->parent)
							menu = menu->parent;

						if (menu->checkHotKey(event)) {
							done = true;
							break;
							}

						// Check for keyboard shortcut's that could be
						// valid for this menu (non-Alted ones).

						if ((selection = findShortCut(event)) != -1) {
							if (!items[selection]->disabled) {
								if (items[selection]->isSubMenu()) {
									refresh(selection);
									if ((result = executeSubMenu(event)) != cmMenu)
										done = true;
									}
								else {
									current = selection;
									result = items[selection]->command;
									done = true;
									}
								}
							}
						else {
							selection = current;
							tracking = autoTracking;
							}
						break;
					}
				autoTracking = tracking;
				break;
			case evCommand:
				// Must have received a cmMenu command to get here

				PRECONDITION(event.message.command == cmMenu);
				if (current == -1)
					selection = findNextItem(items.numberOfItems()-1);
				autoTracking = false;
				break;
			}

		// Refresh the state of the menu if it has changed

		if (!done && current != selection)
			refresh(selection);
		}

	restoreMenu();					// Restore stuff behind subMenu's
	eventQueue.mouseMove(oldMove);
	return result;
}

void TMenu::doSelection(TEvent& event)
/****************************************************************************
*
* Function:		TMenu::doSelection
* Parameters:	event	- Event causing the selection process to begin
*
* Description:	Sets up for a selection from the menu.
*
****************************************************************************/
{
	putEvent(event);			// Re-post the event to begin selection
	ushort command = execute();
	if (command != cmValid && command != cmMenu /* && commandEnabled(...) */) {
		event.what = evCommand;
		event.message.command = command;
		event.message.infoPtr = NULL;
		putEvent(event);
		}
	clearEvent(event);
}

int TMenu::findShortCut(TEvent& event)
/****************************************************************************
*
* Function:		TMenu::findShortCut
* Parameters:	event	- Event to check
* Returns:		Index of menu item with specified hot character
*
****************************************************************************/
{
	for (int i = 0; i < items.numberOfItems(); i++) {
		if (items[i]->hotChar &&
				items[i]->hotChar == event.key.charScan.charCode)
			return i;
		}
	return -1;
}

bool TMenu::checkHotKey(TEvent& event)
/****************************************************************************
*
* Function:		TMenu::checkHotKey
* Parameters:	event	- Event to check
* Returns:		True if command was posted
*
* Description:	Checks for a valid hot key combination, and posts the
*				appropriate command if one is found. In order to make the
*				hot key processing valid anywhere in the menu, we search
*				all subMenu's for hotkey values.
*
****************************************************************************/
{
	for (int i = 0; i < items.numberOfItems(); i++) {
		TMenuItem& item = *items[i];
		if (item.isSubMenu()) {
			if (item.subMenu->checkHotKey(event))
				return true;
			}
		else if (!item.disabled && item.hotKey == event.key) {
			event.what = evCommand;
			event.message.command = items[i]->command;
			event.message.infoPtr = NULL;
			putEvent(event);
			clearEvent(event);
			return true;
			}
		}
	return false;
}

void TMenu::handleEvent(TEvent& event,phaseType)
/****************************************************************************
*
* Function:		TMenu::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Default event handling routine for the menu class.
*
****************************************************************************/
{
	if (items.numberOfItems() == 0)
		return;

	switch (event.what) {
		case evMouseDown:
			doSelection(event);
			break;
		case evKeyDown:
			if (!checkHotKey(event)) {
				// Check for Alt-?? combination keyboard shortcut to start
				// menu selection.

				int index;
				if (event.key.modifiers & mdAlt) {
					if ((index = findShortCut(event)) != -1) {
						current = index;
						doSelection(event);
						}
					}
				}
			break;
		case evCommand:
			if (event.message.command == cmMenu)
				doSelection(event);
			break;
		case evBroadcast:
			// Handle change in active command set here!

			break;
		}
}

void TMenu::drawItem(int index)
/****************************************************************************
*
* Function:		TMenu::drawItem
* Parameters:	index	- Index of the menu item to draw
*
* Description:	Draws the menu item 'index'. If the item is the currently
*				selected item, we draw the item as selected, otherwise
*				we draw it as non-selected.
*
****************************************************************************/
{
	TMenuItem& item = *items[index];

	MGL_pushViewport();
	MGL_setRelViewport(item.bounds);

	int maxx = MGL_maxx();
	int maxy = MGL_maxy();
	int depressed = (index == current);
	int offsetx = 3 + depressed,offsety = 3 + depressed;

	// Highlight the item if it is currently selected

	if (index == current) {
		MGL_setColor(getColor(4));
		MGL_fillRectCoord(0,0,maxx+1,maxy+1);
		MGL_setBorderColors(getColor(1),getColor(2));
		drawBorderCoord(0,0,maxx+1,maxy+1,BDR_INSET,1);
		}

	if (item.name[0] == '-') {
		// Draw the separator item

		MGL_setColor(getColor(2));
		drawLineCoord(0,maxy/2,maxx-_MVIS_sysLineWidth+1,maxy/2);
		}
	else {
		// Draw the item's text

		MGL_setColor(getColor(item.disabled ? 6 : ((index == current) ? 7 : 5)));
		MGL_setTextJustify(LEFT_TEXT,TOP_TEXT);
		MGL_drawStrXY(offsetx,offsety,item.name);
		if (item.isSubMenu()) {
			// Draw an arrow to indicate that a submenu is present

			MGL_fillPolygon(7,arrow,maxx-10+depressed,maxy/2-4+depressed);
			}
		else if (item.hotKeyText) {
			MGL_setTextJustify(RIGHT_TEXT,TOP_TEXT);
			MGL_drawStrXY(maxx-3+depressed,offsety,item.hotKeyText);
			}

		// Underline the keyboard shortcut character if present

		if (item.hotChar)
			drawLineCoord(item.hot1.x + depressed,
						  item.hot1.y + depressed,
						  item.hot2.x + depressed - _MVIS_sysLineWidth + 1,
						  item.hot2.y + depressed);
		}

	MGL_popViewport();
}

void TMenu::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TMenu::draw
* Parameters:	clip	- Clipping rectangle for the view
*
* Description:	Method to draw the menu representation.
*
****************************************************************************/
{
	mouse.obscure();

	MGL_setClipRect(clip);

	// Draw the background for the menu

	MGL_setColor(getColor(3));
	MGL_fillRect(bounds);

	// Draw the border around the menu

	MGL_setColor(getColor(2));
	drawRect(bounds);
	MGL_setColor(getColor(1));
	drawLineCoord(bounds.left()+_MVIS_sysLineWidth,
		bounds.top()+_MVIS_sysLineWidth,
		bounds.left()+_MVIS_sysLineWidth,
		bounds.bottom()-2*_MVIS_sysLineWidth);
	drawLineCoord(bounds.left()+_MVIS_sysLineWidth,
		bounds.top()+_MVIS_sysLineWidth,
		bounds.right()-2*_MVIS_sysLineWidth,
		bounds.top()+_MVIS_sysLineWidth);

	// Draw the menu items in the menu

	TextJust old;			// Save current text settings
	fontManager.useFont(fmSystemFont);

	for (int i = 0; i < items.numberOfItems(); i++)
		drawItem(i);

	old.use();				// Restore old text settings
	mouse.unobscure();
}

void TMenu::refresh(int selection)
/****************************************************************************
*
* Function:		TMenu::refresh
* Parameters:	selection	- New value for the current selection
*
* Description:	Refreshes the menu by drawing as little as possible. Assumes
*				that the menu is currently draw in the correct state and
*				is fully visible.
*
****************************************************************************/
{
	if (current == selection)
		return;

	// Draw the menu items in the menu

	TextJust old;			// Save current text settings
	fontManager.useFont(fmSystemFont);

	mouse.obscure();
	int oldSelect = current;
	if ((current = selection) != -1)
		drawItem(current);
	if (oldSelect != -1) {
		MGL_setColor(getColor(3));
		MGL_fillRect(items[oldSelect]->bounds);
		drawItem(oldSelect);
		}

	old.use();				// Restore old text settings
	mouse.unobscure();
}

ushort TMenu::getHelpCtx() const
/****************************************************************************
*
* Function:		TMenu::getHelpCtx
* Returns:		Help context number for the current menu item.
*
* Description:
*
****************************************************************************/
{
	return items[current]->helpCtx;
}

TRect& TMenu::setItemBounds(int index,const TPoint& start)
/****************************************************************************
*
* Function:		TMenu::setItemBounds
* Parameters:	index	- Index of menu item to compute bounds for
*				start	- Starting point for the item's bounds
* Returns		Reference to the bounding box
*
* Description:	Sets the bounding box for the indexed item.
*
****************************************************************************/
{
	TRect& b = items[index]->bounds;

	b.topLeft = start;
	b.right() = b.left() + itemWidth + 6;
	if (items[index]->name[0] == '-')
		b.bottom() = b.top() + 5;
	else
		b.bottom() = b.top() + itemHeight;
	return b;
}

void TMenu::doneDefinition()
/****************************************************************************
*
* Function:		TMenu::doneDefinition
*
* Description:	Completes the definition of the menu structure. Here we
*				compute the size of the bounding box given the menu items
*				stored in it.
*
****************************************************************************/
{
	int		i,width;
	metrics	m;

	TextJust old;			// Save current text settings
	fontManager.useFont(fmSystemFont);

	// Compute the height and width of the menu items

	MGL_getFontMetrics(&m);
	itemHeight = m.ascent-m.descent+1+7;

	itemWidth = 0;
	for (i = 0; i < items.numberOfItems(); i++) {
		TMenuItem& item = *items[i];
		width = MGL_textWidth(item.name) + MGL_textWidth("  ");
		if (item.isSubMenu())
			width += 10;
		else
			width += MGL_textWidth(item.hotKeyText);
		itemWidth = max(itemWidth,width);

		// Compute the position of the hot character underscore if
		// present.

		if (item.hotChar) {
			char old = item.name[item.hotIndex];
			item.name[item.hotIndex] = '\0';
			item.hot1.x = item.hot1.y = 3;
			MGL_underScoreLocation(&item.hot1.x,&item.hot1.y,item.name);
			item.hot1.x += MGL_textWidth(item.name);
			item.hot2 = item.hot1;
			metrics m;
			MGL_getCharMetrics(old,&m);
			item.hot2.x += m.fontWidth-1;
			item.name[item.hotIndex] = old;
			}
		}

	// Compute the bounding box for each item, and compute the bounding
	// box for the entire menu.

	bounds.right() = bounds.left()+1;
	bounds.bottom() = bounds.top()+1;
	TPoint start(1 + 2*_MVIS_sysLineWidth,1 + 2*_MVIS_sysLineWidth);
	for (i = 0; i < items.numberOfItems(); i++) {
		TRect& b = setItemBounds(i,start);
		bounds += b;
		start.y = b.bottom();
		}

	// Add a small buffer to the right and bottom edges

	bounds.right() += 2 * _MVIS_sysLineWidth;
	bounds.bottom() += 2 * _MVIS_sysLineWidth;
	setBounds(bounds);

	// Now fixup any submenus that are attached to this menu, moving them
	// to the correct location

	for (i = 0; i < items.numberOfItems(); i++) {
		TMenuItem& item = *items[i];
		if (item.isSubMenu())
			item.subMenu->moveTo(bounds.right(),item.bounds.top());
		}

	old.use();
}

color_t TMenu::getColor(uchar index)
/****************************************************************************
*
* Function:		TMenu::getColor
* Parameters:	index	- Index of color value to lookup
* Returns:		Actual color value to use.
*
* Description:	Overloaded version for menu's. We need to search the
*				parent hierarchy first to find the group that contains the
*				menu.
*
****************************************************************************/
{
	if (parent)
		return parent->getColor(index);
	else
		return TView::getColor(index);
}

void TMenu::moveTo(int x,int y)
/****************************************************************************
*
* Function:		TMenu::moveTo
* Parameters:	x,y	- New position to move view to
*
* Description:	As well as moving the view, we also move all of the
*				menu items.
*
****************************************************************************/
{
	int	dx = x-bounds.left();
	int dy = y-bounds.top();

	// Ensure that the menu is still within the bounds of the current
	// viewport.

	if (bounds.right()+dx > MGL_maxx()+1)
		dx = (MGL_maxx()-size.x) - bounds.left();
	if (bounds.bottom()+dy > MGL_maxy()+1)
		dy = (MGL_maxy()-size.y) - bounds.top();

	// Move each item, recursively moving submenu items.

	bounds.offset(dx,dy);
	for (int i = 0; i < items.numberOfItems(); i++) {
		TMenuItem& item = *items[i];
		item.bounds.offset(dx,dy);
		if (item.isSubMenu())
			item.subMenu->moveTo(bounds.right(),item.bounds.top());
		}
}

TPalette& TMenu::getPalette() const
/****************************************************************************
*
* Function:		TMenu::getPalette
* Returns:		Pointer to the standard palette for menus.
*
****************************************************************************/
{
	static char cpMenu[] = {3,4,5,6,7,8,9};
	static TPalette palette(cpMenu,sizeof(cpMenu));
	return palette;
}
