/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tevent.hpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Header file for the event collection subsystem. We rely on
*				the MegaToobox event module to collect the events for us,
*				which when then stick into a more suitable structure for
*				C++.
*
* $Id: tevent.hpp 1.2 1994/03/09 11:50:10 kjb Exp $
*
****************************************************************************/

#ifndef	__TEVENT_HPP
#define	__TEVENT_HPP

#ifndef	__EVENT_H
#include "event.h"
#endif

#ifndef	__TPOINT_HPP
#include "tpoint.hpp"
#endif

/*------------------------------ Constants --------------------------------*/

const ushort

// Event codes

	evKeyDown		= KEYDOWN,		// Key was depressed
	evKeyAuto		= KEYREPEAT,	// Key auto repeat
	evKeyUp			= KEYUP,		// Key was released
	evMouseDown		= MOUSEDOWN,	// Mouse button was depressed
	evMouseAuto		= 0x0080,		// Mouse button auto repeat
	evMouseUp		= MOUSEUP,		// Mouse button was released
	evMouseMove		= MOUSEMOVE,	// Mouse cursor was moved
	evTimerTick		= TIMERTICK,	// A scheduled timer tick event occurred
	evCommand		= 0x0100,		// Event is a command message
	evBroadcast		= 0x0200,		// Event is a broadcast message
	evRepaint		= 0x0400,		// Event is a repaint message

// Event code masks

	evNothing		= NULLEVT,
	evMouse			= evMouseDown | evMouseUp | evMouseMove | evMouseAuto,
	evKeyboard		= evKeyDown | evKeyAuto | evKeyUp,
	evMessage		= 0xFF00,		// Event was a message of some type
	evAll			= 0xFFFF,		// All events mask

// Mouse button masks

	mbLeftButton	= LEFTBMASK,
	mbRightButton	= RIGHTBMASK,
	mbBothButtons	= mbLeftButton | mbRightButton,

// Event modifier masks

	mdLeftButton	= LEFTBUT,
	mdRightButton	= RIGHTBUT,
	mdLeftShift		= LEFTSHIFT,
	mdRightShift	= RIGHTSHIFT,
	mdLeftCtrl		= LEFTCTRL,
	mdLeftAlt		= LEFTALT,
	mdShift			= mdLeftShift | mdRightShift,
	mdCtrl			= CTRLSTATE,
	mdAlt			= ALTSTATE,
	mdMouse			= mdLeftButton | mdRightButton,
	mdKey			= mdShift | mdCtrl | mdAlt,

//  Event masks

	positionalEvents= evMouse,
	focusedEvents	= evKeyboard | evCommand;

/*-------------------------- Class definitions ----------------------------*/

struct MouseEvent {
	uchar	buttons;				// Button code mask
	bool	doubleClick;			// Event was a double click
	ushort	modifiers;				// Modifier flags
	ulong	when;					// Clock ticks since midnight
	};

struct KeyEvent {
	union {
		ushort		keyCode;
		struct {
			uchar	charCode;
			uchar	scanCode;
			} charScan;
		};
	ushort	modifiers;				// Modifier flags
	ulong	when;					// Clock ticks since midnight
	};

struct TimerEvent {
	ulong	when;					// Clock ticks since midnight
	};

struct MessageEvent {
	ushort	command;				// Command for this message
	union {
		void*	infoPtr;			// Various message data types
		long	infoLong;
		ushort	infoWord;
		short	infoInt;
		uchar	infoByte;
		char	infoChar;
		};
	};

struct TEvent {
	ushort	what;					// Event code
	TPoint	where;					// Where the event occurred (has constr.)
	union {
		MouseEvent		mouse;		// Mouse event information
		KeyEvent		key;		// Key event information
		TimerEvent		timer;		// Timer event information
		MessageEvent	message;	// Message event information
		};
	};

//---------------------------------------------------------------------------
// The following class implements the event queue mechanism for the
// class library. It is basically C++ wrapper to the stuff in the Event
// module of the MegaToolbox, with a few extras. Note that there is no
// destructor, since the event module already ensures that the event
// system will be shut down upon proper program termination.
//
// It automatically handles generating mouse double click and auto repeat
// events, which are not handled directly by the Event module.
//---------------------------------------------------------------------------

class TEventQueue {
private:
	ushort	doubleClick;		// Ticks between double clicks
	ushort	doubleClickThresh;	// Movement allowed for clicks (in pixels)
	ushort	autoRepeat;			// Ticks betwen mouse auto repeats
	ushort	autoDelay;			// Ticks before mouse auto repeat begins
	bool	firstAuto;			// True for first auto mouse repeat
	ulong	autoTicks;			// Current auto tick value
	TEvent	downMouse;			// Last mouse down event

public:
			TEventQueue();

			// Suspend/resume the event queue
			void suspend()	{ EVT_suspend(); };
			void resume()	{ EVT_resume(); };

			// Obtain or look at the next event
			bool getNext(TEvent& event,ushort mask = evAll);
			bool peekNext(TEvent& event,ushort mask = evAll);

			// Post an event to the event queue
			bool post(const TEvent& event);

			// Flush all events specified by the mask
			void flush(ushort mask = evAll)	{ EVT_flush(mask); };

			// Turn on or off collection of mouse movement events
			bool mouseMove(bool set)		{ return EVT_mouseMove(set); };

			// Turn collection of timer tick events on or off
			void timerTickOn(ushort ticks)	{ EVT_setTimerTick(ticks); };
			void timerTickOff()				{ EVT_setTimerTick(0); };

			// Set/get the double click speed
			void setDoubleClick(ushort ticks)	{ doubleClick = ticks; };
			ushort getDoubleClick()				{ return doubleClick; };

			// Set/get the double click threshold
			void setDoubleClickThreshold(ushort thresh)
				{ doubleClickThresh = thresh; };
			ushort getDoubleClickThreshold()
				{ return doubleClickThresh; };

			// Set/get the mouse auto repeat speed
			void setAutoRepeat(ushort ticks)	{ autoRepeat = ticks; };
			ushort getAutoRepeat()				{ return autoRepeat; };

			// Set/get the mouse auto repeat delay
			void setAutoRepeatDelay(ushort ticks)	{ autoDelay = ticks; };
			ushort getAutoRepeatDelay()				{ return autoDelay; };
	};

/*--------------------------- Global Variables ----------------------------*/

extern TEventQueue	eventQueue;

#endif	// __TEVENT_HPP
