/*****************************************************************
Module name: Cntl-DE.C
Programmer : Jeffrey M. Richter.
*****************************************************************/

#include <windows.h>
#include <custcntl.h>

#include "cntl-de.h"


// Property string used internally to store local handle of 
// CTLSTYLEDLG data structure.
static char _szCtlProp[] = "CtlDlgStyleData";


// Data structure used internally to get information into the 
// style dialog box function.
typedef struct {
   HGLOBAL      hGlblCtlStyle;// Handle holds control's CTLSTYLE
   LPFNSTRTOID  lpfnStrToId;  // Func to cnvrt string ID to #
   LPFNIDTOSTR  lpfnIdToStr;  // Func to cnvrt # ID to string
} CTLSTYLEDLG, FAR *LPCTLSTYLEDLG, NEAR *NPCTLSTYLEDLG;


// This function should be called first in the ClassInfo
// function to initialize the new control.
HGLOBAL WINAPI ControlInfo
        (WORD wVersion, LPSTR szClass, LPSTR szTitle) {

   HGLOBAL hGlbl = (HGLOBAL) NULL;
   LPCTLINFO lpCtlInfo;

   hGlbl = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
                        (DWORD) sizeof(CTLINFO));
   if (hGlbl == NULL) return(hGlbl);
   lpCtlInfo = (LPCTLINFO) GlobalLock(hGlbl);
   lpCtlInfo->wVersion = wVersion;

   // Initialize wCtlTypes to zero, incremented by 
   // AddControlType function.
   lpCtlInfo->wCtlTypes = 0;
   lstrcpy(lpCtlInfo->szClass, szClass);
   lstrcpy(lpCtlInfo->szTitle, szTitle);
   GlobalUnlock(hGlbl);
   return(hGlbl);
}



// This function should be called repeatedly to add new control 
// types to the structure returned by the ControlInfo function.  
// This function should be called in the ClassInfo function.
BOOL WINAPI AddControlType (HGLOBAL hGlbl, WORD wType,
        WORD wWidth, WORD wHeight, DWORD dwStyle, LPSTR szDescr) {

   LPCTLINFO lpCtlInfo; WORD wNumTypes;
   lpCtlInfo = (LPCTLINFO) GlobalLock(hGlbl);
   wNumTypes = lpCtlInfo->wCtlTypes;
   if (wNumTypes == CTLTYPES) {
      GlobalUnlock(hGlbl);
      return(FALSE);
   }
   lpCtlInfo->Type[wNumTypes].wType   = wType;
   lpCtlInfo->Type[wNumTypes].wWidth  = wWidth;
   lpCtlInfo->Type[wNumTypes].wHeight = wHeight;
   lpCtlInfo->Type[wNumTypes].dwStyle = dwStyle;
   lstrcpy(lpCtlInfo->Type[wNumTypes].szDescr, szDescr);
   lpCtlInfo->wCtlTypes++;
   GlobalUnlock(hGlbl);
   return(TRUE);
}



// This function displays the control's style dialog box and 
// should be called from the ClassStyle function.
int WINAPI ShowStyleDlg (HINSTANCE hInstance, LPCSTR szTemplate,
          HWND hWndParent, DLGPROC fpDlgProc, LPARAM lParam,
          HGLOBAL hGlblCtlStyle, LPFNSTRTOID lpfnStrToId,
          LPFNIDTOSTR lpfnIdToStr) {

   HLOCAL hLclCtlStyleDlg;
   NPCTLSTYLEDLG npCtlStyleDlg;
   int x;

   hLclCtlStyleDlg = LocalAlloc(LMEM_MOVEABLE | LMEM_ZEROINIT,
      sizeof(CTLSTYLEDLG));
   if (hLclCtlStyleDlg == NULL) return(FALSE);

   npCtlStyleDlg = (NPCTLSTYLEDLG) LocalLock(hLclCtlStyleDlg);
   npCtlStyleDlg->hGlblCtlStyle = hGlblCtlStyle;
   npCtlStyleDlg->lpfnStrToId = lpfnStrToId;
   npCtlStyleDlg->lpfnIdToStr = lpfnIdToStr;
   LocalUnlock(hLclCtlStyleDlg);

   // Associate property with Dialog Editor's window.
   SetProp(hWndParent, _szCtlProp, hLclCtlStyleDlg);

   // Display control's Styles Dialog Box.
   x = DialogBoxParam(hInstance, szTemplate, hWndParent,
      fpDlgProc, lParam);

   // Remove property associated with Dialog Editor's window.
   RemoveProp(hWndParent, _szCtlProp);

   LocalFree(hLclCtlStyleDlg);
   return(x == IDOK);   // Return whether CTLSTYLE has changed.
}



// This function should only be called from the ClassDlgFn
// function.  It locks the memory block containing the CTLSTYLE 
// structure for the selected control and returns the FAR address
// to that structure.
LPCTLSTYLE WINAPI CtlStyleLock (HWND hDlg) {
   HLOCAL hLclCtlStyleDlg;
   NPCTLSTYLEDLG npCtlStyleDlg;
   LPCTLSTYLE lpCtlStyle = NULL;

   // Property is associated with Dialog Editor's window.
   // Parent of the dialog box is the Dialog Editor.
   hLclCtlStyleDlg = GetProp(GetParent(hDlg), _szCtlProp);

   if (hLclCtlStyleDlg == NULL) return(lpCtlStyle);
   npCtlStyleDlg = (NPCTLSTYLEDLG) LocalLock(hLclCtlStyleDlg);
   lpCtlStyle = (LPCTLSTYLE)
      GlobalLock(npCtlStyleDlg->hGlblCtlStyle);
   LocalUnlock(hLclCtlStyleDlg);
   return(lpCtlStyle);
}



// This function should only be called from the ClassDlgFn
// function.  It unlocks the memory block containing the CTLSTYLE
// structure for the selected control and returns whether the 
// block was successfully unlocked.
BOOL WINAPI CtlStyleUnlock (HWND hDlg) {
   HLOCAL hLclCtlStyleDlg;
   NPCTLSTYLEDLG npCtlStyleDlg;
   BOOL fOk = FALSE;

   // Property is associated with Dialog Editor's window.
   // Parent of the dialog box is the Dialog Editor.
   hLclCtlStyleDlg = GetProp(GetParent(hDlg), _szCtlProp);

   if (hLclCtlStyleDlg == NULL) return(fOk);
   npCtlStyleDlg = (NPCTLSTYLEDLG) LocalLock(hLclCtlStyleDlg);
   fOk = GlobalUnlock(npCtlStyleDlg->hGlblCtlStyle);
   LocalUnlock(hLclCtlStyleDlg);
   return(fOk);
}



// This function should only be called from the ClassDlgFn
// function.  It converts the ID value for the control into a
// identifier string and stores the string in the address passed
// in.  The number of characters in the string is returned.
WORD WINAPI GetIdString (HWND hDlg, LPSTR szId, WORD wIdMaxLen) {
   HLOCAL hLclCtlStyleDlg;
   NPCTLSTYLEDLG npCtlStyleDlg;
   LPCTLSTYLE lpCtlStyle;
   WORD wIdLen;

   // Property is associated with Dialog Editor's window.
   // Parent of the dialog box is the Dialog Editor.
   hLclCtlStyleDlg = GetProp(GetParent(hDlg), _szCtlProp);
   if (hLclCtlStyleDlg == NULL) return(0);

   npCtlStyleDlg = (NPCTLSTYLEDLG) LocalLock(hLclCtlStyleDlg);
   lpCtlStyle = (LPCTLSTYLE)
      GlobalLock(npCtlStyleDlg->hGlblCtlStyle);

   // Call the lpfnIdToStr function to convert the numeric ID 
   // to its string equivalent.
   wIdLen = (*npCtlStyleDlg->lpfnIdToStr)
      (lpCtlStyle->wId, szId, wIdMaxLen);

   GlobalUnlock(npCtlStyleDlg->hGlblCtlStyle);
   LocalUnlock(hLclCtlStyleDlg);
   return(wIdLen);
}




// This function should only be called from the ClassDlgFn
// function.  It converts an ID string value into its numeric 
// equivalent and stores the numeric value in the CTLSTYLE
// structure for the control.  If the loword of the result is 0,
// the ID is invalid, otherwise, the hiword contains the numeric 
// value of the ID.
DWORD WINAPI SetIdValue (HWND hDlg, LPSTR szId) {
   HLOCAL hLclCtlStyleDlg;
   NPCTLSTYLEDLG npCtlStyleDlg;
   LPCTLSTYLE lpCtlStyle;
   DWORD dwResult = 0;

   hLclCtlStyleDlg = GetProp(GetParent(hDlg), _szCtlProp);
   if (hLclCtlStyleDlg == NULL) return(dwResult);

   npCtlStyleDlg = (NPCTLSTYLEDLG) LocalLock(hLclCtlStyleDlg);

   // Call the lpfnStrToId function to convert the string ID
   // to its numeric equivalent.
   dwResult = (*npCtlStyleDlg->lpfnStrToId)(szId);

   LocalUnlock(hLclCtlStyleDlg);

   // If LOWORD is zero, string NOT found.
   if (LOWORD(dwResult) == 0)
      return(dwResult);

   // LOWORD is not zero, numeric ID is in the HIWORD.
   lpCtlStyle = CtlStyleLock(hDlg);
   lpCtlStyle->wId = HIWORD(dwResult);
   CtlStyleUnlock(hDlg);
   return(dwResult);
}
