/*
  dosdir.c:

  Implementation of the Turbo C directory functions (findfirst and findnext)
  on MS-DOS, UNIX and VMS platforms using the appropriate file & directory
  structure.

  Provides the DOS directory framework for MS-DOS/UNIX/VMS application
  portability.

  Supports MS-DOS with Borland C++ or Turbo C, Sun with GNU C compiler,
  DEC Alpha (OSF-1), VAX/VMS C, and other comparable platforms.

  Jason Mathews
  NASA/Goddard Space Flight Center
  Code 633.2
  Greenbelt, Maryland  20771-0001 USA

  Internet: mathews@nssdca.gsfc.nasa.gov

  Copyright (C) 1994 by Jason Mathews.
  Permission is granted to any individual or institution to use, copy,
  or redistribute this software so long as it is not sold for profit,
  provided this copyright notice is retained.

  Modification history:
   02-May-91  - Original version.
   13-May-94  - Reimplemented findfirst/findnext with ffblk structure
		to match MS-DOS externally,
	      - Fixed wildcard evaluation function.
   08-Jun-94  - Replaced wildcard evaluation function with recursive
		function provided by Info-ZIP's portable UnZip,
	      - Added VMS functions.
  */

#include "dosdir.h"

#if (defined(UNIX) || defined(VMS))
#include <time.h>
#include "match.h"

/* stub functions for get/set disk
 * fake MS-DOS functions that do not apply to unix or vms:
 */

int getdisk()
{
  return 0;
}

int setdisk( drive )
    int drive;
{
  return 0;
}
#endif

#ifdef UNIX

/*
 * Function:    findnext
 *
 * Purpose:     Use findnext after findfirst to find the remaining files
 *
 * Returns:     zero if successful, otherwise, it returns a -1
 *              and sets errno either to the constant ENOENT indicating
 *              that the file could not be found, or that there are
 *		no more files.
 *
 *  Parms:
 *      struct ffblk* fb  = structure to hold results of search
 */
int findnext(fb)
    struct ffblk* fb;
{
  fb->ff_attrib = 0;
  if (!fb->ff_dirp) goto findnext_err;
  while ((fb->ff_dp = readdir(fb->ff_dirp)) != NULL)
  {
    struct stat statbuf;        /* stat structure */
    if (stat(fb->ff_dp->d_name, &statbuf)) continue;
    if (statbuf.st_mode & S_IFDIR)
      {
	if (!(fb->ff_attribs & FA_DIREC)) continue;
	fb->ff_attrib |= FA_DIREC;
      }
    if (match(fb->ff_dp->d_name, fb->ff_filespec, 0))
      {
	struct tm* t = localtime(&statbuf.st_mtime);
	/* fill in file info */
	fb->ff_name  = fb->ff_dp->d_name;
	fb->ff_fsize = statbuf.st_size;
	fb->ff_fdate = ((unsigned) t->tm_year-80) << 9 |
	  (unsigned) (t->tm_mon+1) << 5 | t->tm_mday;
	fb->ff_ftime = (unsigned) t->tm_hour << 11 |
	  (unsigned) t->tm_min << 5 | t->tm_sec/2;
	if ( !(statbuf.st_mode & S_IWRITE) )
	  fb->ff_attrib |= FA_RDONLY;
	return 0;       /* successful match */
      }
  }  /* while */

  closedir(fb->ff_dirp);

 findnext_err:

  errno = ENOENT;	/* no file found */
  memset(fb, 0x0, sizeof(struct ffblk)); /* invalidate structure */
  return -1;
}  /** findnext **/


/*
 * Function:    findfirst
 *
 * Purpose:     Find file matching specification in specified directory
 *              given directory information
 *
 * Returns:     zero if successful, otherwise, it returns a -1
 *              and sets errno either to the constant ENOENT indicating
 *              that the file could not be found, or that there are
 *		no more files.
 *
 *  Parms:
 *      char *filespec    = filename to search for including path
 *      struct ffblk* fb  = structure to hold results of search
 *      int attrib        = file attributes to match
 */
int findfirst(pathname, fb, attrib)
    const char *pathname;
    struct ffblk* fb;
    int attrib;
{
  char *s = strrchr(pathname, '/');
  char dir[MAXDIR];     /* directory path */
  if (s)
    {
      strcpy(fb->ff_filespec, s+1);
      strncpy(dir, pathname, s-pathname);
    }
  else
    {
      strcpy(dir, ".");
      strcpy(fb->ff_filespec, pathname);
    }
  fb->ff_attribs = attrib;
  fb->ff_dirp = opendir(dir);
  return findnext(fb);
}  /** findfirst **/

#endif /* ?UNIX */

#ifdef VMS

int findnext( struct ffblk* fb )
{
    int rc;
    fb->ff_attrib = 0;
    while ((rc = SYS$SEARCH(&fb->fab, 0, 0)) == RMS$_NORMAL)
      {
	struct stat statbuf;    /* stat structure */
	int len = fb->nam.nam$b_rsl;
	char *s;
	if (len < 1) continue;
	fb->rsa[len] = '\0';
	if (stat(fb->rsa, &statbuf))
	  continue; /* no priviledge to read file info - skip it */
	if (statbuf.st_mode & S_IFDIR)
	  {
	    if (!(fb->attribs & FA_DIREC))
	      continue;
	    fb->ff_attrib |= FA_DIREC;
	  }
	s = strchr(fb->rsa, DIR_END);
	fb->ff_name = (s) ? s + 1 : fb->rsa;
	if (match(fb->ff_name, fb->filespec, 1))
	  {
	    struct tm* t = localtime(&statbuf.st_mtime);
	    fb->ff_fsize = statbuf.st_size;
	    fb->ff_fdate = ((unsigned) t->tm_year-80) << 9 |
	      (unsigned) (t->tm_mon+1) << 5 | t->tm_mday;
	    fb->ff_ftime = (unsigned) t->tm_hour << 11 |
	      (unsigned) t->tm_min << 5 | t->tm_sec/2;
	    if ( !(statbuf.st_mode & S_IWRITE) )
	      fb->ff_attrib |= FA_RDONLY;
	    return 0;
	  }
      }

    vaxc$errno   = ENOENT;	/* no file found */
    memset(fb, 0x0, sizeof(struct ffblk)); /* invalidate structure */
    return -1;
}

int findfirst( const char *path, struct ffblk *fb, int attrib )
{
  char *s;
  fb->fab = cc$rms_fab;
  fb->nam = cc$rms_nam;
  strcpy(fb->path, path);
  s = strchr(fb->path, DIR_END);
  if (!s)
    s = strrchr(fb->path, ':'); /* check device/node */
  if (s)
    {
      strncpy(fb->rms_filespec, fb->path, s - fb->path + 1);
      strcat(fb->rms_filespec, "*.*");
      fb->filespec = s+1;
      printf("filespec = %s\n", fb->rms_filespec);
    }
  else
    {
      strcpy(fb->rms_filespec, "*.*");
      fb->filespec = fb->path;
    }

  /* if no version number specified then add one */
  s = strchr(fb->path, ';');
  if (!s) strcat(fb->path, ";*");

  fb->fab.fab$l_dna = &fb->rms_filespec;
  fb->fab.fab$b_dns = strlen(fb->rms_filespec);
  fb->fab.fab$l_nam = &fb->nam;
  fb->nam.nam$l_esa = &fb->esa;
  fb->nam.nam$b_ess = MAXPATH;
  fb->nam.nam$l_rsa = &fb->rsa;
  fb->nam.nam$b_rss = MAXPATH;
  fb->attribs       = attrib;
  if (SYS$PARSE(&fb->fab, 0, 0) != RMS$_NORMAL)
    {
	memset(fb, 0x0, sizeof(struct ffblk)); /* invalidate structure */
	vaxc$errno  = ENOENT;     /* no file found */
	return -1;
    }
    return findnext(fb);
}

#endif /* ?VMS */
