// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.

/////////////////////////////////// uniform random number objects


#include "urandvec.h"

double URand::frand()
{
  // 16 or 32 bit Uniform(0,1) rng
  // return a uniform 0-1 floating point
  // prime modulus multiplicative linear
  // congruential random number generator
  // f(z) = (16807 * z) mod 2147483647;
  // Reference: Communications of the
  // ACM vol 31 no 10
  // 127773 is 2147483647 / 16807
  // 2836 is 2147483647 % 16807
  // Note you can replace this with whatever
  // U(0,1) rng you want. This one is popular.
  
  // check the seed for admissability
  if (!seed) seed++;
  if (seed == 2147483647l) seed = 2147483646l;
  
  high = seed / 127773l;
  low = seed % 127773l;
  test = 16807 * low - 2836l * high;
  seed = (test > 0l) ? test : test + 2147483647l;
  t = ((double) seed) / 2147483647.0;
  
  // do not return 1.0
  return (t == 1.0) ? 1-1.0e-32 : t;
}
////////////////////////////////////////////////////
void URandVec::Makeit(int nn, unsigned baseseed, int InConstructor)
{
  n = nn;
  int i, j;
  // using the same baseseed gives reproducable
  // streams otherwise use the current time as the
  // elapsed number of seconds since 00:00:00 GMT
  // 1/1/1970 note that this constructor provides
  // at most UINT_MAX streams.
  if (baseseed)
    srand(baseseed);
  else
    srand((unsigned) time(NULL));
  
  // allocate the vector of random number streams
  if (InConstructor) uvect = new URand[n];
  if (!uvect) Matrix::Nrerror("URandVec allocation failure");
  
  // separate all seeds by a minimum of mindel
  unsigned *seedarray = new unsigned[n];
  unsigned mindel = UINT_MAX / (10*n);
  mindel = (mindel < 10) ? 10 : mindel;
  for (i = 0; i < n; i++) {
    unsigned t = rand();
    int badseed = 1;
    int cnter = 0;
    while (badseed && i > 0) {
      badseed = 0;
      // search for first seed that is too
      // close to t
      for (j = 0; j < i; j++) {
        if (abs(seedarray[j] - t) < mindel) {
          badseed = 1;
          break;
        }
      }
      if (badseed) t = rand();
      // stop after 20 searches if
      // mindel is too large
      if (++ cnter > 20) break;
    }
    seedarray[i] = t;
    uvect[i].frand(t);
  }
  delete[] seedarray;
  
  // jumble
  URand temp;
  for (i = 0; i < n; i++) {
    j = rand() % n;
    temp = uvect[j];
    uvect[j] = uvect[i];
    uvect[i] = temp;
  }
  // set u and nextj
  u = uvect[rand() % n].frand();
  GetNextJ();
  
  // burn in by running it some
  for (i = 1; i <= 10; i++)
    for (j = 0; j < n; j++) URandV();
  
}
//////////////////// ostream insertion
ostream &operator << (ostream &s, NormalRV &r)
{
  s << "Normal Random Number generator: mean = " << r.mean << ", std = "
    << r.stddev << endl;
  return s;
}
ostream &operator << (ostream &s, NCGammaRV &r)
{
  s << "NCGamma Random Number generator: alpha = " << r.alpha << ", beta = "
    << r.beta << ", ncp = " << r.ncp << endl;
  return s;
}
ostream &operator << (ostream &s, NCChiRV &r)
{
  s << "NC Chi Square Random Number generator: df = " << r.df
    << ", ncp = " << r.ncp << endl;
  return s;
}
ostream &operator << (ostream &s, NCFRV &r)
{
  s << "NC F Random Number generator: df1 = " << r.df1 << ", df2 = "
    << r.df2 << ", ncp = " << r.ncp << endl;
  return s;
}
ostream &operator << (ostream &s, NCBetaRV &r)
{
  s << "NC Beta Random Number generator: alpha = " << r.alpha << ", beta = "
    << r.beta << ", ncp = " << r.ncp << endl;
  return s;
}
ostream &operator << (ostream &s, NCtRV &r)
{
  s << "NC t Random Number generator: df = " << r.df
    << ", ncp = " << r.ncp << endl;
  return s;
}
ostream &operator << (ostream &s, ExpRV &r)
{
  s << "Exponential Random Number generator: lambda = " << r.lambda << endl;
  return s;
}
ostream &operator << (ostream &s, MvnRV &r)
{
  s << "Multivariate Normal Random Number generator: Mu \n" << r.Mu()
    << "V \n" << r.V() << endl;
  return s;
}

