// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.

//////////////// eigenvalues and eigen vectors of a real general matrix

#include "rg.h"

////////// note this does not work for complex matrices

#ifndef USE_COMPLEX

FLOAT rg::d_sign(FLOAT a, FLOAT b)
{
  FLOAT x;
  x = (a > 0) ? a : - a;
  return ((b >= 0) ? x : - x);
}
int rg::cdiv_(FLOAT ar, FLOAT ai, FLOAT br, FLOAT bi, FLOAT &cr, FLOAT &ci)
{
  // Local variables
  FLOAT s, ais, bis, ars, brs;
  
  
  //     complex division, (cr,ci) = (ar,ai)/(br,bi)
  
  s = fabs(br) + fabs(bi);
  ars = ar / s;
  ais = ai / s;
  brs = br / s;
  bis = bi / s;
  s = brs*brs + bis*bis;
  cr = (ars * brs + ais * bis) / s;
  ci = (ais * brs - ars * bis) / s;
  return 0;
} // cdiv_

void rg::cmult_(FLOAT a, FLOAT b, FLOAT c, FLOAT d, FLOAT &e, FLOAT &f)
{
    e = a*c-b*d;
    f = a*d+b*c;
}

// Subroutine
int rg::balanc_(int &n, Matrix &a, int *low, int *igh, Matrix &scale)
{
  // Local variables
  int iexc;
  FLOAT c, f, g;
  int i, j, k, l, m;
  FLOAT r, s, radix, b2;
  int jj;
  bboolean noconv;
  
  
  //     this subroutine is a translation of the algol procedure balance,
  
  //     num. math. 13, 293-304(1969) by parlett and reinsch.
  //     handbook for auto. comp., vol.ii-linear algebra, 315-326(1971).
  
  //     this subroutine balances a real matrix and isolates
  //     eigenvalues whenever possible.
  
  //     on input
  
  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.
  
  //        n is the order of the matrix.
  
  //        a contains the input matrix to be balanced.
  
  //     on output
  
  //        a contains the balanced matrix.
  
  //        low and igh are two ints such that a(i,j)
  //          is equal to zero if
  //           (1) i is greater than j and
  //           (2) j=1,...,low-1 or i=igh+1,...,n.
  
  //        scale contains information determining the
  //           permutations and scaling factors used.
  
  //     suppose that the principal submatrix in rows low through igh
  //     has been balanced, that p(j) denotes the index interchanged
  //     with j during the permutation step, and that the elements
  //     of the diagonal matrix used are denoted by d(i,j).  then
  //        scale(j) = p(j),    for j = 1,...,low-1
  //                 = d(j,j),      j = low,...,igh
  //                 = p(j)         j = igh+1,...,n.
  //     the order in which the interchanges are made is n to igh+1,
  //     then 1 to low-1.
  
  //     note that 1 is returned for igh if igh is zero formally.
  
  //     the algol procedure exc contained in balance appears in
  //     balanc  in line.  (note that the algol roles of identifiers
  //     k,l have been reversed.)
  
  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory
  
  
  //     this version dated august 1983.
  
  //     ------------------------------------------------------------------
  
  
  // Function Body
  radix = 16.;
  
  b2 = radix * radix;
  k = 1;
  l = n;
  goto L100;
  //     .......... in-line procedure for row and
  //                column exchange ..........
  L20 :
  scale(m) = (FLOAT) j;
  if (j == m) goto L50;
  
  for (i = 1; i <= l;++ i) {
    f = a(i, j);
    a(i, j) = a(i, m);
    a(i, m) = f;
  }
  
  for (i = k; i <= n;++ i) {
    f = a(j, i);
    a(j, i) = a(m, i);
    a(m, i) = f;
  }
  
  L50 :
  switch ((int) iexc) {
    case 1 : goto L80;
    case 2 : goto L130;
  }
  //     .......... search for rows isolating an eigenvalue
  //                and push them down ..........
  L80 :
  if (l == 1) goto L280;
  -- l;
  //     .......... for j=l step -1 until 1 do -- ..........
  L100 :
  for (jj = 1; jj <= l;++ jj) {
    j = l + 1 - jj;
    
    for (i = 1; i <= l;++ i) {
      if (i == j) goto L110;
      if (a(j, i) != 0.) goto L120;
      L110 :
      ;
    }
    
    m = l;
    iexc = 1;
    goto L20;
    L120 :
    ;
  }
  
  goto L140;
  //     .......... search for columns isolating an eigenvalue
  //                and push them left ..........
  L130 :
  ++ k;
  
  L140 :
  for (j = k; j <= l;++ j) {
    
    for (i = k; i <= l;++ i) {
      if (i == j) goto L150;
      if (a(i, j) != 0.) goto L170;
      L150 :
      ;
    }
    
    m = k;
    iexc = 2;
    goto L20;
    L170 :
    ;
  }
  //     .......... now balance the submatrix in rows k to l ..........
  for (i = k; i <= l;++ i) scale(i) = 1.;
  //     .......... iterative loop for norm reduction ..........
  L190 :
  noconv = FFALSE;
  
  for (i = k; i <= l;++ i) {
    c = 0.;
    r = 0.;
    for (j = k; j <= l;++ j) {
      if (j == i) goto L200;
      c += fabs(a(j, i));
      r += fabs(a(i, j));
      L200 :
      ;
    }
    //     .......... guard against zero c or r due to underflow .........
    
    if (c == 0. || r == 0.) goto L270;
    g = r / radix;
    f = 1.;
    s = c + r;
    L210 :
    if (c >= g) goto L220;
    f *= radix;
    c *= b2;
    goto L210;
    L220 :
    g = r * radix;
    L230 :
    if (c < g) goto L240;
    f /= radix;
    c /= b2;
    goto L230;
    //     .......... now balance ..........
    L240 :
    if ((c + r) / f >= s *.95) goto L270;
    g = 1. / f;
    scale(i) *=f;
    noconv = TTRUE;
    
    for (j = k; j <= n;++ j) a(i, j) *= g;
    for (j = 1; j <= l;++ j) a(j, i) *= f;
    
    L270 :
    ;
  }
  
  if (noconv) goto L190;
  
  L280 :
  *low = k;
  *igh = l;
  return 0;
} // balanc_

// Subroutine
int rg::balbak_(int &n, int *low, int *igh, Matrix &scale, int &m, Matrix &z)
{
  // Local variables
  int i, j, k;
  FLOAT s;
  int ii;
  
  
  
  //     this subroutine is a translation of the algol procedure balbak,
  //     num. math. 13, 293-304(1969) by parlett and reinsch.
  //     handbook for auto. comp., vol.ii-linear algebra, 315-326(1971).
  
  //     this subroutine forms the eigenvectors of a real general
  //     matrix by back transforming those of the corresponding
  //     balanced matrix determined by  balanc.
  
  //     on input
  
  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.
  
  //        n is the order of the matrix.
  
  //        low and igh are ints determined by  balanc.
  
  //        scale contains information determining the permutations
  //          and scaling factors used by  balanc.
  
  //        m is the number of columns of z to be back transformed.
  
  //        z contains the real and imaginary parts of the eigen-
  //          vectors to be back transformed in its first m columns.
  
  //     on output
  
  //        z contains the real and imaginary parts of the
  //          transformed eigenvectors in its first m columns.
  
  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory
  
  
  //     this version dated august 1983.
  
  //     ------------------------------------------------------------------
  
  
  // Parameter adjustments
  // Function Body
  if (m == 0) goto L200;
  if (*igh == *low) goto L120;
  
  for (i = *low; i <= *igh;++ i) {
    s = scale(i);
    //     .......... left hand eigenvectors are back transformed
    //                if the foregoing statement is replaced by
    //                s=1.0d0/scale(i). ..........
    for (j = 1; j <= m;++ j) z(i, j) *= s;
  }
  //     ......... for i=low-1 step -1 until 1,
  //               igh+1 step 1 until n do -- ..........
  L120 :
  for (ii = 1; ii <= n;++ ii) {
    i = ii;
    if (i >= *low && i <= *igh) goto L140;
    if (i < *low) i = *low - ii;
    k = (int) scale(i);
    if (k == i) goto L140;
    for (j = 1; j <= m;++ j) {
      s = z(i, j);
      z(i, j) = z(k, j);
      z(k, j) = s;
    }
    L140 :
    ;
  }
  
  L200 :
  return 0;
} // balbak_

// Subroutine
int rg::elmhes_(int &n, int *low, int *igh, Matrix &a, int *int_)
{
  // Local variables
  int i, j, m;
  FLOAT x, y;
  int la, mm1, kp1, mp1;
  
  
  
  //     this subroutine is a translation of the algol procedure elmhes,
  //     num. math. 12, 349-368(1968) by martin and wilkinson.
  //     handbook for auto. comp., vol.ii-linear algebra, 339-358(1971).
  
  //     given a real general matrix, this subroutine
  //     reduces a submatrix situated in rows and columns
  //     low through igh to upper hessenberg form by
  //     stabilized elementary similarity transformations.
  
  //     on input
  
  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.
  
  //        n is the order of the matrix.
  
  //        low and igh are ints determined by the balancing
  //          subroutine  balanc.  if  balanc  has not been used,
  //          set low=1, igh=n.
  
  //        a contains the input matrix.
  
  //     on output
  
  //        a contains the hessenberg matrix.  the multipliers
  //          which were used in the reduction are stored in the
  //          remaining triangle under the hessenberg matrix.
  
  //        int contains information on the rows and columns
  //          interchanged in the reduction.
  //          only elements low through igh are used.
  
  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory
  
  
  //     this version dated august 1983.
  
  //     ------------------------------------------------------------------
  
  
  // Parameter adjustments
  -- int_;
  
  // Function Body
  la = *igh - 1;
  kp1 = *low + 1;
  if (la < kp1) goto L200;
  
  for (m = kp1; m <= la;++ m) {
    mm1 = m - 1;
    x = 0.;
    i = m;
    
    for (j = m; j <= *igh;++ j) {
      if (fabs(a(j, mm1)) <= fabs(x)) goto L100;
      x = a(j, mm1);
      i = j;
      L100 :
      ;
    }
    
    int_[m] = i;
    if (i == m) goto L130;
    //     .......... interchange rows and columns of a ..........
    for (j = mm1; j <= n;++ j) {
      y = a(i, j);
      a(i, j) = a(m, j);
      a(m, j) = y;
    }
    for (j = 1; j <= *igh;++ j) {
      y = a(j, i);
      a(j, i) = a(j, m);
      a(j, m) = y;
    }
    //     .......... end interchange ..........
    L130 :
    if (x == 0.) goto L180;
    mp1 = m + 1;
    
    for (i = mp1; i <= *igh;++ i) {
      y = a(i, mm1);
      if (y == 0.) goto L160;
      y /= x;
      a(i, mm1) = y;
      for (j = m; j <= n;++ j) a(i, j) -= y * a(m, j);
      for (j = 1; j <= *igh;++ j) a(j, m) += y * a(j, i);
      L160 :
      ;
    }

    L180 :
    ;
  }

  L200 :
  return 0;
} // elmhes_

// Subroutine
int rg::eltran_(int &n, int *low, int *igh, Matrix &a, int *int_, Matrix &z)
{
  // Local variables
  int i, j, kl, mm, mp, mp1;

  //     this subroutine is a translation of the algol procedure elmtrans,

  //     num. math. 16, 181-204(1970) by peters and wilkinson.
  //     handbook for auto. comp., vol.ii-linear algebra, 372-395(1971).

  //     this subroutine accumulates the stabilized elementary
  //     similarity transformations used in the reduction of a
  //     real general matrix to upper hessenberg form by  elmhes.

  //     on input

  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.

  //        n is the order of the matrix.

  //        low and igh are ints determined by the balancing
  //          subroutine  balanc.  if  balanc  has not been used,
  //          set low=1, igh=n.

  //        a contains the multipliers which were used in the
  //          reduction by  elmhes  in its lower triangle
  //          below the subdiagonal.

  //        int contains information on the rows and columns
  //          interchanged in the reduction by  elmhes.
  //          only elements low through igh are used.

  //     on output

  //        z contains the transformation matrix produced in the
  //          reduction by  elmhes.

  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory


  //     this version dated august 1983.

  //     ------------------------------------------------------------------


  //     .......... initialize z to identity matrix ..........
  // Parameter adjustments
  -- int_;

  // Function Body
  for (j = 1; j <= n;++ j)
    for (i = 1; i <= n;++ i) z(i, j) = (i == j) ? 1 : 0;

    kl = *igh - *low - 1;
    if (kl >= 1) {
      //     .......... for mp=igh-1 step -1 until low+1 do -- ..........
      for (mm = 1; mm <= kl;++ mm) {
        mp = *igh - mm;
        mp1 = mp + 1;
        for (i = mp1; i <= *igh;++ i) z(i, mp) = a(i, mp - 1);
        i = int_[mp];
        if (i != mp) {
          for (j = mp; j <= *igh;++ j) {
            z(mp, j) = z(i, j);
            z(i, j) = 0.;
          }
          z(i, mp) = 1.;
        }
      }
    }
    return 0;
} // eltran_

// Subroutine
int rg::hqr_(int &n, int *low, int *igh, Matrix &h, Matrix &wr, 
             Matrix &wi, int &ierr)
  
{
  // System generated locals
  int i__1, i__2;
  FLOAT d__1, d__2;
  
  // Local variables
  FLOAT norm;
  int i, j, k, l, m;
  FLOAT p, q, r, s, t, w, x, y;
  int na, en, ll, mm;
  FLOAT zz;
  bboolean notlas;
  int mp2, itn, its, enm2;
  FLOAT tst1, tst2;
  
  
  
  //     this subroutine is a translation of the algol procedure hqr,
  //     num. math. 14, 219-231(1970) by martin, peters, and wilkinson.
  //     handbook for auto. comp., vol.ii-linear algebra, 359-371(1971).
  
  //     this subroutine finds the eigenvalues of a real
  //     upper hessenberg matrix by the qr method.
  
  //     on input
  
  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.
  
  //        n is the order of the matrix.
  
  //        low and igh are ints determined by the balancing
  //          subroutine  balanc.  if  balanc  has not been used,
  //          set low=1, igh=n.
  
  //        h contains the upper hessenberg matrix.  information about
  //          the transformations used in the reduction to hessenberg
  //          form by  elmhes  or  orthes, if performed, is stored
  //          in the remaining triangle under the hessenberg matrix.
  
  //     on output
  
  //        h has been destroyed.  therefore, it must be saved
  //          before calling  hqr  if subsequent calculation and
  //          back transformation of eigenvectors is to be performed.
  
  //        wr and wi contain the real and imaginary parts,
  //          respectively, of the eigenvalues.  the eigenvalues
  //          are unordered except that complex conjugate pairs
  //          of values appear consecutively with the eigenvalue
  //          having the positive imaginary part first.  if an
  //          error exit is made, the eigenvalues should be correct
  //          for indices ierr+1,...,n.
  
  //        ierr is set to
  //          zero       for normal return,
  //          j          if the limit of 30*n iterations is exhausted
  //                     while the j-th eigenvalue is being sought.
  
  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory
  
  
  //     this version dated august 1983.
  
  //     ------------------------------------------------------------------
  
  
  
  // Function Body
  ierr = 0;
  norm = 0.;
  k = 1;
  //     .......... store roots isolated by balanc
  //                and compute matrix norm ..........
  i__1 = n;
  for (i = 1; i <= i__1;++ i) {
    i__2 = n;
    for (j = k; j <= i__2;++ j) // L40:
      norm += (d__1 = h(i, j), fabs(d__1));
    k = i;
    if (i >= *low && i <= *igh) goto L50;
    wr(i) = h(i, i);
    wi(i) = 0.;
    L50 :
    ;
  }
  
  en = *igh;
  t = 0.;
  itn = n * 30;
  //     .......... search for next eigenvalues ..........
  L60 :
  if (en < *low) goto L1001;
  its = 0;
  na = en - 1;
  enm2 = na - 1;
  //     .......... look for single small sub-diagonal element
  //                for l=en step -1 until low do -- ..........
  L70 :
  for (ll = *low; ll <= en;++ ll) {
    l = en + *low - ll;
    if (l == *low) goto L100;
    s = (d__1 = h(l - 1, l - 1), fabs(d__1)) + (d__2 = h(l, l), fabs(d__2));
    if (s == 0.) s = norm;
    tst1 = s;
    tst2 = tst1 +(d__1 = h(l, (l - 1)), fabs(d__1));
    if (tst2 == tst1) goto L100;
  }
  //     .......... form shift ..........
  L100 :
  x = h(en, en);
  if (l == en) goto L270;
  y = h(na, na);
  w = h(en, na) * h(na, en);
  if (l == na) goto L280;
  if (itn == 0) goto L1000;
  if (its != 10 && its != 20) goto L130;
  //     .......... form exceptional shift ..........
  t += x;
  
  for (i = *low; i <= en;++ i) h(i, i) -= x;
  
  s = fabs(h(en, na)) + fabs(h(na, enm2));
  x = s *.75;
  y = x;
  w = s * -.4375* s;
  L130 :
  ++ its;
  -- itn;
  //     .......... look for two consecutive small
  //                sub-diagonal elements.
  //                for m=en-2 step -1 until l do -- ..........
  for (mm = l; mm <= enm2;++ mm) {
    m = enm2 + l - mm;
    zz = h(m, m);
    r = x - zz;
    s = y - zz;
    p = (r * s - w) / h(m + 1, m) + h(m, m + 1);
    q = h(m + 1, m + 1) - zz - r - s;
    r = h(m + 2, m + 1);
    s = fabs(p) + fabs(q) + fabs(r);
    p /= s;
    q /= s;
    r /= s;
    if (m == l) goto L150;
    tst1 = fabs(p) * ((d__1 = h(m - 1, m - 1), fabs(d__1)) +
      fabs(zz) + (d__2 = h(m + 1, m + 1), fabs(d__2)));
    tst2 = tst1 +(d__1 = h(m, m - 1), fabs(d__1)) * (fabs(q) + fabs(r));
    if (tst2 == tst1) goto L150;
  }
  
  L150 :
  mp2 = m + 2;
  for (i = mp2; i <= en;++ i) {
    h(i, (i - 2)) = 0.;
    if (i == mp2) goto L160;
    h(i, (i - 3)) = 0.;
    L160 :
    ;
  }
  //     .......... double qr step involving rows l to en and
  //                columns m to en ..........
  i__1 = na;
  for (k = m; k <= i__1;++ k) {
    notlas = (k != na) ? TTRUE : FFALSE;
    if (k == m) goto L170;
    p = h(k, (k - 1));
    q = h(k + 1, (k - 1));
    r = 0.;
    if (notlas) r = h(k + 2, (k - 1));
    x = fabs(p) + fabs(q) + fabs(r);
    if (x == 0.) goto L260;
    p /= x;
    q /= x;
    r /= x;
    L170 :
    d__1 = sqrt(p * p + q * q + r * r);
    s = d_sign(d__1, p);
    if (k == m) goto L180;
    h(k, (k - 1)) = -s * x;
    goto L190;
    L180 :
    if (l != m) h(k, (k - 1)) = -h(k, (k - 1));
    L190 :
    p += s;
    x = p / s;
    y = q / s;
    zz = r / s;
    q /= p;
    r /= p;
    if (notlas) goto L225;
    //     .......... row modification ..........
    for (j = k; j <= n;++ j) {
      p = h(k, j) + q * h(k + 1, j);
      h(k, j) -= p * x;
      h(k + 1, j) -= p * y;
    }
    
    // Computing MIN
    j = (en < k + 3) ? en : k + 3;
    //     .......... column modification ..........
    i__2 = j;
    for (i = 1; i <= i__2;++ i) {
      p = x * h(i, k) + y * h(i, k + 1);
      h(i, k) -= p;
      h(i, (k + 1)) -= p * q;
    }
    goto L255;
    L225 :
    //     .......... row modification ..........
    i__2 = n;
    for (j = k; j <= i__2;++ j) {
      p = h(k, j) + q * h(k + 1, j) + r * h(k + 2, j);
      h(k, j) -= p * x;
      h(k + 1, j) -= p * y;
      h(k + 2, j) -= p * zz;
    }
    
    // Computing MIN
    j = (en < k + 3) ? en : k + 3;
    //     .......... column modification ..........
    for (i = 1; i <= j;++ i) {
      p = x * h(i, k) + y * h(i,(k + 1)) + zz * h(i, (k + 2));
      h(i, k) -= p;
      h(i, (k + 1)) -= p * q;
      h(i, (k + 2)) -= p * r;
      // L240:
    }
    L255 :
    
    L260 :
    ;
  }
  
  goto L70;
  //     .......... one root found ..........
  L270 :
  wr(en) = x + t;
  wi(en) = 0.;
  en = na;
  goto L60;
  //     .......... two roots found ..........
  L280 :
  p = (y - x) / 2.;
  q = p * p + w;
  zz = sqrt(fabs(q));
  x += t;
  if (q < 0.) goto L320;
  //     .......... real pair ..........
  zz = p + d_sign(zz, p);
  wr(na) = x + zz;
  wr(en) = wr(na);
  if (zz != 0.) {
    wr(en) = x - w / zz;
  }
  wi(na) = 0.;
  wi(en) = 0.;
  goto L330;
  //     .......... complex pair ..........
  L320 :
  wr(na) = x + p;
  wr(en) = x + p;
  wi(na) = zz;
  wi(en) = -zz;
  L330 :
  en = enm2;
  goto L60;
  //     .......... set error -- all eigenvalues have not
  //                converged after 30*n iterations ..........
  L1000 :
  ierr = en;
  L1001 :
  return 0;
} // hqr_

// Subroutine
int rg::hqr2_(int n, int *low, int *igh, Matrix &h, Matrix &wr, Matrix &wi,
  Matrix &z, int &ierr)
{
  // System generated locals
  FLOAT d__1, d__2, d__3, d__4;
  
  // Local variables
  FLOAT norm;
  int i, j, k, l, m;
  FLOAT p, q, r, s, t, w, x, y;
  int na, ii, en, jj;
  FLOAT ra, sa;
  int ll, mm, nn;
  FLOAT vi, vr, zz;
  bboolean notlas;
  int mp2, itn, its, enm2;
  FLOAT tst1, tst2;
  FLOAT xxxx, yyyy;
  
  
  //     this subroutine is a translation of the algol procedure hqr2,
  //     num. math. 16, 181-204(1970) by peters and wilkinson.
  //     handbook for auto. comp., vol.ii-linear algebra, 372-395(1971).
  
  //     this subroutine finds the eigenvalues and eigenvectors
  //     of a real upper hessenberg matrix by the qr method.  the
  //     eigenvectors of a real general matrix can also be found
  //     if  elmhes  and  eltran  or  orthes  and  ortran  have
  //     been used to reduce this general matrix to hessenberg form
  //     and to accumulate the similarity transformations.
  
  //     on input
  
  //        nm must be set to the row dimension of two-dimensional
  //          array parameters as declared in the calling program
  //          dimension statement.
  
  //        n is the order of the matrix.
  
  //        low and igh are ints determined by the balancing
  //          subroutine  balanc.  if  balanc  has not been used,
  //          set low=1, igh=n.
  
  //        h contains the upper hessenberg matrix.
  
  //        z contains the transformation matrix produced by  eltran
  //          after the reduction by  elmhes, or by  ortran  after the
  //          reduction by  orthes, if performed.  if the eigenvectors
  //          of the hessenberg matrix are desired, z must contain the
  //          identity matrix.
  
  //     on output
  
  //        h has been destroyed.
  
  //        wr and wi contain the real and imaginary parts,
  //          respectively, of the eigenvalues.  the eigenvalues
  //          are unordered except that complex conjugate pairs
  //          of values appear consecutively with the eigenvalue
  //          having the positive imaginary part first.  if an
  //          error exit is made, the eigenvalues should be correct
  //          for indices ierr+1,...,n.
  
  //        z contains the real and imaginary parts of the eigenvectors.
  //          if the i-th eigenvalue is real, the i-th column of z
  //          contains its eigenvector.  if the i-th eigenvalue is complex
  
  //          with positive imaginary part, the i-th and (i+1)-th
  //          columns of z contain the real and imaginary parts of its
  //          eigenvector.  the eigenvectors are unnormalized.  if an
  //          error exit is made, none of the eigenvectors has been found.
  
  
  //        ierr is set to
  //          zero       for normal return,
  //          j          if the limit of 30*n iterations is exhausted
  //                     while the j-th eigenvalue is being sought.
  
  //     calls cdiv for complex division.
  
  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory
  
  
  //     this version dated august 1983.
  
  //     ------------------------------------------------------------------
  
  
  // Parameter adjustments
  
  
  // Function Body
  ierr = 0;
  norm = 0.;
  k = 1;
  //     .......... store roots isolated by balanc
  //                and compute matrix norm ..........
  for (i = 1; i <= n;++ i) {
    for (j = k; j <= n;++ j) norm += fabs(h(i, j));
    k = i;
    if (i < *low || i > *igh) {
      wr(i) = h(i, i);
      wi(i) = 0.;
    }
  }
  en = *igh;
  t = 0.;
  itn = n * 30;
  //     .......... search for next eigenvalues ..........
  L60 :
  if (en < *low) goto L340;
  its = 0;
  na = en - 1;
  enm2 = na - 1;
  //     .......... look for single small sub-diagonal element
  //                for l=en step -1 until low do -- ..........
  L70 :
  for (ll = *low; ll <= en;++ ll) {
    l = en + *low - ll;
    if (l == *low) goto L100;
    s = fabs(h(l - 1, l - 1)) + fabs(h(l, l));
    if (s == 0.) s = norm;
    tst1 = s;
    tst2 = tst1 + fabs(h(l, l - 1));
    if (tst2 == tst1) goto L100;
  }
  //     .......... form shift ..........
  L100 :
  x = h(en, en);
  if (l == en) goto L270;
  y = h(na, na);
  w = h(en, na) * h(na, en);
  if (l == na) goto L280;
  if (itn == 0) goto L1000;
  if (its != 10 && its != 20) goto L130;
  //     .......... form exceptional shift ..........
  t += x;
  
  for (i = *low; i <= en;++ i) h(i, i) -= x;
  
  s = fabs(h(en, na)) + fabs(h(na, enm2));
  x = s *.75;
  y = x;
  w = s * (- 0.4375) * s;
  L130 :
  ++ its;
  -- itn;
  //     .......... look for two consecutive small
  //                sub-diagonal elements.
  //                for m=en-2 step -1 until l do -- ..........
  for (mm = l; mm <= enm2;++ mm) {
    m = enm2 + l - mm;
    zz = h(m, m);
    r = x - zz;
    s = y - zz;
    p = (r * s - w) / h(m + 1, m) + h(m, m + 1);
    q = h(m + 1, m + 1) - zz - r - s;
    r = h(m + 2, m + 1);
    s = fabs(p) + fabs(q) + fabs(r);
    p /= s;
    q /= s;
    r /= s;
    if (m == l) goto L150;
    
    tst1 = fabs(p) * (fabs(h(m - 1, m - 1)) + fabs(zz) + fabs(h(m + 1, m + 1))
        )
      ;
      
    tst2 = tst1 + (fabs(h(m, m - 1)) * (fabs(q) + fabs(r)));
    if (tst2 == tst1) goto L150;
  }
  
  L150 :
  mp2 = m + 2;
  for (i = mp2; i <= en;++ i) {
    h(i, i - 2) = 0.;
    if (i != mp2) h(i, i - 3) = 0.;
  }
  //     .......... double qr step involving rows l to en and
  //                columns m to en ..........
  //  do 260 k=m,na
  for (k = m; k <= na;++ k) {
    notlas = (k != na) ? TTRUE : FFALSE;
    if (k == m) goto L170;
    p = h(k, k - 1);
    q = h(k + 1, k - 1);
    r = 0.;
    if (notlas) r = h(k + 2, k - 1);
    x = fabs(p) + fabs(q) + fabs(r);
    if (x == 0.) goto L260;
    p /= x;
    q /= x;
    r /= x;
    L170 :
    d__1 = sqrt(p * p + q * q + r * r);
    s = d_sign(d__1, p);
    if (k == m) goto L180;
    h(k, k - 1) = -s * x;
    goto L190;
    L180 :
    if (l != m) h(k, k - 1) = -h(k, k - 1);
    L190 :
    p += s;
    x = p / s;
    y = q / s;
    zz = r / s;
    q /= p;
    r /= p;
    if (notlas) goto L225;
    //    .......... row modification ..........
    //    section for when k==na
    for (j = k; j <= n;++ j) {
      p = h(k, j) + q * h(k + 1, j);
      h(k, j) -= p * x;
      h(k + 1, j) -= p * y;
    }
    
    // Computing MIN
    j = (en < k + 3) ? en : k + 3;
    //     .......... column modification ..........
    for (i = 1; i <= j;++ i) {
      p = x * h(i, k) + y * h(i, k + 1);
      h(i, k) -= p;
      h(i, (k + 1)) -= p * q;
    }
    //     .......... accumulate transformations ..........
    for (i = *low; i <= *igh;++ i) {
      p = x * z(i, k) + y * z(i, k + 1);
      z(i, k) -= p;
      z(i, k + 1) -= p * q;
    }
    goto L255;
    L225 :
    //     .......... row modification ..........
    //    section for when k < na
    for (j = k; j <= n;++ j) {
      p = h(k, j) + q * h(k + 1, j) + r * h(k + 2, j);
      h(k, j) -= p * x;
      h(k + 1, j) -= p * y;
      h(k + 2, j) -= p * zz;
    }
    
    // Computing MIN
    j = (en < k + 3) ? en : k + 3;
    //     .......... column modification ..........
    for (i = 1; i <= j;++ i) {
      p = x * h(i, k) + y * h(i, (k + 1)) + zz * h(i, (k + 2));
      h(i, k) -= p;
      h(i, (k + 1)) -= p * q;
      h(i, (k + 2)) -= p * r;
    }
    //     .......... accumulate transformations ..........
    for (i = *low; i <= *igh;++ i) {
      p = x * z(i, k) + y * z(i, (k + 1)) + zz * z(i, (k + 2));
      z(i, k) -= p;
      z(i, (k + 1)) -= p * q;
      z(i, (k + 2)) -= p * r;
    }
    L255 :
    ;
    L260 :
    ;
  }
  
  goto L70;
  //     .......... one root found ..........
  L270 :
  h(en, en) = x + t;
  wr(en) = h(en, en);
  wi(en) = 0.;
  en = na;
  goto L60;
  //     .......... two roots found ..........
  L280 :
  p = (y - x) / 2.;
  q = p * p + w;
  zz = sqrt(fabs(q));
  h(en, en) = x + t;
  x = h(en, en);
  h(na, na) = y + t;
  if (q < 0.) goto L320;
  //     .......... real pair ..........
  zz = p + d_sign(zz, p);
  wr(na) = x + zz;
  wr(en) = wr(na);
  if (zz != 0.) wr(en) = x - w / zz;
  wi(na) = 0.;
  wi(en) = 0.;
  x = h(en, na);
  s = fabs(x) + fabs(zz);
  p = x / s;
  q = zz / s;
  r = sqrt(p * p + q * q);
  p /= r;
  q /= r;
  //     .......... row modification ..........
  for (j = na; j <= n;++ j) {
    zz = h(na, j);
    h(na, j) = q * zz + p * h(en, j);
    h(en, j) = q * h(en, j) - p * zz;
  }
  //     .......... column modification ..........
  for (i = 1; i <= en;++ i) {
    zz = h(i, na);
    h(i, na) = q * zz + p * h(i, en);
    h(i, en) = q * h(i, en) - p * zz;
  }
  //     .......... accumulate transformations ..........
  for (i = *low; i <= *igh;++ i) {
    zz = z(i, na);
    z(i, na) = q * zz + p * z(i, en);
    z(i, en) = q * z(i, en) - p * zz;
  }
  
  goto L330;
  //     .......... complex pair ..........
  L320 :
  wr(na) = x + p;
  wr(en) = x + p;
  wi(na) = zz;
  wi(en) = -zz;
  L330 :
  en = enm2;
  goto L60;
  //     .......... all roots found.  backsubstitute to find
  //                vectors of upper triangular form ..........
  L340 :
  if (norm == 0.) goto L1001;
  //     .......... for en=n step -1 until 1 do -- ..........
  //  do 800 nn=1,n
  for (nn = 1; nn <= n;++ nn) {
    en = n + 1 - nn;
    p = wr(en);
    q = wi(en);
    na = en - 1;
    // if (q) 710, 600, 800
    if (q < 0.) {
      goto L710;
    } else if (q == 0) {
      goto L600;
    } else {
      goto L800;
    }
    //     .......... real vector ..........
    L600 :
    m = en;
    h(en, en) = 1.;
    if (na == 0) goto L800;
    //     .......... for i=en-1 step -1 until 1 do -- ..........
    // d0 700 ii=1,na
    for (ii = 1; ii <= na;++ ii) {
      i = en - ii;
      w = h(i, i) - p;
      r = 0.;
      for (j = m; j <= en;++ j) r += h(i, j) * h(j, en);
      if (wi(i) >= 0.) goto L630;
      zz = w;
      s = r;
      goto L700;
      L630 :
      m = i;
      if (wi(i) != 0.) goto L640;
      t = w;
      if (t != 0.) goto L635;
      tst1 = norm;
      t = tst1;
      L632 :
      t *=.01;
      tst2 = norm + t;
      if (tst2 > tst1) goto L632;
      L635 :
      h(i, en) = -r / t;
      goto L680;
      //     .......... solve real equations ..........
      L640 :
      x = h(i, i + 1);
      y = h(i + 1, i);
      q = (wr(i) - p) * (wr(i) - p) + wi(i) * wi(i);
      t = (x * s - zz * r) / q;
      h(i, en) = t;
      if (fabs(x) <= fabs(zz)) goto L650;
      h(i + 1, en) = (- r - w * t) / x;
      goto L680;
      L650 :
      h(i + 1, en) = (- s - y * t) / zz;
      
      //     .......... overflow control ..........
      L680 :
      t = fabs(h(i, en));
      if (t == 0.) goto L700;
      tst1 = t;
      tst2 = tst1 + 1. / tst1;
      if (tst2 > tst1) goto L700;
      for (j = i; j <= en;++ j) h(j, en) /= t;
      L700 :
      ;
    }
    //     .......... end real vector ..........
    goto L800;
    //     .......... complex vector ..........
    L710 :
    m = na;
    //     .......... last vector component chosen imaginary so that
    //                eigenvector matrix is triangular ..........
    if (fabs(h(en, na)) <= fabs(h(na, en))) goto L720;
    h(na, na) = q / h(en, na);
    h(na, en) = -(h(en, en) - p) / h(en, na);
    goto L730;
    L720 :
    d__2 = h(na, na) - p;
    cdiv_(0.0, - h(na, en), d__2, q, xxxx, yyyy);
    h(na, na) = xxxx;
    h(na, en) = yyyy;
    L730 :
    h(en, na) = 0.;
    h(en, en) = 1.;
    enm2 = na - 1;
    if (enm2 == 0) goto L800;
    //     .......... for i=en-2 step -1 until 1 do -- ..........
    for (ii = 1; ii <= enm2;++ ii) {
      i = na - ii;
      w = h(i, i) - p;
      ra = 0.;
      sa = 0.;
      
      for (j = m; j <= en;++ j) {
        ra += h(i, j) * h(j, na);
        sa += h(i, j) * h(j, en);
      }
      
      if (wi(i) >= 0.) goto L770;
      zz = w;
      r = ra;
      s = sa;
      goto L795;
      L770 :
      m = i;
      if (wi(i) != 0.) goto L780;
      cdiv_(- ra, - sa, w, q, xxxx, yyyy);
      h(i, na) = xxxx;
      h(i, en) = yyyy;
      goto L790;
      //     .......... solve complex equations ..........
      L780 :
      x = h(i, i + 1);
      y = h(i + 1, i);
      vr = (wr(i) - p) *(wr(i) - p) + wi(i) *wi(i) - q*q;
      vi = (wr(i) - p) * 2.0 * q;
      if (vr != 0. || vi != 0.) goto L784;
      tst1 = norm * (fabs(w) + fabs(q) + fabs(x) + fabs(y) + fabs(zz));
      vr = tst1;
      L783 :
      vr *=.01;
      tst2 = tst1 + vr;
      if (tst2 > tst1) goto L783;
      L784 :
      d__1 = x * r - zz * ra + q * sa;
      d__2 = x * s - zz * sa - q * ra;
      cdiv_(d__1, d__2, vr, vi, xxxx, yyyy);
      h(i, na) = xxxx;
      h(i, en) = yyyy;
      if (fabs(x) <= fabs(zz) + fabs(q)) goto L785;
      h(i + 1, na) = (- ra - w * h(i, na) + q * h(i, en)) / x;
      h(i + 1, en) = (- sa - w * h(i, en) - q * h(i, na)) / x;
      goto L790;
      L785 :
      d__1 = -r - y * h(i, na);
      d__2 = -s - y * h(i, en);
      cdiv_(d__1, d__2, zz, q, xxxx, yyyy);
      h(i + 1, na) = xxxx;
      h(i + 1, en) = yyyy;
      //     .......... overflow control ..........
      L790 :
      // Computing MAX
      d__3 = fabs(h(i, na));
      d__4 = fabs(h(i, en));
      t = (d__3 > d__4) ? d__3 : d__4; //max(d__3,d__4);
      if (t == 0.) goto L795;
      tst1 = t;
      tst2 = tst1 + 1. / tst1;
      if (tst2 > tst1) goto L795;
      for (j = i; j <= en;++ j) {
        h(j, na) /= t;
        h(j, en) /= t;
      }
      L795 :
      ;
    }
    //     .......... end complex vector ..........
    L800 :
    ;
  }
  //     .......... end back substitution.
  //                vectors of isolated roots ..........
  for (i = 1; i <= n;++ i)
    if (i < *low || i > *igh)
      for (j = i; j <= n;++ j) z(i, j) = h(i, j);
    
    //     .......... multiply by transformation matrix to give
    //                vectors of original full matrix.
    //                for j=n step -1 until low do -- ..........
    for (jj = *low; jj <= n;++ jj) {
      j = n + *low - jj;
      m = (j < *igh) ? j : *igh; //min(j,*igh);
      for (i = *low; i <= *igh;++ i) {
        zz = 0.;
        for (k = *low; k <= m;++ k) zz += z(i, k) * h(k, j);
        z(i, j) = zz;
      }
    }
    
    goto L1001;
    //     .......... set error -- all eigenvalues have not
    //                converged after 30n iterations ..........
    L1000 :
    ierr = en;
    L1001 :
    return 0;
} // hqr2_

// Subroutine

// Subroutine

rg::rg(Matrix &aa, Matrix &wr, Matrix &wi,
  int matz, Matrix &z, int &ierr)
{

  // Local variables
  int is1 = 0, is2 = 0;
  int n = aa.r;
  Matrix a = aa; // don't trash aa;
  wr = wi = Fill(n,1,0.0);
  Matrix fv1 = Fill(n, 1, 0);
  int *iv1 = new int[n];
  if (!iv1) aa.Nrerror("rg_: workspace allocation failure");

  //     this subroutine calls the recommended sequence of
  //     subroutines from the eigensystem subroutine package (eispack)
  //     to find the eigenvalues and eigenvectors (if desired)
  //     of a real general matrix.

  //     on input

  //        n  is the order of the matrix  a.

  //        a  contains the real general matrix.

  //        matz  is an int variable set equal to zero if
  //        only eigenvalues are desired.  otherwise it is set to
  //        any non-zero int for both eigenvalues and eigenvectors.

  //     on output

  //        wr  and  wi  contain the real and imaginary parts,
  //        respectively, of the eigenvalues.  complex conjugate
  //        pairs of eigenvalues appear consecutively with the
  //        eigenvalue having the positive imaginary part first.

  //        z  contains the real and imaginary parts of the eigenvectors
  //        if matz is not zero.  if the j-th eigenvalue is real, the
  //        j-th column of  z  contains its eigenvector.  if the j-th
  //        eigenvalue is complex with positive imaginary part, the
  //        j-th and (j+1)-th columns of  z  contain the real and
  //        imaginary parts of its eigenvector.  the conjugate of this
  //        vector is the eigenvector for the conjugate eigenvalue.

  //        ierr  is an int output variable set equal to an error
  //           completion cde described in the documentat-ion for hqr
  //           and hqr2.  the normal completion code is zero.

  //        iv1  and  fv1  are temporary storage arrays.

  //     questions and comments should be directed to burton s. garbow,
  //     mathematics and computer science div, argonne national laboratory


  //     this version dated august 1983.
  //     translation 1 by f2c, with cleanup by Mark Von Tress (5-95)
  //     ------------------------------------------------------------------


  // Parameter adjustments
  -- iv1;

  // Function Body
  ierr = n * 10;
  balanc_(n, a, &is1, &is2, fv1);
  elmhes_(n, &is1, &is2, a, &iv1[1]);
  if (!matz) //     .......... find eigenvalues only ..........
    hqr_(n, &is1, &is2, a, wr, wi, ierr);
  else {
    //     .......... find both eigenvalues and eigenvectors ..........
    z = Fill(n,n,0.0);
    eltran_(n, &is1, &is2, a, &iv1[1], z);
    hqr2_(n, &is1, &is2, a, wr, wi, z, ierr);
    if (!ierr) balbak_(n, &is1, &is2, fv1, n, z);
  }
  iv1++;
  delete[] iv1;
} // rg_


// sweep out k1 to k2 of a complex matrix (a,b)
void rg::Sweep2mats(int k1, int k2, Matrix& a, Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Sweep2mats: a is garbage");
        b.Garbage("Sweep2mats: b is garbage");
#endif

        FLOAT eps = 1.0e-8, c=0,d=0,e=0,f=0;
        FLOAT one = 1.0, zero = 0.0;
        FLOAT dr=0, di=0;
        int i, j, k, n, it;

        if (a.c != a.r)
          a.Nrerror("Sweep2mats: matrix a not square");
        if (b.c != b.r)
          b.Nrerror("Sweep2mats: matrix b not square");
        if (a.c != b.c || a.r != b.r )
          a.Nrerror("Sweep2mats: matrices a and b are not the same dimension");

        n = a.r;
        if (k2 < k1) { k = k1; k1 = k2; k2 = k; }

        for (k = k1; k <= k2; k++) {
          if ( fabs( a(k, k)+b(k,k) ) < eps)
                 for (it = 1; it <= n; it++)
                         a(it, k) = a(k, it) =
                         b(it, k) = b(k, it) = 0.0;
          else {
                 cdiv_(one,zero,a(k,k),b(k,k),dr,di);
                 a(k, k) = dr;
                 b(k, k) = di;
                 for (i = 1; i <= n; i++) {
                    if (i != k){
                        //a(i, k) *=  -d;
                        cmult_(a(i,k),b(i,k), -dr, -di, c,d);
                        a(i, k) = c;
                        b(i, k) = d;
                    }
                 }
                 for (j = 1; j <= n; j++) {
                     if (j != k){
                        // a(k, j) *= d;
                        cmult_(a(k,j), b(k,j), dr, di, c,d);
                        a(k, j) = c;
                        b(k, j) = d;
                     }
                 }
                 for (i = 1; i <= n; i++) {
                        if (i != k) {
                                for (j = 1; j <= n; j++) {
                                  if (j != k){
                                     //a(i, j) += a(i, k) *a(k, j) / d;
                                     cmult_(a(i,k),b(i,k),a(k,j),b(k,j),c,d);
                                     cdiv_(c,d,dr,di,e,f);
                                     a(i,j) += e;
                                     b(i,j) += f;
                                  }
                                } // end for j
                        } // end for i != k
                 } // end for i
          } // end else
        } // end for k
}   /*sweep*/

///// inversion of a complex matrix (a,b) to produce (ai,bi)
void rg::Inv2mats(Matrix &a, Matrix &b, Matrix &ai, Matrix &bi)
{
#ifndef NO_CHECKING
        a.Garbage("Inv2mats: a is garbage");
        b.Garbage("Inv2mats: b is garbage");
#endif
        if (a.c != a.r)
          a.Nrerror("Inv2mats: matrix a not square");
        if (b.c != b.r)
          b.Nrerror("Inv2mats: matrix b not square");
        if (a.c != b.c || a.r != b.r )
          a.Nrerror("Inv2mats: matrices a and b are not the same dimension");

        ai = a;
        bi = b;
        Sweep2mats(1, b.r, ai, bi);
}
void rg::Unpack(Matrix &wr, Matrix &wi, Matrix &z, Matrix &Pr, Matrix &Pi,
                Matrix &Pinvr, Matrix &Pinvi)
{  //section 2.3.2 of the EISPACK Manual
   // unpack the eigenvectors if there are complex eigenvalues
   Pr=Pi=Pinvr=Pinvi=Fill(z.r,z.c,0.0);
   int i, j;
   for( j=1; j<=z.c; j++){
      if( fabs( wi(j) ) <1.0e-15)
         for( i=1; i<=z.r; i++){
            Pr(i,j) = z(i,j);
            Pi(i,j) = 0.0;
         }
      if( wi(j) >= 1.0e-15 )
         for( i=1; i<=z.r; i++){
           Pr(i,j) = z(i,j);
           Pi(i,j) = z(i,j+1);
         }
      if( wi(j) <= -1.0e-15 )
         for( i=1; i<=z.r; i++){
           Pr(i,j) =  Pr(i,j-1);
           Pi(i,j) = -Pi(i,j-1);
         }
   }
   Inv2mats(Pr,Pi,Pinvr,Pinvi);
}


// #endif for inclusion when not using complex matrices
#endif
