// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.

////////////////// Uniform random number generator objects

#include "matv.h"
#include "dist.h"
#include <time.h>
#include <limits.h>

#ifndef _URANDVEC_
#define _URANDVEC_

class URand
{ // c++ class to return a U(0,1) number
  // period of 2^31
  public:
         // use clock to seed the rng
         URand(){
                seed = (unsigned) time(NULL);
                frand();
         }
         // use an explicit seed
         URand(unsigned s): seed(s){ frand();}

         // generate a uniform random number generator,
         // and reset the seed
         double frand( unsigned s ){
                seed=s;
                return frand();
         }
         //generate a uniform random number
         double frand();
         //return the value of the seed
         long Seed() { return seed; }

         // operator versions of frand()
         double operator()(){return frand();}
         double operator()(unsigned s){ return frand(s);}

  private:
         long seed, high, low, test;
         double t;
};

/////////////////////////////////////////////////
class URandVec
{ // work on a jumbled array of URands
  // this should have a very long period since it
  // has n independent random number streams,
  // each of a long period. And the stream pointer in
  // each tends to move along uniformly.
  // Shuffle array of random number streams by
  // Algorithm B Knuth - Seminumerical Algorithms-
  // page 32
  //
  // The default period of this is
  // 32*(2*31-2)  = (2**36-2**6)
  // Also note that n should be a small power of 2, say 4,8,16,32,64,
  // so that each stream has nearly equal probability of being
  // selected since (n-1 or 0) == RAND_MAX mod n.  

  public:
         URandVec(){ Makeit(32,0,1); }
         URandVec(unsigned baseseed){ Makeit(32,baseseed,1); }
         URandVec(int nn, unsigned baseseed) { Makeit(nn,baseseed,1); }
         ~URandVec(){delete[] uvect; }
         // take a U(0,1) from stream nextj
         double URandV(){
                u = uvect[ nextj ].frand();
                GetNextJ();
                return u;
         }
         // take a U(0,1) from stream nextj and
         // re-seed uvect[ nextj ]
         double URandV( unsigned s ){
                u = uvect[ nextj ].frand(s);
                GetNextJ();
                return u;
         }
         // operator forms of URandV()
         double operator ()(){ return URandV(); }
         double operator ()(unsigned s){
                return URandV(s);
         }
         void ReSeed( unsigned s) { Makeit( n, s, 0); }

  private:
         URand *uvect;  // vector of rng streams
         int n;         // number of elements in uvect
         int nextj;     // next element to use
         double u;      // Uniform(0,1) to return
         // Algorithm B shuffle method
         void GetNextJ(){ nextj = (int)(n * u); }
         void Makeit(int nn, unsigned baseseed, int InConstructor );
};

/////////////////// Derived Classes of RNG Using Probability Integral Transform
/////////////////// These are cheap to write, but slow.
/////////////////// note that the inverse distribution function does
/////////////////// the range checking if it comes from dist.cpp.

/////////////////// normal random number
class NormalRV : virtual public URandVec
{
  public:
         NormalRV(){ mean=0; stddev=1; }
         NormalRV(double m, double ss): mean(m), stddev(ss) {}
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NormalRV &r);
         double mean,stddev;
  private:
         double B(){
                //a u(0,1) has mean 0.5, and variance 1/12, so sum
                // of 12 iid u(0,1) has mean=6 and variance = 12*(1/12) = 1
                // so sum of 12 u(0,1) - 6 has approximate N(0,1)
                double sum=0;
                for( int i=1; i<=12; i++) sum += URandV();
                return mean+stddev*(sum-6.0);
         }
};

////////////////////////// gamma variate
class NCGammaRV : virtual public URandVec
{
  public:
         NCGammaRV(){ alpha = 0.5; beta=2.0; ncp=0; } // chisquare(1)
         NCGammaRV(double a, double b): alpha(a), beta(b), ncp(0) {}
         NCGammaRV(double a, double b, double n): alpha(a), beta(b), ncp(n) {}
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NCGammaRV &r);
         double alpha, beta, ncp;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                return ncgammai(u,alpha,beta,ncp);;
         }
};

//////////////////////// non-central chi-square
class NCChiRV : virtual public URandVec
{
  public:
         NCChiRV(){ df = 1; ncp=0; } // chisquare(1)
         NCChiRV(double a) : df(a), ncp(0) {}
         NCChiRV(double a, double n): df(a), ncp(n) {}
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NCChiRV &r);
         double df, ncp;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                return ncchii(u,df,ncp);
         }
};

////////////////////////// noncentral F variate
class NCFRV : virtual public URandVec
{
  public:
         NCFRV(){ df1 = 1; df2 = 1; ncp=0; }
         NCFRV(double d1, double d2): df1(d1), df2(d2), ncp(0) {}
         NCFRV(double d1, double d2, double n): df1(d1), df2(d2), ncp(n) {}
         double operator ()(){ return B();}
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NCFRV &r);
         double df1, df2, ncp;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                return ncfinv(URandV(),df1,df2,ncp);
         }
};

////////////////////// non-central beta variate
/// use relationship of F to a Beta.
class NCBetaRV : virtual public URandVec
{
  public:
         NCBetaRV(){ alpha = 1.0; beta= 1.0; ncp=0; } //Uniform
         NCBetaRV(double a, double b): alpha(a), beta(b), ncp(0) {}
         NCBetaRV(double a, double b, double n): alpha(a), beta(b), ncp(n) {}
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NCBetaRV &r);
         double alpha, beta, ncp;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                double f = ncfinv(u, 2*alpha, 2*beta, 2*ncp);
                return (alpha*f)/(alpha*f+beta);
         }
};

//////////////////////// non-central T
class NCtRV : virtual public URandVec
{
  public:
         NCtRV(){ df = 1; ncp=0; } // Cauchy
         NCtRV(double a) : df(a), ncp(0) {}
         NCtRV(double a, double n): df(a), ncp(n) {}
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed); return B();}
         friend ostream & operator << (ostream &s, NCtRV &r);
         double df, ncp;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                return ncti(u,df,ncp);
         }
};

//////////////////////exponential
class ExpRV : virtual public URandVec
{
  public:
         ExpRV(){ lambda = 1; }
         ExpRV(double a) : lambda(a){
                 if(lambda <= 0) nrerror("ExpRV: Lambda <= 0");
         }
         double operator ()(){ return B(); }
         double operator ()(unsigned seed){ReSeed(seed);  return B(); }
         friend ostream & operator << (ostream &s, ExpRV &r);
         double lambda;
  private:
         double B(){
                double u=URandV();
                u = (u<0.0001) ? 0.0001 : u;
                u = (u>0.9999) ? 0.9999 : u;
                return -log(1.0-u)/lambda;
         }
};

////////////////////// Multivariate Normal Random Vector
class MvnRV : virtual public NormalRV
{
  public:
         // constructors
         MvnRV(){ mu=Fill(2,1,0); v=half=Ident(2); }
         MvnRV(Matrix &a) : mu(a) {v=half=Ident(a.r); }
         MvnRV(Matrix &a, Matrix &b) : mu(a), v(b){
                if (b.r != b.c ) b.Nrerror("MvnRV: V is not square");
                half=Tran(Cholesky(v));
         }

         // random number functions
         Matrix operator ()(){ return B(); }
         Matrix operator ()(unsigned seed){ReSeed(seed);  return B(); }

         // output the matrices
         friend ostream & operator << (ostream &s, MvnRV &r);
         Matrix Mu() { return mu;}
         Matrix V()  { return v; }
         Matrix Half(){ return half; }

         // reset values of mu or v
         void ResetMu( Matrix &m ){ mu = m; };
         void ResetV( Matrix &vv){
                v = vv;
                if (v.r != v.c ) v.Nrerror("MvnRV: V is not square");
                half=Tran(Cholesky(v));
         }
         void ResetMuV( Matrix &mu, Matrix &vv ){ ResetMu(mu); ResetV(vv); }

  private:
         Matrix mu, v, half;
         Matrix B(){
                if (mu.r != v.r ) mu.Nrerror("MvnRV: mu.r != V.r");
                Matrix temp = Fill(mu.r,1,0);
                for( int i=1; i<=mu.r; i++) temp(i) = NormalRV::operator()();
                temp = mu + half*temp;
                temp.release();
                return temp;
         }
};


#endif
